module CGIKit

# SessionStore is a class for saving session into storage.
class SessionStore
  def initialize( application )
    @application = application
  end

  def checkin( context )
    save(context)
  end

  def checkout( session_id, request )
    if session_id.nil? then
      return nil
    end
    restore(session_id, request)
  end

  # abstract
  def save( context ); end

  # abstract.
  def restore( session_id, request ); end

  # abstract.
  def remove( session_id ); end

  def sweep_sessions; end

end


# A FileSessionStore object saves a session to a file with marshaling.
class FileSessionStore < SessionStore
  TMPDIR = 'session'

  def initialize( application )
    super
    @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
  end

  def save( context )
    unless FileTest.directory? @tmpdir
      require 'ftools'
      File.makedirs @tmpdir
    end
    FileLock.exclusive_lock(tmpfile(context.session.session_id)) do |file|
      Marshal.dump(context.session, file)
    end
  end

  def restore( session_id, request )
    session = nil
    if exist?(session_id) then
      FileLock.shared_lock(tmpfile(session_id)) do | file |
        session = Marshal.load file
      end
      session.session_store = self
      session.application = @application
    end
    session
  end

  def tmpfile( session_id )
    File.join(@tmpdir, session_id).untaint
  end

  def exist?( session_id )
    FileTest.exist?(tmpfile(session_id))
  end

  def remove( session_id )
    if FileTest.exist?(tmpfile(session_id))
      File.delete(tmpfile(session_id))
    end
  end

  def sweep_sessions
    successed = 0
    failed = 0
    Dir.foreach(@tmpdir) do |file|
      begin
        unless /\A\./ === file
          path = File.join(@tmpdir, file)
          session = nil
          FileLock.shared_lock(path) do |f|
            session = Marshal.load(f)
          end
          if session.timeout? then
            begin
              File.delete(path)
              successed += 1
            rescue
              failed += 1
            end
          end
        end
      rescue
      end
    end
    [successed, failed]
  end

end


class MemorySessionStore < SessionStore
  def initialize( application )
    super
    @caches = {}
  end

  def save( context )
    @caches[context.session.session_id] = context.session
  end

  def restore( session_id, request )
    if session = @caches[session_id] then
      session.session_store = self
      session.application = @application
    end
    session
  end

  def remove( session_id )
    @caches.delete(session_id)
  end

  def exist?( session_id )
    @caches.key?(session_id)
  end

  def sweep_sessions
    before = @caches.size
    @caches.delete_if do |id, session|
      session.timeout?
    end
    [before - @caches.size, 0]
  end

end

end
