module CGIKit

# ByteData objects manage bytes.
class ByteData

  # Path of a file saving the bytes.
  attr_accessor :path

  # Content type of the bytes.
  attr_accessor :content_type

  class << self

    # Create an instance from specified IO object.
    # If you give this a File object, sets "path" attribute of the instance.
    def new_with_io( io, offset = nil, count = nil )
      io.pos = offset if offset
      bytes  = new io.read(count)
      if io.respond_to? 'path' then
        bytes.path = io.path
      end
      bytes
    end

    # Create an instance from specified file.
    def new_with_file( filename )
      bytes = nil
      open(filename) do |f|
        bytes      = new(f.read)
        bytes.path = f.path
        if ext = File.extname(filename) then
          ext.tr!('.', '')
          bytes.content_type = ResourceManager.mime(ext)
        end
      end
      bytes
    end

  end

  def initialize( string = nil )
    @bytes = string || ''
  end

  def tempfile?
    false
  end

  # Returns bytes with spcified length or whole length if you omit it.
  def bytes( length = nil )
    if length then
      @bytes.slice(0, length)
    else
      @bytes.to_s
    end
  end

  # Executes the block for every byte.
  def each
    @bytes.each_byte do |byte|
      yield byte
    end
  end

  # Returns true if the bytes of each objects are equal.
  def ==( bytes )
    @bytes == bytes.bytes
  end

  # Returns length of the bytes.
  def length
    @bytes.size
  end
  alias size length

  # Appends bytes to the bytes.
  def <<( bytes )
    if bytes.is_a?(ByteData) then
      @bytes << bytes.bytes
    else
      @bytes << bytes
    end
    self
  end

  # Writes the bytes to a specified file.
  def write_to_file( filename, lock = true )
    if lock then
      FileLock.exclusive_lock(filename, 'w+b') do |file|
        file.write to_s
      end
    else
      File.open(filename, 'w+b') do |file|
        file.write to_s
      end
    end      
  end

  # Returns the object as a string.
  def to_s
    @bytes.to_s
  end

  def open; end
  def close; end

end


class TempfileByteData < ByteData

  attr_accessor :tempfile

  def initialize( tempfile )
    @tempfile = tempfile
    close
  end

  def tempfile?
    true
  end

  # Returns bytes with spcified length or whole length if you omit it.
  def bytes( length = nil )
    open do
      @tempfile.read(length)
    end
  end

  # Executes the block for every byte.
  def each
    open do
      @tempfile.each_byte do |byte|
        yield byte
      end
    end
  end

  def ==( bytes )
    @tempfile == bytes.tempfile
  end

  def length
    open do
      @tempfile.size
    end
  end

  def <<( bytes )
    open do
      @tempfile.seek(0, IO::SEEK_END)
      if ByteData === bytes then
        @tempfile << bytes.bytes
      else
        @tempfile << bytes
      end
      self
    end
  end

  def to_s
    bytes
  end

  def open( &block )
    @tempfile.open if @tempfile.closed?
    if block_given? then
      @tempfile.rewind
      value = block.call
      close
      return value
    end
  end

  def close
    @tempfile.close unless @tempfile.closed?
  end

  def _dump( limit )
    Marshal.dump(bytes(), limit)
  end

  def _load( object )
    ByteData.new(Marshal.load(object))
  end

end


# ResourceManager class manages resources of an application.
class ResourceManager
  MIME = {
    'ez'      => 'application/andrew-inset',
    'hqx'     => 'application/mac-binhex40',
    'cpt'     => 'application/mac-compactpro',
    'doc'     => 'application/msword',
    'bin'     => 'application/octet-stream',
    'dms'     => 'application/octet-stream',
    'lha'     => 'application/octet-stream',
    'lzh'     => 'application/octet-stream',
    'exe'     => 'application/octet-stream',
    'class'   => 'application/octet-stream',
    'so'      => 'application/octet-stream',
    'dll'     => 'application/octet-stream',
    'oda'     => 'application/oda',
    'pdf'     => 'application/pdf',
    'ai'      => 'application/postscript',
    'eps'     => 'application/postscript',
    'ps'      => 'application/postscript',
    'smi'     => 'application/smil',
    'smil'    => 'application/smil',
    'mif'     => 'application/vnd.mif',
    'xls'     => 'application/vnd.ms-excel',
    'ppt'     => 'application/vnd.ms-powerpoint',
    'wbxml'   => 'application/vnd.wap.wbxml',
    'wmlc'    => 'application/vnd.wap.wmlc',
    'wmlsc'   => 'application/vnd.wap.wmlscriptc',
    'bcpio'   => 'application/x-bcpio',
    'vcd'     => 'application/x-cdlink',
    'pgn'     => 'application/x-chess-pgn',
    'cpio'    => 'application/x-cpio',
    'csh'     => 'application/x-csh',
    'dcr'     => 'application/x-director',
    'dir'     => 'application/x-director',
    'dxr'     => 'application/x-director',
    'dvi'     => 'application/x-dvi',
    'spl'     => 'application/x-futuresplash',
    'gtar'    => 'application/x-gtar',
    'hdf'     => 'application/x-hdf',
    'js'      => 'application/x-javascript',
    'skp'     => 'application/x-koan',
    'skd'     => 'application/x-koan',
    'skt'     => 'application/x-koan',
    'skm'     => 'application/x-koan',
    'latex'   => 'application/x-latex',
    'nc'      => 'application/x-netcdf',
    'cdf'     => 'application/x-netcdf',
    'sh'      => 'application/x-sh',
    'shar'    => 'application/x-shar',
    'swf'     => 'application/x-shockwave-flash',
    'sit'     => 'application/x-stuffit',
    'sv4cpio' => 'application/x-sv4cpio',
    'sv4crc'  => 'application/x-sv4crc',
    'tar'     => 'application/x-tar',
    'tcl'     => 'application/x-tcl',
    'tex'     => 'application/x-tex',
    'texinfo' => 'application/x-texinfo',
    'texi'    => 'application/x-texinfo',
    't'       => 'application/x-troff',
    'tr'      => 'application/x-troff',
    'roff'    => 'application/x-troff',
    'man'     => 'application/x-troff-man',
    'me'      => 'application/x-troff-me',
    'ms'      => 'application/x-troff-ms',
    'ustar'   => 'application/x-ustar',
    'src'     => 'application/x-wais-source',
    'xhtml'   => 'application/xhtml+xml',
    'xht'     => 'application/xhtml+xml',
    'zip'     => 'application/zip',
    'au'      => 'audio/basic',
    'snd'     => 'audio/basic',
    'mid'     => 'audio/midi',
    'midi'    => 'audio/midi',
    'kar'     => 'audio/midi',
    'mpga'    => 'audio/mpeg',
    'mp2'     => 'audio/mpeg',
    'mp3'     => 'audio/mpeg',
    'aif'     => 'audio/x-aiff',
    'aiff'    => 'audio/x-aiff',
    'aifc'    => 'audio/x-aiff',
    'm3u'     => 'audio/x-mpegurl',
    'ram'     => 'audio/x-pn-realaudio',
    'rm'      => 'audio/x-pn-realaudio',
    'rpm'     => 'audio/x-pn-realaudio-plugin',
    'ra'      => 'audio/x-realaudio',
    'wav'     => 'audio/x-wav',
    'pdb'     => 'chemical/x-pdb',
    'xyz'     => 'chemical/x-xyz',
    'bmp'     => 'image/bmp',
    'gif'     => 'image/gif',
    'ief'     => 'image/ief',
    'jpeg'    => 'image/jpeg',
    'jpg'     => 'image/jpeg',
    'jpe'     => 'image/jpeg',
    'png'     => 'image/png',
    'tiff'    => 'image/tiff',
    'tif'     => 'image/tiff',
    'djvu'    => 'image/vnd.djvu',
    'djv'     => 'image/vnd.djvu',
    'wbmp'    => 'image/vnd.wap.wbmp',
    'ras'     => 'image/x-cmu-raster',
    'pnm'     => 'image/x-portable-anymap',
    'pbm'     => 'image/x-portable-bitmap',
    'pgm'     => 'image/x-portable-graymap',
    'ppm'     => 'image/x-portable-pixmap',
    'rgb'     => 'image/x-rgb',
    'xbm'     => 'image/x-xbitmap',
    'xpm'     => 'image/x-xpixmap',
    'xwd'     => 'image/x-xwindowdump',
    'igs'     => 'model/iges',
    'iges'    => 'model/iges',
    'msh'     => 'model/mesh',
    'mesh'    => 'model/mesh',
    'silo'    => 'model/mesh',
    'wrl'     => 'model/vrml',
    'vrml'    => 'model/vrml',
    'css'     => 'text/css',
    'html'    => 'text/html',
    'htm'     => 'text/html',
    'asc'     => 'text/plain',
    'txt'     => 'text/plain',
    'rtx'     => 'text/richtext',
    'rtf'     => 'text/rtf',
    'sgml'    => 'text/sgml',
    'sgm'     => 'text/sgml',
    'tsv'     => 'text/tab-separated-values',
    'wml'     => 'text/vnd.wap.wml',
    'wmls'    => 'text/vnd.wap.wmlscript',
    'etx'     => 'text/x-setext',
    'xml'     => 'text/xml',
    'xsl'     => 'text/xml',
    'mpeg'    => 'video/mpeg',
    'mpg'     => 'video/mpeg',
    'mpe'     => 'video/mpeg',
    'qt'      => 'video/quicktime',
    'mov'     => 'video/quicktime',
    'mxu'     => 'video/vnd.mpegurl',
    'avi'     => 'video/x-msvideo',
    'movie'   => 'video/x-sgi-movie',
    'ice'     => 'x-conference/x-cooltalk'
  }

  TMPDIR = 'resource'
  DEFAULT_TMP_DATA_KEY_FIGURES = 16

  class << self
    def create_tmp_data_key
      md5 = Digest::MD5::new
      md5.update Time.now.to_s
      md5.update rand(0).to_s
      md5.update $$.to_s
      md5.hexdigest[0, tmp_data_key_figures]
    end

    def tmp_data_key_figures
      DEFAULT_TMP_DATA_KEY_FIGURES
    end

    def mime( extension )
      MIME[extension]
    end
  end

  def initialize( application )
    @application          = application
    @resources            = @application.resources
    @web_server_resources = @application.web_server_resources
    @document_root        = @application.document_root
    @tmpdir = File.join(@application.tmpdir, TMPDIR).untaint
  end

  # Returns the public URL for the specified resource
  # when it is under the web server resources directory.
  # Otherwise returns nil.
  def url( name, request = nil )
    url = nil
    if filepath = path(name) then
      if @web_server_resources and \
        /\A#{File.expand_path(@web_server_resources)}/ === filepath then
        url = filepath.sub(File.expand_path(@document_root), '') if @document_root
        return url
      elsif @tmpdir and /\A#{File.expand_path(@tmpdir)}/ === filepath then
        url = @application.resource_request_handler.resource_url(name, request)
      end
    end
    url
  end

  # Returns the file path of the specified resource.
  def path( name )
    resourcedir = [@tmpdir, @resources, @web_server_resources]
    resourcedir.each do |dir|
      filepath = File.join(dir, name).untaint
      absolute = File.expand_path(filepath).untaint
      if FileTest.exist?(absolute) then
        return absolute
      end
    end
    nil
  end

  # Returns a ByteData object for the specified resource.
  def bytedata( name )
    data = nil
    if filepath = path(name) then
      if filepath =~ /#{File.expand_path(@tmpdir)}/ then
        cache = nil
        FileLock.shared_lock(filepath) do |file|
          cache = Marshal.load(file)
        end
        data = ByteData.new(cache[:data])
        data.path = filepath
        data.content_type = cache[:mime]
      else
        data = ByteData.new_with_file(filepath)
        data.content_type = content_type(filepath)
      end
    end
    data
  end

  # Returns a String object for the specified resource.
  def string( name )
    bytedata(name).to_s
  end

  def write( path, content, lock = true )
    if lock then
      FileLock.exclusive_lock(path, 'wb') do |file|
        file.write(content.to_s)
      end
    else
      File.open(path, 'wb') do |file|
        file.write(content.to_s)
      end
    end      
  end

  def write_resource( name, content, lock = true )
    path = File.join(@resources, name)
    write(path, content, lock)
  end

  def write_web_server_resource( name, content, lock = true )
    path = File.join(@web_server_resources, name)
    write(path, content, lock)
  end

  # Finds the content type for extension of the specified path.
  # If the path don't have extension, returns nil.
  def content_type( path )
    # for ruby 1.6
    base = File.basename path
    type = nil
    if base =~ /\.([^\.]+)$/ then
      type = MIME[$1]
    end
    type
  end

  def set_data( data, key, mime )
    unless FileTest.directory? @tmpdir
      require 'ftools'
      File.makedirs @tmpdir
    end
    cache = {}
    cache[:data] = data
    cache[:key]  = key
    cache[:mime] = mime
    FileLock.exclusive_lock(tmpfile(key)) do |file|
      Marshal.dump(cache, file)
    end
  end

  def tmpfile( filename )
    File.join(@tmpdir, filename).untaint
  end

  def exist?( filename )
    FileTest.exist?(tmpfile(filename))
  end

  def remove_data( key )
    path = tmpfile(key)
    if FileTest.exist?(path)
      File.delete(path)
    end
  end

end

end


