module CGIKit

# An API object has information of dynamic element's or component's bindings.
# The API object can validate declaration of the element or component at runtime.
class API

  attr_accessor :element_type, :content, :bindings, :validations

  def initialize( element_type, content = false )
    @element_type = element_type
    @content      = content
    @bindings     = {}
    @validations  = []
  end


  #
  # accessing
  #

  def []( name )
    b = @bindings[name]
    if b
      return b
    end
    
    nil
  end

  def <<( object )
    case object
    when Binding
      @bindings[object.name] = object
    when Validation
      @validations << object
    end
  end


  #
  # testing
  #

  def has_binding?( name )
    if @bindings[name]
      return true
    else
      false
    end
  end


  #
  # validating
  #

  # Returns nil or array contains errors.
  def validate( associations, component_name, element_name )
    errors = []
    bindings = @bindings.values
    target = bindings + @validations
    target.each do |t|
      begin
        t.validate(associations, component_name, element_name, @element_type)
      rescue ValidationError => e
        errors << e
      end
    end

    if errors.empty? then
      nil
    else
      errors
    end
  end


  #
  # generating string of other tags
  #

  # Return a string of values of optional attributes with
  # "other" attribute string excepting the element's own attributes.
  def other( component, associations )
    optional = ''
    associations.each do |key, as|
      if (has_binding?(key) == false) and (key != Declaration::ELEMENT_KEY) and \
        (key != Declaration::OTHER_KEY) then
        value = as.value(component).to_s
        optional << " #{key}=\"#{value}\""
      end
    end

    if as = associations[Declaration::OTHER_KEY] then
      other = ' ' + as.value(component).to_s
    else
      other = ''
    end      

    optional + other
  end

  def enabled( component, associations )
    enabled = true
    associations.each do |key, as|
      if key == Declaration::ENABLED_KEY then
        enabled = as.bool(component)
      end
    end
    if enabled then
      ''
    else
      ' disabled="disabled"'
    end
  end

end


class Binding
  BOOLEAN        = 'Boolean'
  PAGE_NAMES     = 'Page Names'
  MIME_TYPES     = 'MIME Types'
  RESOURCES      = 'Resources'
  DIRECT_ACTIONS = 'Direct Actions'

  attr_accessor :name, :required, :settable

  # Value type to set.
  attr_accessor :value_set

  # Default value.
  attr_writer :default

  def initialize( name )
    @name     = name
    @required = false
    @settable = false
    @default  = nil
  end

  # Returns the default value.
  # If the value is Symbol, it is resolved with key-value coding.
  def default( component = nil )
    if (Symbol === @default) and component then
      component.value_for_keypath(@default)
    else
      @default
    end
  end

  def validate( associations, component_name, element_name, element_type )
    msg = nil
    isbind = false

    associations.each do |key, as|
      if (key == @name) 
        if (@settable == true) and (as.settable? == false) then
          msg = "'#@name' must be bound to a settable value."
        end
        isbind = true
        break
      end
    end

    if (isbind == false) and (@required == true) then
      msg = "'#@name' is a required binding."
    end

    if msg then
      raise ValidationError.new(msg, component_name, element_name, element_type)
    end
  end
end


class Validation
  attr_accessor :message, :condition

  def initialize( message, condition = nil )
    @message = message
    @condition = condition
  end

  def validate( associations, component_name, element_name, element_type )
    if @condition.eval?(associations) then
      raise ValidationError.new(@message, component_name, element_name, element_type)
    end
  end
end


class KeyErrorCondition
  BOUND      = 'bound'
  UNBOUND    = 'unbound'
  SETTABLE   = 'settable'
  UNSETTABLE = 'unsettable'

  attr_reader :key, :error

  def initialize( key, error )
    @key = key
    @error = error
  end

  def eval?( associations )
    as = associations[@key]
    if ((as and (@error == BOUND)) or
        (as.nil? and (@error == UNBOUND)) or
       (as and as.settable? and (@error == SETTABLE)) or
       (as and as.constant? and (@error == UNSETTABLE))) then
      true
    else
      false
    end
  end
end


class AndCondition
  attr_reader :conditions

  def initialize( conditions )
    @conditions = conditions
  end

  def eval?( associations )
    @conditions.each do |condition|
      unless condition.eval? associations then
        return false
      end
    end
    true
  end
end


class OrCondition
  attr_reader :conditions

  def initialize( conditions )
    @conditions = conditions
  end

  def eval?( associations )
    @conditions.each do |condition|
      if condition.eval? associations then
        return true
      end
    end
    false
  end
end


class NotCondition
  attr_reader :condition

  def initialize( condition )
    @condition = condition
  end

  def eval?( associations )
    unless @condition.eval? associations then
      true
    else
      false
    end
  end
end


class ValidationError < StandardError
  attr_accessor :errors, :element_name, :element_type, :component_name

  def initialize( message = '',
                  component_name = nil,
                  element_name = nil,
                  element_type = nil )
    super(message)
    @component_name = component_name
    @element_name = element_name
    @element_type = element_type
    @errors = []
  end

  def <<( error )
    if Array === error then
      @errors.concat(error)
    else
      @errors << error
    end
  end

end


module APIUtilities

  #
  # binding
  #

  def href_binding
    Binding.new(:href)
  end

  def name_binding( required = false )
    binding = Binding.new(:name)
    binding.required = required
    binding
  end

  def page_binding( required = false )
    binding = Binding.new(:page)
    binding.required = required
    binding
  end

  def src_binding( required = false )
    binding = Binding.new(:src)
    binding.required = required
    binding
  end

  def action_binding( required = false )
    binding = Binding.new(:action)
    binding.required = required
    binding
  end

  def list_binding( required = true )
    binding = Binding.new(:list)
    binding.required = required
    binding
  end

  def item_binding()
    binding = Binding.new(:item)
    binding.settable = true
    binding
  end

  def display_binding()
    binding = Binding.new(:display)
    binding
  end

  def value_binding( required = true, settable = false )
    binding = Binding.new(:value)
    binding.required = required
    binding.settable = settable
    binding
  end

  def enabled_binding
    binding = Binding.new(:enabled)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end

  def escape_binding
    binding = Binding.new(:escape)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end

  def checked_binding( default = false )
    binding = Binding.new(:checked)
    binding.default = default
    binding.value_set = Binding::BOOLEAN
    binding.settable = true
    binding
  end

  def selection_binding
    binding = Binding.new(:selection)
    binding.settable = true
    binding
  end

  def selections_binding
    binding = Binding.new(:selections)
    binding.settable = true
    binding
  end

  def selected_value_binding
    binding = Binding.new(:selected_value)
    binding.settable = true
    binding
  end

  def selected_values_binding
    binding = Binding.new(:selected_values)
    binding.settable = true
    binding
  end

  def string_binding
    binding = Binding.new(:string)
    binding
  end

  def validate_binding
    binding = Binding.new(:validate)
    binding
  end

  def pass_binding
    binding = Binding.new(:pass)
    binding.value_set = Binding::BOOLEAN
    binding.settable = true
    binding.default = false
    binding
  end

  def direct_action_binding
    binding = Binding.new(:direct_action)
    binding.value_set = Binding::DIRECT_ACTIONS
    binding
  end

  def action_class_binding
    binding = Binding.new(:action_class)
    binding
  end

  def session_id_binding
    binding = Binding.new(:session_id)
    binding.value_set = Binding::BOOLEAN
    binding.default = true
    binding
  end


  #
  # setting
  #

  def set_validation( api )
    api << validate_binding()
    api << pass_binding()
    api << universal_validation(:validate, :pass)
  end

  def set_direct_action( api )
    api << direct_action_binding()
    api << action_class_binding()
    api
  end


  #
  # condition
  #

  def existential_condition( *names )
    conditions = []
    names.each do |name|
      bound = KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
      other_bounds = []
      names.each do |other|
        if name != other then
          other_bounds << KeyErrorCondition.new(other, KeyErrorCondition::BOUND)
        end
      end
      existentials = OrCondition.new(other_bounds)
      conditions << AndCondition.new([bound, existentials])
    end
    OrCondition.new(conditions)
  end

  def universal_condition( *names )
    conditions = []
    names.each do |name|
      bound = KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
      other_bounds = []
      names.each do |other|
        if name != other then
          other_bounds << KeyErrorCondition.new(other, KeyErrorCondition::UNBOUND)
        end
      end
      universals = OrCondition.new(other_bounds)
      conditions << AndCondition.new([bound, universals])
    end
    OrCondition.new(conditions)
  end

  # * or(ex1:bound, ex2:bound) or(un1:bound, un2:bound) or(both1:bound, both2:bound)
  # * on(and(ex, un), and(ex, both), and(un, both))
  def required_condition( existential = [], universal = [], both = [] )
    all_unbound = unbound_condition(existential + universal + both)
    any = any_condition(existential, universal, both)
    OrCondition.new([all_unbound, any])
  end

  def any_condition( existential = [], universal = [], both = [] )
    ex_any = any_bound_condition(existential)
    un_any = any_bound_condition(universal)
    both_any = any_bound_condition(both)

    ex_un = AndCondition.new([ex_any, un_any])
    ex_both = AndCondition.new([ex_any, both_any])
    un_both = AndCondition.new([un_any, both_any])

    ex_cond = existential_condition(*existential)
    un_cond = universal_condition(*universal)

    OrCondition.new([ex_un, ex_both, un_both, ex_cond, un_cond])
  end

  def unbound_condition( names )
    and_condition(names, KeyErrorCondition::UNBOUND)
  end

  def and_condition( names, error )
    conditions = []
    names.each do |name|
      conditions << KeyErrorCondition.new(name, error)
    end
    AndCondition.new(conditions)
  end

  def any_bound_condition( names )
    conditions = []
    names.each do |name|
      conditions << KeyErrorCondition.new(name, KeyErrorCondition::BOUND)
    end
    OrCondition.new(conditions)
  end


  #
  # validation
  #

  def existential_validation( *names )
    list = join_names(names, 'and')
    msg = "#{list} can't both be bound."
    Validation.new(msg, existential_condition(*names))
  end

  def universal_validation( *names )
    list = join_names(names, 'and')
    msg = "#{list} must be bound when one of the both is bound."
    Validation.new(msg, universal_condition(*names))
  end

  def required_validation( existential = [], universal = [], both = [] )
    composed_validation(true, existential, universal, both)
  end

  def any_validation( existential = [], universal = [], both = [] )
    composed_validation(false, existential, universal, both)
  end

  def composed_validation( is_required, existential, universal, both )
    list_existential = join_names(existential, 'or')
    list_universal = join_names(universal, 'and')
    list_both = join_names(both, 'and')

    if is_required and universal.empty? and both.empty? then
      prefix = 'exactly'
      aux = 'must'
    else
      prefix = 'either'
      aux = 'may'
    end
    msg = "#{prefix} "
    unless existential.empty? then
      msg << "one of #{list_existential} #{aux} be bound"
      if (universal.empty? == false) or (both.empty? == false) then
        msg << ', or '
      end
    end
    unless both.empty? then
      msg << "both of #{list_universal} #{aux} be bound"
      if (both.empty? == false) then
        msg << ', or '
      end
    end
    unless both.empty? then
      msg << "either or both of #{list_both} #{aux} be bound"
    end
    msg << '.'

    if is_required then
      Validation.new(msg, required_condition(existential, universal, both))
    else
      Validation.new(msg, any_condition(existential, universal, both))
    end
  end

  def join_names( names, keyword )
    list = ''
    names.each do |name|
      if names.first == name then
        list << "'#{name}'"
      elsif names.last == name then
        list << " #{keyword} '#{name}'"
      else
        list << ", '#{name}'"
      end
    end
    list
  end

  def item_display_value_validation
    msg = "'item' must be bound when 'display' or 'value' is bound."
    item_condition = KeyErrorCondition.new(:item, KeyErrorCondition::UNBOUND)
    display_condition = KeyErrorCondition.new(:display, KeyErrorCondition::BOUND)
    value_condition = KeyErrorCondition.new(:value, KeyErrorCondition::BOUND)
    display_value = OrCondition.new([display_condition, value_condition])
    condition = AndCondition.new([item_condition, display_value])
    Validation.new(msg, condition)
  end

end

end

