/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitAssert.c,v 1.8 2003/10/05 09:56:25 tsutsumi Exp $
 */

/** @file
 * Assert function implementations.
 */

#include <stdio.h>
#include <ccunit/CCUnitAssert.h>
#include <ccunit/CCUnitTestFailure.h>
#include <setjmp.h>
#include <stdarg.h>
#include <assert.h>

/**
 * @addtogroup CCUnitAssert
 * @{
 */

extern jmp_buf _ccunit_runTest_env;

void ccunit_assert (const char* file, unsigned int line,
		    bool cond, const char* condstr)
{
  CCUnitTestFailure* f;
  if (cond)
    return;
  f = ccunit_newTestFailure (file, line, condstr, NULL, NULL);
  longjmp (_ccunit_runTest_env, (int)f);
}

/**
 * @defgroup CCUnitValueToString Convert value to string
 * Make value to string.
 * @{
 */

/**
 * @name ccunit_assert_format_TYPE
 * snprintf format string for each types.
 * @{
 */
const char* _ccunit_assert_format_char = "%c";
const char* _ccunit_assert_format_u_char = "%c";
const char* _ccunit_assert_format_int = "%d";
const char* _ccunit_assert_format_u_int = "%u";
const char* _ccunit_assert_format_short = "%hd";
const char* _ccunit_assert_format_u_short = "%hu";
const char* _ccunit_assert_format_long = "%ld";
const char* _ccunit_assert_format_u_long = "%lu";
const char* _ccunit_assert_format_float = "%f";
const char* _ccunit_assert_format_double = "%f";
const char* _ccunit_assert_format__ccunit_str_t = "%s";
const char* _ccunit_assert_format__ccunit_ptr_t = "%p";
#if CCUNIT_HAVE_QUAD_T
const char* _ccunit_assert_format_quad_t = "%lld";
#endif
#if CCUNIT_HAVE_U_QUAD_T
const char* _ccunit_assert_format_u_quad_t = "%llu";
#endif
/** @} */

/** @name Value to string definitions
 * Make value to string.
 * @param TYP type of value.
 * @param CAPACITY string size.
 * @{
 */
#define DEF_VALUE_TO_STRING(TYP, CAPACITY)			\
  static char* TYP ## _to_string (TYP value)			\
  {								\
    char* str = NULL;						\
    int require = 0;						\
    int capacity = CAPACITY;					\
    for (str = malloc (capacity);				\
	 str != NULL;						\
	 str = realloc (str, capacity))				\
      {								\
	require = snprintf (str, capacity,			\
			    _ccunit_assert_format_ ## TYP,	\
			    value);				\
	if (require < 0)					\
	  capacity *= 2;					\
	else if (require < capacity)				\
	  break;						\
	else							\
	  capacity = require + 1;				\
      }								\
    return str;							\
  }

DEF_VALUE_TO_STRING(char, 6);
DEF_VALUE_TO_STRING(u_char, 6);
DEF_VALUE_TO_STRING(int, 12);
DEF_VALUE_TO_STRING(u_int, 12);
DEF_VALUE_TO_STRING(short, 8);
DEF_VALUE_TO_STRING(u_short, 8);
DEF_VALUE_TO_STRING(long, 12);
DEF_VALUE_TO_STRING(u_long, 12);
DEF_VALUE_TO_STRING(float, 24);
DEF_VALUE_TO_STRING(double, 24);
#if CCUNIT_HAVE_QUAD_T
DEF_VALUE_TO_STRING(quad_t, 24);
#endif
#if CCUNIT_HAVE_U_QUAD_T
DEF_VALUE_TO_STRING(u_quad_t, 24);
#endif
DEF_VALUE_TO_STRING(_ccunit_str_t, 32);
DEF_VALUE_TO_STRING(_ccunit_ptr_t, 24);

/** @}
 * end of DEF_VALUE_TO_STRING
 */

/** @}
 * end of CCUnitValueToString
 */

/**
 * @name Assert test type
 * @{
 * Assert test function.
 * @param TYP type of test value.
 */
#define DEF_CCUNIT_ASSERT_TEST_TYPE(TYP)				\
  void ccunit_assert_test_ ## TYP (const char* file,			\
				   unsigned int line,			\
				   bool cond,				\
				   const char* condstr,			\
				   TYP expect,				\
				   TYP actual)				\
  {									\
    if (cond)								\
      return;								\
    else								\
      {									\
	const char* ex = TYP ## _to_string (expect);			\
	const char* ac = TYP ## _to_string (actual);			\
	CCUnitTestFailure* f;						\
	f = ccunit_newTestFailure (file, line, condstr, ex, ac);	\
	safe_free ((char*)ex);						\
	safe_free ((char*)ac);						\
	assert (f != NULL);						\
	longjmp (_ccunit_runTest_env, (int)f);				\
      }									\
  }

void ccunit_assert_test__ccunit_obj_t (const char* file,
				       unsigned int line,
				       bool cond,
				       const char* condstr,
				       _ccunit_obj_t expect,
				       _ccunit_obj_t actual,
				       char* (*to_string)(_ccunit_obj_t))
{
  if (cond)
    return;
  else
    {
      char* ex = (!to_string ? _ccunit_ptr_t_to_string (expect)
		  : to_string (expect));
      char* ac = (!to_string ? _ccunit_ptr_t_to_string (actual)
		  : to_string (actual));
      CCUnitTestFailure* f;
      f = ccunit_newTestFailure (file, line, condstr, ex, ac);
      safe_free (ex);
      safe_free (ac);
      assert (f != NULL);
      longjmp (_ccunit_runTest_env, (int)f);
    }
}


DEF_CCUNIT_ASSERT_TEST_TYPE(char);
DEF_CCUNIT_ASSERT_TEST_TYPE(u_char);
DEF_CCUNIT_ASSERT_TEST_TYPE(int);
DEF_CCUNIT_ASSERT_TEST_TYPE(u_int);
DEF_CCUNIT_ASSERT_TEST_TYPE(short);
DEF_CCUNIT_ASSERT_TEST_TYPE(u_short);
DEF_CCUNIT_ASSERT_TEST_TYPE(long);
DEF_CCUNIT_ASSERT_TEST_TYPE(u_long);
#if CCUNIT_HAVE_QUAD_T
DEF_CCUNIT_ASSERT_TEST_TYPE(quad_t);
#endif
#if CCUNIT_HAVE_U_QUAD_T
DEF_CCUNIT_ASSERT_TEST_TYPE(u_quad_t);
#endif
DEF_CCUNIT_ASSERT_TEST_TYPE(float);
DEF_CCUNIT_ASSERT_TEST_TYPE(double);
DEF_CCUNIT_ASSERT_TEST_TYPE(_ccunit_str_t);
DEF_CCUNIT_ASSERT_TEST_TYPE(_ccunit_ptr_t);

/** @} */

/** @} */
