/* Copyright (C) 2003, 2010 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/
/*
 * $Id: CCUnitReadSuite.c,v 1.19 2010/08/24 21:15:22 tsutsumi Exp $
 */

/** @file
 * ReadSuite module implementation.
 */
#include <ccunit/CCUnitMakeSuite.h>
#include <ccunit/CCUnitLogMessage.h>
#include <ctype.h>
#include <errno.h>

/**
 * @addtogroup CCUnitReadSuite
 * @{
 */

/**
 * destroy test def.
 *
 * @param test testdef to destroy.
 */
static void destroyTestDef (_CCUnitTestDef* test)
{
  if (!test)
    return;
  safe_free (test->name);
  safe_free (test->idname);
}

/**
 * init test def.
 *
 * @param test testdef.
 * @param type test type.
 * @param name test name.
 */
static _CCUnitTestDef* initTestDef (_CCUnitTestDef* test,
				    CCUnitTestType_t type,
				    const char* name)
{
  test->type = type;
  test->name = safe_strdup (name);
  test->idname = NULL;
  return test;
}

/**
 * delete test def.
 *
 * @param test testdef to delete.
 */
static void deleteTestDef (_CCUnitTestDef* test)
{
  if (!test)
    return;
  if (!test->dtor)
    ;
  else
    test->dtor (test);
  safe_free (test);
}

/**
 * destroy test suite def.
 *
 * @param suite test suite def.
 */
static void destroyTestSuiteDef (_CCUnitTestSuiteDef* suite)
{
  ccunit_deleteList (&suite->testdefs, (void(*)(void*))deleteTestDef);
  destroyTestDef (&suite->testdef);
}

_CCUnitTestSuiteDef* ccunit_newTestSuiteDef (const char* name)
{
  _CCUnitTestSuiteDef* suite = calloc (1, sizeof (*suite));
  if (!suite)
    return suite;
  initTestDef (&suite->testdef, ccunitTypeSuite, name);
  suite->testdef.dtor = (void(*)(_CCUnitTestDef*))destroyTestSuiteDef;
  ccunit_initList (&suite->testdefs);
  return suite;
}

inline void ccunit_deleteTestSuiteDef (_CCUnitTestSuiteDef* suite)
{
  deleteTestDef (&suite->testdef);
}

/**
 * add test to test suite.
 *
 * @param suite test suite to add.
 * @param test test group.
 * @return added test.
 */
static _CCUnitTestDef* addTestDef (_CCUnitTestSuiteDef* suite,
				   _CCUnitTestDef* test)
{
  if (!suite || !test)
    return NULL;
  ccunit_addList (&suite->testdefs, test);
  return test;
}

/**
 * add test suite to test suite.
 *
 * @param suite test suite to add.
 * @param test test suite.
 * @return added test.
 */
static inline _CCUnitTestDef* addTestSuiteDef (_CCUnitTestSuiteDef* suite,
					       _CCUnitTestSuiteDef* test)
{
  const char* name;
  if (!suite || !test)
    return NULL;
  name = test->testdef.name;
  if (!name)
    name = "";
  ccunit_log ("add test suite: %s", name);
  return addTestDef (suite, &test->testdef);
}

/**
 * add test case to test suite.
 *
 * @param suite test suite to add.
 * @param test test case.
 * @return added test.
 */
static inline _CCUnitTestDef* addTestCaseDef (_CCUnitTestSuiteDef* suite,
						 _CCUnitTestCaseDef* test)
{
  const char* name;
  if (!suite || !test)
    return NULL;
  name = test->testdef.name;
  if (!name)
    name = "";
  ccunit_log ("add test case: %s", name);
  return addTestDef (suite, &test->testdef);
}

/**
 * create new test func.
 *
 * @param scope scope.
 * @param type return type of func.
 * @param name func name.
 * @param desc description.
 * @return new test func def.
 */
static _CCUnitFuncDef* newFuncDef (const char* scope,
				   const char* type,
				   const char* name,
				   const char* desc)
{
  struct _CCUnitFuncDef* f = calloc (1, sizeof (*f));
  ccunit_log ("create new test func: %s %s", type, name);
  if (!f)
    return f;
  f->scope = !scope ? strdup ("extern") : safe_strdup (scope);
  f->type = safe_strdup (type);
  f->name = safe_strdup (name);
  f->desc = !desc ? safe_strdup (name) : strdup (desc);
  return f;
}

/**
 * delete test func def.
 * @param func test func def to delete.
 */
static void deleteFuncDef (_CCUnitFuncDef* func)
{
  if (!func)
    return;
  safe_free (func->scope);
  safe_free (func->type);
  safe_free (func->name);
  safe_free (func->desc);
  free (func);
}

/**
 * destroy test case def.
 * @param testCase test case def to destroy.
 */
static void destroyTestCaseDef (_CCUnitTestCaseDef* testCase)
{
  ccunit_deleteList (&testCase->testFuncs, (void(*)(void*))deleteFuncDef);
  destroyTestDef (&testCase->testdef);
}

/**
 * create new test case def.
 *
 * @param name test case name.
 */
static _CCUnitTestCaseDef* newTestCaseDef (const char* name)
{
  _CCUnitTestCaseDef* testCase = calloc (1, sizeof (*testCase));
  ccunit_log ("create new test case: %s", name);
  if (!testCase)
    return NULL;
  initTestDef (&testCase->testdef, ccunitTypeTestCase, name);
  testCase->testdef.dtor = (void(*)(_CCUnitTestDef*))destroyTestCaseDef;
  ccunit_initList (&testCase->testFuncs);
  return testCase;
}

/**
 * @defgroup _CCUnitLine _Line
 * Read one line module.
 * @{
 */

/**
 * Read line.
 */
struct _CCUnitLine
{
  char* str;					/**< read line buffer */
  size_t length;				/**< line length */
  size_t capacity;				/**< buffer capacity */
  unsigned long lno;				/**< line number */
  FILE* ifp;					/**< input stream */
  const char* fname;				/**< input file name */
};

/**
 * Current processing line.
 */
static struct _CCUnitLine line;

/**
 * Get one line from stream.
 * This func copies a read line on the global variable <code>line</code>.
 *
 * @return When reading succeeds, value except for the zero is
 * returned. When an error occurs, a zero is returned.
 */
static int readline ()
{
  static const size_t MIN_LINE_BUF_LEN = 512;
  char* insertAt;
  size_t restSize;
  char* sp;
  char* tail;
  /* buffer hasn't been allocate yet */
  if (line.str == NULL)
    {
      line.capacity = MIN_LINE_BUF_LEN;
      line.str = calloc (line.capacity, sizeof(line.str[0]));
      line.length = 0;
      line.lno = 0;
    }
  /* shrink to minimum size */
  else if (line.capacity > MIN_LINE_BUF_LEN)
    {
      char* str;
      line.capacity = MIN_LINE_BUF_LEN;
      str = realloc (line.str, line.capacity);
      if (str)
	line.str = str;
    }
  insertAt = line.str;
  restSize = line.capacity;
  line.length = 0;
  sp = 0;
  while ((sp = fgets (insertAt, restSize, line.ifp)) != NULL)
    {
      line.length += strlen(insertAt);
      /* read complete whole line */
      if (line.str[line.length-1] == '\n'
	  || line.str[line.length-1] == '\r')	/* for mac? */
	{
	  break;
	}
      else
	{
	  /* to extend capacity for rest of line */
	  size_t newCapacity = line.capacity * 2 / 3;
	  char* str = realloc (line.str, newCapacity);
	  if (!str)
	    {
	      ccunit_log ("/* no more memory */");
	      return 0;
	    }
	  line.str = str;
	  restSize = newCapacity - line.capacity;
	  insertAt = str + line.capacity;
	  line.capacity = newCapacity;
	}
    }
  if (!sp)
    return 0;
  /* chomp CR/LF */
  if (line.length > 0)
    {
      tail = line.str + line.length - 1;
      if (*tail == '\n')
	{
	  line.length --;
	  *tail = '\0';
	  if (line.length > 0)
	    tail --;
	}
      if (*tail == '\r')	/* for dos and/or mac? */
	{
	  line.length --;
	  *tail = '\0';
	  if (line.length > 0)
	    tail --;
	}
    }
  line.lno ++;
  return 1;
}

/**
 * read contents of doc comment.
 *
 * @return comment string. or NULL when error occurred.
 */
static char* readDocCommentContents ()
{
  bool eoc = false;				/* reach end of comment */
  char* content = NULL;				/* comment content */
  size_t length = 0;				/* content length */
  char* start = NULL;				/* start of content */
  char* end = NULL;				/* end of content */
  ccunit_log ("readDocCommentContent");
  start = line.str + 2;
  while (!eoc)
    {
      ccunit_dbg ("read from:%lu: \"%s\"", line.lno, start);
      /* skip white spaces */
      for (; *start && isspace ((int)*start); start ++)
	;
      /* skip leading block comment '*'<WSP>... */
      if (*start != '*')
	;
      else if (start[1] == '/')			/* eoc */
	;
      else					/* skip white spaces */
	for (start ++; *start && isspace ((int)*start); start ++)
	  ;
      /* skip doxygen command line */
      if (*start == '@')
	{
	  ccunit_log ("skip doxygen javadoc style comment");
	  for (end = start + 1; *end; end ++)
	    if (end[0] == '*' && end[1] == '/')
	      {
		ccunit_log ("end of comment");
		eoc = true;
		break;
	      }
	  start = end;
	}
      /* seek to eol or end of comment */
      for (end = start; *end; end ++)
	if (end[0] == '*' && end[1] == '/')
	  {
	    ccunit_log ("end of comment");
	    eoc = true;
	    break;
	  }
      /* trim trailing white space */
      for (end --; start < end; end --)
	if (!isspace ((int)*end))
	  {
	    end ++;
	    break;
	  }
      /* did a comment exist? */
      if (start < end)
	{
	  int len = (int)(end - start);
	  char* newContent = realloc (content, length + len + 2);
	  if (!newContent)
	    {
	      ccunit_err ("no more memory");
	      break;
	    }
	  if (length > 0)
	    newContent[length ++] = ' ';	/* word space */
	  memcpy (newContent + length, start, len);
	  length += len;
	  newContent[length] = '\0';
	  content = newContent;
	  ccunit_log ("get: \"%*.*s\"", len, len, start);
	}
      if (eoc || !readline ())
	break;
      start = line.str;
    }
  ccunit_log ("comment content: \"%s\"", !content ? "" : content);
  return content;
}

/**
 * read document comment.
 *
 * @return comment content if matched, or NULL if not matched.
 */
static char* readDocComment ()
{
  const char* cmnt = "/**";
  if (strncmp (line.str, cmnt, strlen(cmnt)) != 0) /* not a comment */
    ;
  else if (line.str[3] == '*' || line.str[3] == '/') /* not doc */
    ;
  else
    {
      ccunit_dbg ("found doc comment:%lu: %s", line.lno, line.str);
      return readDocCommentContents ();
    }
  return NULL;
}

static const char* testTypeStr[] = {
  "case", "suite"
};

/**
 * get test def.
 *
 * @param type test type.
 * @param str comment string.
 * @return test name.
 */
static const char* getTestName (CCUnitTestType_t type, const char* str)
{
  static const char* prefixStr[] = {
    "TEST CASE:", "TEST SUITE:",
  };
  const int typeid = (type == ccunitTypeSuite) ? 1 : 0;
  const char* testType = testTypeStr[typeid];
  const char* prefix = prefixStr[typeid];
  const size_t prefixLen = strlen (prefix);
  const char* name = NULL;
  if (strncasecmp (str, prefix, prefixLen) == 0)
    {
      for (name = str + prefixLen; *name; name ++)
	if (!isspace ((int)*name))
	  break;
      if (!*name)
	{
	  name = NULL;
	  ccunit_err ("no test %s name: %s. near line %lu",
		      testType, str, line.lno);
	}
    }
  else
    ccunit_dbg ("not a test %s name: %s", testType, str);
  return name;
}

/**
 * get end of test string.
 * @param type test type.
 * @param str string.
 * @return name of test.
 */
static const char*
getEndOfTest (CCUnitTestType_t type, const char* str)
{
  static const char* prefixStr[] = {
    "END TEST CASE", "END TEST SUITE"
  };
  const int typeid = (type == ccunitTypeSuite) ? 1 : 0;
  const char* testType = testTypeStr[typeid];
  const char* prefix = prefixStr[typeid];
  const size_t prefixLen = strlen (prefix);
  const char* name = NULL;
  if (strncasecmp (str, prefix, prefixLen) == 0)
    {
      name = str + prefixLen;
      if (*name && !isspace ((int)*name) && !ispunct ((int)*name))
	{
	  name = NULL;
	  ccunit_dbg ("not a end of test %s: %s", testType, str);
	}
      else
	{
	  for (; *name; name ++)
	    if (!isspace ((int)*name))
	      break;
	  if (!*name)
	    ;
	  else
	    ccunit_log ("end of test %s: %s", testType, name);
	}
    }
  else
    ccunit_dbg ("not a end of test %s: %s", testType, str);
  return name;
}

/**
 * read test funcdef.
 *
 * @param type required type string.
 * @param prefix required func name prefix.
 * @param desc description.
 * @return funcdef object.
 */
static _CCUnitFuncDef* readTestFunc (const char* type,
				    const char* prefix,
				    const char* desc)
{
  const char* scope = "static";
  char* typ;
  char* name;
  ccunit_dbg ("read func: %s %s... from '%s'", type, prefix, line.str);
  for (typ = line.str; *typ; typ ++)
    if (!isspace ((int)*typ))
      break;
  if (strncmp (typ, scope, strlen (scope)) != 0)
    scope = "extern";
  else
    {
      typ += strlen (scope);
      if (*typ && !isspace ((int)*typ))
	{
	  ccunit_dbg ("type mismatch: %s %s", type, typ);
	  return NULL;
	}
      for (;;)
	{
	  for (; *typ; typ ++)
	    if (!isspace ((int)*typ))
	      break;
	  if (*typ)
	    break;
	  if (!readline ())
	    {
	      ccunit_err ("unexpected EOF");
	      return NULL;
	    }
	  typ = line.str;
	}
    }
  if (strncmp (typ, type, strlen (type)) != 0)
    {
      ccunit_dbg ("type mismatch: %s %s", type, typ);
      return NULL;
    }
  name = typ + strlen (type);
  if (*name && !isspace ((int)*name))
    {
      ccunit_dbg ("type mismatch: %s %s", type, name);
      return NULL;
    }
  for (;;)
    {
      for (; *name; name ++)
	if (!isspace ((int)*name))
	  break;
      if (*name)
	break;
      if (!readline ())
	{
	  ccunit_err ("unexpected EOF");
	  return NULL;
	}
      name = line.str;
    }
  if (strncmp (name, prefix, strlen(prefix)) == 0)
    {
      char* tail;
      for (tail = name + 1; *tail; tail ++)
	if (isspace ((int)*tail) || *tail == '(')
	  {
	    *tail = '\0';
	    break;
	  }
      return newFuncDef (scope, type, name, desc);
    }
  else
    ccunit_dbg ("name mismatch: %s %s", prefix, name);
  return NULL;
}

/**
 * read test case function.
 *
 * @param suite parent suite.
 * @param cname test case name to read.
 */
static void readTestCase (_CCUnitTestSuiteDef* suite, const char* cname)
{
  _CCUnitTestCaseDef* testCase;
  _CCUnitFuncDef* f = NULL;
  const char* name;
  char* doc;
  char* desc = NULL;
  if (!suite)
    return;
  testCase = newTestCaseDef (cname);
  if (!testCase)
    return;
  addTestCaseDef (suite, testCase);
  while (readline ())
    {
      /* setUp function def */
      if ((f = readTestFunc ("void", "setUp", desc)) != NULL)
	{
	  ccunit_addList (&testCase->testFuncs, f);
	  safe_free (desc);
	}
      /* tearDown function def */
      else if ((f = readTestFunc ("void", "tearDown", desc)) != NULL)
	{
	  ccunit_addList (&testCase->testFuncs, f);
	  safe_free (desc);
	}
      /* setup_setUp function def */
      else if ((f = readTestFunc ("void", "setup_setUp", desc)) != NULL)
	{
	  ccunit_addList (&testCase->testFuncs, f);
	  safe_free (desc);
	}
      /* setup_tearDown function def */
      else if ((f = readTestFunc ("void", "setup_tearDown", desc)) != NULL)
	{
	  ccunit_addList (&testCase->testFuncs, f);
	  safe_free (desc);
	}
      /* if test case function def, then read as test func. */
      else if ((f = readTestFunc ("void", "test", desc)) != NULL)
	{
	  ccunit_addList (&testCase->testFuncs, f);
	  safe_free (desc);
	}
      /* if current line is javaDoc comment, then read as description. */
      else if ((doc = readDocComment ()) != NULL)
	{
	  if ((name = getTestName (ccunitTypeTestCase, doc)) != NULL)
	    {
	      readTestCase (suite, name);
	      safe_free (doc);
	    }
	  else if ((name = getEndOfTest (ccunitTypeTestCase, doc))
		   != NULL)
	    {
	      ccunit_log ("end test case: %s", testCase->testdef.name);
	      safe_free (doc);
	      break;
	    }
	  safe_free (desc);
	  desc = doc;
	  doc = NULL;
	}
      else
	;
    }
  safe_free (desc);
}

/**
 * read test suite def.
 *
 * @param suite test suitedef.
 */
static void readSuite (_CCUnitTestSuiteDef* suite)
{
  _CCUnitFuncDef* f;
  const char* name;
  char* doc = NULL;
  char* desc = NULL;
  while (readline ())
    {
      /* if current line is javaDoc comment, then read as description. */
      if ((doc = readDocComment ()) != NULL)
	{
	  if ((name = getTestName (ccunitTypeTestCase, doc)) != NULL)
	    {
	      readTestCase (suite, name);
	      safe_free (doc);
	    }
	  else if ((name = getEndOfTest (ccunitTypeTestCase, doc)) != NULL)
	    {
	      ccunit_err ("%s:%lu: invalid end test case comment '%s'",
			  line.fname, line.lno, doc);
	      safe_free (doc);
	    }
	  else if ((name = getTestName (ccunitTypeSuite, doc)) != NULL)
	    {
	      _CCUnitTestSuiteDef* newSuite;
	      newSuite = ccunit_newTestSuiteDef (name);
	      if (!newSuite)
		break;
	      addTestSuiteDef (suite, newSuite);
	      readSuite (newSuite);
	      safe_free (doc);
	    }
	  else if ((name = getEndOfTest (ccunitTypeSuite, doc)) != NULL)
	    {
	      break;
	    }
	  safe_free (desc);
	  desc = doc;
	  doc = NULL;
	}
      else if ((f = readTestFunc ("void", "test", desc)) != NULL
	       || (f = readTestFunc ("void", "setUp", desc)) != NULL
	       || (f = readTestFunc ("void", "tearDown", desc)) != NULL
	       || (f = readTestFunc ("void", "setup_setUp", desc)) != NULL
	       || (f = readTestFunc ("void", "setup_tearDown", desc)) != NULL)
	{
	  ccunit_err ("%s:%lu: missing test case start comment '%s': ignored",
		      line.fname, line.lno, line.str);
	  deleteFuncDef (f);
	  safe_free (desc);
	}
      else
	;
    }
  safe_free (doc);
  safe_free (desc);
}

/** @} */

void ccunit_readSuite (const char* fname, _CCUnitTestSuiteDef* suite)
{
  if (strcmp (fname, "-") == 0) /* special file name '-' as stdin  */
    {
      line.ifp = stdin;
      line.fname = "(stdin)";
    }
  else
    {
      line.ifp = fopen (fname, "r");
      if (!line.ifp)				/* open error */
	{
	  ccunit_err ("can't open file '%s': %s.  skipped.\n",
		      fname, strerror (errno));
	  return;
	}
      line.fname = fname;
    }
  readSuite (suite);
  safe_free (line.str);
  if (line.ifp != NULL && line.ifp != stdin)
    fclose (line.ifp);
  memset (&line, 0, sizeof (line));
}

/** @} */
