/* -*- mode: C; -*- */
/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestCase.h,v 1.12 2003/10/05 11:29:59 tsutsumi Exp $
 */

/** @file
 * TestCase module.
 */

#ifndef CCUNITTESTCASE_H
#define CCUNITTESTCASE_H

#include <ccunit/CCUnitConfig.h>

/**
 * @ingroup CCUnitTestFixture
 * @defgroup CCUnitTestCase TestCase
 *
 * A single test object.
 *
 * This class is used to implement a simple test case.
 *
 * @{
 */

/**
 * A single test object.
 *
 * For each test implement a function which interacts with the
 * case. Verify the expected results with assertions specified by
 * calling CCUNIT_ASSERT on the expression you want to test:
 * 
 * @code
 * void testAdd ()
 * {
 *   int result = value1 + value2;
 *   CCUNIT_ASSERT (result == 5);
 * }
 *
 * ...
 *
 * void MathTest_newTestCase_testAdd ()
 * {
 *   return ccunit_newTestCase ("addTest", "add test", addTest);
 * }
 * @endcode
 *
 * @see CCUnitTestFixture, CCUnitTestSuite, CCUnitMakeSuite
 * 
 * @ingroup WritingTestFixture
 */
typedef struct CCUnitTestCase
{
  const char* name;				/**< test case name */
  const char* desc;				/**< test description */
  void (*runTest) ();				/**< run test function */
} CCUnitTestCase;

typedef CCUnitTestCase CCUnitTestFunc;

/**
 * Create new test case.
 * @param name case name.
 * @param desc case description.
 * @param runTest run test function.
 * @return new test case
 * @ingroup WritingTestFixture
 */
extern CCUnitTestCase* ccunit_newTestCase (const char* name,
					   const char* desc,
					   void (*runTest)());

/**
 * Create new test function.
 * @param name func name.
 * @param desc func description.
 * @param func run test function.
 * @return new test func.
 * @ingroup WritingTestFixture
 */
extern inline CCUnitTestFunc* ccunit_newTestFunc (const char* name,
						  const char* desc,
						  void (*func)());
#define CCUNIT_NEWTESTFUNC(FUNC) ccunit_newTestFunc (#FUNC, NULL, FUNC)

/**
 * Delete test case.
 * @param testCase deleting case.
 * @ingroup WritingTestFixture
 */
extern void ccunit_deleteTestCase (CCUnitTestCase* testCase);

/**
 * Delete test func.
 * @param testFunc deleting func.
 * @ingroup WritingTestFixture
 */
extern inline void ccunit_deleteTestFunc (CCUnitTestFunc* testFunc);

/** @} */

#endif	/* !CCUNITTESTCASE_H */
