/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/
/*
 * $Id: CCUnitMakeSuite.c,v 1.10 2003/10/05 12:16:58 tsutsumi Exp $
 */

/** @file
 * Make test suite object code from test source code.
 */
#include <ccunit/CCUnitMakeSuite.h>
#include <ccunit/CCUnitLogMessage.h>
#if CCUNIT_HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdio.h>
#include <errno.h>
#if CCUNIT_STDC_HEADERS
#include <stdarg.h>
#endif

/**
 * If `setvbuf' takes the buffering type as its second argument and
 * the buffer pointer as the third, instead of the other way around.
 */
#if CCUNIT_SETVBUF_REVERSED
#  define SETVBUF(FP,PTR,TYP,SZ) setvbuf(FP,TYP,PTR,SZ)
#else
#  define SETVBUF(FP,PTR,TYP,SZ) setvbuf(FP,PTR,TYP,SZ)
#endif

#if defined(__MACH__) || defined(__CYGWIN__)
#define DIRSEP '/'				/**< directory separetor */
#elif defined (__WIN32__) || defined (__MSDOS__)
#define DIRSEP '\\'				/**< directory separetor */
#else
#define DIRSEP '/'				/**< directory separetor */
#endif

#if CCUNIT_HAVE_GETOPT_LONG
#  define LONGOPTNAME(LO) "      --" #LO "\n"	      /**< long name option */
#else
#  define LONGOPTNAME(LO) /* -- LO | */		/**< long name option */
#endif

/**
 * @addtogroup CCUnitMakeSuite
 * @{
 */

/**
 * print help massage.
 * @param progname program name.
 */
static void usage (const char* progname)
{
  fprintf (stdout,
	   "%s\n"
	   "  USAGE: %s [OPTIONS] FILES...\n"
	   "    OPTION:\n"
	   LONGOPTNAME(output OUTFILE)
	   "      -o OUTFILE\n"
	   "          output file name (default stdout)\n"
	   LONGOPTNAME(function FUNCTION)
	   "      -f FUNCTION\n"
	   "          created function name\n"
	   LONGOPTNAME(verbose)
	   "      -v\n"
	   "          output verbose message\n"
	   LONGOPTNAME(debug)
	   "      -d\n"
	   "          output debug message\n"
	   LONGOPTNAME(version)
	   "      -V\n"
	   "          print version\n"
	   LONGOPTNAME(help)
	   "      -h\n"
	   "          print this message\n"
	   , CCUNIT_PACKAGE_STRING, progname);
}

int ccunit_makeSuite (int ac, char** av)
{
  _CCUnitTestSuiteDef* suite;
  char* outputFile = "-";
  const char* functionName = "ccunit_suite";
  FILE* ofp = stdout;
  int opt;
  char* progname = NULL;
#ifdef CCUNIT_HAVE_GETOPT_LONG
  int option_index = 0;
  static struct option long_options[] = {
    { "output", required_argument, NULL, 'o' },
    { "function", required_argument, NULL, 'f' },
    { "verbose", no_argument, NULL, 'v' },
    { "debug", no_argument, NULL, 'd' },
    { "version", no_argument, NULL, 'V' },
    { "help", no_argument, NULL, 'h' },
    { NULL, 0, NULL, 0 }
  };
#endif
  if (av[0])
    {
      progname = strrchr (av[0], DIRSEP);
      if (!progname)
	{
	  progname = av[0];
#         if defined (__WIN32__) || defined (__MSDOS__)
	  if (progname[2] == ':')
	    progname += 2;
#         endif
	}
      else
	progname ++;
    }
  for (;;)
    {
#ifdef CCUNIT_HAVE_GETOPT_LONG
      opt = getopt_long (ac, av, "o:f:vdVh", long_options, &option_index);
#else
      opt = getopt (ac, av, "o:f:vdVh");
#endif
      if (opt == -1)
	break;
      switch (opt)
	{
	case 0:
#ifdef CCUNIT_HAVE_GETOPT_LONG
	  puts (long_options[option_index].name);
#endif
	  break;
	case 'o':
	  outputFile = optarg;
	  break;
	case 'f':
	  functionName = optarg;
	  break;
	case 'v':
	  _ccunit_verbose_message = !_ccunit_verbose_message;
	  break;
	case 'd':
	  _ccunit_debug_message = !_ccunit_debug_message;
	  break;
	case 'V':
	  fprintf (stdout, "%s - %s\n", CCUNIT_PACKAGE_STRING, progname);
	  return 0;
	case 'h':
	  usage (progname);
	  return 0;
	case '?':
	  fprintf (stderr, "unknown option %s\n", av[optind]);
	  return 1;
	case ':':
	  fprintf (stderr, "bad option argument: %s\n", av[optind]);
	  return 1;
	}
    }
  if (optind >= ac)
    {
      usage (progname);
      return 0;
    }
  suite = ccunit_newTestSuiteDef (NULL);
  if (!suite)
    {
      ccunit_err ("can't create test suite def. not enough memory.");
      return -1;
    }
  if (strcmp (outputFile, "-") == 0)
    ofp = stdout;
  else
    {
      ofp = fopen (outputFile, "w");
      if (!ofp)
	{
	  fprintf (stderr, "can't open file '%s': %s.\n",
		   outputFile, strerror (errno));
	  return 2;
	}
    }
  SETVBUF(ofp, NULL, _IONBF, 0);
  SETVBUF(stderr, NULL, _IONBF, 0);
  /* process source files */
  for (; optind < ac; optind ++)
    ccunit_readSuite (av[optind], suite);
  ccunit_printSuite (ofp, functionName, suite);
  if (ofp != stdout)
    fclose (ofp);
  ccunit_deleteTestSuiteDef (suite);
  return 0;
}

int ccunit_va_makeSuite (const char* prg, ...)
{
  va_list args;
  size_t capacity = 10;
  char** av = calloc (10, sizeof (char*));
  int ac = 0;
  int rc;
  av[0] = (char*)prg;
  va_start (args, prg);
  while (av[ac])
    {
      ac ++;
      if (ac >= capacity)
	{
	  capacity *= 2;
	  av = realloc (av, capacity * sizeof (char*));
	}
      av[ac] = va_arg (args, char*);
    }
  va_end (args);
  rc = ccunit_makeSuite (ac, av);
  free (av);
  return rc;
}

/** @} */
