/*
 * Copyright (c) 2002-2004 David Keiichi Watanabe
 * davew@xlife.org
 *
 * Modified by (c) 2004-2005 heavy_baby
 * heavy_baby@users.sourceforge.jp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.sourceforge.cabos;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import com.limegroup.gnutella.ErrorService;
import com.limegroup.gnutella.MessageService;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.UPnPManager;
import com.limegroup.gnutella.Uploader;
import com.limegroup.gnutella.downloader.ManagedDownloader;
import com.limegroup.gnutella.io.NIODispatcher;
import com.limegroup.gnutella.settings.ApplicationSettings;
import com.limegroup.gnutella.settings.ConnectionSettings;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.FileUtils;

public class AqMain {
	/* Instances */

	private static InputStreamReader reader = null;

	private static OutputStreamWriter writer = null;

	protected static List dispatches = Collections.EMPTY_LIST;

	/* Main */

	public static void main(String[] args) {
		/* open streams */

		try {
			reader = new InputStreamReader(System.in, "UTF-8");
			writer = new OutputStreamWriter(System.out, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			System.exit(0);
		}
		AqEvent.signalEvent(AqEvent.kLWEventCoreConnected);

		/* set error callback */

		AqEventHandler handler = new AqEventHandler();
		ErrorService.setErrorCallback(handler);
		MessageService.setCallback(handler);

		/* make writable app support directories */

		if (CommonUtils.getUserHomeDir().exists())
			FileUtils.setWriteable(CommonUtils.getUserHomeDir());
		if (CommonUtils.getUserSettingsDir().exists())
			FileUtils.setWriteable(CommonUtils.getUserSettingsDir());

		/* check personal firewall blocking */

		if (!NIODispatcher.instance().isRunning())
			System.exit(0);

		/* start UPnP */

		if (!ConnectionSettings.DISABLE_UPNP.getValue())
			UPnPManager.instance().start();

		/* start LW */

		RouterService router = new RouterService(handler);
		router.start();

		/* schedule threads */

		RouterService.schedule(new DispatchCommand(), 100, 100);
		RouterService.schedule(new ConnectionUpdate(), 60 * 1000, 60 * 1000);
		RouterService.schedule(new TransferUpdate(), 1 * 1000, 1 * 1000);

		/* read loop */

		AqEvent.signalEvent(AqEvent.kLWEventCoreInitialized);
		
		readLoop();
		System.exit(0);
	}

	private static void readLoop() {

		LinkedList args = new LinkedList();
		StringBuffer fragment = new StringBuffer();
		Thread.currentThread().setPriority(Thread.MIN_PRIORITY);

		try {
			while (true) {
				char c = (char) reader.read();

				if (c == '|' || c == '\n') {
					args.add(fragment.toString());
					fragment.setLength(0);

					if (c == '\n') {
						List newDispatches = new LinkedList(dispatches);
						newDispatches.add(args.clone());
						dispatches = Collections
								.unmodifiableList(newDispatches);
						args.clear();
					}

				} else {
					fragment.append(c);
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
		}

	}

	protected static void writeEvent(String event) {
		synchronized (writer) {
			try {
				writer.write(event);
				writer.flush();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	protected static void shutdown() {
		synchronized (writer) {
			try {
				writer.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		RouterService.shutdown();
		System.exit(0);
	}
}

/* DispatchCommand */

class DispatchCommand implements Runnable {

	public void run() {

		List newDispatches = new LinkedList(AqMain.dispatches);
		if (!newDispatches.isEmpty()) {
			try {
				AqDispatcher.dispatchCommand((List) newDispatches.get(0));
			} catch (Exception e) {
				e.printStackTrace();
			}
			newDispatches.remove(0);
			AqMain.dispatches = Collections.unmodifiableList(newDispatches);
		}

	}

}

/* ConnectionUpdate */

class ConnectionUpdate implements Runnable {
	public void run() {
		AqEvent.signalEvent(AqEvent.kLWEventConnectionsUpdated);
	}
}

/* TransferUpdate */

class TransferUpdate implements Runnable {

	public void run() {
		/* Downloads */

		for (Iterator i = AqEventHandler.downloads.iterator(); i.hasNext(); publishUpdate((ManagedDownloader) i
				.next()))
			;
		AqEvent.signalEvent(AqEvent.kLWEventDownloadsUpdated);

		/* Uploads */

		for (Iterator i = AqEventHandler.uploads.iterator(); i.hasNext(); publishUpdate((Uploader) i
				.next()))
			;
		AqEvent.signalEvent(AqEvent.kLWEventUploadsUpdated);

		/* calculate timer */

		int totalUptime = ApplicationSettings.TOTAL_UPTIME.getValue() + 1;
		ApplicationSettings.TOTAL_UPTIME.setValue(totalUptime);
		ApplicationSettings.AVERAGE_UPTIME.setValue(totalUptime
				/ ApplicationSettings.SESSIONS.getValue());
	}

	private void publishUpdate(ManagedDownloader d) {
		if (d != null)
			AqEvent.signalEvent(AqEvent.kLWEventUpdateDownloadStats, d);
	}

	private void publishUpdate(Uploader u) {
		if (u != null)
			AqEvent.signalEvent(AqEvent.kLWEventUpdateUploadStats, u);
	}

}
