package com.limegroup.gnutella.gui.menu;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import javax.swing.Box;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.WindowConstants;

import com.limegroup.gnutella.gui.BoxPanel;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.GUIUtils;
import com.limegroup.gnutella.gui.MultiLineLabel;
import com.limegroup.gnutella.gui.ResourceManager;
import com.limegroup.gnutella.gui.themes.ThemeFileHandler;
import com.limegroup.gnutella.gui.themes.ThemeMediator;
import com.limegroup.gnutella.gui.themes.ThemeSettings;
import com.limegroup.gnutella.util.CommonUtils;

/**
 * The menu to be used for themes.
 */
final class ThemeMenu extends AbstractMenu {
    
    /**
     * The client property to use for theme changing items.
     */
    private static final String THEME_PROPERTY = "THEME_NAME";
    
    /**
     * The listener for changing the theme.
     */
    private static final ActionListener THEME_CHANGER =
        new ThemeChangeListener();
    
    /**
     * The ButtonGroup to store the theme options in.
     */
    private static final ButtonGroup GROUP = new ButtonGroup();
    
    /**
     * Constructs the menu.
     */
    ThemeMenu(String key) {
        super(key);
        
        addMenuItem("VIEW_THEMES_GET_MORE", new GetThemesListener());
        addMenuItem("VIEW_THEMES_REFRESH", new RefreshThemesListener());
        
        JMenuItem def =
            addMenuItem("VIEW_THEMES_USE_DEFAULT", THEME_CHANGER);            
        final Object defaultVal = ThemeSettings.THEME_DEFAULT.getAbsolutePath();
        def.putClientProperty(THEME_PROPERTY, defaultVal);
        
        // Add a listener to set the new theme as selected.
        def.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ae) {
                setSelection(defaultVal);
            }
        });
        addSeparator();
        addThemeItems();
    }
    
    /**
     * Sets the default theme.
     */
    private static void setSelection(Object value) {
        Enumeration items = GROUP.getElements();
        while(items.hasMoreElements()) {
            JMenuItem item = (JMenuItem)items.nextElement();
            if(value.equals(item.getClientProperty(THEME_PROPERTY))) {
                item.setSelected(true);
                break;
            }
        }
    }        
    
    /**
     * Scans through the theme directory for .lwtp files & adds them
     * as menu items to the menu.
     */ 
    private void addThemeItems() {
        File themeDir = ThemeSettings.THEME_DIR_FILE;
        if(!themeDir.exists()) return;
       
        String[] allThemes =
            themeDir.list(new ThemeFileFilter());
        if(allThemes == null) return;
        
        JMenuItem theme;
        File themeFile;
        for(int i = 0; i < allThemes.length; i++) {
            themeFile = new File(themeDir, allThemes[i]);
            theme = new JRadioButtonMenuItem(
                ThemeSettings.formatName(themeFile.getName()));
            theme.setFont(AbstractMenu.FONT);
            GROUP.add(theme);
            if( themeFile.equals(ThemeSettings.THEME_FILE.getValue()) )
                theme.setSelected(true);
            theme.addActionListener(THEME_CHANGER);
            theme.putClientProperty(THEME_PROPERTY, themeFile.getAbsolutePath());
            MENU.add(theme);
        }
    }
    
    /**
     * Removes all items in the group from the menu.  Used for refreshing.
     */
    private void removeThemeItems() {
        Enumeration items = GROUP.getElements();
        List removed = new LinkedList();
        while(items.hasMoreElements()) {
            JMenuItem item = (JMenuItem)items.nextElement();
            MENU.remove(item);
            removed.add(item);
        }
        
        for(Iterator itr = removed.iterator(); itr.hasNext();)
            GROUP.remove((JMenuItem)itr.next());
          
        
    }
    
    /**
     * Opens the themes page in the default browser, displaying
     * an error message if the browser could not be launched 
     * successfully.
     */
    private static class GetThemesListener implements ActionListener {
    	public void actionPerformed(ActionEvent e) {
    		String url = "http://www.limewire.com/skins2";
    	    GUIMediator.openURL(url);
    	}
    }
    
    /**
     * Refreshes the theme menu options to those on the disk.
     */
    private class RefreshThemesListener implements ActionListener {
    	public void actionPerformed(ActionEvent e) {
            removeThemeItems();
            addThemeItems();
    	}
    }    
    
    /**
     * ActionListener to change the theme based on the client property.
     */
    protected static class ThemeChangeListener implements ActionListener {    
        public void actionPerformed(ActionEvent e) {
            JMenuItem item = (JMenuItem)e.getSource();
            String themePath = 
                (String)item.getClientProperty(THEME_PROPERTY);
            File themeFile = new File(themePath);
	    File oldTheme = ThemeSettings.THEME_FILE.getValue();
	    ThemeMediator.changeTheme(themeFile);
        }
    }
    
    /**
     * <tt>FileNameFilter</tt> class for only displaying theme file types.
     */
    public static class ThemeFileFilter implements FilenameFilter {
        public boolean accept(File dir, String name) {
            // don't allow anything that isn't a theme file
            if(!name.endsWith(ThemeSettings.EXTENSION))
                return false;
                
            // if this is one of the old 'default_X' themes
            // we used to ship with, ignore it.
            if(name.startsWith("default_"))
                return false;
                
            // only allow the osx theme if we're on osx.
            if(!CommonUtils.isMacOSX() && 
              name.equals(ThemeSettings.PINSTRIPES_OSX_THEME_NAME))
                return false;
                
            // only allow the brushed metal theme if we're on
            // osx with 10.3 or above w/ java 1.4.2+.
            if(name.equals(ThemeSettings.BRUSHED_METAL_OSX_THEME_NAME) &&
               (!CommonUtils.isPantherOrAbove() ||
                !CommonUtils.isJava14OrLater()))
                return false;
                
            // only allow the windows theme if we're on windows.
            if(!CommonUtils.isWindows() &&
              name.equals(ThemeSettings.WINDOWS_LAF_THEME_NAME))
                return false;
            
            // only show pro theme if we're on pro.
            if(!CommonUtils.isPro() &&
               name.equals(ThemeSettings.PRO_THEME_NAME))
                return false;
                
            // only show GTK theme on linux with 1.5
            if(name.equals(ThemeSettings.GTK_LAF_THEME_NAME) && 
               (!CommonUtils.isLinux() ||
                !CommonUtils.isJava15OrLater()))
                return false;
                
            // everything's okay -- allow it.                
            return true;
        }
    }
    
}
