package com.limegroup.gnutella.gui.library;

import java.awt.Color;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.util.ArrayList;
import java.util.Enumeration;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import com.limegroup.gnutella.FileManager;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.playlist.PlaylistMediator;
import com.limegroup.gnutella.gui.tables.DragManager;
import com.limegroup.gnutella.gui.tables.FileTransfer;
import com.limegroup.gnutella.gui.themes.ThemeFileHandler;
import com.limegroup.gnutella.settings.FileSetting;
import com.limegroup.gnutella.settings.QuestionsHandler;
import com.limegroup.gnutella.settings.SharingSettings;

/**
 * This class forms a wrapper around the tree that controls 
 * navigation between shared folders.  It constructs the tree and
 * supplies access to it.  It also controls tree directory selection,
 * deletion, etc.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class LibraryTree {

	/**
	 * Constant name for the incomplete folder
	 */
	private final String INCOMPLETE_FILE_NAME = "Incomplete";

	/**
	 * Constant handle to the library mediator class.
	 */
	private final LibraryMediator LIBRARY_MEDIATOR = 
		LibraryMediator.instance();
	
	/**
	 * Constant for the <tt>JTree</tt>.
	 */
	private final JTree TREE = new JTree();

	/**
	 * Constant for the root node of the tree.
	 */
	private final LibraryTreeNode ROOT_NODE = 
		new LibraryTreeNode();

	/**
	 * Constant for the tree model.
	 */
	private final DefaultTreeModel TREE_MODEL =
		new DefaultTreeModel(ROOT_NODE);
		
    /**
     * The incomplete node.
     */
    private LibraryTreeNode incompleteNode;
    
    /**
     * The saved node.
     */
    private LibraryTreeNode savedNode;

	/**
	 * Constant for the popup menu.
	 */
	private final JPopupMenu DIRECTORY_POPUP = new JPopupMenu();


	/**
	 * Constructs the tree and its primary listeners,visualization options, 
	 * editors, etc. 
	 */
	LibraryTree() {
	
		TREE.setModel(TREE_MODEL);
		TREE.setRootVisible(false);
		TREE.getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
		TREE.setEditable(false);
		TREE.setInvokesStopCellEditing(true);
		TREE.setShowsRootHandles(true);	
		TREE.putClientProperty("JTree.lineStyle", "None");	

		makePopupMenu();
		TREE.addTreeSelectionListener(new LibraryTreeSelectionListener());

		// add the incomplete directory to the tree
		addIncompleteDirectory();
		addSaveDirectory();
		updateTheme();
		
		DragManager.install(TREE);
	}

	// inherit doc comment
	public void updateTheme() {
		Color tableColor = ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue();
		TREE.setBackground(tableColor);	    
	    DefaultTreeCellRenderer rnd = new DefaultTreeCellRenderer();
	    rnd.setOpaque(false);
	    TREE.setCellRenderer(rnd);
	}


	/**
	 * Cancels all editing of shared directory folder names.
	 */
	void cancelEditing() {
		if(TREE.isEditing()) {
			TREE.getCellEditor().cancelCellEditing();
		}        
	}

	/**
	 * Adds the visual representation of this folder to the library.
	 *
	 * @param dir    the <tt>File</tt> instance denoting the abstract
	 *               pathname of the new shared directory to add to the 
	 *               library. 
	 *
	 * @param parent the <tt>File</tt> instance denoting the abstract
	 *               pathname of the parent of the new shared directory to 
	 *               add to the library. 
	 */
	void addSharedDirectory(final File dir, final File parent) {
		LibraryTreeNode curNode = null;
		
		// the node we should reload in the tree model
		LibraryTreeNode nodeToLoad = null;

		// the holder for a potential new current node
		AbstractFileHolder holder = null;

		// if this is a "root" shared directory
		if(parent == null) {
			if(!ROOT_NODE.isChild(dir)) {
				holder = new InternalNodeFileHolder(dir);
				curNode = new LibraryTreeNode(holder);

				// insert this node one behind the end, as we
				// have already inserted the incomplete & save
				// directory, and that should go last
				TREE_MODEL.insertNodeInto(curNode, ROOT_NODE,
										  ROOT_NODE.getChildCount()-2);
				nodeToLoad = ROOT_NODE;
			}
		}
		
		// otherwise "dir" is a subdirectory of a directory
		// that is itself shared
		else {
			AbstractFileHolder curHolder = null;
			Enumeration elems = ROOT_NODE.breadthFirstEnumeration();
			if(elems.hasMoreElements()) {
				while(elems.hasMoreElements()) {
					curNode = (LibraryTreeNode)elems.nextElement();
					curHolder = curNode.getFileHolder();
					if(curHolder.matchesFile(parent)) {
						if(!curNode.isChild(dir)) {
							holder = new InternalNodeFileHolder(dir);
							LibraryTreeNode newNode = new LibraryTreeNode(holder);
							TREE_MODEL.insertNodeInto(newNode, curNode,
													  curNode.getChildCount());
							nodeToLoad = curNode;
						}
						break;
					}
				}
			}
			// handle the case that should never occur --
			// where a subdirectory of a "root" shared
			// directory is added before its root.
			else {
				InternalNodeFileHolder newHolder = 
				    new InternalNodeFileHolder(dir);
				curNode = new LibraryTreeNode(newHolder);
				TREE_MODEL.insertNodeInto(curNode, ROOT_NODE,
										  ROOT_NODE.getChildCount()-2);
				nodeToLoad = ROOT_NODE;
			}
		}

        // make sure we're not editing the shared file names.
		cancelEditing();

        int[] selected = TREE.getSelectionRows();
		TREE_MODEL.reload(nodeToLoad);
		
        //maintain the selected row.
        //the else is needed for the case of two shared folders..
        //calling TREE_MODEL.reload() will unselect your selection
	    if ( selected == null || selected.length == 0 )
	        setSelectionRow(0);
	    else
	        setSelectionRow(selected[0]);
	}

    /**
     * Adds files to the playlist recursively.
     */
    void addPlayListEntries() {
		LibraryTreeNode selectedNode = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();

		// return if nothing is selected
		if(selectedNode == null) return;
		if(incompleteDirectoryIsSelected())	return;

		final File currFile = selectedNode.getFileHolder().getFile();
		GUIMediator.instance().schedule(new Runnable() {
	        public void run() {
                String[] filter = {"mp3", "ogg"};
                File[] filesToAdd =
                    FileManager.getFilesRecursive(currFile, filter);
                PlaylistMediator pm = GUIMediator.getPlayList();
                if(pm != null)
                    pm.addFilesToPlaylist(filesToAdd);
            }
        });
    }

	/**
	 * Returns the file object associated with the mousePoint
	 * parameter if the mousePoint is over a valid file.
	 *
	 * @return a <tt>File</tt> instance contained in the 
	 *         node wrapper object, or <tt>null</tt> if the wrapped
	 *         <tt>File</tt> object is not a directory or is null
	 */
	File getFileForPoint(Point mousePoint) {
		TreePath path = TREE.getPathForLocation(mousePoint.x,mousePoint.y);
		if(path == null) return null;

		LibraryTreeNode node = (LibraryTreeNode)path.getLastPathComponent();		
		AbstractFileHolder fh = node.getFileHolder();
		File file = fh.getFile();
		if(file.isDirectory())
			return file;
		else
			return null;
	}

	/**
	 * Returns a boolean indicating whether or not the current mouse drop 
	 * event is dropping to the incomplete folder.
	 *
	 * @param mousePoint the <tt>Point</tt> instance representing the
	 *                   location of the mouse release
	 *
     * @return  <tt>true</tt> if the mouse was released on the Incomplete
	 *          folder, <tt>false</tt> otherwise 
	 */
	boolean droppingToIncompleteFolder(Point mousePoint) {
	    File f = getFileForPoint(mousePoint);
	    if( f == null)
	        return false;
        return f.getName().equals(INCOMPLETE_FILE_NAME);
	}

	/** 
	 * Returns the File object associated with the currently selected 
	 * directory.
	 *
	 * @return the currently selected directory in the library, or 
	 *  <tt>null</tt> if no directory is selected
	 */ 
	File getSelectedDirectory() {
		LibraryTreeNode selectedNode = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();

		// return if nothing is selected
		if(selectedNode == null) return null;

		return selectedNode.getFileHolder().getFile();
	}	

	/** 
	 * Returns the top-level directories as an array of <tt>File</tt> objects
	 * for updating the shared directories in the <tt>SettingsManager</tt>.
	 *
	 * @return the array of top-level directories as <tt>File</tt> objects
	 */
	File[] getSharedDirectories() {
		LibraryTreeNode curNode; 

		int length = ROOT_NODE.getChildCount();
		ArrayList newFiles = new ArrayList();
		for(int i=0; i<length; i++) {
			curNode = (LibraryTreeNode)ROOT_NODE.getChildAt(i);
			AbstractFileHolder fh = curNode.getFileHolder();
			if(!isIncompleteDirectory(curNode) &&
			   !isSavedDirectory(curNode)) {
				File f = fh.getFile();
				if(f.isDirectory()) {		
					newFiles.add(f);
				}
			}
		}
		File[] files = new File[newFiles.size()];
		for(int r=0; r<newFiles.size(); r++) {
			files[r] = (File)newFiles.get(r);
		}
		return files;
	}

	/** 
	 * Sets the selected row in the tree. 
	 *
	 * @param row the row to select
	 */
	void setSelectionRow(int row) {
		TREE.setSelectionRow(row);
	}

	/**
	 * Returns a boolean specifying whether or not the File "parent" 
	 * parameter is the same as the File object associated with the 
	 * currently selected row.
	 *
	 * @param parent the <tt>File</tt> instance associated with the
	 *               parent of the node in question
	 * @return <tt>true</tt> if the <tt>parent</tt> argument is the 
	 *         currently selected <tt>File</tt> instance, <tt>false</tt>
	 *         otherwise
	 */
	boolean parentIsSelected(final File parent) {
		LibraryTreeNode node = (LibraryTreeNode)TREE.getLastSelectedPathComponent();
		if(node == null) return false;
		
		return node.getFileHolder().matchesFile(parent);		
	}

	/** 
	 * Removes all shared directories from the visual display.
	 */ 
	void clear() {
		ROOT_NODE.removeAllChildren();	
		addIncompleteDirectory();
		addSaveDirectory();
	}

	/**
	 * Stops sharing the selected folder in the library if there is a 
	 * folder selected, if the folder is not the save folder, or if the
	 * folder is not a subdirectory of a "root" shared folder.
	 */
	void unshareLibraryFolder() {
		LibraryTreeNode node = 
		    (LibraryTreeNode)TREE.getLastSelectedPathComponent();
		if(node == null) return;

		if(incompleteDirectoryIsSelected()) {
			showIncompleteFolderMessage("delete");
		}
		else if(!node.getLibraryTreeNodeParent().isRoot()) {
			GUIMediator.showMessage("MESSAGE_CANNOT_UNSHARE_SUBDIRS");
		}						
		else {
			String msgKey = "MESSAGE_CONFIRM_UNSHARE_DIRECTORY";				
			int response = GUIMediator.showYesNoMessage(
			    msgKey, QuestionsHandler.UNSHARE_DIRECTORY);
			if(response != GUIMediator.YES_OPTION) return;	
			TREE_MODEL.removeNodeFromParent(node); 
			LIBRARY_MEDIATOR.handleRootSharedDirectoryChange();
		}
	}


	/**
	 * Returns whether or not the incomplete directory is
	 * selected in the tree. 
	 *
	 * @return <tt>true</tt> if the incomplete directory is selected,
	 *  <tt>false</tt> otherwise
	 */
	boolean incompleteDirectoryIsSelected() {
		LibraryTreeNode node = (LibraryTreeNode)TREE.getLastSelectedPathComponent();
		if(node == null) return false;

		return isIncompleteDirectory(node);
	}
	
	/**
	 * Returns whether or not the saved directory is selected
	 * in the tree.
	 */
	boolean savedDirectoryIsSelected() {
	    LibraryTreeNode node = (LibraryTreeNode)TREE.getLastSelectedPathComponent();
	    if(node == null) return false;
	    
	    return isSavedDirectory(node);
	}

    /** 
	 * Accessor for the <tt>JTree</tt> that this class wraps. 
	 *
	 * @return the <tt>JTree</tt> instance used by the library.
	 */
    JTree getTree() {
        return TREE;
    }

	/**
	 * Determines whether the LibraryTreeNode parameter is the 
	 * holder for the incomplete folder.
	 * 
	 * @param holder the <tt>LibraryTreeNode</tt> class to check
	 *  for whether or not it is the incomplete directory
	 * @return <tt>true</tt> if it does contain the incomplete
	 *  directory, <tt>false</tt> otherwise
	 */
	private boolean isIncompleteDirectory(LibraryTreeNode node) {
	    return node == incompleteNode;
	}
	
	/**
	 * Determines whether the LibraryTreeNode parameter is the 
	 * holder for the saved folder.
	 * 
	 * @param holder the <tt>LibraryTreeNode</tt> class to check
	 *  for whether or not it is the saved directory
	 * @return <tt>true</tt> if it does contain the saved
	 *  directory, <tt>false</tt> otherwise
	 */
	private boolean isSavedDirectory(LibraryTreeNode node) {
	    return node == savedNode;
	}

	/**
	 * Adds the incomplete directory (but not any subdirectories
	 * the incomplete directory may have) to the tree.
	 */
	private void addIncompleteDirectory() {
		AbstractFileHolder fh = new SettingFileHolder(
		    SharingSettings.INCOMPLETE_DIRECTORY,
		    GUIMediator.getStringResource("LIBRARY_TREE_INCOMPLETE_DIRECTORY"));
		incompleteNode = new LibraryTreeNode(fh, false);
		TREE_MODEL.insertNodeInto(incompleteNode, ROOT_NODE, 0);
		TREE_MODEL.reload(ROOT_NODE);
	}
	
	/**
	 * Adds the save directory (but not any subdirectories) to the tree.
	 */
	private void addSaveDirectory() {
		AbstractFileHolder fh = new SettingFileHolder(
		    SharingSettings.DIRECTORY_FOR_SAVING_FILES, 
		    GUIMediator.getStringResource("LIBRARY_TREE_SAVED_DIRECTORY"));
	    savedNode = new LibraryTreeNode(fh, false);
	    TREE_MODEL.insertNodeInto(savedNode, ROOT_NODE, 0);
	    TREE_MODEL.reload(ROOT_NODE);
	}

	/**
	 * Shows a message indicating that a specific action cannot
	 * be performed on the incomplete directory (such as changing 
	 * its name).
	 *
	 * @param action the error that occurred
	 */
	private void showIncompleteFolderMessage(String action) {
		String key1 = "MESSAGE_INCOMPLETE_DIRECTORY_START";
		String key2 = "MESSAGE_INCOMPLETE_DIRECTORY_END";
		GUIMediator.showError(key1, action, key2);
	}

	/** 
	 * Displays an error deleting the folder whose name is passed
	 * in as a parameter.
	 *
	 * @param folderName the name of the folder the encountered an error
	 *  during deletion.
	 */
	private void showFolderDeletionError(String folderName) {
		final String key1 = "MESSAGE_UNABLE_TO_DELETE_DIRECTORY_START";
		final String key2 = "MESSAGE_UNABLE_TO_CHANGE_DIRECTORY_END";
		final String msg = "'"+folderName+"'.";
		GUIMediator.showError(key1, msg, key2);
	}

	/**
	 * Handles a change to the LibraryTreeNode passed in as a
	 * parameter, either calling LibraryView for a "full" reload
	 * of the node and SettingsManager values when there is a 
	 * change to one of the "root" shared directories, and simply
	 * reloading the settings in the FileManager otherwise.
	 *
	 * @param node the LibraryTreeNode whose directory has changed.
	 */
	private void handleDirectoryChange(LibraryTreeNode node) {
		if(node.getLibraryTreeNodeParent().isRoot()) {
			// make sure the shared directories in
			// the settings manager and file
			// manager match the root (shared)
			// directories displayed in the tree						
			LIBRARY_MEDIATOR.handleRootSharedDirectoryChange();
		}
		else {
			// we don't need to reload SettingsManager
			// since the names of the "root" shared
			// directories have not changed, so simply 
			// reload the FileManager and clear the 
			// library table.
			LIBRARY_MEDIATOR.refresh();
		}
	}

	/**
	 * Constructs the popup menu that appears in the tree on
	 * a right mouse click.
	 */
	private void makePopupMenu() {
		JMenuItem unshareItem = 
		    new JMenuItem( 
                GUIMediator.getStringResource("LIBRARY_TREE_UNSHARE_FOLDER_LABEL"));
        JMenuItem addPlayListItem = 
		    new JMenuItem(
                GUIMediator.getStringResource("LIBRARY_TREE_TO_PLAYLIST_FOLDER_LABEL"));

		unshareItem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent ae) {
				unshareLibraryFolder();
			}
		});
	   
        addPlayListItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ae) {
				addPlayListEntries();
            }
        });

		DIRECTORY_POPUP.add(unshareItem);
		if(GUIMediator.isPlaylistVisible())
            DIRECTORY_POPUP.add(addPlayListItem);
		MouseListener ml = new MouseAdapter() {
  			public void checkPopUp(MouseEvent e) {
				int row = TREE.getRowForLocation(e.getX(), e.getY());
				if(row == -1) return;
				
				TREE.setSelectionRow(row);
				if(e.isPopupTrigger() || SwingUtilities.isRightMouseButton(e)) {
				   // only allow the unsharing item if it wasn't incomplete or saved
				   boolean allowUnshare = !incompleteDirectoryIsSelected() &&
				                          !savedDirectoryIsSelected();
				   DIRECTORY_POPUP.getComponent(0).setEnabled(allowUnshare);
				   DIRECTORY_POPUP.show(TREE,e.getX(),e.getY());
		        }			
  			}
  			public void mouseClicked(MouseEvent e) {
				checkPopUp(e);
			}
  			public void mousePressed(MouseEvent e) {
				checkPopUp(e);
			}
  		};
  		TREE.addMouseListener(ml);		
	}


	/**
	 * Selection listener that changes the files displayed in the table
	 * if the user chooses a new directory in the tree.
	 */
	private class LibraryTreeSelectionListener implements TreeSelectionListener {
		public void valueChanged(TreeSelectionEvent e) {	
			LibraryTreeNode node = (LibraryTreeNode)
			TREE.getLastSelectedPathComponent();

			// return if we cannot get the selected node
			if(node == null) return;
			
			if(!incompleteDirectoryIsSelected() &&
			   !savedDirectoryIsSelected()) {
				LIBRARY_MEDIATOR.updateTableFiles(
				    node.getFileHolder().getFile());
			} else {
				LIBRARY_MEDIATOR.refresh();
			}			
		}
	}

	/**
	 * Private class that extends a DefaultMutableTreeNode. Using
	 * this class ensures that the "UserObjects" associated with
	 * the tree nodes will always be File objects.
	 */
	private final class LibraryTreeNode extends DefaultMutableTreeNode
	                                    implements FileTransfer {
		private AbstractFileHolder _holder;
		private final boolean hasParents;

		private LibraryTreeNode() {
			_holder = new RootNodeFileHolder();
			hasParents = true;
		}

		private LibraryTreeNode(AbstractFileHolder holder) {
		    this(holder, true);
		}
		
		private LibraryTreeNode(AbstractFileHolder holder, boolean hasParents) {
			super(holder);
			if(holder == null)
				throw new NullPointerException("null holder in LibraryTreeNode");
			_holder = holder;
			this.hasParents = hasParents;
        }
		
		/**
		 * Returns the file.
		 */
		public File getFile() {
		    return _holder.getFile();
		}

		/** 
		 * Returns the <tt>AbstractFileHolder</tt> object contained in this 
		 * node.
		 *
		 * @return the <tt>AbstractFileHolder</tt> object contained in this 
		 * node
		 */
		private AbstractFileHolder getFileHolder() {
			return _holder;
		}
		
		/**
		 * Determines whether or not the File parameter is the 
		 * <tt>File</tt> contained in the FileHolder user object 
		 * of any of this node's children.
		 *
		 * @param file The <tt>File</tt> instance to check against
		 *             this node's children to see if it is a child.
		 *
		 * @return <tt>true</tt> if and only if the file parameter
		 *         is equal to a <tt>File</tt> object in one of 
		 *         this node's children.
		 */
		private boolean isChild(File file) {
			int count = getChildCount();			
			for(int i=0; i<count; i++) {
				LibraryTreeNode curNode = (LibraryTreeNode)getChildAt(i);
				if(!curNode.hasParents)
				    return false;
				File curFile = curNode.getFileHolder().getFile();
				if(curFile.equals(file))
					return true;
			}
			return false;
		}

		/**
		 * Overrides the setUserObject method of the 
		 * DefaultMutableTreeNode superclass to ensure that
		 * the user object must be a FileHolder and to reset
		 * the "HOLDER" variable to the new holder.
		 *
		 * @param holder the AbstractFileHolder instance to set as
		 *  the user object for this node.
		 */
		public void setUserObject(Object holder) {			          
            
            // ignore attempts to set this to null..
			if(holder == null) return;
            if(!(holder instanceof AbstractFileHolder)) {
                throw new IllegalArgumentException("unexpected object: "+
                                                   holder);
            }
			super.setUserObject(holder);
			_holder = (AbstractFileHolder)holder;
		}

		/**
		 * Replaces the getParent() method so that we handle all of the
		 * casting internally for convenience.  
		 *
		 * @return the <tt>LibraryTreeNode</tt> parent for this node, or
		 *  <tt>null</tt> if this node has no parent
		 */
		private LibraryTreeNode getLibraryTreeNodeParent() {
			TreeNode parentNode = super.getParent();
			if(parentNode == null) return null;
			return (LibraryTreeNode)parentNode;
		}	  
	}

	/**
	 * A private abstract class that enables the use of 
	 * specialized root and internal node subclasses.
	 */
	private abstract class AbstractFileHolder {
		protected abstract File getFile();
		protected abstract boolean matchesFile(File file);
	}

	/**
	 * This class wraps a File object for a node in the tree.
	 * it redefines the "toString" method so that the tree will
	 * display the name of the associated File object as the 
	 * name of the node in the tree.  this class also handles 
	 * determining equality among File objects for directory
	 * insertion.
	 */
	private class InternalNodeFileHolder extends AbstractFileHolder {	
		private final File FILE;

		/**
		 * Constructs a new <tt>InternalNodeFileHolder</tt> with the 
		 * given <tt>File</tt> instance.
		 *
		 * @param file the <tt>File</tt> instance for this holder
		 */
		InternalNodeFileHolder(File file) {
		    FILE = file;
		}

		/**
		 * Returns the File object that this FileHolder holds.
		 */
		protected File getFile() {
			return FILE;
		}
		
		/**
		 * Returns a boolean indicating whether the "file"
		 * parameter is the same as the File object for 
		 * this file holder.
		 */
		protected boolean matchesFile(File file) {
			return FILE.equals(file);
		}


		/** 
		 * This is the method that JTree calls to get the
		 * display name for each node in the tree.  
		 */
		public String toString() {
			return FILE.getName();
		}
	}

	/**
	 * Root node class the extends AbstractFileHolder
	 */
	private class RootNodeFileHolder extends AbstractFileHolder {
		protected File getFile() {
			return null;
		}		
		protected boolean matchesFile(File file) {
			return false;
		}
		public String toString() {
			return "";
		}
	}
	
	/**
	 * File holder for a setting (a file that can change)
	 */
	private class SettingFileHolder extends AbstractFileHolder {
	    private final FileSetting SETTING;
	    private final String DISPLAY;
	    SettingFileHolder(FileSetting setting, String name) {
	        SETTING = setting;
	        DISPLAY = name;
	    }
	    protected File getFile() {
	        return SETTING.getValue();
	    }
	    protected boolean matchesFile(File file) {
	        return getFile().equals(file);
	    }
	    public String toString() {
	        return DISPLAY;
	    }
	}

}


