package com.limegroup.gnutella.gui.library;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.Vector;

import javax.swing.JComponent;
import javax.swing.SwingUtilities;

import com.limegroup.gnutella.FileDesc;
import com.limegroup.gnutella.FileManager;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.gui.FileChooserHandler;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.themes.ThemeMediator;
import com.limegroup.gnutella.gui.themes.ThemeObserver;
import com.limegroup.gnutella.settings.SharingSettings;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.FileUtils;


/**
 * This class functions as an initializer for all of the elements
 * of the library and as a mediator between library objects.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
public final class LibraryMediator implements ThemeObserver {

	/**
	 * Singleton instance of this class.
	 */
	private static final LibraryMediator INSTANCE = new LibraryMediator();
    
    /**
     * Constant handle to the <tt>LibraryTree</tt> library controller.
     */
    private static final LibraryTree  LIBRARY_TREE = new LibraryTree();
    
    /**
     * Constant handle to the <tt>LibraryTable</tt> that displays the files
     * in a given directory.
     */
    private static final LibraryTableMediator LIBRARY_TABLE =
        LibraryTableMediator.instance();


    /**
     * Constant handle to the class the constructs the library components.
     */
	private static final LibraryConstructor LIBRARY_CONSTRUCTOR =
		new LibraryConstructor(LIBRARY_TABLE, LIBRARY_TREE);
    
    /**
     * Constant handle to the file update handler.
     */
    private final HandleFileUpdate FILE_UPDATER = new HandleFileUpdate();
    

	/**
	 * Instance accessor following singleton.  Returns the 
	 * <tt>LibraryView</tt> instance.
	 *
	 * @return the <tt>LibraryView</tt> instance
	 */
	public static LibraryMediator instance() {
		return INSTANCE;
	}

    /** 
	 * Constructs a new <tt>LibraryMediator</tt> instance to manage calls
	 * between library components.
	 */
    private LibraryMediator() {		
		GUIMediator.setSplashScreenString(
		    GUIMediator.getStringResource("SPLASH_STATUS_LIBRARY_WINDOW"));
		ThemeMediator.addThemeObserver(this);
    }

	// inherit doc comment
	public void updateTheme() {
		LIBRARY_CONSTRUCTOR.updateTheme();
		LIBRARY_TREE.updateTheme();
	}

	/**
	 * Returns the <tt>JComponent</tt> that contains all of the elements of
	 * the library.
	 *
	 * @return the <tt>JComponent</tt> that contains all of the elements of
	 * the library.
	 */
	public JComponent getComponent() {
		return LIBRARY_CONSTRUCTOR.getComponent();
	}
	
    /**
	 * Tells the library to launch the application associated with the 
	 * selected row in the library. 
	 */
    public void launch() {
		LIBRARY_TABLE.launch();
    }
    
    /**
	 * Deletes the currently selected rows in the table. 
	 */
    public void deleteLibraryFile() {
        LIBRARY_TABLE.removeSelection();
    }
        
	/**
	 * Removes the gui elements of the library tree and table.
	 */
	public void clearLibrary() {
		LIBRARY_TREE.clear();
		LIBRARY_TABLE.clearTable();
	}

	/**
	 * Reloads all file and directories in the library.
	 */
    public void refresh() {
		LIBRARY_TABLE.clearTable();
		if(LIBRARY_TREE.incompleteDirectoryIsSelected()) {
			showIncompleteFiles();
		} else if(LIBRARY_TREE.savedDirectoryIsSelected()) {
		    showSavedFile();
		} else {
		    if ( LIBRARY_TREE.getSelectedDirectory() != null ) {
		        loadShareableFiles( LIBRARY_TREE.getSelectedDirectory() );
		        LIBRARY_TABLE.forceResort();
		    }
            FileManager fm = RouterService.getFileManager();
            fm.loadSettings(false);		   
		}
    }
    
    /**
     * Quickly refreshes the library -- only done if incomplete
     * or saved is selected.
     */
     public void quickRefresh() {
        if ( LIBRARY_TREE.incompleteDirectoryIsSelected() )
            refresh();
        else if ( LIBRARY_TREE.savedDirectoryIsSelected() )
            refresh();
    }

	/**
	 * Cancels all editing of fields in the tree and table.
	 */
	public void cancelEditing() {
		LIBRARY_TREE.cancelEditing();
	}

	/** 
	 * Launches explorer on PC in selected Shared directory
	 *
	 */
	public static void launchExplorer()
	{
		File exploreDir = null;
		if ( LIBRARY_TREE.getSelectedDirectory() != null ) 
		    exploreDir = LIBRARY_TREE.getSelectedDirectory();

		if ( exploreDir == null ) return;

		try {
		    String explorePath = exploreDir.getCanonicalPath();	   
			String cmdStr = "";
			if(CommonUtils.isWindows())
			    cmdStr = "explorer"; 
			else if (CommonUtils.isMacOSX()) 	
			    cmdStr = "open"; 
			Runtime.getRuntime().exec(new String[] {cmdStr, explorePath});
		} catch(SecurityException se) {
		} catch(IOException se) {
		}
	}

    /** 
	 * Displays a file chooser for selecting a new folder to share and 
	 * adds that new folder to the settings and FileManager.
	 */
    public void addSharedLibraryFolder() {
		File dir = FileChooserHandler.getInputDirectory();

		if(dir == null || !dir.isDirectory() || !dir.canRead()) {
			GUIMediator.showError("ERROR_INVALID_SHARED_DIRECTORY");
			return;
		}
		
		try {
			SharingSettings.addDirectory(dir);
			// Rescan the directores into the FileManager.  
			// The loadSettings method is non-blocking, 
			// so this is safe to call here.  
			RouterService.getFileManager().loadSettings(false);
		} catch(IOException ioe) {				
		}			
    }

    /** 
	 * Refreshes the shared directories in the settings manager as well 
	 * as the files in the file manager based on the directories in the
	 * library tree.
	 */
    public void handleRootSharedDirectoryChange() {
        final File[] sharedDirectories = LIBRARY_TREE.getSharedDirectories();
        SharingSettings.DIRECTORIES_TO_SHARE.
            setValue(sharedDirectories);
        RouterService.getFileManager().loadSettings(true);
    }

	/**
	 * Adds a shared directory to the library.
	 * 
	 * @param dir     a <tt>File</tt> instance denoting the abstract
	 *                pathname of the shared directory to add
	 *
	 * @param parent  a <tt>File</tt> instance denoting the abstract
	 *                pathname of the shared directory's parent directory
	 */
	public void addSharedDirectory(final File dir, final File parent) {
		LIBRARY_TREE.addSharedDirectory(dir, parent);
	}

	/**
	 * Adds a shared file to the library if its parent directory is
	 * currently selected in the library tree.
	 * 
	 * @param file    a <tt>FileDesc</tt> instance denoting the abstract
	 *                pathname of the shared file to add
	 *
	 * @param parent  a <tt>File</tt> instance denoting the abstract
	 *                pathname of the shared file's parent directory
	 *
	 */
	public void addSharedFile(final FileDesc file, final File parent) {
		if(LIBRARY_TREE.parentIsSelected(parent))
			LIBRARY_TABLE.add(file);
	}
	
	/**
	 * Update the this file's statistic
	 */
	public void updateSharedFile(final File file) {
	    File selDir = LIBRARY_TREE.getSelectedDirectory();
	    // if the library table is visible, and
	    // if the selected directory is null
	    // or if we the file exists in a directory
	    // other than the one we selected, then there
	    // is no need to update.
	    // the user will see the newest stats when he/she 
	    // selects the directory.
		if( selDir != null && 
		    selDir.equals(FileUtils.getParentFile(file)) &&
		    LIBRARY_TABLE.getTable().isShowing() ) {
		    // pass the update off to the file updater
		    // this way, only one Runnable is ever created,
		    // instead of allocating a new one every single time
		    // a query is hit.
		    // Very useful for large libraries and generic searches (ala: mp3)
		    FILE_UPDATER.addFileUpdate(file);
	    }
	}
	
	public void setAnnotateEnabled(boolean enabled) {
	    LIBRARY_TABLE.setAnnotateEnabled(enabled);
	}

    /** 
	 * Removesthe selected folder from the shared folder group.. 
	 */
    public void unshareLibraryFolder() {
        LIBRARY_TREE.unshareLibraryFolder();
    }

    /**
     * Adds a file to the playlist.
     */
    void addFileToPlayList(File toAdd) {
        GUIMediator.getPlayList().addFileToPlaylist(toAdd);
    }


    /**
     * Launches the specified file.
     */
    void launchAudio(File toPlay) {
        GUIMediator.instance().launchAudio(toPlay);
    }


    /** 
	 * Obtains the shared files for the given directory and updates the 
	 * table accordingly.
	 *
	 * @param selectedDir the currently selected directory in
	 *        the library
	 */
    void updateTableFiles(File selectedDir) {
		FileDesc[] files = 
		    RouterService.getSharedFileDescriptors(selectedDir);
		LIBRARY_TABLE.clearTable();
		
		// this is called when the non-incomplete directory is selected,
		LIBRARY_TABLE.setIncompleteSelected(false);
		
		// Add stuff unsorted and then sort afterwards so we don't
		// rewrite the hashmap after every add.
		if(files != null)
		    for( int i = 0; i < files.length; i++)
				LIBRARY_TABLE.addUnsorted(files[i]);
				
        //after the shared files are added, add the non-shared files.
        // any already shared files are ignored.
        //Small optimization:
        if ( RouterService.getNumPendingShared() > 0 ) {
            loadShareableFiles(selectedDir);
        }
        
        LIBRARY_TABLE.forceResort(); 
    }
    
    /**
     * Loads all sharable files in the specified directory into
     * the table.
     *
     * To maintain sorting, must force a resort after adding.
     */
    void loadShareableFiles(File dir) {
        File[] pending = FileUtils.listFiles( dir );
        if ( pending != null ) {
            for(int i = 0; i < pending.length; i++) {
                if(FileManager.isFileShareable(pending[i],
                                               pending[i].length())
                  )
                    LIBRARY_TABLE.addUnsorted(pending[i]);
            }
        } 
    }
    
    /**
     * Displays the file in the saved directory.
     */
    private void showSavedFile() {
        LIBRARY_TABLE.clearTable();
        File savedFile = SharingSettings.getSaveDirectory();
        if(savedFile == null)
            return;
        LIBRARY_TABLE.setIncompleteSelected(false);
        
        FileDesc[] sharedFiles = RouterService.getSharedFileDescriptors(savedFile);
        if(sharedFiles != null)
            for(int i = 0; i < sharedFiles.length; i++)
                LIBRARY_TABLE.addUnsorted(sharedFiles[i]);

		File[] files = savedFile.listFiles(new FileFilter() {
		    public boolean accept(File file) {
		        return file.isFile();
		    }
		});
		if(files != null)
			for( int i = 0 ; i < files.length; i++ )
                LIBRARY_TABLE.addUnsorted(files[i]);
                
        LIBRARY_TABLE.forceResort();
    }
        

	/**
	 * Displays the files in the incomplete directory.
	 */
	private void showIncompleteFiles() {
		LIBRARY_TABLE.clearTable();
  		File incFile = SharingSettings.INCOMPLETE_DIRECTORY.getValue();
  		if( incFile == null ) {
			// we cannot do anything if we could not get the directory,
			// so simply return
			return;
		}
		
		// this is called when the incomplete directory is selected,
		// so enable resume
		LIBRARY_TABLE.setIncompleteSelected(true);		
		
		// Ask for the File Descriptors of shared incomplete files ...
		FileDesc[] sharedFiles = RouterService.getIncompleteFileDescriptors();		
		// Add stuff unsorted and then sort afterwards so we don't
		// rewrite the hashmap after every add.
		if(sharedFiles != null)
		    for( int i = 0; i < sharedFiles.length; i++)
				LIBRARY_TABLE.addUnsorted(sharedFiles[i]);		
		
		
		File[] files = incFile.listFiles(new FileFilter() {
		    public boolean accept(File file) {
			    String name = file.getName();
			    return !file.isHidden() &&
			           !name.startsWith(".") &&
			           file.isFile() &&
			           !name.equals("downloads.dat") &&
			           !name.equals("downloads.bak");
            }
        });
		if(files != null)
			for( int i = 0 ; i < files.length; i++ )
                LIBRARY_TABLE.addUnsorted(files[i]);
		
		LIBRARY_TABLE.forceResort();
	}
	

    /** Returns true if this is showing the special incomplete directory,
     *  false if showing normal files. */
    public boolean incompleteDirectoryIsSelected() {
        return LIBRARY_TREE.incompleteDirectoryIsSelected();        
    }
    
    /**
     *  Class to handle updates to shared file stats
     *  without creating tons of runnables.
     *  Idea taken from HandleQueryString in VisualConnectionCallback
     */
    private static final class HandleFileUpdate implements Runnable 
    {
        private Vector  list;
        private boolean active;
    
        public HandleFileUpdate( ) {
            list   = new Vector();
            active = false;
        }
    
        public void addFileUpdate(File f) {
            list.addElement(f);
            if(active == false) {
                active = true;
                SwingUtilities.invokeLater(this);
            }
        }
    
        public void run() {
            try {
                File f;
                while (list.size() > 0) {
                    f = (File) list.firstElement();
                    list.removeElementAt(0);
    			    LIBRARY_TABLE.update(f);
                }
             } catch (IndexOutOfBoundsException e) {
        	    //this really should never happen, but
        	    //who really cares if we're not sharing it?
             } catch (Exception e) {
                //no other errors could happen, so if one does, something's wrong
                GUIMediator.showInternalError(e);
             }
             active = false;
        }
    }    
}
