package com.limegroup.gnutella.gui;

import com.limegroup.gnutella.licenses.License;
import com.limegroup.gnutella.licenses.VerificationListener;
import com.limegroup.gnutella.URN;
import com.limegroup.gnutella.gui.themes.ThemeFileHandler;
import com.limegroup.gnutella.gui.themes.ThemeObserver;

import org.apache.commons.httpclient.URI;

import javax.swing.JDialog;
import javax.swing.JPanel;
import javax.swing.JLabel;
import javax.swing.JComponent;
import javax.swing.JTextArea;
import javax.swing.JButton;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.UIManager;
import java.awt.Font;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Container;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 * A dialog displaying Creative Commons License information.
 */
public class LicenseWindow extends JDialog implements VerificationListener {    
    
    /** The URN that the license is expected to match. */
    private final URN URN;
    /** The License itself. */
    private final License LICENSE;
    /** The panel where the ongoing status of the license verification is shown */
    private final JPanel DETAILS;
    /** A listener who wants to receive notification about the license being verified */
    private final VerificationListener LISTENER;
    
    /**
     * Constructs a new LicenseWindow.
     * @param license the License being displayed
     * @param urn the URN the license is validating against
     * @param listener a VerificationListener this license can forward
     *                 licenseVerified events to.
     */
    public LicenseWindow(License license, URN urn, VerificationListener listener) {
        super(GUIMediator.getAppFrame());
        URN = urn;
        LICENSE = license;
        DETAILS = new JPanel(new GridBagLayout());
        LISTENER = listener;
        
        setModal(false);
        setResizable(false);
        setTitle(GUIMediator.getStringResource("LICENSE_CC_TITLE"));
        setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        
        JComponent pane = (JComponent)getContentPane();
        GUIUtils.addHideAction(pane);
        pane.setPreferredSize(new Dimension(400, 180));            
        DETAILS.setPreferredSize(new Dimension(400, 150));

        getContentPane().setLayout(new GridBagLayout());
        constructDialog(getContentPane());
        validate();
        
        if(GUIMediator.isAppVisible())
            setLocationRelativeTo(GUIMediator.getAppFrame());
        else
            setLocation(GUIMediator.getScreenCenterPoint(this));        
    }
    
    /**
     * Notification that a license has been verified.
     *
     * Rebuilds the details panel to match the new status.
     */
    public void licenseVerified(License license) {
        if(license == LICENSE) {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    buildDetails();
                }
            });
        }

        if(LISTENER != null)
            LISTENER.licenseVerified(license);
    }
    
    
    /**
     * Builds the details panel.
     *
     * This will show either a message that
     * 1) The license is currently being located.
     * 2) The license is not yet verified, asking the user to verify.
     * 3) The license could not be validated (with guess-work links)
     * 4) The license is valid (with the details, and links)
     */
    private void buildDetails() {
        DETAILS.removeAll();
        GridBagLayout bag = (GridBagLayout)DETAILS.getLayout();
        GridBagConstraints c = new GridBagConstraints();
        
        if(LICENSE.isVerifying()) {
            
            // TODO:  Add a pretty animation.
            JTextArea text = newTextArea(GUIMediator.getStringResource("LICENSE_CC_LOCATING"));
            c.anchor = GridBagConstraints.CENTER;
            c.fill = GridBagConstraints.BOTH;
            c.weightx = 1;
            c.weighty = 1;
            DETAILS.add(text, c);
            
        } else if(!LICENSE.isVerified()) {
            
            JTextArea text = newTextArea(GUIMediator.getStringResource("LICENSE_CC_UNVERIFIED"));
            c.gridwidth = GridBagConstraints.REMAINDER;
            c.fill = GridBagConstraints.BOTH;
            c.weightx = 1;
            c.weighty = 1;
            c.anchor = GridBagConstraints.NORTHWEST;
            DETAILS.add(text, c);
            
            JButton button = new JButton(GUIMediator.getStringResource("LICENSE_CC_VERIFY"));
            button.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent ev) {
                    LICENSE.verify(LicenseWindow.this);
                    buildDetails();
                }
            });
            c.gridwidth = GridBagConstraints.REMAINDER;
            c.fill = GridBagConstraints.NONE;
            c.weightx = 0;
            c.weighty = 0;
            c.anchor = GridBagConstraints.SOUTHWEST;
            DETAILS.add(button, c);

        } else if(!LICENSE.isValid(URN)) {
            URI licenseURI = LICENSE.getLicenseURI();
            JComponent comp;

            comp = new JLabel(GUIMediator.getThemeImage("warning"));
            c.anchor = GridBagConstraints.NORTH;
            c.insets = new Insets(0, 0, 0, 5);
            DETAILS.add(comp, c);
            
            String invalidText = GUIMediator.getStringResource("LICENSE_CC_INVALID");
            if(licenseURI != null)
                invalidText += "  " + GUIMediator.getStringResource("LICENSE_CC_RETRY");
            comp = newTextArea(invalidText);
            c.gridwidth = GridBagConstraints.REMAINDER;
            c.anchor = GridBagConstraints.NORTHWEST;
            c.fill = GridBagConstraints.BOTH;
            c.weightx = 1;
            c.weighty = 1;
            c.insets = new Insets(0, 0, 0, 0);
            DETAILS.add(comp, c);
            
            c.gridwidth = 1;
            c.gridheight = 2;
            c.weightx = 0;
            c.weighty = 0;
            DETAILS.add(javax.swing.Box.createGlue(), c);
            

            JButton button = new JButton(GUIMediator.getStringResource("LICENSE_CC_VERIFY"));
            button.addActionListener(new ActionListener() {
                public void actionPerformed(ActionEvent ev) {
                    LICENSE.verify(LicenseWindow.this);
                    buildDetails();
                }
            });
            button.setVisible(licenseURI != null);
            c.gridwidth = GridBagConstraints.RELATIVE;
            c.gridheight = 2;
            c.fill = GridBagConstraints.NONE;
            c.anchor = GridBagConstraints.CENTER;
            c.insets = new Insets(0, 0, 0, 100);
            DETAILS.add(button, c);

            if(LICENSE.getLicenseDeed(URN) == null) {
                comp = new JLabel();
                comp.setVisible(false);
            } else {
                comp = new URLLabel(LICENSE.getLicenseDeed(URN), GUIMediator.getStringResource("LICENSE_CC_CLAIMED_DEED"));
            }
            c.gridwidth = GridBagConstraints.REMAINDER;
            c.gridheight = 1;
            c.anchor = GridBagConstraints.WEST;
            c.fill = GridBagConstraints.BOTH;
            c.weightx = 0;
            c.weighty = 0;
            c.insets = new Insets(0, 0, 3, 0);
            DETAILS.add(comp, c);
            
            if(licenseURI == null) {
                comp = new JLabel();
                comp.setVisible(false);
            } else {
                comp = new URLLabel(licenseURI, GUIMediator.getStringResource("LICENSE_CC_VERIFICATION"));
            }
            c.insets = new Insets(0, 0, 3, 0);
            DETAILS.add(comp, c);
            
        } else { // LICENSE.isValid()
            
            JComponent label = new JLabel(GUIMediator.getStringResource("LICENSE_CC_DETAILS"));
            Font f = label.getFont();
            label.setFont(new Font(f.getName(), Font.BOLD, f.getSize()));
            c.fill = GridBagConstraints.HORIZONTAL;
            c.gridwidth = GridBagConstraints.REMAINDER;
            c.anchor = GridBagConstraints.NORTHWEST;
            DETAILS.add(label, c);
            
            JTextArea text = newTextArea(LICENSE.getLicenseDescription(URN));
            c.gridwidth = GridBagConstraints.RELATIVE;
            c.gridheight = 2;
            c.weighty = 1;
            c.weightx = .7;
            c.anchor = GridBagConstraints.NORTHWEST;
            c.fill = GridBagConstraints.BOTH;
            DETAILS.add(text, c);
            
            if(LICENSE.getLicenseDeed(URN) != null) {
                label = new URLLabel(LICENSE.getLicenseDeed(URN), GUIMediator.getStringResource("LICENSE_CC_DEED"));
            } else {
                label = new JLabel();
                label.setVisible(false);
            }
            c.gridwidth = GridBagConstraints.REMAINDER;
            c.gridheight = 1;
            c.weighty = 0;
            c.weightx = .3;
            c.anchor = GridBagConstraints.NORTHWEST;
            c.fill = GridBagConstraints.BOTH;
            c.insets = new Insets(0, 0, 3, 0);
            DETAILS.add(label, c);
            
            label = new URLLabel(LICENSE.getLicenseURI(), GUIMediator.getStringResource("LICENSE_CC_VERIFICATION"));
            DETAILS.add(label, c);
        }
        
        validate();
        repaint();
    }
    
    /**
     * Constructs the dialog itself.
     */
    private void constructDialog(Container parent) {
        GridBagLayout bag = (GridBagLayout)parent.getLayout();
        GridBagConstraints c = new GridBagConstraints();
        
        JLabel img = new URLLabel("http://www.creativecommons.org", GUIMediator.getThemeImage("cc_window"));
        c.insets = new Insets(4, 4, 2, 0);
        parent.add(img, c);
        
        JComponent line = new Line(ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue());
        c.insets = new Insets(0, 2, 0, 2);
        c.gridwidth = GridBagConstraints.RELATIVE;
        c.fill = GridBagConstraints.VERTICAL;
        parent.add(line, c);
        
        JTextArea text = newTextArea(LICENSE.getLicense());
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.BOTH;
        c.insets = new Insets(4, 0, 2, 4);
        parent.add(text, c);
        
        line = new Line(ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue());
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.insets = new Insets(0, 0, 2, 0);
        parent.add(line, c);
        
        buildDetails();
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.fill = GridBagConstraints.BOTH;
        c.insets = new Insets(4, 4, 0, 4);
        c.weighty = 1;
        c.weightx = 1;
        parent.add(DETAILS, c);
        
        line = new Line(ThemeFileHandler.TABLE_BACKGROUND_COLOR.getValue());
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.gridheight = GridBagConstraints.RELATIVE;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.insets = new Insets(2, 0, 2, 0);
        c.ipady = 0;
        c.weighty = 0;
        c.weightx = 0;
        parent.add(line, c);
        
        JButton button = new JButton(GUIMediator.getStringResource("GENERAL_OK_BUTTON_LABEL"));
        button.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent ev) {
                LicenseWindow.this.dispose();
                LicenseWindow.this.setVisible(false);
           }
        });
        c.gridheight = GridBagConstraints.REMAINDER;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.insets = new Insets(0, 0, 4, 4);
        c.fill = GridBagConstraints.NONE;
        c.anchor = GridBagConstraints.EAST;
        parent.add(button, c);

        pack();
    }
    
    /**
     * Builds a new JTextArea with the appropriate values set.
     */
    private JTextArea newTextArea(String msg) {
        JTextArea text = new JTextArea();
        text.setLineWrap(true);
        text.setWrapStyleWord(true);
        text.setEditable(false);
        text.setFont(UIManager.getFont("Table.font"));
        JLabel label = new JLabel();
        text.setForeground(label.getForeground());
        text.setBackground(label.getBackground());
        text.setText(msg);
        return text;
    }
}