package bodybuilder.util;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import bodybuilder.exception.BodyBuilderException;

/**
 * オブジェクトユーティリティ
 */
public class ObjectUtils {

    /////////////////////////////////////////////////////////////////
    // object getter

    /**
     * オブジェクトを取得する。
     * 
     * @param type オブジェクトの型
     * @return オブジェクト
     */
    public static Object getObject(String type) {
        return getObject(null, type);
    }

    /**
     * オブジェクトを取得する。
     * 
     * @param value コンストラクタに渡す文字列
     * @param type オブジェクトの型
     * @return オブジェクト
     */
    public static Object getObject(String value, String type) {
        // 型が指定されていない、またはStringクラスの場合、文字列をそのまま返す。
        if (type == null || type.equals(String.class.getName())) {
            return value;
        }

        // プリミティブ型の場合はラッパーを返す。
        if (isPrimitive(type)) {
            return getPrimitiveWrapperInstance(value, type);
        }

        // オブジェクトを生成。
        if (value != null) {
            return getInstance(type, new Class[] { String.class },
                    new Object[] { value });
        } else {
            return getInstance(type, new Class[0], new Object[0]);
        }
    }

    /**
     * オブジェクトを取得する。
     * 
     * @param type オブジェクトの型
     * @return オブジェクト
     */
    public static Object getObject(Class type) {
        return getObject(null, type);
    }

    /**
     * オブジェクトを取得する。
     * 
     * @param value コンストラクタに渡す文字列
     * @param type オブジェクトの型
     * @return オブジェクト
     */
    public static Object getObject(String value, Class type) {
        // 型が指定されていない、またはStringクラスの場合、文字列をそのまま返す。
        if (type == null || type.equals(String.class.getName())) {
            return value;
        }

        // オブジェクトを生成。
        if (value != null) {
            return getInstance(type, new Class[] { String.class },
                    new Object[] { value });
        } else {
            return getInstance(type, new Class[0], new Object[0]);
        }
    }

    /**
     * プリミティブ型かどうかを取得する。
     * 
     * @param type 型
     * @return プリミティブ型の場合はtrue
     */
    public static boolean isPrimitive(String type) {
        boolean primitive = false;

        // いずれかの文字列と等しい場合はプリミティブ型。
        if ("bool".equals(type) || "boolean".equals(type)) {
            primitive = true;
        } else if ("byte".equals(type)) {
            primitive = true;
        } else if ("char".equals(type) || "character".equals(type)) {
            primitive = true;
        } else if ("short".equals(type)) {
            primitive = true;
        } else if ("int".equals(type) || "integer".equals(type)) {
            primitive = true;
        } else if ("long".equals(type)) {
            primitive = true;
        } else if ("float".equals(type)) {
            primitive = true;
        } else if ("double".equals(type)) {
            primitive = true;
        }

        return primitive;
    }

    /**
     * プリミティブラッパーを取得する。
     * 
     * @param value 値
     * @param type 型
     * @return プリミティブラッパー
     */
    public static Object getPrimitiveWrapperInstance(String value, String type) {
        Object obj = null;

        // ラッパーを取得する。
        if ("bool".equals(type) || "boolean".equals(type)) {
            obj = Boolean.valueOf(value);
        } else if ("byte".equals(type)) {
            obj = Byte.valueOf(value);
        } else if ("char".equals(type) || "character".equals(type)) {
            obj = new Character((char) Integer.parseInt(value));
        } else if ("short".equals(type)) {
            obj = Short.valueOf(value);
        } else if ("int".equals(type) || "integer".equals(type)) {
            obj = Integer.valueOf(value);
        } else if ("long".equals(type)) {
            obj = Long.valueOf(value);
        } else if ("float".equals(type)) {
            obj = Float.valueOf(value);
        } else if ("double".equals(type)) {
            obj = Double.valueOf(value);
        }

        return obj;
    }

    /**
     * インスタンスを取得する。
     * 
     * @param type クラス名
     * @param classes コンストラクタの引数の型
     * @param arguments コンストラクタの引数
     * @return インスタンス
     */
    public static Object getInstance(String type, Class[] classes,
            Object[] arguments) {
        Class clazz = getClass(type);
        return getInstance(clazz, classes, arguments);
    }

    /**
     * インスタンスを取得する。
     * 
     * @param clazz クラス
     * @param classes コンストラクタの引数の型
     * @param arguments コンストラクタの引数
     * @return インスタンス
     */
    public static Object getInstance(Class clazz, Class[] classes,
            Object[] arguments) {
        try {
            Constructor constructor = clazz.getDeclaredConstructor(classes);
            constructor.setAccessible(true);
            return constructor.newInstance(arguments);
        } catch (SecurityException e) {
            throw new BodyBuilderException(e);
        } catch (NoSuchMethodException e) {
            throw new BodyBuilderException(e);
        } catch (IllegalArgumentException e) {
            throw new BodyBuilderException(e);
        } catch (InstantiationException e) {
            throw new BodyBuilderException(e);
        } catch (IllegalAccessException e) {
            throw new BodyBuilderException(e);
        } catch (InvocationTargetException e) {
            throw new BodyBuilderException(e);
        }
    }

    /////////////////////////////////////////////////////////////////
    // class getter

    /**
     * クラスを取得する。
     * 
     * @param type クラス名
     * @return クラス
     */
    public static Class getClass(String type) {
        try {
            if (isPrimitive(type)) {
                return getPrimitiveClass(type);
            }

            return Class.forName(type);
        } catch (ClassNotFoundException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * プリミティブクラスを取得する。
     * 
     * @param type クラス名
     * @return プリミティブクラス
     */
    public static Class getPrimitiveClass(String type) {
        Class clazz = null;

        // プリミティブクラスを取得。
        if ("bool".equals(type) || "boolean".equals(type)) {
            clazz = boolean.class;
        } else if ("byte".equals(type)) {
            clazz = byte.class;
        } else if ("char".equals(type) || "character".equals(type)) {
            clazz = char.class;
        } else if ("short".equals(type)) {
            clazz = short.class;
        } else if ("int".equals(type) || "integer".equals(type)) {
            clazz = int.class;
        } else if ("long".equals(type)) {
            clazz = long.class;
        } else if ("float".equals(type)) {
            clazz = float.class;
        } else if ("double".equals(type)) {
            clazz = double.class;
        }

        return clazz;
    }

    /////////////////////////////////////////////////////////////////
    // object info utility

    /**
     * クラス名を取得する。
     * 
     * @param object オブジェクト
     * @param omit 省略する場合はtrue
     * @return クラス名
     */
    public static String getName(Object object, boolean omit) {
        if (object == null) {
            return "null";
        }

        return getName(object.getClass(), omit);
    }

    /**
     * クラス名を取得する。
     * 
     * @param clazz クラス
     * @param omit 省略する場合はtrue
     * @return クラス名
     */
    public static String getName(Class clazz, boolean omit) {
        String className = null;

        if (clazz.isArray()) {
            // 配列の場合
            className = clazz.getComponentType().getName() + "[]";
        } else {
            className = clazz.getName();
        }

        // クラス名を省略する場合、パッケージ名を削除する。
        if (omit) {
            className = omit(className);
        }

        if (Proxy.isProxyClass(clazz)) {
            StringBuffer proxyName = new StringBuffer();
            Class[] interfaces = clazz.getInterfaces();
            proxyName.append(className);
            proxyName.append("(");

            for (int i = 0; i < interfaces.length; i++) {
                String interfaceName = interfaces[i].getName();
                proxyName.append(omit(interfaceName));

                if (i > 0) {
                    proxyName.append(",");
                }
            }

            proxyName.append(")");
            className = proxyName.toString();
        }

        return className;
    }

    /**
     * 完全修飾クラス名からパッケージ名を削除する。
     * 
     * @param className 完全修飾クラス名
     * @return パッケージ名を削除したクラス名
     */
    private static String omit(String className) {
        int pos = className.lastIndexOf('.');

        if (pos > 0) {
            className = className.substring(pos + 1);
        }

        return className;
    }

    /**
     * サイズを取得する。
     * 
     * @param object オブジェクト
     * @return サイズ
     */
    public static String getSize(Object object) {
        if (object == null) {
            return null;
        }

        Class clazz = object.getClass();

        // 配列の場合
        if (clazz.isArray()) {
            return String.valueOf(Array.getLength(object));
        }

        // 配列以外の場合はsize()、またはlength()の値を返す。
        Method method = null;

        // size()
        try {
            method = clazz.getMethod("size", new Class[0]);
        } catch (NoSuchMethodException e) {
            method = null;
        }

        // length()
        try {
            if (method == null) {
                method = clazz.getMethod("length", new Class[0]);
            }
        } catch (NoSuchMethodException e) {
            method = null;
        }

        if (method == null) {
            return null;
        }

        String size = null;

        // サイズを取得。
        try {
            Object ret = method.invoke(object, new Object[0]);

            if (ret != null) {
                size = ret.toString();
            }
        } catch (NumberFormatException e) {
            size = null;
        } catch (IllegalAccessException e) {
            size = null;
        } catch (InvocationTargetException e) {
            size = null;
        }

        return size;
    }

    /**
     * オブジェクトの情報を取得する。
     * 
     * @param object オブジェクト
     * @return オブジェクトの情報
     */
    public static String getInfo(Object object) {
        return getInfo(object, false);
    }

    /**
     * オブジェクトの情報を取得する。
     * 
     * @param object オブジェクト
     * @param appendValue オブジェクトの値を出力する場合はtrue
     * @return オブジェクトの情報
     */
    public static String getInfo(Object object, boolean appendValue) {
        if (object == null) {
            return "null";
        }

        // クラス名を取得。
        StringBuffer buffer = new StringBuffer();
        buffer.append(ObjectUtils.getName(object, true));

        // サイズを取得。
        String size = ObjectUtils.getSize(object);

        if (size != null) {
            buffer.append("(" + size + ")");
        }

        // 値を取得。
        if (appendValue) {
            buffer.append(" \"");
            buffer.append(object.toString());
            buffer.append("\"");
        }

        return buffer.toString();
    }

    /////////////////////////////////////////////////////////////////
    // class list getter

    /**
     * 実装するクラスのリストを取得する。
     * 
     * @param object オブジェクト
     * @return 実装するクラスのリスト
     */
    public static List getClassNames(Object object) {
        return getClassNames(object, false);
    }

    /**
     * 実装するクラスのリストを取得する。
     * 
     * @param object オブジェクト
     * @param isReverse 逆順の場合はtrue
     * @return 実装するクラスのリスト
     */
    public static List getClassNames(Object object, boolean isReverse) {
        if (object == null) {
            List classNames = new ArrayList();
            classNames.add("null");
            return classNames;
        }

        return getClassNames(object.getClass(), isReverse);
    }

    /**
     * 実装するクラスのリストを取得する。
     * 
     * @param clazz クラス
     * @return 実装するクラスのリスト
     */
    public static List getClassNames(Class clazz) {
        return getClassNames(clazz, false);
    }

    /**
     * 実装するクラスのリストを取得する。
     * 
     * @param clazz クラス
     * @param isReverse 逆順の場合はtrue
     * @return 実装するクラスのリスト
     */
    public static List getClassNames(Class clazz, boolean isReverse) {
        List classNames = new ArrayList();
        Class cls = clazz;

        // プリミティブ型の場合
        if (clazz.isPrimitive()) {
            classNames.add("primitive");
            return classNames;
        }

        // 配列の場合
        if (clazz.isArray()) {
            classNames.add("array");
            return classNames;
        }

        // それ以外の場合
        final String objectClassName = Object.class.getName();
        List interfaceNames = new ArrayList();

        // 親クラス名を取得。
        while (!cls.getName().equals(objectClassName)) {
            classNames.add(cls.getName());
            // インターフェイス名を取得。
            addInterfaceNames(cls, interfaceNames);
            cls = cls.getSuperclass();
        }

        // インターフェース名をリストに追加。
        classNames.addAll(interfaceNames);

        return classNames;
    }

    /**
     * インターフェース名をクラス名リストに追加する。
     * 
     * @param clazz クラス
     * @param names クラス名リスト
     */
    private static void addInterfaceNames(Class clazz, List names) {
        // インターフェースを取得。
        Class[] interfaces = clazz.getInterfaces();

        // インターフェース名を再帰的に取得。
        for (int i = 0; i < interfaces.length; i++) {
            String interfaceName = interfaces[i].getName();

            if (names.contains(interfaceName)) {
                continue;
            }

            names.add(interfaceName);
            addInterfaceNames(interfaces[i], names);
        }
    }

    /////////////////////////////////////////////////////////////////
    // method utility

    /**
     * メソッドを取得する。
     * 
     * @param object オブジェクト
     * @param name メソッド名
     * @param classes 引数のクラス
     * @return メソッド
     */
    public static Method getMethod(Object object, String name, Class[] classes) {
        if (object == null) {
            throw new BodyBuilderException("object that '" + name
                    + "' defined is null.");
        }

        return getMethod(object.getClass(), name, classes);
    }

    /**
     * メソッドを取得する。
     * 
     * @param clazz クラス
     * @param name メソッド名
     * @param classes 引数のクラス
     * @return メソッド
     */
    public static Method getMethod(Class clazz, String name, Class[] classes) {
        try {
            Method method = null;

            try {
                // publicメソッド(スパークラスを含む)を取得。
                method = clazz.getMethod(name, classes);
            } catch (NoSuchMethodException e) {
                // public以外のメソッドを取得。
                method = clazz.getDeclaredMethod(name, classes);
            }

            return method;
        } catch (SecurityException e) {
            throw new BodyBuilderException("cannot get method '" + name + " "
                    + Arrays.asList(classes) + "' as '" + clazz.getName()
                    + "'.", e);
        } catch (NoSuchMethodException e) {
            throw new BodyBuilderException("cannot get method '" + name + " "
                    + Arrays.asList(classes) + "' as '" + clazz.getName()
                    + "'.", e);
        }
    }

    /**
     * メソッドを実行する。
     * 
     * @param type クラス名
     * @param name メソッド名
     * @param classes 引数のクラス
     * @param arguments 引数
     * @return メソッドの戻り値
     */
    public static Object invokeMethod(String type, String name,
            Class[] classes, Object[] arguments) {
        return invokeMethod(getClass(type), null, name, classes, arguments);
    }

    /**
     * メソッドを実行する。
     * 
     * @param object オブジェクト
     * @param name メソッド名
     * @param classes 引数のクラス
     * @param arguments 引数
     * @return メソッドの戻り値
     */
    public static Object invokeMethod(Object object, String name,
            Class[] classes, Object[] arguments) {
        if (object == null) {
            throw new BodyBuilderException("object that '" + name
                    + "' defined is null.");
        }

        return invokeMethod(object.getClass(), object, name, classes, arguments);
    }

    /**
     * メソッドを実行する。
     * 
     * @param clazz クラス
     * @param object オブジェクト
     * @param name メソッド名
     * @param classes 引数のクラス
     * @param arguments 引数
     * @return メソッドの戻り値
     */
    public static Object invokeMethod(Class clazz, Object object, String name,
            Class[] classes, Object[] arguments) {
        Method method = getMethod(clazz, name, classes);
        return invokeMethod(clazz, object, method, arguments);
    }

    /**
     * メソッドを実行する。
     * 
     * @param clazz クラス
     * @param object オブジェクト
     * @param method メソッド
     * @param arguments 引数
     * @return メソッドの戻り値
     */
    public static Object invokeMethod(Class clazz, Object object,
            Method method, Object[] arguments) {
        try {
            // privateメソッドも実行する。
            method.setAccessible(true);
            return method.invoke(object, arguments);
        } catch (SecurityException e) {
            throw new BodyBuilderException("cannot invole method '"
                    + method.getName() + "' as '" + object + "'.", e);
        } catch (IllegalArgumentException e) {
            throw new BodyBuilderException("cannot invole method '"
                    + method.getName() + "' as '" + object + "'.", e);
        } catch (IllegalAccessException e) {
            throw new BodyBuilderException("cannot invole method '"
                    + method.getName() + "' as '" + object + "'.", e);
        } catch (InvocationTargetException e) {
            throw new BodyBuilderException("cannot invole method '"
                    + method.getName() + "' as '" + object + "'.", e);
        }
    }

    /**
     * スタティックメソッドかどうかを取得する。
     * 
     * @param method メソッド
     * @return スタティックメソッドの場合はtrue
     */
    public static boolean isStaticMethod(Method method) {
        return method.toString().matches("^.* static .*$");
    }

    /////////////////////////////////////////////////////////////////
    // field utility

    /**
     * フィールドの値を取得する。
     * 
     * @param type クラス名
     * @param name フィールド名
     * @return フィールドの値
     */
    public static Object getFiledValue(String type, String name) {
        return getFiledValue(getClass(type), null, name);
    }

    /**
     * フィールドの値を取得する。
     * 
     * @param object オブジェクト
     * @param name フィールド名
     * @return フィールドの値
     */
    public static Object getFiledValue(Object object, String name) {
        if (object == null) {
            throw new BodyBuilderException("object that '" + name
                    + "' defined is null.");
        }

        return getFiledValue(object.getClass(), object, name);
    }

    /**
     * フィールドの値を取得する。
     * 
     * @param clazz クラス
     * @param object オブジェクト
     * @param name フィールド名
     * @return フィールドの値
     */
    public static Object getFiledValue(Class clazz, Object object, String name) {
        try {
            Field field = null;

            try {
                // publicフィールド(スーパークラス含む)を取得。
                field = clazz.getField(name);
            } catch (NoSuchFieldException e) {
                // public以外のフィールドを取得。
                field = clazz.getDeclaredField(name);
                field.setAccessible(true);
            }

            return field.get(object);
        } catch (SecurityException e) {
            throw new BodyBuilderException("cannot get field '" + name
                    + "' as '" + object + "(" + clazz + ")'.", e);
        } catch (NoSuchFieldException e) {
            throw new BodyBuilderException("cannot get field '" + name
                    + "' as '" + object + "(" + clazz + ")'.", e);
        } catch (IllegalArgumentException e) {
            throw new BodyBuilderException("cannot get field '" + name
                    + "' as '" + object + "(" + clazz + ")'.", e);
        } catch (IllegalAccessException e) {
            throw new BodyBuilderException("cannot get field '" + name
                    + "' as '" + object + "(" + clazz + ")'.", e);
        }
    }

    /////////////////////////////////////////////////////////////////
    // instanceof

    /**
     * クラスが指定された型のインスタンスかどうかを返す。
     * 
     * @param clazz クラス
     * @param type 型
     * @return インスタンスの場合はtrue
     */
    public static boolean instance_of(Class clazz, String type) {
        List classes = getClassNames(clazz);
        return classes.contains(type);
    }

    /**
     * クラスが指定された型のインスタンスかどうかを返す。
     * 
     * @param clazz クラス
     * @param type 型
     * @return インスタンスの場合はtrue
     */
    public static boolean instance_of(Class clazz, Class type) {
        return instance_of(clazz, type.getName());
    }

    /**
     * オブジェクトが指定された型のインスタンスかどうかを返す。
     * 
     * @param obj オブジェクト
     * @param type 型
     * @return インスタンスの場合はtrue
     */
    public static boolean instance_of(Object obj, String type) {
        List classes = getClassNames(obj);
        return classes.contains(type);
    }

    /**
     * オブジェクトが指定された型のインスタンスかどうかを返す。
     * 
     * @param obj オブジェクト
     * @param type 型
     * @return インスタンスの場合はtrue
     */
    public static boolean instance_of(Object obj, Class type) {
        return instance_of(obj, type.getName());
    }

    /////////////////////////////////////////////////////////////////
    // package getter

    /**
     * 指定されたオブジェクトのパッケージ名を取得する。
     * 
     * @param clazz クラス
     * @param asterisk アスタリスクをつけるかどうかのフラグ
     * @return パッケージ名
     */
    public static String getPackage(String clazz, boolean asterisk) {
        int index = clazz.lastIndexOf('.');

        if (index < 0) {
            return "";
        }

        String pkg = clazz.substring(0, index);

        return asterisk ? pkg + ".*" : pkg;
    }

    /**
     * 指定されたクラスのパッケージ名を取得する。
     * 
     * @param clazz クラス
     * @return パッケージ名
     */
    public static String getPackage(String clazz) {
        return getPackage(clazz, false);
    }

    /**
     * 指定されたクラスのパッケージ名を取得する。
     * 
     * @param clazz クラス
     * @return パッケージ名
     */
    public static String getPackage(Class clazz) {
        return getPackage(clazz.getName());
    }

    /**
     * 指定されたオブジェクトのパッケージ名を取得する。
     * 
     * @param obj オブジェクト
     * @return パッケージ名
     */
    public static String getPackage(Object obj) {
        if (obj == null) {
            return "";
        }

        return getPackage(obj.getClass().getName());
    }

}