package bodybuilder.builder;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.Config;
import bodybuilder.util.jdom.ExtendedSAXBuilder;

/**
 * ブルワーカ
 */
public class Bullworker {

    /////////////////////////////////////////////////////////////////
    // constant field

    /**
     * SAXビルダー
     */
    private static final ExtendedSAXBuilder builder = new ExtendedSAXBuilder(
            Config.isValidate());

    /**
     * オブジェクト要素
     */
    private static final String OBJECT_ELEMENT = "object";

    /**
     * name属性
     */
    private static final String OBJECT_NAME_ATTRIBUTE = "name";

    /////////////////////////////////////////////////////////////////
    // variable field

    /**
     * オブジェクトリスト
     */
    private List objectList = new ArrayList();

    /**
     * オブジェクトマップ
     */
    private Map objectMap = new HashMap();

    /////////////////////////////////////////////////////////////////
    // constructor

    /**
     * コンストラクタ。
     * 
     * @param filename XMLファイル名
     */
    public Bullworker(String filename) {
        this(new File(filename));
    }

    /**
     * コンストラクタ。
     * 
     * @param file XMLファイル
     */
    public Bullworker(File file) {
        try {
            loadObjects(builder.build(file));
        } catch (JDOMException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * コンストラクタ。
     * 
     * @param in 入力ストリーム
     */
    public Bullworker(InputStream in) {
        try {
            loadObjects(builder.build(in));
        } catch (JDOMException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * コンストラクタ。
     * 
     * @param reader リーダー
     */
    public Bullworker(Reader reader) {
        try {
            loadObjects(builder.build(reader));
        } catch (JDOMException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /////////////////////////////////////////////////////////////////
    // public method

    /**
     * オブジェクトを取得する。
     * 
     * @return オブジェクト
     */
    public Object getMuscle() {
        // オブジェクトリストが空の場合はnullを返す。
        if (objectList.size() < 1) {
            return null;
        }

        // <object>を取得。
        Element object = (Element) objectList.get(0);
        // オブジェクトを取得。
        return Builder.getValue(object);
    }

    /**
     * オブジェクトを取得する。
     * 
     * @param index オブジェクト要素のインデックス
     * @return オブジェクト
     */
    public Object getMuscle(int index) {
        // <object>を取得。
        Element object = (Element) objectList.get(index);
        // オブジェクトを取得。
        return Builder.getValue(object);
    }

    /**
     * オブジェクトを取得する。
     * 
     * @param name オブジェクト要素の名前
     * @return オブジェクト
     */
    public Object getMuscle(String name) {
        // オブジェクト要素を取得。
        Element object = (Element) objectMap.get(name);

        // 取得できなかった場合はnullを返す。
        if (object == null) {
            return null;
        }

        // オブジェクトを取得。
        return Builder.getValue(object);
    }

    /////////////////////////////////////////////////////////////////
    // private method

    /**
     * オブジェクト要素を読み込む。
     * 
     * @param doc XMLドキュメント
     */
    private void loadObjects(Document doc) {
        // ルート要素を取得。
        Element root = doc.getRootElement();

        // <object>を取得。
        List children = root.getChildren(OBJECT_ELEMENT);

        // <object>をフィールドに格納。
        for (int i = 0; i < children.size(); i++) {
            add((Element) children.get(i));
        }
    }

    /**
     * オブジェクト要素をフィールドに格納する。
     * 
     * @param object オブジェクト要素
     */
    private void add(Element object) {
        // オブジェクトリストにオブジェクト要素を格納。
        objectList.add(object);

        // name属性を取得。
        String name = object.getAttributeValue(OBJECT_NAME_ATTRIBUTE);

        // nameが存在する場合、オブジェクトマップに<object>を格納。
        if (name != null) {
            objectMap.put(name, object);
        }
    }

}