package bodybuilder.test.dicon;

import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.Map;

import bodybuilder.builder.argument.Argument;
import bodybuilder.exception.BodyBuilderException;
import bodybuilder.inspector.Inspector;
import bodybuilder.test.XMLTestCase;
import bodybuilder.test.TestCaseXML;
import bodybuilder.test.common.Catch;
import bodybuilder.test.common.Execute;
import bodybuilder.test.common.Return;
import bodybuilder.test.database.DatabaseAssertion;
import bodybuilder.test.database.DatabaseSetUp;
import bodybuilder.util.ExtendedPropertyUtils;
import bodybuilder.util.ObjectUtils;

/**
 * DIConテストケース
 */
public class DIConTestCase extends XMLTestCase {

    /**
     * DIConテストケースXML
     */
    protected DIConTestCaseXML xml = null;

    /**
     * DIConテストケースXMLをセットする。
     * 
     * @param xml DIConテストケースXML
     */
    protected void setXML(TestCaseXML xml) {
        this.xml = (DIConTestCaseXML) xml;
    }

    /**
     * テストを実行する。
     */
    protected void runTest() {
        info("'" + getName() + "' is started .");
        debug("description is:\n" + xml.getDescription());

        // ターゲットとインジェクションタイプを取得。
        String target = xml.getTarget();
        int type = xml.getType();
        debug("target is '" + target + "'(Type" + type + ").");

        InjectedObjects injectedObjects = xml.getInjectedObjects();

        // 注入するオブジェクトを取得してコンポーネントを生成。
        Object component = buildComponent(target, type, injectedObjects);

        // データベースをセットアップ。
        DatabaseSetUp databaseSetUp = xml.getDatabaseSetUp();

        if (databaseSetUp != null) {
            debug("set up database by '" + databaseSetUp.getFile() + "'.");
            databaseSetUp.setUp();
        }

        // 実行するメソッドを取得。
        Execute execute = xml.getExecute();
        String method = execute.getMethod();
        Argument argument = execute.getArgument();
        debug("invoked method is '" + method + "' as '" + target + "'.");

        Object actualReturn = null;
        Throwable cause = null;

        // メソッドを実行。
        try {
            actualReturn = ObjectUtils.invokeMethod(component, method, argument
                    .getClasses(), argument.getArguments());
        } catch (BodyBuilderException e) {
            // メソッドの例外ではない場合は上位に投げる。
            if (!(e.getCause() instanceof InvocationTargetException)) {
                throw e;
            }

            // メソッドの例外を取得。
            cause = e.getCause().getCause();
            debug("'" + cause + "' was catched.");
        } finally {
            if (injectedObjects != null) {
                windup(injectedObjects);
            }
        }

        debug("'" + method + "' was invoked.");

        // 例外を検査。
        Catch expectedCatch = xml.getCatch();

        if (expectedCatch != null
                && ObjectUtils.instance_of(cause, expectedCatch.getType())) {
            debug("verify catched exception.");
            Inspector.assertObjectEquals(expectedCatch.getException(), cause);
            debug("catched exception value was corrected.");
        } else if (expectedCatch != null) {
            // 例外を検査しない場合は上位に投げる。
            throw new BodyBuilderException("unexpected exception '" + cause
                    + "' but was '" + expectedCatch.getType() + "'.", cause);
        } else if (cause != null) {
            // 例外を検査しない場合は上位に投げる。
            throw new BodyBuilderException("unexpected exception '" + cause
                    + "'.", cause);
        }

        // 戻り値を検査。
        Return expectedReturn = xml.getReturn();

        if (expectedReturn != null) {
            debug("verify return value.");
            Inspector.assertObjectEquals(expectedReturn.getValue(),
                    actualReturn);
            debug("return value was corrected.");
        }

        // インジェクションオブジェクトを検査。
        InjectedObjects expectedObjects = xml.getExpectedObjects();

        if (expectedObjects != null) {
            String[] names = expectedObjects.getNames();

            for (int i = 0; i < names.length; i++) {
                debug("verify object '" + names[i] + "' as '" + component
                        + "'.");
                Object expectedObject = expectedObjects.getObject(names[i]);
                Object actualObject = injectedObjects.getObject(names[i]);
                Inspector.assertObjectEquals(expectedObject, actualObject);
                debug("'" + names[i] + "' was corrected.");
            }
        }

        // データベースを検査。
        DatabaseAssertion databaseAssertion = xml.getDatabaseAssertion();

        if (databaseAssertion != null) {
            debug("verify database by '" + databaseAssertion.getFile() + "'.");
            databaseAssertion.assertDataSetEquals();
            debug("database was corrected.");
        }

        info("test was ended.");
    }

    /**
     * コンポーネントを生成する。
     * 
     * @param target ターゲット
     * @param type インジェクションタイプ
     * @param injectedObjects インジェクションオブジェクト
     * @return コンポーネント
     */
    private Object buildComponent(String target, int type,
            InjectedObjects injectedObjects) {
        Object component = null;

        switch (type) {
        case DIConTestCaseXML.SETTER_INJECTION:
            // セッター・インジェクション(Type 2)
            // コンポーネントを生成してオブジェクトを注入。
            debug("build component.");
            component = ObjectUtils.getObject(target);

            if (injectedObjects == null) {
                break;
            }

            Map objects = injectedObjects.getObjectMap();
            Iterator names = objects.keySet().iterator();

            while (names.hasNext()) {
                String name = (String) names.next();
                Object object = objects.get(name);
                debug("inject object '" + name + "'.");
                ExtendedPropertyUtils.setProperty(component, name, object);
            }

            break;

        case DIConTestCaseXML.CONSTRUCTOR_INJECTION:
            // コンストラクタ・インジェクション(Type 3)
            // コンポーネントが未定義の場合はエラー。
            if (injectedObjects == null) {
                throw new BodyBuilderException(
                        "undefined injection objects for type 3.");
            }

            // コンポーネントを生成してオブジェクトを注入。
            Class[] classes = injectedObjects.getClasses();
            Object[] arguments = injectedObjects.getObjects();
            debug("build component and inject objects.");
            component = ObjectUtils.getInstance(target, classes, arguments);
            break;

        default:
            // Type 2、Type 3以外には未対応。
            throw new BodyBuilderException("uninplemented injection type '"
                    + type + "'.");
        }

        // コンポーネントを返す。
        return component;
    }

    /**
     * コンポーネントの後始末をする。
     * 
     * @param injectedObjects コンポーネント
     */
    private void windup(InjectedObjects injectedObjects) {
        String[] names = injectedObjects.getNames();

        // 後始末をする
        for (int i = 0; i < names.length; i++) {
            Object component = injectedObjects.getObject(names[i]);
            Execute[] windup = injectedObjects.getWindUp(names[i]);

            // 未定義の場合はスキップ。
            if (windup == null || windup.length < 1) {
                continue;
            }

            debug("wind up component '" + names[i] + "'.");

            // 後始末するメソッドを実行。
            for (int j = 0; j < windup.length; j++) {
                String method = windup[j].getMethod();
                Argument argument = windup[j].getArgument();
                debug("execute method '" + method + "' as '"
                        + component.getClass().getName() + "' for wind up.");
                ObjectUtils.invokeMethod(component, windup[j].getMethod(),
                        argument.getClasses(), argument.getArguments());
            }
        }
    }

}