package bodybuilder.inspector;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import bodybuilder.util.Config;
import bodybuilder.util.ExtendedPropertyUtils;
import bodybuilder.util.ObjectUtils;

/**
 * ビーンインスペクター
 */
public class BeanInspector extends Inspector {

    /**
     * 検査しないプロパティマップのキャッシュ
     */
    private static Map ignorePropertiesCache = null;

    /**
     * 二つのビーンが等しいことを表明する。
     * 
     * @param expected 期待する値
     * @param actual 現実の値
     * @param trace バックトレース
     */
    public void assertEquals(Object expected, Object actual,
            ObjectBackTrace trace) {
        // プロパティを取得。
        Map expectedProps = ExtendedPropertyUtils.getProperties(expected);
        Map actualProps = ExtendedPropertyUtils.getProperties(actual);

        // プロパティ名を検査。
        Set expectedNameSet = expectedProps.keySet();
        assertKeySetEquals(expectedNameSet, actualProps.keySet(), trace);
        Iterator names = expectedNameSet.iterator();

        // 検査しないプロパティのリストを取得。

        // プロパティ値を検査。
        while (names.hasNext()) {
            String name = (String) names.next();

            if(isIgnoreProperty(expected, name)) {
                continue;
            }

            trace.appendKey(name);
            Object expectedElement = expectedProps.get(name);
            Object actualElement = actualProps.get(name);
            assertObjectEquals(expectedElement, actualElement, trace);
        }
    }

    /**
     * プロパティが検査対象外かどうかを返す。
     * 
     * @param clazz 型
     * @param name プロパティ名
     * @return 検査対象外の場合はtrue
     */
    private boolean isIgnoreProperty(Object obj, String name) {
        if (ignorePropertiesCache == null) {
            ignorePropertiesCache = Config.getIgnoreProperties();
        }

        List classNames = ObjectUtils.getClassNames(obj.getClass());

        for (int i = 0; i < classNames.size(); i++) {
            String className = (String) classNames.get(i);
            String pkg = ObjectUtils.getPackage(className, true);
            List ignoreProps = (List) ignorePropertiesCache.get(className);

            if (ignoreProps == null) {
                ignoreProps = (List) ignorePropertiesCache.get(pkg);
            }

            if (ignoreProps == null) {
                continue;
            }

            if (ignoreProps.contains(name)) {
                return true;
            }
        }

        return false;
    }

}