package bodybuilder.builder.servlet;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import bodybuilder.builder.Builder;
import bodybuilder.exception.BodyBuilderException;
import bodybuilder.util.Config;
import bodybuilder.util.jdom.ExtendedSAXBuilder;

/**
 * サーブレットオブジェクトブルワーカ
 */
public class ServletObjectsBullworker {

    /////////////////////////////////////////////////////////////////
    // constant field

    /**
     * SAXビルダー
     */
    private static final ExtendedSAXBuilder builder = new ExtendedSAXBuilder(
            Config.isValidate());

    /**
     * サーブレットコンテキスト要素
     */
    private static final String SERVLETCONTEXT_ELEMENT = "servletcontext";

    /**
     * リクエスト要素
     */
    private static final String REQUEST_ELEMENT = "request";

    /**
     * セッション要素
     */
    private static final String SESSION_ELEMENT = "session";

    /////////////////////////////////////////////////////////////////
    // variable field

    /**
     * サーブレットコンテキスト
     */
    private ServletContext servletcontext = null;

    /**
     * リクエスト
     */
    private HttpServletRequest request = null;

    /**
     * セッション
     */
    private HttpSession session = null;

    /////////////////////////////////////////////////////////////////
    // constructor

    /**
     * コンストラクタ。
     * 
     * @param filename XMLファイル名
     */
    public ServletObjectsBullworker(String filename) {
        this(new File(filename));
    }

    /**
     * コンストラクタ。
     * 
     * @param file XMLファイル
     */
    public ServletObjectsBullworker(File file) {
        try {
            loadObjects(builder.build(file));
        } catch (JDOMException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * コンストラクタ。
     * 
     * @param in 入力ストリーム
     */
    public ServletObjectsBullworker(InputStream in) {
        try {
            loadObjects(builder.build(in));
        } catch (JDOMException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /**
     * コンストラクタ。
     * 
     * @param reader リーダ
     */
    public ServletObjectsBullworker(Reader reader) {
        try {
            loadObjects(builder.build(reader));
        } catch (JDOMException e) {
            throw new BodyBuilderException(e);
        } catch (IOException e) {
            throw new BodyBuilderException(e);
        }
    }

    /////////////////////////////////////////////////////////////////
    // public method

    /**
     * サーブレットコンテキストを取得する。
     * 
     * @return サーブレットコンテキスト
     */
    public ServletContext getServletContext() {
        return servletcontext;
    }

    /**
     * リクエストを取得する。
     * 
     * @return リクエスト
     */
    public HttpServletRequest getRequest() {
        return request;
    }

    /**
     * セッションを取得する。
     * 
     * @return セッション
     */
    public HttpSession getSession() {
        return session;
    }

    /////////////////////////////////////////////////////////////////
    // private method

    /**
     * リクエスト要素、セッション要素を読み込む。
     * 
     * @param doc XMLドキュメント
     */
    private void loadObjects(Document doc) {
        // ルート要素を取得。
        Element root = doc.getRootElement();

        // <servletcontext>を取得。
        Element servletcontext = root.getChild(SERVLETCONTEXT_ELEMENT);

        // サーブレットコンテキストを取得。
        if (servletcontext != null) {
            this.servletcontext = (ServletContext) Builder
                    .build(servletcontext);
        }

        // <request>を取得。
        Element request = root.getChild(REQUEST_ELEMENT);

        // リクエストを取得。
        if (request != null) {
            this.request = (HttpServletRequest) Builder.build(request);
        }

        // <session>を取得。
        Element session = root.getChild(SESSION_ELEMENT);

        // セッションを取得。
        if (session != null) {
            this.session = (HttpSession) Builder.build(session);
        }
    }

}