/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.value;

import jp.bitmeister.asn1.exception.ASN1IllegalArgument;

/**
 * Represents ASN.1 'Hex string(hString)' item.
 * 
 * <p>
 * An instance of this class represents a 'hString' item.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public class HexString implements StringItem {

	/**
	 * The hexadecimal string.
	 */
	private String string;

	/**
	 * Instantiates a {@code HexString} and initialize it with a {@code String}.
	 * 
	 * @param string
	 *            The {@code String} that consist of an arbitrary numbers of
	 *            hexadecimal characters.
	 */
	public HexString(String string) {
		if (!string.matches("[0-9A-Fa-f]*")) {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage("Invalid string '" + string
					+ "'. hString must consist of hexadecimal string.", null,
					null, null, null);
			throw ex;
		}
		this.string = string.trim().toUpperCase();
	}

	/**
	 * Instantiates a {@code HexString} and initialize it with an array of
	 * {@code byte}.
	 * 
	 * @param array
	 *            The array of {@code byte}.
	 */
	public HexString(byte... array) {
		StringBuilder builder = new StringBuilder();
		for (Byte b : array) {
			builder.append(String.format("%02X", b));
		}
		string = builder.toString();
	}

	/**
	 * Returns the string value of this hString.
	 * 
	 * @return The hexadecimal string.
	 */
	public String string() {
		return string;
	}

	/**
	 * Converts the value of this hString to an array of {@code boolean}.
	 * 
	 * @return An array of {@code boolean}.
	 */
	public boolean[] toBinArray() {
		boolean[] array = new boolean[string.length() * 4];
		int index = 0, mask = 0;
		byte octet = 0;
		for (int i = 0; i < array.length; i++) {
			if (mask == 0) {
				octet = Byte.parseByte(string.substring(index++, index), 16);
				mask = 8;
			}
			array[i] = (octet & mask) > 0;
			mask >>>= 1;
		}
		return array;
	}

	/**
	 * Converts the value of this hString to an array of {@code byte}.
	 * 
	 * @return An array of {@code byte}.
	 */
	public byte[] toByteArray() {
		byte[] array = new byte[string.length() / 2 + string.length() % 2];
		byte octet = 0;
		for (int i = 0; i < string.length(); i++) {
			octet = Integer.valueOf(string.substring(i, i + 1), 16).byteValue();
			if (i % 2 == 0) {
				array[i / 2] = (byte) (octet << 4);
			} else {
				array[i / 2] = (byte) (array[i / 2] + octet);
			}
		}
		return array;
	}

	/**
	 * Returns a string representation of this item.
	 * 
	 * @return A string representation of this item.
	 */
	@Override
	public String toString() {
		return '\'' + string + "\'H";
	}

}
