/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.value;

import jp.bitmeister.asn1.exception.ASN1IllegalArgument;

/**
 * Represents ASN.1 'Binaly string(bString)' item.
 * 
 * <p>
 * An instance of this class represents a 'bString' item.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public class BinString implements StringItem {

	private String string;

	/**
	 * Instantiates a {@code BinString} and initialize it with a {@code String}.
	 * 
	 * @param string
	 *            The {@code String} consist of '0's and '1's.
	 */
	public BinString(String string) {
		if (!string.matches("[01 ]*")) {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage("Invalid string '" + string
					+ "'. bString shall consist of '0's and '1's.", null, null,
					null, null);
			throw ex;
		}
		this.string = string.trim();
	}

	/**
	 * Instantiates a {@code BinString} and initialize it with an array of
	 * {@code boolean}.
	 * 
	 * @param array
	 *            The array of {@code boolean}.
	 */
	public BinString(boolean... array) {
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i < array.length; i++) {
			builder.append(array[i] ? '1' : '0');
		}
		string = builder.toString();
	}

	/**
	 * Returns the string value of this bString.
	 * 
	 * @return The hexadecimal string.
	 */
	public String string() {
		return string;
	}

	/**
	 * Converts the value of this bString to an array of {@code boolean}.
	 * 
	 * @return An array of {@code boolean}.
	 */
	public boolean[] toBinArray() {
		boolean[] array = new boolean[string.length()];
		for (int i = 0; i < array.length; i++) {
			array[i] = string.charAt(i) == '1';
		}
		return array;
	}

	/**
	 * Converts the value of this bString to an array of {@code byte}.
	 * 
	 * @return An array of {@code byte}.
	 */
	public byte[] toByteArray() {
		byte[] array = new byte[string.length() / 8 + 1];
		int index = 0, shift = 7;
		for (int i = 0; i < string.length(); i++) {
			if (string.charAt(i) == '1') {
				array[index] |= 1 << shift;
			}
			shift--;
			if (shift < 0) {
				shift = 7;
				index++;
			}
		}
		return array;
	}

	/**
	 * Returns a string representation of this item.
	 * 
	 * @return A string representation of this item.
	 */
	@Override
	public String toString() {
		return '\'' + string + "'B";
	}

}
