/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.useful;

import jp.bitmeister.asn1.annotation.ASN1Alternative;
import jp.bitmeister.asn1.annotation.ASN1Anonymous;
import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Element;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.builtin.ANY;
import jp.bitmeister.asn1.type.builtin.BIT_STRING;
import jp.bitmeister.asn1.type.builtin.CHOICE;
import jp.bitmeister.asn1.type.builtin.INTEGER;
import jp.bitmeister.asn1.type.builtin.OBJECT_IDENTIFIER;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;
import jp.bitmeister.asn1.type.builtin.SEQUENCE;

/**
 * Represents ASN.1 'EXTERNAL' type.
 * 
 * <p>
 * An instance of this class represents an 'EXTERNAL' type data and the class is
 * a sub-class of {@code SEQUENCE}. 'EXTERNAL' type is defined with following
 * ASN.1 definition :
 * 
 * <pre>
 * EXTERNAL ::= [UNIVERSAL 8] IMPLICIT SEQUENCE {
 *   direct-reference      OBJECT IDENTIFIER OPTIONAL,
 *   indirect-reference    INTEGER           OPTIONAL,
 *   data-value-descriptor OBJECT DESCRIPTOR OPTIONAL,
 *   encoding CHOICE {
 *     single-ASN1-type [0] ANY,
 *     octet-aligned    [1] IMPLICIT OCTET STRING,
 *     arbitary         [2] IMPLICIT BIT STRING } }
 * </pre>
 * 
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 8, tagMode = ASN1TagMode.IMPLICIT)
public class EXTERNAL extends SEQUENCE {

	/**
	 * 'direct-reference' field.
	 */
	@ASN1Element(value = 0, optional = true)
	@ASN1Identifier("direct-reference")
	public OBJECT_IDENTIFIER direct_reference;

	/**
	 * 'indirect-reference' field.
	 */
	@ASN1Element(value = 1, optional = true)
	@ASN1Identifier("indirect-reference")
	public INTEGER indirect_reference;

	/**
	 * 'data-value-descriptor' field.
	 */
	@ASN1Element(value = 2, optional = true)
	@ASN1Identifier("data-value-descriptor")
	public ObjectDescriptor data_value_descriptor;

	/**
	 * 'encoding' field.
	 */
	@ASN1Element(3)
	public Encoding encoding;

	/**
	 * Represents 'CHOICE' type for 'encoding' of 'EXTERNAL' type.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	@ASN1Anonymous
	public static class Encoding extends CHOICE {

		/**
		 * 'single-ASN1-type' field;
		 */
		@ASN1Alternative
		@ASN1Tag(0)
		@ASN1Identifier("single-ASN1-type")
		public ANY single_ASN1_type;

		/**
		 * 'octet-aligned' field.
		 */
		@ASN1Alternative
		@ASN1Tag(1)
		@ASN1Identifier("octet-aligned")
		public OCTET_STRING octet_aligned;

		/**
		 * 'arbitary' field.
		 */
		@ASN1Alternative
		@ASN1Tag(2)
		public BIT_STRING arbitary;

		public Encoding() {
			super();
		}

		public Encoding(ASN1TagClass tagClass, int tagNumber, ASN1Type data) {
			super(tagClass, tagNumber, data);
		}

	}

	/**
	 * Instanciate an empty {@code ExternalType}.
	 */
	public EXTERNAL() {
	}

	public EXTERNAL(OBJECT_IDENTIFIER direct_reference,
			INTEGER indirect_reference, ObjectDescriptor data_value_descriptor,
			Encoding encoding) {
		this.direct_reference = direct_reference;
		this.indirect_reference = indirect_reference;
		this.data_value_descriptor = data_value_descriptor;
		this.encoding = encoding;
	}

}
