/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.builtin;

import java.util.ArrayList;
import java.util.Collection;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.annotation.ASN1XmlTypeName;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.CollectionType;

/**
 * Represents ASN.1 'SET OF' type.
 * 
 * <p>
 * This is the class used for defining 'SET OF' types. Generic type parameter
 * {@code T} indicates component type of the 'SET OF' type. A constructor of a
 * class that extends {@code SET_OF} must call parent constructor with
 * {@code componentType} parameter to assign the {@code Class} object of T.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Identifier("SET")
@ASN1XmlTypeName("SET_OF")
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 17, tagMode = ASN1TagMode.IMPLICIT)
public abstract class SET_OF<T extends ASN1Type> extends CollectionType<T> {

	/**
	 * Instantiate an empty {@code SET_OF}.
	 * 
	 * @param componentType
	 *            The class instance of component type.
	 */
	public SET_OF(Class<T> componentType) {
		super(componentType);
	}
	
	/**
	 * Instantiate a {@code SET_OF} and initialize it with the components.
	 * 
	 * @param componentType
	 *            The class instance of component type.
	 * @param components
	 *            Components to be set this instance.
	 */
	public SET_OF(Class<T> componentType, T... components) {
		super(componentType, components);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.CollectionType#newCollection()
	 */
	@Override
	@SuppressWarnings("serial")
	protected Collection<T> newCollection() {
		return new ArrayList<T>() {

			@Override
			@SuppressWarnings("unchecked")
			public boolean equals(Object other) {
				if (other == null || !getClass().equals(other.getClass())
						|| ((ArrayList<T>) other).size() != size()) {
					return false;
				}
				boolean[] checked = new boolean[size()];
				for (ASN1Type e : (ArrayList<T>) other) {
					for (int i = 0; i < size(); i++) {
						if (!checked[i] && get(i).equals(e)) {
							checked[i] = true;
							break;
						}
					}
				}
				for (int i = 0; i < size(); i++) {
					if (!checked[i]) {
						return false;
					}
				}
				return true;
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#accept(jp.bitmeister.asn1.processor.
	 * ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}

}
