/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.builtin;

import java.util.Arrays;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.annotation.ASN1XmlTypeName;
import jp.bitmeister.asn1.exception.ASN1IllegalArgument;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.Concatenatable;
import jp.bitmeister.asn1.type.PrimitiveType;
import jp.bitmeister.asn1.type.SizeCountable;
import jp.bitmeister.asn1.value.StringItem;

/**
 * Represents ASN.1 'OCTET STRING' type.
 * 
 * <p>
 * An instance of this class represents an 'OCTET STRING' type data, and has an
 * array of {@code byte} value.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Identifier("OCTET STRING")
@ASN1XmlTypeName("OCTET_STRING")
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 4, tagMode = ASN1TagMode.IMPLICIT)
public class OCTET_STRING extends PrimitiveType<byte[]> implements
		Concatenatable<OCTET_STRING>, SizeCountable {

	/**
	 * Instantiates an empty {@code OCTET_STRING}.
	 */
	public OCTET_STRING() {
	}

	/**
	 * Instantiates an {@code OCTET_STRING} and initialize it with the array of
	 * {@code byte} value.
	 * 
	 * @param value
	 *            The value assigned to the instance.
	 */
	public OCTET_STRING(byte... value) {
		set(value);
	}

	/**
	 * Instantiates an {@code OCTET_STRING} and initialize it with the
	 * {@code StringItem}.
	 * 
	 * @param item
	 *            The value assigned to the instance.
	 */
	public OCTET_STRING(StringItem item) {
		set(item);
	}

	/**
	 * Sets the {@code StringItem} value to the instance.
	 * 
	 * @param item
	 *            The value assigned to the instance.
	 */
	public void set(StringItem item) {
		set(item.toByteArray());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.Concatenatable#concatenate(null)
	 */
	public void concatenate(OCTET_STRING data) {
		if (data == null) {
			return;
		}
		if (!getClass().equals(data.getClass())) {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage(
					"The type '"
							+ data.specification().fullIdentifier()
							+ "' of the data to be concatenated is not the same type of this instance.",
					null, getClass(), null, null);
			throw ex;
		}
		if (data.hasValue()) {
			if (!hasValue()) {
				set(data.value());
			} else {
				byte[] newValue = new byte[value().length + data.value().length];
				System.arraycopy(value(), 0, newValue, 0, value().length);
				System.arraycopy(data.value(), 0, newValue, value().length,
						data.value().length);
				set(newValue);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.SizeCountable#size()
	 */
	public int size() {
		return value().length;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.PrimitiveType#cloneValue()
	 */
	@Override
	protected byte[] cloneValue() {
		return value().clone();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.PrimitiveType#valueEquals(java.lang.Object)
	 */
	@Override
	public boolean valueEquals(Object other) {
		if (other instanceof OCTET_STRING) {
			byte[] otherValue = ((OCTET_STRING) other).value();
			if (value() != null) {
				return Arrays.equals(value(), otherValue);
			}
			return otherValue == null;
		}
		return false;
	}

	/* 
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.PrimitiveType#hashCode()
	 */
	@Override
	public int hashCode() {
		return Arrays.hashCode(value());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#accept(jp.bitmeister.asn1.processor.
	 * ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}

}
