/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.util.Arrays;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.codec.ber.BerDecoder;
import jp.bitmeister.asn1.processor.ASN1Visitor;

/**
 * Represents a type that does not have definition.
 * 
 * <p>
 * When a decoder meets a source data with a tag that is not registered to
 * {@code ASN1ModuleManager}, decoder decodes the data as an {@code UnknownType}. An
 * instance of this class contains a tag class, a tag number and an array of
 * {@code byte} represents row contents octets.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1ModuleManager
 * @see BerDecoder
 */
@ASN1BuiltIn
@ASN1Identifier("(unknown)")
public final class UnknownType extends ASN1Type {

	private ASN1TagClass tagClass;

	private int tagNumber;

	private byte[] value;

	/**
	 * Instantiate with a tag class and a tag number.
	 * 
	 * @param tagClass
	 *            ASN.1 tag class.
	 * @param tagNumber
	 *            ASN.1 tag number.
	 */
	public UnknownType(ASN1TagClass tagClass, int tagNumber) {
		this.tagClass = tagClass;
		this.tagNumber = tagNumber;
	}

	/**
	 * Set contents octets.
	 * 
	 * @param value
	 *            The contents octets.
	 */
	public void set(byte[] value) {
		this.value = value;
	}

	/**
	 * Returns contents octets.
	 * 
	 * @return The contents octets.
	 */
	public byte[] value() {
		return value;
	}

	/**
	 * Returns tag class.
	 * 
	 * @return The ASN.1 tag class.
	 */
	public ASN1TagClass tagClass() {
		return tagClass;
	}

	/**
	 * Returns tag number.
	 * 
	 * @return The ASN.1 tag number.
	 */
	public int tagNumber() {
		return tagNumber;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clear()
	 */
	@Override
	public void clear() {
		tagClass = null;
		tagNumber = 0;
		value = null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hasValue()
	 */
	@Override
	public boolean hasValue() {
		return value != null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#accept(jp.bitmeister.asn1.processor.
	 * ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#valueEquals(java.lang.Object)
	 */
	@Override
	public boolean valueEquals(Object other) {
		if (other instanceof UnknownType) {
			UnknownType comparison = (UnknownType) other;
			if (comparison.tagClass == tagClass
					&& comparison.tagNumber == tagNumber) {
				if (Arrays.equals(comparison.value, value)) {
					return true;
				}
			}
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hashCode()
	 */
	@Override
	public int hashCode() {
		return value.hashCode();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clone()
	 */
	@Override
	public Object clone() {
		UnknownType clone = new UnknownType(tagClass, tagNumber);
		clone.value = value.clone();
		return clone;
	}

}
