/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.bitmeister.asn1.annotation.ASN1Alternative;
import jp.bitmeister.asn1.exception.ASN1IllegalArgument;
import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;
import jp.bitmeister.asn1.type.builtin.CHOICE;

/**
 * The base class for structured types defined by referencing a list of distinct
 * ASN.1 types.
 * 
 * <p>
 * This class provides generic interfaces and common methods for classes that
 * represents structured types which defined by referencing a list of distinct
 * ASN.1 types. This class is the parent class of {code CHOICE}.
 * </p>
 * 
 * @see CHOICE
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public abstract class SelectiveType extends StructuredType {

	private static final Map<Class<? extends StructuredType>, NamedTypeSpecification[]> ALTERNATIVES_MAP = new HashMap<Class<? extends StructuredType>, NamedTypeSpecification[]>();

	/**
	 * Returns an array of {@code NamedTypeSpecification} that represents
	 * alternatives of the type.
	 * 
	 * @param type
	 *            The type.
	 * @return An array of {@code NamedTypeSpecification}.
	 */
	private static NamedTypeSpecification[] getAlternativeTypeList(
			Class<? extends SelectiveType> type) {
		if (ALTERNATIVES_MAP.containsKey(type)) {
			return ALTERNATIVES_MAP.get(type);
		}
		List<NamedTypeSpecification> alternatives = new ArrayList<NamedTypeSpecification>();
		for (Field f : type.getDeclaredFields()) {
			if (f.isAnnotationPresent(ASN1Alternative.class)) {
				alternatives.add(new NamedTypeSpecification(f.getAnnotation(
						ASN1Alternative.class).value(), f));
			}
		}
		@SuppressWarnings("unchecked")
		Class<? extends SelectiveType> parent = (Class<? extends SelectiveType>) type
				.getSuperclass();
		NamedTypeSpecification[] array;
		if (parent == CHOICE.class) {
			if (alternatives.isEmpty()) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage(
						"CHOICE type shall have at least one alternative.",
						null, type, null, null);
				throw ex;
			}
			array = alternatives.toArray(new NamedTypeSpecification[0]);
			if (TypeSpecification.getSpecification(type).tagDefault() == ASN1TagDefault.AUTOMATIC_TAGS) {
				generateAutomaticTags(array);
			}
			new UnorderedElementsChecker(type).check(array);
		} else {
			if (!alternatives.isEmpty()) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage(
						"If a class does not extend CHOICE directly, it can not define own alternatives.",
						null, type, null, null);
				throw ex;
			}
			array = getAlternativeTypeList(parent);
		}
		ALTERNATIVES_MAP.put(type, array);
		return array;
	}

	/**
	 * Selected alternative.
	 */
	private NamedTypeSpecification selection;

	/**
	 * Instantiates an empty {@code SelectiveType}.
	 */
	public SelectiveType() {
	}

	/**
     * Instantiates a {@code SelectiveType} and initialize it with the
     * parameter. The type of the data is used for select alternative. If this
     * {@code SelectiveType} has two or more alternatives that are defined as
     * same type, the result of this constructor is undefined.
	 * 
	 * @param data
	 *            The ASN.1 data assigned to this instance.
	 */
	public SelectiveType(ASN1Type data) {
		for (NamedTypeSpecification e : getAlternativeTypeList(getClass())) {
			if (e.type() == data.getClass()) {
				set(e, data);
				return;
			}
		}
		ASN1IllegalArgument ex = new ASN1IllegalArgument();
		ex.setMessage("Can't select alternative by the type of the data.",
				null, getClass(), null, data);
		throw ex;
	}

	/**
	 * Instantiates a {@code SelectiveType} and initialize it with parameters.
	 * 
	 * @param tagClass
	 *            The tag class used for select an alternative.
	 * @param tagNumber
	 *            The tag number used for select an alternative.
	 * @param data
	 *            The data to be assigned.
	 */
	public SelectiveType(ASN1TagClass tagClass, int tagNumber, ASN1Type data) {
		set(alternative(tagClass, tagNumber), data);
	}

	/**
	 * Instantiates a {@code SelectiveType} and initialize it with parameters.
	 * 
	 * @param elementName
	 *            The element name used for select an alternative.
	 * @param data
	 *            The data to be assigned.
	 */
	public SelectiveType(String elementName, ASN1Type data) {
		set(elementName, data);
	}

	/**
	 * Returns the element specified by the ASN.1 tag class and number.
	 * 
	 * @param tagClass
	 *            ASN.1 tag class.
	 * @param tagNumber
	 *            ASN.1 tag number.
	 * @return The element specified by the ASN.1 tag class and number.
	 */
	public NamedTypeSpecification alternative(ASN1TagClass tagClass,
			int tagNumber) {
		for (NamedTypeSpecification e : getAlternativeTypeList(getClass())) {
			if (e.matches(tagClass, tagNumber)) {
				return e;
			}
		}
		ASN1IllegalArgument ex = new ASN1IllegalArgument();
		ex.setMessage("The tag '" + tagClass + " " + tagNumber
				+ "' does not match to any alternatives of this type.", null,
				getClass(), null, null);
		throw ex;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#matches(jp.bitmeister.asn1.type.ASN1TagClass
	 * , int)
	 */
	@Override
	public boolean matches(ASN1TagClass tagClass, int tagNumber) {
		for (NamedTypeSpecification e : getAlternativeTypeList(getClass())) {
			if (e.matches(tagClass, tagNumber)) {
				return true;
			}
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StructuredType#set(jp.bitmeister.asn1.type.
	 * NamedTypeSpecification, jp.bitmeister.asn1.type.ASN1Type)
	 */
	@Override
	public void set(NamedTypeSpecification alternative, ASN1Type data) {
		alternative.assign(this, data);
		selection = alternative;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StructuredType#set(java.lang.String,
	 * jp.bitmeister.asn1.type.ASN1Type)
	 */
	@Override
	public void set(String elementName, ASN1Type component) {
		set(getElement(elementName), component);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StructuredType#get(java.lang.String)
	 */
	@Override
	public ASN1Type get(String elementName) {
		return getElement(elementName).retrieve(this);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.StructuredType#getElement(java.lang.String)
	 */
	@Override
	public NamedTypeSpecification getElement(String elementName) {
		for (NamedTypeSpecification e : getAlternativeTypeList(getClass())) {
			if (e.identifier().equals(elementName)) {
				return e;
			}
		}
		ASN1IllegalArgument ex = new ASN1IllegalArgument();
		ex.setMessage(
				"No such alternative '" + elementName + "' in this type.",
				null, getClass(), null, null);
		throw ex;
	}

	/**
	 * Returns current selection.
	 * 
	 * @return An alternative currently selected.
	 */
	private NamedTypeSpecification selection() {
		if (selection == null) {
			for (NamedTypeSpecification e : getAlternativeTypeList(getClass())) {
				if (e.retrieve(this) != null) {
					selection = e;
					break;
				}
			}
		}
		return selection;
	}

	/**
	 * Returns the ASN.1 data of selected alternative.
	 * 
	 * @return The ASN.1 data of selected alternative.
	 */
	public ASN1Type selectedValue() {
		if (selection() != null) {
			return selection.retrieve(this);
		}
		return null;
	}

	/**
	 * Returns the identifier of selected alternative.
	 * 
	 * @return The identifier of selected alternative.
	 */
	public String selectedIdentifier() {
		if (selection() != null) {
			return selection.identifier();
		}
		return null;
	}

	/**
	 * Returns the ASN.1 tag of selected alternative.
	 * 
	 * @return The tag of selected alternative.
	 */
	public ASN1TagValue selectedTag() {
		if (selection() != null) {
			return selection.tag();
		}
		return null;
	}

	/**
	 * Clears the selection of this instance.
	 */
	public void clearSelection() {
		if (selection() != null) {
			selection.assign(this, null);
			selection = null;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clear()
	 */
	@Override
	public void clear() {
		selection = null;
		for (NamedTypeSpecification e : getAlternativeTypeList(getClass())) {
			e.assign(this, null);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hasValue()
	 */
	@Override
	public boolean hasValue() {
		return selection() != null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#valueEquals(java.lang.Object)
	 */
	@Override
	public boolean valueEquals(Object other) {
		if (other instanceof SelectiveType) {
			ASN1Type thisSelection = selectedValue();
			ASN1Type otherSelection = ((SelectiveType) other).selectedValue();
			if (thisSelection != null) {
				return thisSelection.equals(otherSelection);
			}
			return otherSelection == null;
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hashCode()
	 */
	@Override
	public int hashCode() {
		ASN1Type data = selectedValue();
		if (data == null) {
			return 0;
		}
		return data.hashCode();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clone()
	 */
	@Override
	public Object clone() {
		SelectiveType clone = ASN1Type.instantiate(getClass());
		ASN1Type value = selectedValue();
		if (value != null) {
			selection.assign(clone, (ASN1Type) value.clone());
			clone.selection = selection;
		}
		return clone;
	}

}
