/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;
import jp.bitmeister.asn1.type.builtin.ANY;
import jp.bitmeister.asn1.type.builtin.CHOICE;

/**
 * Contains a set of ASN.1 tag value.
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Tag
 * @see ASN1TagClass
 * @see ASN1TagMode
 */
public class ASN1TagValue {

	/**
	 * Tag class.
	 */
	private ASN1TagClass tagClass;

	/**
	 * Tag number.
	 */
	private int tagNumber;

	/**
	 * Indicates tagging mode.
	 */
	private ASN1TagMode tagMode;

	/**
	 * Instantiates an {@code ASN1TagValue} of an ASN.1 type.
	 * 
	 * @param type
	 *            The {@code Class} object of an ASN.1 type.
	 */
	ASN1TagValue(Class<? extends ASN1Type> type,
			Class<? extends ASN1Module> module) {
		ASN1Tag tag = type.getAnnotation(ASN1Tag.class);
		tagClass = tag.tagClass();
		tagNumber = tag.value();
		setTagMode(tag.tagMode(), ASN1Module.tagDefault(module), type, null);
	}

	/**
	 * Instantiates an {@code ASN1TagValue} of an element of ASN.1 type.
	 * 
	 * @param namedType
	 *            The element of an ASN.1 type.
	 */
	ASN1TagValue(NamedTypeSpecification namedType) {
		ASN1Tag tag = namedType.field().getAnnotation(ASN1Tag.class);
		if (tag.tagClass() != ASN1TagClass.CONTEXT_SPECIFIC
				&& tag.tagClass() != ASN1TagClass.PRIVATE) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage(
					"Only context-specific or private tagging is allowed to an elements of structured type.",
					null, namedType.enclosingType(), namedType.identifier(),
					null);
			throw ex;
		}
		tagClass = tag.tagClass();
		tagNumber = tag.value();
		setTagMode(tag.tagMode(),
				TypeSpecification.getSpecification(namedType.enclosingType())
						.tagDefault(), namedType.enclosingType(),
				namedType.identifier());
	}

	/**
	 * Instantiates an {@code ASN1TagValue} with automatic tagging rules.
	 * 
	 * @param order
	 *            The appearing order of a field.
	 * @param type
	 *            The ASN.1 type of a field.
	 */
	ASN1TagValue(int order, Class<? extends ASN1Type> type) {
		tagClass = ASN1TagClass.CONTEXT_SPECIFIC;
		tagNumber = order;
		if (canImplicitTagging(type)) {
			tagMode = ASN1TagMode.IMPLICIT;
		} else {
			tagMode = ASN1TagMode.EXPLICIT;
		}
	}

	/**
	 * Returns tag class of this tag.
	 * 
	 * @return The tag class.
	 */
	public ASN1TagClass tagClass() {
		return tagClass;
	}

	/**
	 * Returns tag number of this tag.
	 * 
	 * @return The tag number.
	 */
	public int tagNumber() {
		return tagNumber;
	}

	/**
	 * Returns tagging mode.
	 * 
	 * @return The tagging mode.
	 */
	public ASN1TagMode tagMode() {
		return tagMode;
	}

	/**
	 * Sets tagging mode of this tag.
	 * 
	 * @param tagMode
	 *            The tagging mode applied to the type or the field.
	 * @param type
	 *            The ASN.1 type.
	 * @param fieldName
	 *            The field name.
	 */
	private void setTagMode(ASN1TagMode tagMode, ASN1TagDefault tagDefault,
			Class<? extends ASN1Type> type, String fieldName) {
		if (tagMode == ASN1TagMode.DEFAULT) {
			if (tagDefault == ASN1TagDefault.IMPLICIT_TAGS) {
				this.tagMode = ASN1TagMode.IMPLICIT;
			} else {
				this.tagMode = ASN1TagMode.EXPLICIT;
			}
		} else {
			this.tagMode = tagMode;
		}
		if (tagMode == ASN1TagMode.IMPLICIT && !canImplicitTagging(type)) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage(
					"Implicit tagging is not allowed to 'CHOICE' types and 'ANY' types.",
					null, type, fieldName, null);
			throw ex;
		}
	}

	/**
	 * Tests if the type can be tagged implicitly.
	 * 
	 * @param type
	 *            The ASN.1 type to be tested.
	 * @return {@code true} when the type can be tagged implicitly.
	 */
	private boolean canImplicitTagging(Class<?> type) {
		if (CHOICE.class.isAssignableFrom(type)
				|| ANY.class.isAssignableFrom(type)) {
			return false;
		}
		return true;
	}

}
