/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.processor;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Iterator;

import jp.bitmeister.asn1.exception.ASN1RuntimeException;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.ASN1TagValue;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.CollectionType;
import jp.bitmeister.asn1.type.ConstructiveType;
import jp.bitmeister.asn1.type.ElementSpecification;
import jp.bitmeister.asn1.type.PrimitiveType;
import jp.bitmeister.asn1.type.StringType;
import jp.bitmeister.asn1.type.TimeType;
import jp.bitmeister.asn1.type.TypeSpecification;
import jp.bitmeister.asn1.type.UnknownType;
import jp.bitmeister.asn1.type.builtin.ANY;
import jp.bitmeister.asn1.type.builtin.BIT_STRING;
import jp.bitmeister.asn1.type.builtin.BOOLEAN;
import jp.bitmeister.asn1.type.builtin.CHOICE;
import jp.bitmeister.asn1.type.builtin.ENUMERATED;
import jp.bitmeister.asn1.type.builtin.INTEGER;
import jp.bitmeister.asn1.type.builtin.NULL;
import jp.bitmeister.asn1.type.builtin.OBJECT_IDENTIFIER;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;
import jp.bitmeister.asn1.type.builtin.REAL;
import jp.bitmeister.asn1.type.builtin.RELATIVE_OID;
import jp.bitmeister.asn1.type.builtin.SEQUENCE;
import jp.bitmeister.asn1.type.builtin.SEQUENCE_OF;
import jp.bitmeister.asn1.type.builtin.SET;
import jp.bitmeister.asn1.type.builtin.SET_OF;
import jp.bitmeister.asn1.value.BinString;
import jp.bitmeister.asn1.value.HexString;

/**
 * Processor that builds string representation of ASN.1 data.
 * 
 * <p>
 * An instance of this class is set to the {@code stringBuilder} field of
 * {@code ASN1Type} class and used for {@code toString} method.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Type
 */
public class ASN1StringBuilder implements
		ASN1Processor<String, ASN1RuntimeException>,
		ASN1Visitor<Void, ASN1RuntimeException> {

	private static final String NEW_LINE = System.getProperty("line.separator");
	private static final String SPACE = " ";
	private static final String INDENT = "\t";
	private static final String SEPARATOR = "\t";
	private static final String ASSIGN = SPACE + "::=" + SPACE;
	private static final String CONSTRUCTIVE_OPEN = SPACE + "{" + NEW_LINE;
	private static final String CONSTRUCTIVE_SEPARATOR = "," + NEW_LINE;
	private static final String CONSTRUCTIVE_CLOSE = SPACE + "}";
	private static final String COLLECTION_OF = SPACE + "OF" + SPACE;
	private static final String IMPLICIT = SPACE + "IMPLICIT";
	private static final String OPTIONAL = SPACE + "OPTIONAL";

	private static final String VALUE_OPEN = SEPARATOR;
	private static final String VALUE_CLOSE = "";
	private static final String TAG_OPEN = "[";
	private static final String TAG_CLOSE = "]";
	private static final String NUMBER_OPEN = "(";
	private static final String NUMBER_CLOSE = ")";
	private static final String NULL = "null";
	private static final String NAMED_BITS_OPEN = "{";
	private static final String NAMED_BITS_CLOSE = "}";
	private static final String NAMED_BITS_SEPARATOR = ",";
	private static final String OID_OPEN = "{";
	private static final String OID_CLOSE = "}";
	private static final String STRING_OPEN = "\"";
	private static final String STRING_CLOSE = "\"";
	private static final String NOT_SET = "<No value>";
	private static final ThreadLocal<DateFormat> DATE_FORMAT = new ThreadLocal<DateFormat>() {
		@Override
		protected DateFormat initialValue() {
			return new SimpleDateFormat("[yyyy/MM/dd HH:mm:ss.S z]");
		}
	};

	private StringBuilder builder;

	private int indent;

	/**
	 * Instantiates an {@code ASN1StringBuilder}.
	 */
	public ASN1StringBuilder() {
	}

	/**
	 * Returns a string representation of the ASN.1 data.
	 * 
	 * @param data
	 *            The ASN.1 data.
	 * @return A string representation of the ASN.1 data.
	 */
	public String process(ASN1Type data) {
		indent = 0;
		builder = new StringBuilder();
		write(data);
		return builder.toString();
	}

	/**
	 * Writes descriptions of the data to the 'builder'.
	 * 
	 * @param data
	 *            The ASN.1 data.
	 */
	private void write(ASN1Type data) {
		TypeSpecification specification = data.specification();
		while (true) {
			if (specification.hasIdentifier()) {
				builder.append(specification.identifier());
				if (specification.reference() == null) {
					if (data instanceof CollectionType<?>) {
						CollectionType<?> collection = (CollectionType<?>) data;
						if (!collection.componentType().equals(ANY.class)) {
							builder.append(COLLECTION_OF).append(
									collection.componentSpecification()
											.identifier());
						}
					}
					break;
				} else {
					builder.append(ASSIGN);
					if (specification.tag() != null) {
						writeTag(specification.tag());
						builder.append(SPACE);
					}
				}
			}
			specification = specification.reference();
		}
		if (data instanceof PrimitiveType<?>) {
			builder.append(VALUE_OPEN);
			if (data.hasValue()) {
				data.accept(this);
			} else {
				builder.append(NOT_SET);
			}
			builder.append(VALUE_CLOSE);
		} else {
			builder.append(CONSTRUCTIVE_OPEN);
			indent++;
			data.accept(this);
			indent--;
			builder.append(CONSTRUCTIVE_CLOSE);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BOOLEAN)
	 */
	public Void visit(BOOLEAN data) {
		builder.append(data.value() ? "TRUE" : "FALSE");
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.INTEGER)
	 */
	public Void visit(INTEGER data) {
		String numberId = data.identifier();
		if (numberId != null) {
			builder.append(numberId);
		}
		builder.append(NUMBER_OPEN).append(data.value()).append(NUMBER_CLOSE);
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.ENUMERATED)
	 */
	public Void visit(ENUMERATED data) {
		visit((INTEGER) data);
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.REAL)
	 */
	public Void visit(REAL data) {
		builder.append(NUMBER_OPEN).append(data.value()).append(NUMBER_CLOSE);
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BIT_STRING)
	 */
	public Void visit(BIT_STRING data) {
		if (data.hasNamedBits()) {
			builder.append(NAMED_BITS_OPEN);
			boolean trailing = false;
			for (int i = 0; i < data.size(); i++) {
				if (data.bit(i)) {
					String identifier = data.nameOfBit(i);
					if (identifier != null) {
						if (trailing) {
							builder.append(NAMED_BITS_SEPARATOR).append(SPACE);
						} else {
							builder.append(SPACE);
							trailing = true;
						}
						builder.append(identifier);
					}
				}
			}
			builder.append(SPACE).append(NAMED_BITS_CLOSE).append(SEPARATOR);
		}
		builder.append(new BinString(data.value()).toString());
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.OCTET_STRING)
	 */
	public Void visit(OCTET_STRING data) {
		builder.append(new HexString(data.value()).toString());
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.NULL)
	 */
	public Void visit(NULL data) {
		builder.append(NULL);
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SEQUENCE)
	 */
	public Void visit(SEQUENCE data) {
		writeConstructive(data);
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SEQUENCE_OF)
	 */
	public Void visit(SEQUENCE_OF<? extends ASN1Type> data) {
		writeCollection(data);
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET)
	 */
	public Void visit(SET data) {
		writeConstructive(data);
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET_OF)
	 */
	public Void visit(SET_OF<? extends ASN1Type> data) {
		writeCollection(data);
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.CHOICE)
	 */
	public Void visit(CHOICE data) {
		indent();
		if (!data.hasValue()) {
			builder.append(NOT_SET);
		} else {
			builder.append(data.selectedIdentifier());
			if (data.selectedTag() != null) {
				builder.append(SEPARATOR);
				writeTag(data.selectedTag());
			}
			builder.append(SEPARATOR);
			write(data.selectedValue());
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.OBJECT_IDENTIFIER)
	 */
	public Void visit(OBJECT_IDENTIFIER data) {
		builder.append(OID_OPEN);
		if (data.value().size() > 0) {
			builder.append(SPACE).append(data.value().get(0));
			for (int i = 1; i < data.value().size(); i++) {
				builder.append('.').append(data.value().get(i));
			}
			builder.append(SPACE);
		}
		builder.append(OID_CLOSE);
		return null;
	}
	
	/* (non-Javadoc)
	 * @see jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type.builtin.RELATIVE_OID)
	 */
	public Void visit(RELATIVE_OID data) {
		return visit((OBJECT_IDENTIFIER)data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .StringType)
	 */
	public Void visit(StringType data) {
		try {
			String value = data.stringValue();
			builder.append(STRING_OPEN).append(value).append(STRING_CLOSE);
		} catch (Exception e) {
			builder.append(new HexString(data.value()).toString());
		}
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .TimeType)
	 */
	public Void visit(TimeType data) {
		visit((StringType)data);
		builder.append(SPACE).append(DATE_FORMAT.get().format(data.date()));
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.ANY)
	 */
	public Void visit(ANY data) {
		write(data.value());
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .UnknownType)
	 */
	public Void visit(UnknownType data) {
		for (int i = 0; i < data.value().length; i++) {
			if (i % 16 == 0) {
				indent();
			}
			builder.append(String.format("%02X", data.value()[i]));
			builder.append(i % 16 == 15 ? NEW_LINE : SPACE);
		}
		return null;
	}

	/**
	 * Writes descriptions of each element of {@code ConstructiveType}.
	 * 
	 * @param data
	 *            The constructive data.
	 */
	private void writeConstructive(ConstructiveType data) {
		ElementSpecification[] elements = data.getElementTypeList();
		for (int i = 0; i < elements.length; i++) {
			if (i != 0) {
				builder.append(CONSTRUCTIVE_SEPARATOR);
			}
			indent();
			builder.append(elements[i].identifier()).append(SEPARATOR);
			if (elements[i].tag() != null) {
				writeTag(elements[i].tag());
				builder.append(SEPARATOR);
			}
			ASN1Type element = data.getComponent(elements[i]);
			if (element == null) {
				builder.append(NOT_SET);
			} else {
				write(element);
			}
			if (elements[i].optional()) {
				builder.append(OPTIONAL);
			}
		}
	}

	/**
	 * Writes descriptions of each member of {@code CollectionType}.
	 * 
	 * @param data
	 *            The collection data.
	 */
	private void writeCollection(CollectionType<? extends ASN1Type> data) {
		Iterator<? extends ASN1Type> itr = data.collection().iterator();
		if (itr.hasNext()) {
			while (true) {
				indent();
				write(itr.next());
				if (!itr.hasNext()) {
					return;
				}
				builder.append(CONSTRUCTIVE_SEPARATOR);
			}
		}
		indent();
		builder.append(NOT_SET);
	}

	/**
	 * Writes tag description.
	 * 
	 * @param tag
	 *            The tag.
	 */
	private void writeTag(ASN1TagValue tag) {
		builder.append(TAG_OPEN);
		if (tag.tagClass() != ASN1TagClass.CONTEXT_SPECIFIC) {
			builder.append(tag.tagClass()).append(SPACE);
		}
		builder.append(tag.tagNumber()).append(TAG_CLOSE);
		if (tag.tagMode() == ASN1TagMode.IMPLICIT) {
			builder.append(IMPLICIT);
		}
	}

	/**
	 * Writes indents.
	 */
	private void indent() {
		for (int i = 0; i < indent; i++) {
			builder.append(INDENT);
		}
	}

}
