/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.codec.ber;

import java.io.IOException;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.TimeZone;

import jp.bitmeister.asn1.codec.ASN1Encoder;
import jp.bitmeister.asn1.exception.ASN1EncodingException;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.ConstructiveType;
import jp.bitmeister.asn1.type.ElementSpecification;
import jp.bitmeister.asn1.type.TimeType;
import jp.bitmeister.asn1.type.builtin.BIT_STRING;
import jp.bitmeister.asn1.type.builtin.REAL;
import jp.bitmeister.asn1.type.builtin.SET;
import jp.bitmeister.asn1.type.builtin.SET_OF;
import jp.bitmeister.asn1.type.useful.UTCTime;

/**
 * DER (Distinguished Encoding Rules) encoder.
 * 
 * <p>
 * {@code DerEncoder} is an implementation of {@code ASN1Encoder}. It encodes an
 * ASN.1 data to an array of {@code byte} with Distinguished Encoding Rules(DER) and
 * writes the result to {@code OutputStream} that is specified when the encoder
 * was instantiated.
 * </p>
 * <p>
 * DER applies some restrictions on Basic Encoding Rules(BER). The result octets
 * can be decoded by any BER decoder because DER is a sub-set of BER and
 * compatible with it.
 * </p>
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Encoder
 * @see BerEncoder
 * @see BerDecoder
 */
public class DerEncoder extends BerEncoder {

	/**
	 * Instantiates a DER encoder.
	 * 
	 * @param out
	 *            The {@code OutputStream} that encoded octets will be written.
	 */
	public DerEncoder(OutputStream out) {
		super(out);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.REAL)
	 */
	@Override
	public EncodedOctets visit(REAL data) {
		byte[] encoded;
		if (data.value() == 0) { // zero.
			encoded = new byte[0];
		} else if (data.value().isInfinite()) { // special value.
			encoded = new byte[] { data.value() == Double.POSITIVE_INFINITY ? (byte) 0x40
					: 0x41 };
		} else if (data.isBinary()) { // IEEE754 double form.
			long bits = Double.doubleToLongBits(data.value());

			// calculate E and M value.
			int exponent = (int) ((bits >> 52) & 0x7ff) - (1023 + 52);
			long mantissa = bits & 0xfffffffffffffL | 0x10000000000000L;
			while ((mantissa & 0x01) == 0) {
				mantissa >>= 1;
				exponent++;
			}
			int expLen = (exponent > Byte.MAX_VALUE)
					|| (exponent < Byte.MIN_VALUE) ? 2 : 1;
			byte[] mOctets = BigInteger.valueOf(mantissa).toByteArray();
			encoded = new byte[1 + expLen + mOctets.length];

			// binary encoding.
			encoded[0] = (byte) 0x80;
			if ((bits & 0x8000000000000000L) != 0) {
				// value is negative.
				encoded[0] |= 0x40;
			}
			if (expLen == 2) {
				encoded[0] |= 0x01;
				encoded[1] = (byte) (exponent >> 8);
				encoded[2] = (byte) exponent;
			} else {
				encoded[1] = (byte) exponent;
			}
			System.arraycopy(mOctets, 0, encoded, 1 + expLen, mOctets.length);
		}
		else { // ISO6093 NR3
			BigDecimal value = BigDecimal.valueOf(data.value()).stripTrailingZeros();
			int exponent = -value.scale();
			long mantissa = value.unscaledValue().longValue();
			StringBuilder builder = new StringBuilder().append(mantissa).append(".E");
			if (exponent == 0) {
				builder.append('+');
			}
			builder.append(exponent);
			encoded = new byte[builder.length() + 1];
			encoded[0] = 0x03;
			System.arraycopy(builder.toString().getBytes(), 0, encoded, 1, encoded.length - 1);
		}
		return newPrimitiveOctets(encoded);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BIT_STRING)
	 */
	@Override	
	public EncodedOctets visit(BIT_STRING data) {
		if (data.hasNamedBits()) {
			data.contract();
		}
		return super.visit(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .TimeType)
	 */
	@Override
	public EncodedOctets visit(TimeType data) throws ASN1EncodingException {
		if (data instanceof UTCTime) {
			return super.visit(data);
		}
		DateFormat form = data.form();
		form.setTimeZone(TimeZone.getTimeZone("GMT"));
		String[] gmt = form.format(data.date()).split("[+-]");
		String[] timestr = gmt[0].split("[.,]");
		StringBuffer buf = new StringBuffer(timestr[0]);
		if (timestr.length == 2) {
			buf.append('.').append(timestr[1]);
			while (buf.lastIndexOf("0") == buf.length() - 1) {
				buf.setLength(buf.length() - 1);
			}
			if (buf.lastIndexOf(".") == buf.length() - 1) {
				buf.setLength(buf.length() - 1);
			}
		}
		buf.append('Z');
		return newPrimitiveOctets(buf.toString().getBytes());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET)
	 */
	@Override	
	public ConstructedOctets visit(SET data) throws ASN1EncodingException {
		return ((DerConstructedOctets)super.visit(data)).sortByTag();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET_OF)
	 */
	@Override	
	public ConstructedOctets visit(SET_OF<? extends ASN1Type> data)
			throws ASN1EncodingException {
		return ((DerConstructedOctets)super.visit(data)).sortByContents();
	}
	
	/**
	 * Encodes each element of the {@code ConstructiveType} data.
	 * 
	 * @param data
	 *            The data to be encoded.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */
	ConstructedOctets processConstructive(ConstructiveType data)
			throws ASN1EncodingException {
		ConstructedOctets octets = newConstructedOctets();
		ConstructiveType defaultData = null;
		for (ElementSpecification e : data.getElementTypeList()) {
			ASN1Type element = data.getComponent(e);
			if (element == null || !element.hasValue()) {
				continue;
			}
			if (e.hasDefault()) {
				if (defaultData == null) {
					defaultData = ASN1Type.instantiate(data.getClass());
				}
				if (element.equals(defaultData.getComponent(e))) {
					continue;
				}
			}
			octets.addElement(encode(element, e.tag(), null));
		}
		return octets;
	}
	
	/**
	 * Instantiates a new primitive octets.
	 * 
	 * @param contents The content octets.
	 * @return An instance of primitive octets.
	 */
	EncodedOctets newPrimitiveOctets(byte ...contents) {
		return new DerPrimitiveOctets(contents);
	}

	/**
	 * Instantiates a new constructed octets.
	 * @return An instance of constructed octets.
	 */
	ConstructedOctets newConstructedOctets() {
		return new DerConstructedOctets();
	}
	
	/**
	 * Abstract base class for classes represent BER encoded octets.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	abstract class DerOctets implements EncodedOctets {
		
		private ASN1TagClass tagClass;
		
		private int tagNumber;
		
		private byte[] encodedOctets;

		/**
		 * Returns DER encoded octets length includes prefix and contents.
		 * 
		 * @return Total length of BER encoded octets.
		 */
		public int totalLength() {
			return encodedOctets.length;
		}

		/**
		 * Sets identifier and length octets to fix this octets.
		 * 
		 * @param tagClass
		 *            The ASN.1 tag class for this data.
		 * @param tagNumber
		 *            The tag number for this data.
		 */
		public void fix(ASN1TagClass tagClass, int tagNumber) {
			this.tagClass = tagClass;
			this.tagNumber = tagNumber;
			byte[] identifier = encodeTag(tagClass, tagNumber, isConstructed());
			byte[] length = encodeLength(contentsLength());
			encodedOctets = new byte[identifier.length + length.length + contentsLength()];
			System.arraycopy(identifier, 0, encodedOctets, 0, identifier.length);
			System.arraycopy(length, 0, encodedOctets, identifier.length, length.length);
			int index = identifier.length + length.length;
			for (byte[] e: getContents()) {
				System.arraycopy(e, 0, encodedOctets, index, e.length);
				index += e.length;
			}
			clearContents();
		}

		/**
		 * Writes all BER encoded octets to the {@code OutputStream}
		 * 
		 * @param out
		 *            The stream to be written.
		 * @throws IOException
		 *             when {@code IOException} thrown by {@code OutputStream}.
		 */
		public int write(OutputStream out) throws IOException {
			out.write(encodedOctets);
			return encodedOctets.length;
		}

		/**
		 * Returns all content octets.
		 * 
		 * @return The array of content octets.
		 */
		abstract byte[][] getContents();
		
		/**
		 * Clears content octets.
		 */
		abstract void clearContents();

		/**
		 * Comparator used for sorting elements by the canonical order for tags.
		 * 
		 * @author WATANABE, Jun. <jwat at bitmeister.jp>
		 */
		class TagComparator implements Comparator<DerOctets> {

			/*
			 * (non-Javadoc)
			 * 
			 * @see java.util.Comparator#compare(java.lang.Object,
			 * java.lang.Object)
			 */
			public int compare(DerOctets o1, DerOctets o2) {
				if (o1.tagClass != o2.tagClass) {
					return o1.tagClass.ordinal() - o2.tagClass.ordinal();
				}
				return o1.tagNumber - o2.tagNumber;
			}

		}

		/**
		 * Comparator used for sorting elements by the encoded octets being
		 * compared as octet strings.
		 * 
		 * @author WATANABE, Jun. <jwat at bitmeister.jp>
		 */
		class OctetsComparator implements Comparator<DerOctets> {

			/*
			 * (non-Javadoc)
			 * 
			 * @see java.util.Comparator#compare(java.lang.Object,
			 * java.lang.Object)
			 */
			public int compare(DerOctets o1, DerOctets o2) {
				if (o1.encodedOctets.length != o2.encodedOctets.length) {
					return o1.encodedOctets.length - o2.encodedOctets.length;
				}
				for (int i = 0; i < o1.encodedOctets.length; i++) {
					if (o1.encodedOctets[i] != o2.encodedOctets[i]) {
						return (0xff & o1.encodedOctets[i]) - (0xff & o2.encodedOctets[i]);
					}
				}
				return 0;
			}

		}

	}

	/**
	 * Represents primitive BER octets.
	 * 
	 * <p>
	 * This class represents a BER encoded octets that has only one contents
	 * octets.
	 * </p>
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	class DerPrimitiveOctets extends DerOctets {
		 
		private byte[] contents;

		/**
		 * Instantiates a {@code PrimitiveOctets}.
		 * 
		 * @param contents
		 */
		DerPrimitiveOctets(byte... contents) {
			this.contents = contents;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see jp.bitmeister.asn1.codec.ber.BerOctets#isConstructed()
		 */
		public boolean isConstructed() {
			return false;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.ber.BerEncoder2.BerOctets#contentsLength()
		 */
		public int contentsLength() {
			return contents.length;
		}

		/* (non-Javadoc)
		 * @see jp.bitmeister.asn1.codec.ber.DerEncoder.DerOctets#getContents()
		 */
		byte[][] getContents() {
			return new byte[][] {contents};
		}

		/* (non-Javadoc)
		 * @see jp.bitmeister.asn1.codec.ber.DerEncoder.DerOctets#clearContents()
		 */
		void clearContents() {
			contents = null;
		}

	}

	/**
	 * Represents constructed BER octets.
	 * 
	 * <p>
	 * This class represents a BER encoded octets that contains some contents
	 * octets.
	 * </p>
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	class DerConstructedOctets extends DerOctets implements ConstructedOctets {

		private List<DerOctets> elements = new ArrayList<DerOctets>();

		private int length;

		/**
		 * Appends an element to this octets.
		 * 
		 * @param element
		 *            The element to be added.
		 */
		public void addElement(EncodedOctets element) {
			elements.add((DerOctets)element);
			length += element.totalLength();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see jp.bitmeister.asn1.codec.ber.BerOctets#isConstructed()
		 */
		public boolean isConstructed() {
			return true;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see jp.bitmeister.asn1.codec.ber.BerOctets#octetsLength()
		 */
		public int contentsLength() {
			return length;
		}
		
		/* (non-Javadoc)
		 * @see jp.bitmeister.asn1.codec.ber.DerEncoder.DerOctets#getContents()
		 */
		byte[][] getContents() {
			byte[][] contents = new byte[elements.size()][];
			for (int i = 0; i < elements.size(); i++) {
				contents[i] = elements.get(i).encodedOctets;
			}
			return contents;
		}
		
		/* (non-Javadoc)
		 * @see jp.bitmeister.asn1.codec.ber.DerEncoder.DerOctets#clearContents()
		 */
		void clearContents() {
			// nothing to do.
		}

		/**
		 * Sorts elements by the canonical order for tags.
		 * 
		 * @return This object.
		 */
		DerConstructedOctets sortByTag() {
			Collections.sort(elements, new TagComparator());
			return this;
		}

		/**
		 * Sorts elements by the encoded octets being compared as octet strings.
		 * 
		 * @return This object.
		 */
		DerConstructedOctets sortByContents() {
			Collections.sort(elements, new OctetsComparator());
			return this;
		}

	}

}
