/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.builtin;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1Tag;
import jp.bitmeister.asn1.annotation.ASN1XmlTypeName;
import jp.bitmeister.asn1.exception.ASN1IllegalArgument;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.PrimitiveType;

/**
 * Represents ASN.1 'OBJECT IDENTIFIER' type.
 * 
 * <p>
 * An instance of this class represents an 'OBJECT IDENTIFIER' type data and
 * contains a {@code List} of {@code Integer}. Each element of the {@code List}
 * represents an object id component.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
@ASN1BuiltIn
@ASN1Identifier("OBJECT IDENTIFIER")
@ASN1XmlTypeName("OBJECT_IDENTIFIER")
@ASN1Tag(tagClass = ASN1TagClass.UNIVERSAL, value = 6, tagMode = ASN1TagMode.IMPLICIT)
public class OBJECT_IDENTIFIER extends PrimitiveType<List<Integer>> {
	
	private static Map<String, Integer> NAMES_ROOT = new HashMap<String, Integer>();
	private static Map<String, Integer> NAMES_ITU_T = new HashMap<String, Integer>();
	private static Map<String, Integer> NAMES_ISO = new HashMap<String, Integer>();
	private static Map<String, Integer> NAMES_ITU_T_REC = new HashMap<String, Integer>();
	
	static {
		// top level
		NAMES_ROOT.put("itu-t", 0);
		NAMES_ROOT.put("ccitt", 0);
		NAMES_ROOT.put("iso", 1);
		NAMES_ROOT.put("joint-iso-itu-t", 2);
		NAMES_ROOT.put("joint-iso-ccitt", 2);
		// itu-t/ccitt
		NAMES_ITU_T.put("recommendation", 0);
		NAMES_ITU_T.put("question", 1);
		NAMES_ITU_T.put("administration", 2);
		NAMES_ITU_T.put("network-operator", 3);
		NAMES_ITU_T.put("identified-organization", 4);
		NAMES_ITU_T.put("r-recommendation", 5);
		// iso
		NAMES_ISO.put("standard", 0);
		NAMES_ISO.put("registration-authority", 1);
		NAMES_ISO.put("member-body", 2);
		NAMES_ISO.put("identified-organization", 3);
		// itu-t recommendation
		for (char c = 'a'; c <= 'z'; c++) {
			NAMES_ITU_T_REC.put(String.valueOf(c), c - 'a');
		} 
	}
	
	public static int nameFormToInt(List<Integer> list, String nameForm) {
		Integer number = null;
		if (list.size() == 0) {
			number = NAMES_ROOT.get(nameForm);
		}
		else if (list.size() == 1) {
			if (list.get(0) == 0) {
				number = NAMES_ITU_T.get(nameForm);
			}
			else if (list.get(0) == 1) {
				number = NAMES_ISO.get(nameForm);
			}
		}
		else if (list.size() == 2 && list.get(0) == 0 && list.get(1) == 0) {
			number = NAMES_ITU_T_REC.get(nameForm);
		}
		if (number == null) {
			ASN1IllegalArgument e = new ASN1IllegalArgument();
			e.setMessage("Invalid OID name form '" + nameForm + "'.", null, OBJECT_IDENTIFIER.class, null, null);
			throw e;
		}
		return number;
	}
	
	/**
	 * Instantiates an empty {@code OBJECT_IDENTIFIER}.
	 */
	public OBJECT_IDENTIFIER() {
	}

	/**
	 * Instantiates an {@code OBJECT_IDENTIFIER} and initialize it with the
	 * {@code List<Integer>} value.
	 * 
	 * @param value
	 *            the {@code List} of {@code Integer} represents an object
	 *            identifier.
	 */
	public OBJECT_IDENTIFIER(List<Integer> value) {
		set(value);
	}

	/**
	 * Instantiates an {@code OBJECT_IDENTIFIER} and initialize it with the
	 * array of {@code int} value.
	 * 
	 * @param value
	 *            the array of {@code int} represents an object identifier.
	 */
	public OBJECT_IDENTIFIER(int... value) {
		set(value);
	}

	/**
	 * Sets the array of {@code int} to the data.
	 * 
	 * @param value
	 *            the array of {@code int} represents an object identifier.
	 */
	public void set(int... value) {
		List<Integer> oids = new ArrayList<Integer>();
		for (int i : value) {
			oids.add(i);
		}
		set(oids);
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.PrimitiveType#cloneValue()
	 */
	@Override
	protected List<Integer> cloneValue() {
		List<Integer> clone = new ArrayList<Integer>();
		clone.addAll(value());
		return clone;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#accept(jp.bitmeister.asn1.processor.
	 * ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}

}
