/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.util.HashMap;
import java.util.Map;

/**
 * Manager of all ASN.1 modules.
 * 
 * <p>
 * This class manages all of the ASN.1 modules which are sub-class of
 * {@code ASN1Module} and provides {@code instantiate} methods which instantiate
 * an ASN.1 data from an ASN.1 tag or identifier.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Module
 */
public class ASN1ModuleManager {

	/**
	 * The manager of ASN.1 modules.
	 */
	private static ASN1ModuleManager instance = new ASN1ModuleManager();

	/**
	 * Instantiates an ASN.1 data specified by the ASN.1 tag and the tag class.
	 * 
	 * @param module 
	 *            The {@code Class} object of an {@code ASN1Module}.
	 * @param tagClass
	 *            The tag class.
	 * @param tagNumber
	 *            The tag number.
	 * @return An instance of ASN.1 data.
	 */
	public static ASN1Type instantiate(Class<? extends ASN1Module> module,
			ASN1TagClass tagClass, int tagNumber) {
		if (tagClass == ASN1TagClass.UNIVERSAL) {
			return specification(BuiltInModule.class).instantiate(tagClass,
					tagNumber);
		} else if (module != null) {
			return specification(module).instantiate(tagClass, tagNumber);
		}
		return null;
	}

	/**
	 * Instantiates an ASN.1 data specified by the type identifier.
	 * 
	 * @param module 
	 *            The {@code Class} object of an {@code ASN1Module}.
	 * @param typeIdentifier
	 *            The identifier of a type.
	 * @return An instance of ASN.1 data.
	 */
	public static ASN1Type instantiate(Class<? extends ASN1Module> module,
			String typeIdentifier) {
		ASN1Type data = specification(BuiltInModule.class).instantiate(
				typeIdentifier);
		if (data == null && module != null) {
			data = specification(module).instantiate(typeIdentifier);
		}
		return data;
	}

	/**
	 * Returns a {@code ModuleSpecification} instance contains specifications of
	 * the module.
	 * 
	 * @param module
	 *            The {@code Class} object of an {@code ASN1Module}.
	 * @return A {@code ModuleSpecification} instance.
	 */
	static ModuleSpecification specification(Class<? extends ASN1Module> module) {
		if (module == null) {
			module = BuiltInModule.class;
		}
		ModuleSpecification spec = instance.definedModules.get(module);
		if (spec == null) {
			spec = new ModuleSpecification(module);
			instance.definedModules.put(module, spec);
		}
		return spec;
	}

	/**
	 * Returns default tag mode of the module.
	 * 
	 * @param module
	 *            The {@code Class} object of the {@code ASN1Module}.
	 * @return Default tag mode of the module.
	 */
	static ASN1TagDefault tagDefault(Class<? extends ASN1Module> module) {
		return specification(module).tagDefault();
	}

	/**
	 * Returns identifier of the module.
	 * 
	 * @param module
	 *            The {@code Class} object of the {@code ASN1Module}.
	 * @return Identifier of this module.
	 */
	public static String identifier(Class<? extends ASN1Module> module) {
		return specification(module).identifier();
	}

	private Map<Class<? extends ASN1Module>, ModuleSpecification> definedModules = new HashMap<Class<? extends ASN1Module>, ModuleSpecification>();

	/**
	 * Instantiates an {@code ASN1ModuleManager}.
	 */
	private ASN1ModuleManager() {

	}

}
