package base;
import java.io.*;
import java.nio.charset.*;
import java.nio.*;
import java.util.*;
import java.util.logging.Level;

// char => String の変換マップ
class CodeStringTable{
	char[]   codes;
	String[] texts;
	int count;

	static int readChar(InputStream i) throws IOException{
		int a = i.read();
		int b = i.read();
		return ((a&255)<<8)+(b&255);
	}
		
	void load(String file) throws IOException{
		InputStream is = new BufferedInputStream(Util.OpenFile(file));
		count = readChar(is);
		codes = new char[count];
		texts = new String[count];
		for(int i=0;i<count;++i){
			codes[i] = (char)readChar(is);
			int length = is.read();
			byte[] tmp = new byte[length];
			int offset=0;
			while(length>0){
				int delta = is.read(tmp,offset,length);
				if(delta>0){
					length -= delta;
					offset +=delta;
				}
			}
			texts[i] = new String(tmp,"UTF-8");
		}
		is.close();
	}
	String find(int key){
		int low=0;
		int width = count;
		int mid;
		int r;
		while( width >0 ){
			mid = (width>>1);
			int sn = codes[low + mid];
			if(0==(r = key - sn )) return texts[low + mid];
			if(r<0){
				width = mid;
			}else{
				low += ++mid;
				width -= mid;
			}
		}
		return null;
	}
};


public class ConvertChar {
	/////////////////////////////////////////////////////////////////
	// 設定の類

	// JIS→Unicode時に数値実体参照をデコードするなら真
	public static boolean decode_NumericCharacterReferences = false;

	// ISO2022でエスケープされていない部分の文字コード
	public static void setBaseDecodeMode(int nv){ base_decode_mode=nv; }
	static int base_decode_mode =1;
		// 0 == Latin1
		// 1 == JIS X0201 or SJIS 

	// 半角カナ出力時のエスケープ
	static int kana_escapes_type = 0;
		/*
			0: J + shift + 7bit 
			1: J         + 8bit
			2: I         + 7bit
			3: I         + 8bit
		*/

	/////////////////////////////////////////////////////////////////
	// デバッグ

	// ログ出力
	public static java.util.logging.Logger logger = java.util.logging.Logger.getLogger("base.convertchar");

	// バイト配列を表示可能な文字列にする
	public static String HexDump(byte[] ba,int start,int end){
		StringBuffer tmp = new StringBuffer();
		char[] a=new char[1];
		for(int i=start;i<end;++i){
			byte c= ba[i];
			if(c >=0x20 && c <=0x7e && c != '%' ){
				tmp.append((char)c);
			}else{
				tmp.append('%');
				String Hex=Integer.toHexString(c+256);
				tmp.append(Hex.substring(Hex.length()-2));
			}
		}
		return new String(tmp);
	}

	/////////////////////////////////////////////////////////////////

	// 変換できない記号や半角カナを安全な文字列に変換するためのマップ
	static CodeStringTable j2safe = new CodeStringTable();
	static CodeStringTable u2safe = new CodeStringTable();

	/////////////////////////////////////////////////////////////////
	// Unicode(JIS) => Unicode(MS932 or system default)
	// AWT heavy component でも画面に表示できるようにするための変換
	static char[][] ary_ToDisplay={
		{/* \  */ 0x005C, 0x00A5},
		{/* ~  */ 0x007E, 0x203E},
		{/* ￠ */ 0x00A2, 0xFFE0},
		{/* ￡ */ 0x00A3, 0xFFE1},
		{/* ￢ */ 0x00AC, 0xFFE2},
		{/* ― */ 0x2015, 0x2014},
		{/* ∥ */ 0x2016, 0x2225},
		{/* … */ 0x2026, 0x22EF},
		{/* － */ 0x2212, 0xFF0D},
		{/* ～ */ 0x301C, 0xFF5E},
	};
	public static String convertForDisplay(String from){
		StringBuffer dst = new StringBuffer();
		FromLoop: for(int i=0;i<from.length();++i){
			char f=from.charAt(i);
			if(f!=0){
				// JISをそのまま表示できない環境(たぶんMS932)のための変換
				// (表示できる文字は表のキーが0になっててスキップされるはず)
				for(int j=0;j<ary_ToDisplay.length;++j){
					if(f!=ary_ToDisplay[j][0]) continue;
					dst.append(ary_ToDisplay[j][1]);
					continue FromLoop;
				}
			}
			// 機種依存文字を無害化する
			String s = null; // u2safe.find(f);
			if(s!=null) dst.append(s);
			else dst.append(f);
		}
		return new String(dst);
	}

	/////////////////////////////////////////////////////////////////
	// 初期化と環境の調査

	static int FirstCall =0;
	public static void Setup() throws IOException {
		if(FirstCall==0){
			FirstCall=1;

			j2safe.load("j2safe.dat");
			u2safe.load("u2safe.dat");


			// JISとunicodeの変換テーブルを読む
			{
				for(int i=0;i<ary_UnicodeToJIS.length;++i) ary_UnicodeToJIS[i]=0;
				Reader isr = new BufferedReader(new InputStreamReader(Util.OpenFile("u2j208.dat"),"UTF-16BE"));
				int count=0;
				for(;;){
					int c1 = isr.read();
					int c2 = isr.read();
					if(c1==-1 || c2==-1) break;
					ary_UnicodeToJIS[c1] = (char)c2;
					++count;
				}
				logger.info("ary_UnicodeToJIS:read "+count+" entrys");
				isr.close();
			}
			{
				for(int i=0;i<ary_JIS208ToUnicode.length;++i) ary_JIS208ToUnicode[i]=0;
				Reader isr = new BufferedReader(new InputStreamReader(Util.OpenFile("j2u208.dat"),"UTF-16BE"));
				int count=0;
				for(;;){
					int c1 = isr.read();
					int c2 = isr.read();
					if(c1==-1 || c2==-1) break;
					ary_JIS208ToUnicode[c1] = (char)c2;
					++count;
				}
				logger.info("ary_JIS208ToUnicode:read "+count+" entrys");
				isr.close();
			}

			char[] c= new char[1];
			String s;
			String s2;
			byte[] b;
			boolean a1,a2;

			// コンバータの有無を確認する
			// アプレットはプロパティを読めない SysEnc= System.getProperty( "file.encoding" );

			// decorder for other
			try{base_decorder = Charset.forName("ISO-8859-1").newDecoder();}catch(Throwable e){}
			try{x201_decorder = Charset.forName("JIS0201").newDecoder();}catch(Throwable e){}
			//このコンバータは出力専用らしい try{jis212_decorder = Charset.forName("JIS0212").newDecoder();}catch(Throwable e){}

			// ary_ToDisplay の各文字について、
			// 有効かどうか確認する
			StringBuffer log_sb = new StringBuffer();
			log_sb.append("ToDisplay:");
			for(int i=0;i<ary_ToDisplay.length;++i){
				a1=a2=false;
				// in 
				{
					c[0]=ary_ToDisplay[i][0];
					s = new String(c);
					a1= s.equals(new String(s.getBytes()));
				}
				// out 
				{
					c[0]=ary_ToDisplay[i][1];
					s = new String(c);
					a2= s.equals(new String(s.getBytes()));
				}

				if(a1 == a2 ){
					// 両方とも使えるか両方とも使えないなら、変換しない
					ary_ToDisplay[i][0] =0;
				}else{
					if(!a2){
						// 逆方向に変換するべき
						c[0]=ary_ToDisplay[i][0];
						ary_ToDisplay[i][0]=ary_ToDisplay[i][1];
						ary_ToDisplay[i][1]=c[0];
					}
					log_sb.append(" 0x"
						+Integer.toHexString(ary_ToDisplay[i][0])
						+"=>0x"
						+Integer.toHexString(ary_ToDisplay[i][1])
					);
				}
			}
			logger.info(log_sb.toString());
		}
	//	System.err.println(JISToUnicode(UnicodeToJIS("のえ仨のえぷう",0,-1)));
	//	System.exit(0);
	}

	////////////////////////////////////////////////////////////////////////
	// JIS から Unicode への変換

	// 変換テーブル
	public static char[] ary_JIS208ToUnicode =new char[65536];

	private static ByteBuffer decodejis_bb= ByteBuffer.allocate(16);

	// エンティティのデコードが可能かどうかはフォントによって異なる
	static java.awt.Font font=new java.awt.Font("Dialog",0,12);

	// Unicode(MS932) => Unicode(JIS)
	// JISに変換できる状態にするための前変換
	static CharsetDecoder base_decorder;
	static CharsetDecoder jis212_decorder;
	static CharsetDecoder x201_decorder;

	private static final byte[] ary_BrokenJISToX0213_2
	=new byte[]{1, 8, 3, 4, 5,12,13,14,15,78,79,80,81,82,83,84,85,86,87,88,89,90,91,92,93,94,};

	private final static void decodeX201(StringBuffer sb,int b,char mode,boolean shift_out){
		b &= 255;
		// System.err.println("code="+Integer.toHexString(b)+" mode="+mode+" shift="+shift_out);

		if(b<0x80){
			// 7bit コードが渡された場合はカナか英数かを決める必要がある

			// irc.infovalley.ad.jp のMOTDは X0201英数をこのエスケープで送ってくる
			if(mode=='J' && !shift_out ){
				// 英数のまま扱う
				sb.append((char)b);
				return;
			}

			// Cotton と LimeChatへの対応
			if((mode=='J' && shift_out ) // ESC (J + shiftout +7bit
			||  mode=='I'                // ESC (I + 7bit
			){
				// 7bitコードが渡されたと判断する
			}else{
				// 知らないタイプなのでログに記録する
				logger.finer("mode="+mode+" shiftout="+shift_out+" code=0x"+Integer.toHexString(b));
			}

			// 下記で | 0x80 して変換
		}
		// X201の右側で変換
		if(x201_decorder!=null){
			decodejis_bb.clear();
			decodejis_bb.put( (byte)(b|0x80) );
			decodejis_bb.flip();
			try{
				CharBuffer cb = x201_decorder.decode(decodejis_bb);
				sb.append( cb.array(),cb.position(),cb.remaining());
				return;
			}catch(Throwable e){}
		}
		sb.append( "(x201%"+Integer.toHexString(b)+")" );
	}

	private static byte[] jis212_escape=new byte[]{ 0x1b,'$','D'};
	static  byte[] src212 = new byte[2];
	private final static void decodeJIS212(StringBuffer sb,byte b1,byte b2){
		if(jis212_decorder !=null){
			decodejis_bb.clear();
			decodejis_bb.put(jis212_escape);
			decodejis_bb.put(b1);
			decodejis_bb.put(b2);
			decodejis_bb.flip();
			try{
				CharBuffer cb = jis212_decorder.decode(decodejis_bb);
				String s = new String( cb.array(),cb.position(),cb.remaining());
				if(s!=null && s.length()>0){
					// System.err.println("x212 "+s);
					sb.append(s);
					return;
				}
			}catch(Throwable e){}
		}
		sb.append( "(jis212%"+Integer.toHexString(b1&255)+"%"+Integer.toHexString(b2&255)+")");
	}

	public static String JISToUnicode(byte[] ba){ return JISToUnicode(ba,0,ba.length);}
	public static String JISToUnicode(byte[] ba,int arg_start,int end){
		int i=arg_start;
		StringBuffer sb = new StringBuffer();
	//	System.err.println(HexDump(ba,arg_start,end));

		LOOP: while(i<end){
			// ESC が登場するまではbase_decorder でデコード
			if( base_decode_mode ==0 ){
				// Latin1 扱い
				int start=i;while( i<end && ba[i]!=0x1B ) ++i;
				if(i>start){
					try{
						CharBuffer cb = base_decorder.decode(ByteBuffer.wrap(ba,start,i-start));
						sb.append( cb.array(),cb.position(),cb.remaining());
					}catch(Throwable e){
						logger.log(Level.SEVERE,HexDump(ba,start,i),e);
						for(int j=start;j<i;++j) sb.append( ba[j]&255 );
					}
				}
			}else{
				// エスケープが出るまではX201かSJIS
				int b,c1,c2,b2;
				char u;
				while( i<end ){
					b=255&ba[i];
					if(b == 0x1B ) break;
					++i;
					if( b <  0x80 ){ sb.append( (char)b ); continue; } // ascii
					if( b >= 0xa1 && b <= 0xdf ){ decodeX201(sb,b,'b',false); continue; } // x201

					// sjis ?
					if( i<end){
						b2 = 255&ba[i]; // sjis  trail byte
						if( b  >= 0x81 &&  b  <= 0xfc 
						&&  b2 >= 0x40 &&  b2 <= 0xfc && b2 != 0x7f 
						){
							++i;
							int jis = 
								(b<<9)+(b<=0x9f?-(0xe1<<8):-(0x161<<8))
								+b2+
								(b2<=0x7e?-0x1f
								:b2<=0x9e?-0x20
								:256-0x7e)
								;
							// jisに変換してJ2uのテーブルを参照
							u = ary_JIS208ToUnicode[ jis ];
							if( u!=0 ){
								sb.append(u); 
							}else{
								String s = j2safe.find(jis);
								if(s!=null){
									sb.append(s);
								}else{
									sb.append( "(SJIS-0x"+Integer.toHexString(b)+Integer.toHexString(b2)+")");
								}
							 }
							continue;
						}
					}

					// control char
					sb.append( "%"+Integer.toHexString(b) );
				}
			}

			int maxescape = end - 3;
			int page = 0;
			for(;i<end;++i){
				switch(ba[i]){
				// 空白なら必ずASCIIに戻す
				case 0x20: page= 0; continue LOOP;

				case 0x0E: // shift out
					if(page==72010 ){ page=72011; continue; }
					if(page==82010 ){ page=82011; continue; }
					page=0;
					continue LOOP;

				case 0x0F: // shift in
					if(page==72011){ page=72010; continue; }
					if(page==82011){ page=82010; continue; }
					page=0;
					continue LOOP;

				case 0x1B:
					if(i<= maxescape){
						switch(ba[i+1]){
						case '(':
							switch(ba[i+2]){
							case 'B':page=    0;i+=2; continue;	// ASCII ESC (B アスキー
							case 'I':page=72010;i+=2; continue;	// JIS X 0201 片仮名 7ビット半角カナの開始
							case 'J':page=82010;i+=2; continue;	// JIS X0201(LH) ESC (J 半角カナ
							}
							break;
						case '$':
							switch(ba[i+2]){
							case '@': page=2131; i+=2; continue;	// ESC $@ JIS X 0213の1面(include JIS X0208('78)) 
							case 'B': page= 208; i+=2; continue;	// JIS X0208('83) 
							case 'I': page= 201; i+=2; continue;	// 8ビット半角カナの開始
							case '(':
								if(i<maxescape)
								switch(ba[i+3]){
								case 'O': page=2131; i+=3; continue;	// JIS X 0213の1面(include JIS X0212?)
								case 'P': page=2132; i+=3; continue;	// JIS X 0213の2面
								case 'D': page=212;  i+=3; continue;	// JIS X0212  補助漢字
								}
								break;
							}
							break;
						}
						logger.warning(HexDump(ba,i,i+3));
					}
					sb.append(ba[i++]);
					// どれでもないなら一旦エスケープを解除する
					continue LOOP;
				}
				switch(page){
				default: continue LOOP;
				// 半角カナ
				case 201  : decodeX201(sb,ba[i],' ',false); continue;
				// roman ESC (J
				case 82010: decodeX201(sb,ba[i],'J',false); continue;
				case 82011: decodeX201(sb,ba[i],'J',true ); continue;
				// junet ESC (I
				case 72010: decodeX201(sb,ba[i],'I',false); continue;
				case 72011: decodeX201(sb,ba[i],'I',true ); continue;

				case 2131: // 0213-1 いわゆるJIS漢字
				case  208: // 0213-1 は 0208の上位互換
					{
						int c1,c2;
						if( i == end-1
						||  (c1=(255&ba[i]  )) <=0x20
						||  (c2=(255&ba[i+1])) <=0x20
						)   continue LOOP;
						// woolは ローマ数字の3 ⅲ (8ビット目が立っている)などを平気で送ってくる
						int jis = c2|(c1<<8);
						char u = ary_JIS208ToUnicode[jis];
						if(u!=0){
							sb.append(u);
						}else{
							String s = j2safe.find(jis);
							if(s!=null){
								sb.append(s);
							}else{
								sb.append( "(JIS1面-0x"+Integer.toHexString(c2|(c1<<8) )+")");
							}
						}
					}
					++i;continue;
				case 2132: // 0213-2 補助漢字
					{
						int c1,c2;
						if( i == end-1
						||  (c1=(255&ba[i]  )) <=0x20
						||  (c2=(255&ba[i+1])) <=0x20
						)   continue LOOP;

						// 区番号からオフセットを得る
						// オフセットはシフトJISを古い式でJISに変換した際の区番号を意味する
						int k=-1;
						for(int j=0;j<ary_BrokenJISToX0213_2.length;++j){
							if(ary_BrokenJISToX0213_2[j]==c1-0x20){
								k=j+0x7f;
								break;
							}
						}
						if(k==-1) continue LOOP;

						char u = ary_JIS208ToUnicode[c2 |( k <<8)];
						if(u!=0){
							sb.append(u);
						}else{
							sb.append( "(JIS2面-0x"+Integer.toHexString(c1)+Integer.toHexString(c2)+")");
						}
					}
					++i;continue;

				case 212: // sjis には 0212 はない
					if(i==end-1 || ba[i] <= 0x20 ) continue LOOP;
					decodeJIS212(sb,ba[i],ba[i+1]);
					++i;continue;
				}
			}
		}
		String src = sb.toString();
		if(decode_NumericCharacterReferences 
		&& -1 != src.indexOf("&#")
		){
			sb= new StringBuffer();
			// &#H; &#HHHH; をunicodeに変換する
			char entity=0;
			int e_start;
			int e_end=0;
			for(int start=0;start<src.length();){
				e_start=start;
				CHECK: while( e_start<src.length() ){
					e_start=src.indexOf("&#",e_start);
					if(e_start==-1){ e_end=e_start=src.length(); break; }

					int n_start= e_start+2;

					e_end = src.indexOf(';',n_start);
					if(e_end==-1 ){ e_end=e_start=src.length(); break; }

					int radix=10;
					int maxlen = 5;
					if(-1!= "xX".indexOf(src.charAt(n_start))){
						++n_start;
						radix=16;
						maxlen = 4;
					}

					if( e_end-n_start<1 || e_end-n_start >maxlen){ e_start=e_end+1; continue CHECK; }

					// check
					boolean flag=true;
					for(i=n_start;i<e_end;++i){
						char c = src.charAt(i);
						if( !( c>='a' && c<='f')
						&&  !( c>='A' && c<='F')
						&&  !( c>='0' && c<='9')
						){ e_end=e_start=i; continue CHECK; }
					}
					try{
						logger.finer("decode entity "+src.substring(n_start,e_end)+" radix="+radix);
						entity = (char)Integer.parseInt(src.substring(n_start,e_end),radix);
					}catch(Throwable e){ e_start=e_end+1; continue CHECK; }
					break;
				}
				if(e_start > start){
					sb.append(src.substring(start,e_start));
					start=e_start;
				}
				if(e_end>e_start){
					start=e_end+1;
					// 表示できる文字なのか知りたい
					if(!font.canDisplay(entity)){
						// 現在のデフォルトエンコーディングでは表現できない
						logger.fine(src.substring(e_start,start)+"は現在の環境では表示できない");
						sb.append(src.substring(e_start,start));
					}else{
						sb.append(entity);
					}
				}
			}
			src= sb.toString();
		}
		return src;
	}

	///////////////////////////////////////////
	// Unicode => JIS

	static char[] ary_UnicodeToJIS = new char[65536];
	static final byte[] nojis_result =new byte[]{'?'};
	static final byte[] kana_escapesJ = new byte[]{0x1b,'(','J'};
	static final byte[] kana_escapesI = new byte[]{0x1b,'(','I'};
	static final Object[][] encode_escapes=new Object[][]{
		{  null   ,new byte[]{ 0x1b,'(','B'}},
		{"JIS0208",new byte[]{ 0x1b,'$','B'}},
		//出力だけ対応しても仕方ない {"JIS0212",new byte[]{ 0x1b,'$','(','D'}},
		{"JIS0201",null},
	};

	public static byte[] UnicodeToJIS(String src,int start,int end){
		if(end<0) end=src.length();

		ByteArrayOutputStream UnicodeToJIS_bao = new ByteArrayOutputStream();
		byte[] byte2 = new byte[2];

		int mode = 0;
		boolean need_shift_in=false;
		char c;
		String s;
		int i;

		try{
			UnicodeToJIS_bao.reset();
			// unicodeを1文字ずつ
			for(i=start;i<end;++i){
				c = src.charAt(i);
				s=null;

				// 機種依存文字を無害化する
				s = u2safe.find(c);
				int max =(s!=null?s.length():1);
				for(int j=0;j<max;++j){
					if(s!=null) c = s.charAt(j);

					// 1文字ずつ

					int new_mode =0;
					boolean new_need_shift_in = false;
					byte[] encode_result = byte2;
					byte[] new_escape    = null;
					int    encode_result_length= 1;

					if(c<=0x80){ 
						// ASCIIの範囲はそのまま
						byte2[0]=(byte)c;
					}else{
						// テーブル参照
						char jis = ary_UnicodeToJIS[c];
						if(jis == 0 ){
							//変換できないなら数値実体参照
							new_mode=0;
							String hex = "&#x"+Integer.toHexString((int)c)+";";
							try{ encode_result = (hex).getBytes("UTF-8"); }
							catch(Throwable e){
								logger.severe("encode entity failed");
								encode_result=nojis_result;
							}
							encode_result_length = encode_result.length;
							//	System.err.println("code="+c+" entity="+hex);
						}else if( jis >= 0x100 ){
							// JIS 0208
							new_mode=1;
							byte2[0] = (byte)(jis>>8);
							byte2[1] = (byte)(jis&255);
							encode_result_length =2;
						}else if( jis >= 0xa0 && jis <= 0xdf ){
							 // JIS X0201 半角カナ
							switch(kana_escapes_type){
							case 0: new_mode=2010;new_escape=kana_escapesJ;jis&=0x7F;new_need_shift_in=true;break;
							case 1: new_mode=2011;new_escape=kana_escapesJ;          break;
							case 2: new_mode=2012;new_escape=kana_escapesI;jis&=0x7F;break;
							case 3: new_mode=2013;new_escape=kana_escapesI;          break;
							}
							byte2[0]=(byte)jis;
						}else{
							// ascii 
							new_mode=0;
							byte2[0]=(byte)jis;
						}
					}
					// エスケープが異なるならエスケープを出力
					if( mode != new_mode ){
						mode = new_mode;
						if( need_shift_in ) UnicodeToJIS_bao.write(0x0f);
						UnicodeToJIS_bao.write( new_escape!=null ? new_escape : (byte[])encode_escapes[mode][1] );
						if( (need_shift_in = new_need_shift_in) ) UnicodeToJIS_bao.write(0x0e);
					}
					// 変換された文字をバッファに追加
					UnicodeToJIS_bao.write(encode_result,0,encode_result_length);
				}
			}
			// エスケープを閉じる
			if(mode != 0 ){
				if( need_shift_in ) UnicodeToJIS_bao.write(0x0f);
				UnicodeToJIS_bao.write((byte[]) encode_escapes[0][1]);
			}
			return UnicodeToJIS_bao.toByteArray();
		}catch(IOException e){
			logger.log(java.util.logging.Level.WARNING,"UnicodeToJIS",e);
			return new byte[0];
		}
	}
}

/*

JIS X 0201 については
http://www2d.biglobe.ne.jp/~msyk/charcode/jisx0201kana/
を参照

CTCP文字装飾の例
	%03 01  // color
	%1b$B?';XDj%1b(B
	%02 //bold
	%1b$B %25\!<%25k%25I %1b(B
	%16 // 反転
	%1b$B H?E> %1b(B
	%1f // アンダーライン
	%1b$B %25"%25s%25@!<%25i%25$%25s%1b(B
	%0f //CTCP 打ち消し
	%1b$B BG$A>C$7 %1b(B

CTCP文字装飾とSIの絡み
ﾊﾝｶｸ<O>ABC日本語ABC<O>ﾊﾝｶｸ
roman 7bit
	%1b(J %0e J]68 %0f
	%1b(B %0f ABC
	%1b$B F|K\8l
	%1b(B ABC %0f 
	%1b(J %0eJ]68%0f
	%1b(B

roman 8bit
	%1b(J %ca%dd%b6%b8
	%1b(B %0f ABC
	%1b$B F|K\8l
	%1b(B ABC %0f 
	%1b(J %ca%dd%b6%b8
	%1b(B

junet 7bit
	%1b(I J]68
	%1b(B %0f ABC
	%1b$B F|K\8l 
	%1b(B ABC %0f 
	%1b(I J]68
	%1b(B

junet 8bit
	%1b(I %ca%dd%b6%b8 
	%1b(B %0f ABC
	%1b$B F|K\8l 
	%1b(B ABC %0f 
	%1b(I %ca%dd%b6%b8 
	%1b(B

*/
