class ASINFilter < BloggerPost::Filter
  def translate(html)
    amazon_config = @config.config_file_hash['amazon']
    unless amazon_config && amazon_config['awsaccesskeyid'] && amazon_config['associateid']
      puts 'Please configurate for Amazon'
      return html
    end
    html = filter_attr(html, 'bpasin') do |param|
      asin, display_type, image_size = parse_param(param)
      xml = load_cache(asin)
      unless xml
        xml = get_xml(asin, amazon_config['awsaccesskeyid'], amazon_config['associateid']) unless xml
        save_cache(asin, xml)
      end
      url, title, author, small_image, medium_image, large_image, publisher, date, binding = parse_xml(xml)
      case image_size
      when 'small'
        image_url = small_image
        if !image_url || image_url == ''
          image_url = 'http://ec1.images-amazon.com/images/G/09/nav2/dp/no-image-no-ciu._AA80_.gif'
        end
        margin = 0
        line_height = 100
      when 'medium'
        image_url = medium_image
        if !image_url || image_url == ''
          image_url = 'http://ec1.images-amazon.com/images/G/09/nav2/dp/no-image-no-ciu._AA120_.gif'
        end
        margin = 5
        line_height = 120
      when 'large'
        image_url = large_image
        if !image_url || image_url == ''
          image_url = 'http://ec1.images-amazon.com/images/G/09/nav2/dp/no-image-no-ciu.gif'
        end
        margin = 10
        line_height = 150
      end
      title_element = "<a href=\"#{url}\">#{title}</a>"
      image_element = "<a href=\"#{url}\"><img src=\"#{image_url}\" alt=\"#{title}\" title=\"#{title}\" style=\"border: none;\" /></a>"
      case display_type
      when 'title'
        title_element
      when 'image'
        image_element
      when 'detail'
        if author
          author_element = author + '<br />'
        else
          author_element = ''
        end
        "<div><div style=\"float:left;\">#{image_element}</div><div style=\"float:left;margin-left:#{margin}px;line-height:#{line_height}%\"><div style=\"margin-top: #{margin}px;margin-bottom:#{margin}px;\">#{title_element}</div>#{author_element}#{publisher}<br />#{date}<br />#{binding}</div><div style=\"clear: left\"></div></div>"
      end
    end
    html
  end

  private

  def get_xml_text(elements, xpath)
    return nil unless elements[xpath]
    elements[xpath].text
  end

  def parse_xml(xml)
    item = REXML::Document.new(xml).elements['/ItemLookupResponse/Items/Item'].elements
    item_attributes = item['ItemAttributes'].elements
    url = get_xml_text(item, 'DetailPageURL')
    small_image = get_xml_text(item, 'SmallImage/URL')
    medium_image = get_xml_text(item, 'MediumImage/URL')
    large_image = get_xml_text(item, 'LargeImage/URL')
    title = get_xml_text(item_attributes, 'Title')
    author = get_xml_text(item_attributes, 'Author')
    publisher = get_xml_text(item_attributes, 'Publisher')
    date = get_xml_text(item_attributes, 'PublicationDate')
    date = get_xml_text(item_attributes, 'ReleaseDate') unless date
    binding = get_xml_text(item_attributes, 'Binding')
    [url, title, author, small_image, medium_image, large_image, publisher, date, binding]
  end

  def parse_param(param)
    param_list = param.split(':')
    asin = nil
    image_size = 'medium'
    display_type = 'title'
    while param_list.length > 0
      element = param_list.shift
      case element
      when /title|image|detail/
        display_type = element
      when /small|large/
        image_size = element
      else
        asin = element
      end
    end
    [asin, display_type, image_size]
  end

  def cache_path(asin)
    @config.work_dir + 'asin/' + asin + '.xml'
  end

  def load_cache(asin)
    path = cache_path(asin)
    return nil unless File.exist?(path)
    File.read(path)
  end

  def save_cache(asin, xml)
    path = cache_path(asin)
    FileUtils.makedirs(File.dirname(path))
    File.open(path, 'w') {|f| f.write(xml) }
  end

  def get_xml(asin, aws_access_key_id, associate_id)
    if asin =~ /^\d{13}$/
      type = 'ISBN'
    else
      type = 'ASIN'
    end
    request_url = 'http://webservices.amazon.co.jp/onca/xml?'
    option_list = ['Service=AWSECommerceService',
                   'Version=2007-01-15',
                   "AWSAccessKeyId=#{aws_access_key_id}",
                   "AssociateTag=#{associate_id}",
                   'Operation=ItemLookup',
                   "IdType=#{type}",
                   "ItemId=#{asin}",
                   'ResponseGroup=Medium']
    request_url << option_list.join('&')
    res = http_get(@config, URI.parse(request_url))
    res.body
  end
end

BloggerPost.register_filter(ASINFilter)
