/*
 * blanco Framework
 * Copyright (C) 2011 IGA Tosiki
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.beanutils;

import java.util.List;

import blanco.cg.BlancoCgObjectFactory;
import blanco.cg.valueobject.BlancoCgClass;
import blanco.cg.valueobject.BlancoCgField;
import blanco.cg.valueobject.BlancoCgMethod;
import blanco.cg.valueobject.BlancoCgSourceFile;

public class BlancoBeanUtils {
    /**
     * copyTo メソッドを生成します。
     * 
     * @param cgSourceFile
     *            ソースコード情報。入力に利用します。
     * @param cgClass
     *            クラス情報。入出力に利用します。
     */
    public static void generateCopyToMethod(
            final BlancoCgSourceFile cgSourceFile, final BlancoCgClass cgClass) {

        // BlancoCgObjectFactoryクラスのインスタンスを取得します。
        final BlancoCgObjectFactory cgFactory = BlancoCgObjectFactory
                .getInstance();

        final BlancoCgMethod method = cgFactory.createMethod("copyTo",
                "このバリューオブジェクトを指定のターゲットに複写します。");
        cgClass.getMethodList().add(method);

        method.getLangDoc().getDescriptionList().add("<P>使用上の注意</P>");
        method.getLangDoc().getDescriptionList().add("<UL>");
        method.getLangDoc().getDescriptionList().add(
                "<LI>オブジェクトのシャロー範囲のみ複写処理対象となります。");
        method.getLangDoc().getDescriptionList().add(
                "<LI>オブジェクトが循環参照している場合には、このメソッドは使わないでください。");
        method.getLangDoc().getDescriptionList().add("</UL>");

        method.getParameterList().add(
                cgFactory.createParameter("target", cgSourceFile.getPackage()
                        + "." + cgClass.getName(), "target value object."));

        final List<java.lang.String> listLine = method.getLineList();

        // TODO ディープコピーの考慮の追加。

        listLine.add("if (target == null) {");
        listLine.add("throw new IllegalArgumentException(\"Bug: "
                + cgClass.getName()
                + "#copyTo(target): argument 'target' is null\");");
        listLine.add("}");

        listLine.add("");
        listLine.add("// No needs to copy parent class.");
        listLine.add("");

        for (int indexField = 0; indexField < cgClass.getFieldList().size(); indexField++) {
            final BlancoCgField field = cgClass.getFieldList().get(indexField);

            // 配列情報を調整。
            boolean isArray = field.getType().getArray();
            if (isArray == false && field.getType().getName().endsWith("[]")) {
                isArray = true;
            }

            listLine.add("// Name: " + field.getName());
            listLine.add("// Type: " + field.getType().getName());

            if (field.getStatic()) {
                listLine.add("//   skipped (static field)");
                continue;
            }
            if (field.getFinal()) {
                listLine.add("//   skipped (final field)");
                continue;
            }

            if (isArray == false) {
                listLine.add(getCopyFieldLine(field, "this." + field.getName(),
                        "target." + field.getName()));
            } else {
                String typeNameWithoutArray = field.getType().getName();
                for (; typeNameWithoutArray.endsWith("[]");) {
                    typeNameWithoutArray = typeNameWithoutArray.substring(0,
                            typeNameWithoutArray.length() - "[]".length());
                }

                listLine.add("if (this." + field.getName() + " != null) {");
                listLine.add("target." + field.getName() + " = new "
                        + typeNameWithoutArray + "[this." + field.getName()
                        + ".length];");
                listLine.add("for (int index = 0; index < this."
                        + field.getName() + ".length; index++) {");
                listLine.add("// copy contents...");
                listLine.add(getCopyFieldLine(field, "this." + field.getName()
                        + "[index]", "target." + field.getName() + "[index]"));
                listLine.add("}");
                listLine.add("}");
            }
        }
    }

    static String getCopyFieldLine(final BlancoCgField field,
            final String origVarName, final String destVarName) {
        String typeNameWithoutArray = field.getType().getName();
        for (; typeNameWithoutArray.endsWith("[]");) {
            typeNameWithoutArray = typeNameWithoutArray.substring(0,
                    typeNameWithoutArray.length() - "[]".length());
        }

        if (typeNameWithoutArray.equals("boolean")
                || typeNameWithoutArray.equals("char")
                || typeNameWithoutArray.equals("byte")
                || typeNameWithoutArray.equals("short")
                || typeNameWithoutArray.equals("int")
                || typeNameWithoutArray.equals("long")
                || typeNameWithoutArray.equals("float")
                || typeNameWithoutArray.equals("double")) {
            // プリミティブ型
            return (destVarName + " = " + origVarName + ";");
        } else if (typeNameWithoutArray.equals("java.lang.Boolean")
                || typeNameWithoutArray.equals("java.lang.Character")
                || typeNameWithoutArray.equals("java.lang.Byte")
                || typeNameWithoutArray.equals("java.lang.Short")
                || typeNameWithoutArray.equals("java.lang.Integer")
                || typeNameWithoutArray.equals("java.lang.Long")
                || typeNameWithoutArray.equals("java.lang.Float")
                || typeNameWithoutArray.equals("java.lang.Double")) {
            // プリミティブ型のラッパークラス。
            return (destVarName + " = " + origVarName + ";");
        } else if (typeNameWithoutArray.equals("java.lang.String")) {
            return (destVarName + " = " + origVarName + ";");
        } else if (typeNameWithoutArray.equals("java.math.BigDecimal")) {
            return (destVarName + " = " + origVarName + ";");
        } else if (typeNameWithoutArray.equals("java.util.Date")) {
            return (destVarName + " = (" + origVarName
                    + " == null ? null : new Date(" + origVarName + ".getTime()));");
        } else {
            // TODO その他の型をひととおりサポートすること。
            return ("// フィールド[" + field.getName() + "]はサポート外の型["
                    + field.getType().getName() + "]です。");
        }
    }
}
