/*
 * net/balusc/http/multipart/MultipartMap.java
 *
 * Copyright (C) 2009 BalusC
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this library.
 * If not, see <http://www.gnu.org/licenses/>.
 */
package net.balusc.http.multipart;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.Servlet;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.Part;

/**
 * MultipartMap 実装です。<code>HttpServletRequest#getParameterXXX()</code>
 * メソッドをシュミレートして <code>@MultipartConfig</code> サーブレット中の処理を簡単にします。You can access
 * the normal request parameters by <code>{@link #getParameter(String)}</code>
 * and you can access multiple request parameter values by
 * <code>{@link #getParameterValues(String)}</code>.
 * <p>
 * On creation, the <code>MultipartMap</code> will put itself in the request
 * scope, identified by the attribute name <code>parts</code>, so that you can
 * access the parameters in EL by for example <code>${parts.fieldname}</code>
 * where you would have used <code>${param.fieldname}</code>. In case of file
 * fields, the <code>${parts.filefieldname}</code> returns a
 * <code>{@link UploadedFile}</code>.
 * <p>
 * It was a design decision to extend <code>HashMap&lt;String, Object&gt;</code>
 * instead of having just <code>Map&lt;String, String[]&gt;</code> and
 * <code>Map&lt;String, File&gt;</code> properties, because of the accessibility
 * in Expression Language. Also, when the value is obtained by
 * <code>{@link #get(Object)}</code>, as will happen in EL, then multiple
 * parameter values will be converted from <code>String[]</code> to
 * <code>List&lt;String&gt;</code>, so that you can use it in the JSTL
 * <code>fn:contains</code> function.
 * 
 * @author BalusC
 * @link http://balusc.blogspot.com/2009/12/uploading-files-in-servlet-30.html
 */
public class MultipartMap extends HashMap<String, Object> {
	/**
	 * バージョン番号。
	 */
	private static final long serialVersionUID = 1L;

	// 定数
	// ----------------------------------------------------------------------------------

	private static final String ATTRIBUTE_NAME = "parts";
	private static final String CONTENT_DISPOSITION = "content-disposition";
	private static final String CONTENT_DISPOSITION_FILENAME = "filename";
	private static final String DEFAULT_ENCODING = "UTF-8";
	private static final int DEFAULT_BUFFER_SIZE = 10240; // 10KB.

	// 伊賀追加: 最大ファイルサイズ。
	private static final long MAX_FILE_SIZE = 10485760L; // 10MB.

	// 変数
	// ---------------------------------------------------------------------------------------

	private String encoding;
	private boolean multipartConfigured;

	// コンストラクター
	// -------------------------------------------------------------------------------

	/**
	 * Construct multipart map based on the given multipart request and the
	 * servlet associated with the request. When the encoding is not specified
	 * in the given request, then it will default to <tt>UTF-8</tt>.
	 * 
	 * @param multipartRequest
	 *            The multipart request to construct the multipart map for.
	 * @param servlet
	 *            The servlet which is responsible for the given request.
	 * @throws ServletException
	 *             If something fails at Servlet level.
	 * @throws IOException
	 *             If something fails at I/O level.
	 */
	public MultipartMap(final HttpServletRequest multipartRequest,
			final Servlet servlet) throws ServletException, IOException {
		this(multipartRequest, true);
	}

	/**
	 * Construct multipart map based on the given multipart request. When the
	 * encoding is not specified in the given request, then it will default to
	 * <tt>UTF-8</tt>.
	 * 
	 * @param multipartRequest
	 *            The multipart request to construct the multipart map for.
	 * @throws ServletException
	 *             If something fails at Servlet level.
	 * @throws IOException
	 *             If something fails at I/O level.
	 */
	public MultipartMap(final HttpServletRequest multipartRequest)
			throws ServletException, IOException {
		this(multipartRequest, false);
	}

	/**
	 * Global constructor.
	 */
	private MultipartMap(final HttpServletRequest multipartRequest,
			final boolean multipartConfigured) throws ServletException,
			IOException {
		multipartRequest.setAttribute(ATTRIBUTE_NAME, this);

		this.encoding = multipartRequest.getCharacterEncoding();
		if (this.encoding == null) {
			multipartRequest
					.setCharacterEncoding(this.encoding = DEFAULT_ENCODING);
		}
		this.multipartConfigured = multipartConfigured;

		for (Part part : multipartRequest.getParts()) {
			final String filename = getFilename(part);
			if (filename == null) {
				// テキスト part として処理します。
				processTextPart(part);
			} else if (!filename.isEmpty()) {
				// ファイル part として処理します。
				processFilePart(part, filename);
			}
		}
	}

	// アクション。
	// ------------------------------------------------------------------------------------

	@Override
	public Object get(final Object key) {
		final Object value = super.get(key);
		if (value instanceof String[]) {
			final String[] values = (String[]) value;
			return values.length == 1 ? values[0] : Arrays.asList(values);
		} else {
			return value; // UploadFile または null となりえます。
		}
	}

	/**
	 * @see ServletRequest#getParameter(String)
	 */
	public String getParameter(final String name) {
		final Object value = super.get(name);
		if (value instanceof UploadedFile) {
			return ((UploadedFile) value).getName();
		}
		final String[] values = (String[]) value;
		return values != null ? values[0] : null;
	}

	/**
	 * @see ServletRequest#getParameterValues(String)
	 */
	public String[] getParameterValues(final String name) {
		final Object value = super.get(name);
		if (value instanceof UploadedFile) {
			return new String[] { ((UploadedFile) value).getName() };
		}
		return (String[]) value;
	}

	/**
	 * @see ServletRequest#getParameterNames()
	 */
	public Enumeration<String> getParameterNames() {
		return Collections.enumeration(keySet());
	}

	/**
	 * @see ServletRequest#getParameterMap()
	 */
	public Map<String, String[]> getParameterMap() {
		final Map<String, String[]> map = new HashMap<String, String[]>();
		for (Entry<String, Object> entry : entrySet()) {
			final Object value = entry.getValue();
			if (value instanceof String[]) {
				map.put(entry.getKey(), (String[]) value);
			} else {
				map.put(entry.getKey(),
						new String[] { ((UploadedFile) value).getName() });
			}
		}
		return map;
	}

	/**
	 * Returns uploaded file associated with given request parameter name.
	 * 
	 * @param name
	 *            Request parameter name to return the associated uploaded file
	 *            for.
	 * @return Uploaded file associated with given request parameter name.
	 * @throws IllegalArgumentException
	 *             If this field is actually a Text field.
	 */
	public UploadedFile getFile(final String name) {
		final Object value = super.get(name);
		if (value instanceof String[]) {
			throw new IllegalArgumentException(
					"This is a Text field. Use #getParameter() instead.");
		}
		return (UploadedFile) value;
	}

	// ヘルパー
	// ------------------------------------------------------------------------------------

	/**
	 * 与えられた part の content-disposition ヘッダーから filename を戻します。
	 * 
	 * @param part
	 * @return
	 */
	private String getFilename(final Part part) {
		for (String cd : part.getHeader(CONTENT_DISPOSITION).split(";")) {
			if (cd.trim().startsWith(CONTENT_DISPOSITION_FILENAME)) {
				return cd.substring(cd.indexOf('=') + 1).trim()
						.replace("\"", "");
			}
		}
		return null;
	}

	/**
	 * 与えられた part のテキスト値を戻します。
	 * 
	 * @param part
	 * @return
	 * @throws IOException
	 */
	private String getValue(final Part part) throws IOException {
		final BufferedReader reader = new BufferedReader(new InputStreamReader(
				part.getInputStream(), encoding));
		final StringBuilder value = new StringBuilder();
		final char[] buffer = new char[DEFAULT_BUFFER_SIZE];
		for (int length = 0; (length = reader.read(buffer)) > 0;) {
			value.append(buffer, 0, length);
		}
		return value.toString();
	}

	/**
	 * 与えられた part がテキスト part であるものとして処理します。
	 */
	private void processTextPart(final Part part) throws IOException {
		final String name = part.getName();
		final String[] values = (String[]) super.get(name);

		if (values == null) {
			// Not in parameter map yet, so add as new value.
			put(name, new String[] { getValue(part) });
		} else {
			// Multiple field values, so add new value to existing array.
			final int length = values.length;
			final String[] newValues = new String[length + 1];
			System.arraycopy(values, 0, newValues, 0, length);
			newValues[length] = getValue(part);
			put(name, newValues);
		}
	}

	/**
	 * 与えられた part がファイル part であるものとして処理し、与えられたファイル名をもちいてこれを temp ディレクトリに保存します。
	 */
	private void processFilePart(final Part part, String filename)
			throws IOException {
		// 最初にまぬけな MSIE の挙動を訂正 (ファイル名のすべてのクライアント側パスを渡してくるのです)。
		filename = filename.substring(filename.lastIndexOf('/') + 1).substring(
				filename.lastIndexOf('\\') + 1);

		// アップロード・ファイルを取り出して記憶します。
		final UploadedFile file = new UploadedFile(filename);
		if (multipartConfigured) {
			part.write(file.getName()); // とてもよく似たファイルに書き出すことでしょう。
		} else {
			final ByteArrayOutputStream output = new ByteArrayOutputStream();
			InputStream input = null;
			try {
				long totalFileLength = 0;
				input = new BufferedInputStream(part.getInputStream(),
						DEFAULT_BUFFER_SIZE);
				final byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
				for (int length = 0; ((length = input.read(buffer)) > 0);) {
					output.write(buffer, 0, length);

					totalFileLength += length;
					if (totalFileLength > MAX_FILE_SIZE) {
						throw new IOException(
								"アップロード処理において最大ファイルサイズを突破しました。処理中断します。");
					}
				}
				output.flush();
				file.setData(output.toByteArray());
				output.close();
			} finally {
				if (input != null)
					try {
						input.close();
					} catch (IOException ignore) {
						// この例外は無視します。
					}
			}
		}

		put(part.getName(), file);
		part.delete(); // 一時ストレージをクリーンアップします。
	}
}