/*
 * net/balusc/http/multipart/MultipartFilter.java
 * 
 * Copyright (C) 2009 BalusC
 * 
 * This program is free software: you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 * even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this library.
 * If not, see <http://www.gnu.org/licenses/>.
 */
package net.balusc.http.multipart;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;

/**
 * このフィルターは <tt>multipart/form-data</tt> および <tt>multipart/mixed</tt> POST
 * リクエストを検出して、<code>HttpServletRequest</code> から
 * <code>{@link MultipartRequest}</code> に置き換えます。
 * 
 * @author BalusC
 * @link http://balusc.blogspot.com/2009/12/uploading-files-in-servlet-30.html
 */

@WebFilter(urlPatterns = { "/*" })
public class MultipartFilter implements Filter {
	// 定数。
	// ----------------------------------------------------------------------------------

	private static final String REQUEST_METHOD_POST = "POST";
	private static final String CONTENT_TYPE_MULTIPART = "multipart/";

	// アクション。
	// ------------------------------------------------------------------------------------

	@Override
	public void init(final FilterConfig config) throws ServletException {
	}

	@Override
	public void doFilter(ServletRequest request,
			final ServletResponse response, final FilterChain chain)
			throws IOException, ServletException {
		final HttpServletRequest httpRequest = (HttpServletRequest) request;
		if (isMultipartRequest(httpRequest)) {
			request = new MultipartRequest(httpRequest);
		}
		chain.doFilter(request, response);
	}

	@Override
	public void destroy() {
		// 何もしません。
	}

	// ヘルパ・メソッド。
	// ------------------------------------------------------------------------------------

	/**
	 * 与えられたリクエストが multipart リクエストの場合、true を戻します。
	 * 
	 * @param request
	 *            チェックするリクエスト。
	 * @return 与えられたリクエストが multipart リクエストの場合は True。
	 */
	public static final boolean isMultipartRequest(
			final HttpServletRequest request) {
		return REQUEST_METHOD_POST.equalsIgnoreCase(request.getMethod())
				&& request.getContentType() != null
				&& request.getContentType().toLowerCase()
						.startsWith(CONTENT_TYPE_MULTIPART);
	}
}