/*******************************************************************************
 * Copyright (c) 2010  NEC Soft, Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.core;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;

import blanco.xml.bind.BlancoXmlBindingUtil;
import blanco.xml.bind.BlancoXmlMarshaller;
import blanco.xml.bind.BlancoXmlUnmarshaller;
import blanco.xml.bind.valueobject.BlancoXmlAttribute;
import blanco.xml.bind.valueobject.BlancoXmlCharacters;
import blanco.xml.bind.valueobject.BlancoXmlDocument;
import blanco.xml.bind.valueobject.BlancoXmlElement;
import blanco.xml.bind.valueobject.BlancoXmlNode;

/**
 * ECMA-376 の docx 形式ファイルを正常化するためのクラス。
 * 
 * <UL>
 * <LI>日本語の WORD ドキュメントの場合、複数の <w:r> に分かれている <w:t> は、実は 1 つにまとめることが可能なものがあります。このクラスは、この複数の <w:r> を 1 つにまとめるのが主たる目的です。
 * <LI>複数の <w:r> について、<w:t> の内容を集約して 1 つの <w:r> にまとめることにより、翻訳作業の効率が向上します。
 * <LI>w:rFonts 要素の w:hint 属性が、このような <w:r> の細切れ化を引き起こす主たる原因だと考え、w:hint しかない w:rFonts 属性を除去します。
 * <LI>w:hint 属性を取り除いた結果 w:rFonts 要素の中身がなくなった場合には、この要素自身を除去してしまいます。
 * <LI>これら加工を加えた結果、<w:r> の内容が <w:t> 以外で変わらないものが連続していれば、これを集約して 1 つにまとめてしまうことができます。
 * </UL>
 * 
 * @author IGA Tosiki
 */
public class Ecma376DocxNormalizer {
	/**
	 * docx ファイルの正常化をおこないます。
	 *
	 * ○フェーズ1: 「w:rFonts の正規化」
	 *   1.w:rFonts の w:hint は取り去る
	 *	 2.子を持たない w:rFonts は除去する。
	 *
	 * ○フェーズ2: 「空の w:rPr を除去する」
	 *	
	 * ○フェーズ3: 「連続する w:r をまとめる」
	 *   1.前提: w:r 自身の属性が一致すること
	 *	 2.子の w:t 以外の内容 (気にしているのは w:rPr の内容) が一致すること。
	 *   3.w:r 以下に w:t と w:rPr 以外が存在しないこと。
	 *	 4.連続する w:r について、w:t を集約してまとめる。
	 * 
	 * @param fileName ファイル名。
	 * @param bytes 正常化の対象となるデータ。
	 * @return 正常化後のデータ。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	public byte[] normalize(final String fileName, byte[] bytes) throws IOException {
		bytes = phase0101(fileName, bytes);
		bytes = phase0102(fileName, bytes);

		bytes = phase02(fileName, bytes);

		bytes = phase03(fileName, bytes);

		return bytes;
	}

	/**
	 * フェーズ1: 「w:rFonts の正規化」-> w:rFonts の w:hint は取り去る
	 * 
	 * @param fileName ファイル名。
	 * @param bytes 正常化の対象となるデータ。
	 * @return 正常化後のデータ。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	byte[] phase0101(final String fileName, final byte[] bytes) throws IOException {
		final ByteArrayOutputStream bufOut = new ByteArrayOutputStream();
		new Ecma376DocxProcessor() {
			@Override
			public void processWordXml(final InputStream inStream, final OutputStream outStream) {
				final BlancoXmlDocument document = new BlancoXmlUnmarshaller().unmarshal(inStream);
				new Ecma376WordXmlParser() {
					/**
					 * XML ツリーの要素をパースします。
					 * 
					 * @param eleChild 処理対象となる XML 要素。
					 */
					@Override
					void processChild(final BlancoXmlElement eleChild) {
						if (eleChild.getQName().equals("w:rFonts")) { //$NON-NLS-1$
							for (int index = eleChild.getAtts().size() - 1; index >= 0; index--) {
								if (eleChild.getAtts().get(index).getQName().equals("w:hint")) { //$NON-NLS-1$
									// w:hint を取り除きます。
									eleChild.getAtts().remove(index);
								}
							}
						}

						final List<BlancoXmlNode> childs = eleChild.getChildNodes();
						for (BlancoXmlNode nodeChild : childs) {
							if (nodeChild instanceof BlancoXmlElement) {
								processChild((BlancoXmlElement) nodeChild);
							}
						}
					}

					@Override
					public void fireT(BlancoXmlElement eleChild, String text) {
					}
				}.parse(document);

				new BlancoXmlMarshaller().marshal(document, outStream);
			}
		}.processDocx(fileName, bytes, bufOut);
		bufOut.flush();

		return bufOut.toByteArray();
	}

	/**
	 * フェーズ1: 「w:rFonts の正規化」-> 子を持たない w:rFonts は除去する。
	 * 
	 * @param fileName ファイル名。
	 * @param bytes 正常化の対象となるデータ。
	 * @return 正常化後のデータ。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	byte[] phase0102(final String fileName, final byte[] bytes) throws IOException {
		final ByteArrayOutputStream bufOut = new ByteArrayOutputStream();
		new Ecma376DocxProcessor() {
			@Override
			public void processWordXml(final InputStream inStream, final OutputStream outStream) {
				final BlancoXmlDocument document = new BlancoXmlUnmarshaller().unmarshal(inStream);
				new Ecma376WordXmlParser() {
					/**
					 * XML ツリーの要素をパースします。
					 * 
					 * @param eleChild 処理対象となる XML 要素。
					 */
					@Override
					void processChild(final BlancoXmlElement eleChild) {
						final List<BlancoXmlNode> childs = eleChild.getChildNodes();
						for (int index = 0; index < childs.size(); index++) {
							final BlancoXmlNode nodeChild = childs.get(index);
							if (nodeChild instanceof BlancoXmlElement) {
								processChild((BlancoXmlElement) nodeChild);

								final BlancoXmlElement child = (BlancoXmlElement) nodeChild;
								if (child.getQName().equals("w:rFonts")) { //$NON-NLS-1$
									if (child.getChildNodes().size() == 0 && child.getAtts().size() == 0) {
										// 子要素も属性も無い場合には、その要素自体を除去します。
										childs.remove(index);
										index--;
									}
								}
							}
						}
					}

					@Override
					public void fireT(BlancoXmlElement eleChild, String text) {
					}
				}.parse(document);

				new BlancoXmlMarshaller().marshal(document, outStream);
			}
		}.processDocx(fileName, bytes, bufOut);
		bufOut.flush();

		return bufOut.toByteArray();
	}

	/**
	 * ○フェーズ2: 「空の w:rPr を除去する」
	 * 
	 * @param fileName ファイル名。
	 * @param bytes 正常化の対象となるデータ。
	 * @return 正常化後のデータ。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	byte[] phase02(final String fileName, final byte[] bytes) throws IOException {
		final ByteArrayOutputStream bufOut = new ByteArrayOutputStream();
		new Ecma376DocxProcessor() {
			@Override
			public void processWordXml(final InputStream inStream, final OutputStream outStream) {
				final BlancoXmlDocument document = new BlancoXmlUnmarshaller().unmarshal(inStream);
				new Ecma376WordXmlParser() {
					/**
					 * XML ツリーの要素をパースします。
					 * 
					 * @param eleChild 処理対象となる XML 要素。
					 */
					@Override
					void processChild(final BlancoXmlElement eleChild) {
						final List<BlancoXmlNode> childs = eleChild.getChildNodes();
						for (int index = 0; index < childs.size(); index++) {
							final BlancoXmlNode nodeChild = childs.get(index);
							if (nodeChild instanceof BlancoXmlElement) {
								processChild((BlancoXmlElement) nodeChild);

								final BlancoXmlElement child = (BlancoXmlElement) nodeChild;
								if (child.getQName().equals("w:rPr")) { //$NON-NLS-1$
									if (child.getChildNodes().size() == 0 && child.getAtts().size() == 0) {
										// 子要素も属性も無い場合には、その要素自体を除去します。
										childs.remove(index);
										index--;
									}
								}
							}
						}
					}

					@Override
					public void fireT(BlancoXmlElement eleChild, String text) {
					}
				}.parse(document);

				new BlancoXmlMarshaller().marshal(document, outStream);
			}
		}.processDocx(fileName, bytes, bufOut);
		bufOut.flush();

		return bufOut.toByteArray();
	}

	/**
	 * ○フェーズ3: 「連続する w:r をまとめる」
	 *   1.前提: w:r 自身の属性が一致すること
	 *	 2.子の w:t 以外の内容 (気にしているのは w:rPr の内容) が一致すること。
	 *	 3.連続する w:r について、w:t を集約してまとめる。
	 * 
	 * @param fileName ファイル名。
	 * @param bytes 正常化の対象となるデータ。
	 * @return 正常化後のデータ。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	byte[] phase03(final String fileName, final byte[] bytes) throws IOException {
		final ByteArrayOutputStream bufOut = new ByteArrayOutputStream();
		new Ecma376DocxProcessor() {
			@Override
			public void processWordXml(final InputStream inStream, final OutputStream outStream) {
				final BlancoXmlDocument document = new BlancoXmlUnmarshaller().unmarshal(inStream);
				new Ecma376WordXmlParser() {
					/**
					 * XML ツリーの要素をパースします。
					 * 
					 * @param eleChild 処理対象となる XML 要素。
					 */
					@Override
					void processChild(final BlancoXmlElement eleChild) {
						final List<BlancoXmlNode> childs = eleChild.getChildNodes();
						for (int index = 0; index < childs.size(); index++) {
							final BlancoXmlNode nodeChild = childs.get(index);
							if (nodeChild instanceof BlancoXmlElement) {
								processChild((BlancoXmlElement) nodeChild);

								final BlancoXmlElement currentChild = (BlancoXmlElement) nodeChild;
								if (currentChild.getQName().equals("w:r")) { //$NON-NLS-1$
									if (index + 1 >= childs.size()) {
										continue;
									}

									final BlancoXmlNode nodeNextChild = childs.get(index + 1);
									if (nodeNextChild instanceof BlancoXmlElement) {
										final BlancoXmlElement nextChild = (BlancoXmlElement) nodeNextChild;
										if (nextChild.getQName().equals("w:r")) { //$NON-NLS-1$
											// 連続する w:r が現れました。
											final boolean isJoined = phase0301(currentChild, nextChild);
											if (isJoined) {
												childs.remove(index + 1);
												index--;
											}
										}
									}
								}
							}
						}
					}

					@Override
					public void fireT(BlancoXmlElement eleChild, String text) {
					}
				}.parse(document);

				new BlancoXmlMarshaller().marshal(document, outStream);
			}
		}.processDocx(fileName, bytes, bufOut);
		bufOut.flush();

		return bufOut.toByteArray();
	}

	/**
	 * 連続した w:r があらわれた際の処理を行います。
	 * 
	 *   1.前提: w:r 自身の属性が一致すること
	 *	 2.子の w:t 以外の内容 (気にしているのは w:rPr の内容) が一致すること。
	 *   3.w:r 以下に w:t と w:rPr 以外が存在しないこと。
	 *	 4.連続する w:r について、w:t を集約してまとめる。
	 * 
	 * @param currentChild 現在の子要素。
	 * @param nextChild 次の子要素。
	 * @return 2 つの要素を 1 つにまとめたかどうか?
	 */
	boolean phase0301(final BlancoXmlElement currentChild, final BlancoXmlElement nextChild) {
		// w:r 自身の属性が一致することを確認します。
		if (isAttrEquals(currentChild, nextChild) == false) {
			// w:r 自身の属性が一致しません。処理を離脱します。
			return false;
		}

		if (phase0301EqualsLeftOuter(currentChild, nextChild) == false) {
			return false;
		}
		if (phase0301EqualsLeftOuter(nextChild, currentChild) == false) {
			return false;
		}

		// w:r 以下に w:t と w:rPr 以外が存在しないことを確認。
		if (phase0301CheckChildElementForValidJoin(currentChild) == false) {
			return false;
		}
		if (phase0301CheckChildElementForValidJoin(nextChild) == false) {
			return false;
		}

		// ここまで来たら、w:t を合体できるものと判断します。

		final BlancoXmlElement eleCurrentWT = getElement(currentChild, "w:t"); //$NON-NLS-1$
		if (eleCurrentWT == null) {
			return false;
		}

		final BlancoXmlElement eleNextWT = getElement(nextChild, "w:t"); //$NON-NLS-1$
		if (eleNextWT == null) {
			return false;
		}
		final String textNext = BlancoXmlBindingUtil.getTextContent(eleNextWT);

		// 次の要素の w:t を、自分の要素の w:t のテキスト領域に追加します。
		final BlancoXmlCharacters textAfter = new BlancoXmlCharacters();
		textAfter.setValue(textNext);
		eleCurrentWT.getChildNodes().add(textAfter);

		return true;
	}

	/**
	 * w:t を無視して、他は同じ内容であると判断できるかどうか。
	 * 
	 * @param currentChild 左辺要素。
	 * @param nextChild 右辺要素。
	 * @return 一致するとみなせるかどうか。
	 */
	boolean phase0301EqualsLeftOuter(final BlancoXmlElement currentChild, final BlancoXmlElement nextChild) {
		for (BlancoXmlNode nodeLeft : currentChild.getChildNodes()) {
			if (nodeLeft instanceof BlancoXmlElement) {
				final BlancoXmlElement eleLeft = (BlancoXmlElement) nodeLeft;
				if (eleLeft.getQName().equals("w:t")) { //$NON-NLS-1$
					continue;
				}

				boolean isFound = false;
				for (BlancoXmlNode nodeRight : nextChild.getChildNodes()) {
					if (nodeRight instanceof BlancoXmlElement) {
						final BlancoXmlElement eleRight = (BlancoXmlElement) nodeRight;
						if (eleLeft.getQName().equals(eleRight.getQName())) {
							// 名前が一致しました。
							isFound = true;
							if (isElementEquals(eleLeft, eleRight) == false) {
								// 違います。
								return false;
							}
						}
					}
				}
				if (isFound == false) {
					return false;
				}
			}
		}

		return true;
	}

	/**
	 * w:r 以下に w:t と w:rPr 以外が存在しないことを確認。
	 * 
	 * @param currentChild 処理対象のエレメント。
	 * @return w:t を集約可能かどうか。
	 */
	boolean phase0301CheckChildElementForValidJoin(final BlancoXmlElement currentChild) {
		for (BlancoXmlNode objLook : currentChild.getChildNodes()) {
			if (objLook instanceof BlancoXmlElement == false) {
				continue;
			}

			final BlancoXmlElement elementLook = (BlancoXmlElement) objLook;
			if (elementLook.getQName().equals("w:t") || elementLook.getQName().equals("w:rPr")) { //$NON-NLS-1$ //$NON-NLS-2$
				// 問題なし
			} else {
				// w:t と w:rPr 以外が含まれています。この要素は集約しません。
				return false;
			}
		}

		return true;
	}

	/**
	 * 属性値が一致するかどうか調べます。
	 * 
	 * 前提: 同じ名前の属性は 2 個は存在しないこと。
	 * 
	 * @param child1 1 つめの要素。
	 * @param child2 2 つめの要素。
	 * @return 一致するかどうか。
	 */
	boolean isAttrEquals(final BlancoXmlElement child1, final BlancoXmlElement child2) {
		if (isAttrEqualsLeftOuter(child1, child2) == false) {
			return false;
		}
		if (isAttrEqualsLeftOuter(child2, child1) == false) {
			return false;
		}

		return true;
	}

	/**
	 * 左辺を中心に属性値が一致するかどうか調べます。
	 * 
	 * 前提: 同じ名前の属性は 2 個は存在しないこと。
	 * 
	 * @param child1 左辺とする要素。
	 * @param child2 右辺とする要素。
	 * @return 一致するかどうか。
	 */
	boolean isAttrEqualsLeftOuter(final BlancoXmlElement child1, final BlancoXmlElement child2) {
		// 左辺を中心に属性値を比較します。
		for (BlancoXmlAttribute attr1 : child1.getAtts()) {
			boolean isFound = false;
			for (BlancoXmlAttribute attr2 : child2.getAtts()) {
				if (attr1.getQName().equals(attr2.getQName())) {
					if (attr1.getValue().equals(attr2.getValue())) {
						isFound = true;
					} else {
						// QName は一致したが値が異なりました。
						return false;
					}
				}
			}
			if (isFound == false) {
				return false;
			}
		}

		return true;
	}

	/**
	 * 要素が一致するかどうか調べます。
	 * 
	 * 前提: 同じ名前の要素は 2 個は存在しないこと。
	 * 
	 * @param child1 左辺とする要素。
	 * @param child2 右辺とする要素。
	 * @return 一致するかどうか。
	 */
	boolean isElementEquals(final BlancoXmlElement child1, final BlancoXmlElement child2) {
		if (isElementEqualsLeftOuter(child1, child2) == false) {
			return false;
		}
		if (isElementEqualsLeftOuter(child2, child1) == false) {
			return false;
		}

		return true;
	}

	/**
	 * 左辺を中心に要素が一致するかどうか調べます。
	 * 
	 * 前提: 同じ名前の要素は 2 個は存在しないこと。
	 * 
	 * @param child1 左辺とする要素。
	 * @param child2 右辺とする要素。
	 * @return 一致するかどうか。
	 */
	boolean isElementEqualsLeftOuter(final BlancoXmlElement child1, final BlancoXmlElement child2) {
		// 属性値を比較します。
		if (isAttrEquals(child1, child2) == false) {
			return false;
		}

		final String textLeft = BlancoXmlBindingUtil.getTextContent(child1);
		final String textRight = BlancoXmlBindingUtil.getTextContent(child2);
		if (textLeft == null && textRight == null) {
			// ともに内容がありません。
		} else if (textLeft == null || textRight == null) {
			// いずれか一方のみが null で、他方は文字列が入っています。
			return false;
		} else if (textLeft.equals(textRight) == false) {
			// テキストの内容が異なります。
			return false;
		}

		for (BlancoXmlNode nodeLeft : child1.getChildNodes()) {
			if (nodeLeft instanceof BlancoXmlElement) {
				final BlancoXmlElement eleLeft = (BlancoXmlElement) nodeLeft;

				boolean isFound = false;
				for (BlancoXmlNode nodeRight : child2.getChildNodes()) {
					if (nodeRight instanceof BlancoXmlElement) {
						final BlancoXmlElement eleRight = (BlancoXmlElement) nodeRight;
						if (eleLeft.getQName().equals(eleRight.getQName())) {
							// 名前が一致しました。
							isFound = true;
							if (isElementEqualsLeftOuter(eleLeft, eleRight) == false) {
								return false;
							}
						}
					}
				}
				if (isFound == false) {
					// 該当する QName の要素が見つかりませんでした。
					return false;
				}
			}
		}

		return true;
	}

	/**
	 * 指定された名前の要素を取得します。
	 * 
	 * @param argElement 検索対象の要素。
	 * @param argTagname 要素名 (QName)。
	 * @return みつかった要素。見つからなかった場合は null。
	 */
	BlancoXmlElement getElement(final BlancoXmlElement argElement, final String argTagname) {
		for (BlancoXmlNode objLook : argElement.getChildNodes()) {
			if (objLook instanceof BlancoXmlElement == false) {
				continue;
			}

			final BlancoXmlElement elementLook = (BlancoXmlElement) objLook;
			if (elementLook.getQName().equals(argTagname)) {
				return elementLook;
			}
		}

		return null;
	}
}
