/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.filter.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.filter.core.valueobject.BentenConvertToXliffProcessInput;

/**
 * Apache Antタスク [BentenConvertToXliff]のクラス。
 *
 * HTML－XLIFF 変換のための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.filter.core]にBentenConvertToXliffBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenconverttoxliff&quot; classname=&quot;benten.twa.filter.core.BentenConvertToXliffTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenConvertToXliffTask extends Task {
    /**
     * HTML－XLIFF 変換のための Ant タスク。
     */
    protected BentenConvertToXliffProcessInput fInput = new BentenConvertToXliffProcessInput();

    /**
     * フィールド [sourcedir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldSourcedirProcessed = false;

    /**
     * フィールド [targetdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTargetdirProcessed = false;

    /**
     * フィールド [transtargetid] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranstargetidProcessed = false;

    /**
     * フィールド [transsourcelang] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranssourcelangProcessed = false;

    /**
     * フィールド [transtargetlang] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranstargetlangProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[sourcedir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * 変換元となる HTML が格納されたディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setSourcedir(final String arg) {
        fInput.setSourcedir(arg);
        fIsFieldSourcedirProcessed = true;
    }

    /**
     * Antタスクの[sourcedir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * 変換元となる HTML が格納されたディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getSourcedir() {
        return fInput.getSourcedir();
    }

    /**
     * Antタスクの[targetdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * 変換先となる XLIFF を格納するディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setTargetdir(final String arg) {
        fInput.setTargetdir(arg);
        fIsFieldTargetdirProcessed = true;
    }

    /**
     * Antタスクの[targetdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * 変換先となる XLIFF を格納するディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTargetdir() {
        return fInput.getTargetdir();
    }

    /**
     * Antタスクの[transtargetid]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * 翻訳対象 id  を格納します。<br>
     *
     * @param arg セットしたい値
     */
    public void setTranstargetid(final String arg) {
        fInput.setTranstargetid(arg);
        fIsFieldTranstargetidProcessed = true;
    }

    /**
     * Antタスクの[transtargetid]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * 翻訳対象 id  を格納します。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranstargetid() {
        return fInput.getTranstargetid();
    }

    /**
     * Antタスクの[transsourcelang]アトリビュートのセッターメソッド。
     *
     * 項目番号: 4<br>
     * 翻訳元言語<br>
     *
     * @param arg セットしたい値
     */
    public void setTranssourcelang(final String arg) {
        fInput.setTranssourcelang(arg);
        fIsFieldTranssourcelangProcessed = true;
    }

    /**
     * Antタスクの[transsourcelang]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 4<br>
     * 翻訳元言語<br>
     * デフォルト値[en-US]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranssourcelang() {
        return fInput.getTranssourcelang();
    }

    /**
     * Antタスクの[transtargetlang]アトリビュートのセッターメソッド。
     *
     * 項目番号: 5<br>
     * 翻訳先言語<br>
     *
     * @param arg セットしたい値
     */
    public void setTranstargetlang(final String arg) {
        fInput.setTranstargetlang(arg);
        fIsFieldTranstargetlangProcessed = true;
    }

    /**
     * Antタスクの[transtargetlang]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 5<br>
     * 翻訳先言語<br>
     * デフォルト値[ja-JP]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranstargetlang() {
        return fInput.getTranstargetlang();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenConvertToXliffTask begin.");

        // 項目番号[1]、アトリビュート[sourcedir]は必須入力です。入力チェックを行います。
        if (fIsFieldSourcedirProcessed == false) {
            throw new BuildException("必須アトリビュート[sourcedir]が設定されていません。処理を中断します。");
        }
        // 項目番号[2]、アトリビュート[targetdir]は必須入力です。入力チェックを行います。
        if (fIsFieldTargetdirProcessed == false) {
            throw new BuildException("必須アトリビュート[targetdir]が設定されていません。処理を中断します。");
        }
        // 項目番号[3]、アトリビュート[transtargetid]は必須入力です。入力チェックを行います。
        if (fIsFieldTranstargetidProcessed == false) {
            throw new BuildException("必須アトリビュート[transtargetid]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- sourcedir:[" + getSourcedir() + "]");
            System.out.println("- targetdir:[" + getTargetdir() + "]");
            System.out.println("- transtargetid:[" + getTranstargetid() + "]");
            System.out.println("- transsourcelang:[" + getTranssourcelang() + "]");
            System.out.println("- transtargetlang:[" + getTranstargetlang() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenConvertToXliffProcessインタフェースを実装して benten.twa.filter.coreパッケージに BentenConvertToXliffProcessImplクラスを作成することにより解決できる場合があります。
            final BentenConvertToXliffProcess proc = new BentenConvertToXliffProcessImpl();
            if (proc.execute(fInput) != BentenConvertToXliffBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
