/*******************************************************************************
 * Copyright (c) 2010  NEC Soft, Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.ui.viewers;

import org.eclipse.compare.CompareConfiguration;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import benten.cat.ui.compare.TextMergeViewer;

/**
 * 翻訳文字列を比較するためのテキスト・ビューアー。
 *
 * @author KASHIHARA Shinji
 */
public class TransCompareViewer extends TextMergeViewer {

	// 比較コンテンツ。
	private final TransDiffContent diffContent = new TransDiffContent();

	/**
	 * 比較コンテンツを保持するクラス。
	 */
	protected static class TransDiffContent {
		String left;
		String right;
	}

	/**
	 * コンストラクター。
	 * @param parent 親コンポジット。
	 * @param configuration 比較構成。
	 */
	public TransCompareViewer(final Composite parent, final CompareConfiguration configuration) {

		super(parent, configuration);
		final RGB backgroundColor = getControl().getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND).getRGB();
		setBackgroundColor(backgroundColor);

		getCopyButton().addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final String right = getRightSourceViewer().getDocument().get();
				getLeftSourceViewer().getDocument().set(right);
			}
		});
		getLeftSourceViewer().getTextWidget().addModifyListener(new ModifyListener() {
			public void modifyText(final ModifyEvent e) {
				diffContent.left = getLeftSourceViewer().getTextWidget().getText();
			}
		});
	}

	/**
	 * 左コンテンツをセット。
	 * @param left 左コンテンツ文字列。
	 */
	public void setLeftContent(final String left) {
		diffContent.left = left;
	}

	/**
	 * 右コンテンツをセット。
	 * @param right 右コンテンツ文字列。
	 */
	public void setRightContent(final String right) {
		diffContent.right = right;
	}

	/**
	 * コンテンツ変更通知。
	 */
	public void fireContentChange() {

		StyledText widget = getLeftSourceViewer().getTextWidget();
		final int offsetOld = widget.getCaretOffset();
		final String textOld = widget.getText();

		final TransDiffContent newContent = new TransDiffContent();
		newContent.left = diffContent.left;
		newContent.right = diffContent.right;
		super.setInput(newContent);

		// UNDO 対応のため、キャレット位置を元に戻します。
		if (getLeftSourceViewer().isEditable()) {
			widget = getLeftSourceViewer().getTextWidget();
			final String text = widget.getText();
			if (text.length() < offsetOld) {
				widget.setCaretOffset(text.length());
			} else {
				final String textBeforeCaret = text.substring(0, offsetOld);
				final String textOldBeforeCaret = textOld.substring(0, offsetOld);
				int offsetNew = offsetOld;
				if (!textBeforeCaret.equals(textOldBeforeCaret)) {
					offsetNew = offsetOld - (textOld.length() - text.length());
				}
				widget.setCaretOffset(offsetNew);
			}
		}
	}

	/**
	 * 左コンテンツの編集可不可をセット。
	 * @param editable 編集可能にする場合は true。
	 */
	public void setLeftEditable(final boolean editable) {
		final IContentProvider cp = getContentProvider();
		if (cp instanceof TransCompareContentProvider) {
			final TransCompareContentProvider p = (TransCompareContentProvider) cp;
			p.setLeftEditable(editable);
			getCopyButton().setVisible(editable);

			// refresh しないと反映されませんが、パフォーマンス向上のためここではリフレッシュしません。
			// 代わりに fireContentChange で setInput されたときにリフレッシュされます。
			//refresh();
		}
	}

	@Override
	protected void setEditable(final ISourceViewer sourceViewer, final boolean editable) {

		super.setEditable(sourceViewer, editable);

		// 編集不可の場合、背景を灰色にします。
		final Control control = sourceViewer.getTextWidget();
		final int color = editable ? SWT.COLOR_WHITE : SWT.COLOR_WIDGET_BACKGROUND;
		control.setBackground(control.getDisplay().getSystemColor(color));
	}
}
