/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.cat.core;

import java.util.ArrayList;
import java.util.List;

import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;

/**
 * 翻訳メモリーのためのユーティリティ
 *
 * @author IGA Tosiki
 * @author KASHIHARA Shinji
 */
public class BentenTmUtil {

	/**
	 * 完全一致訳を検索します。
	 *
	 * <UL>
	 * <LI>この処理では、文字列の前後の空白は無視して処理を行います。
	 * <LI>source が完全に一致する翻訳が翻訳メモリーにあるかどうか検索します。
	 * <LI>source が完全一致する翻訳のうち、target が異なる複数の翻訳が無いことを確認します。
	 * <LI>fuzzySearch に与える文字列をプリファランスの設定に従って加工する処理については、このメソッドより外側で実施されます。
	 * </UL>
	 *
	 * @param tm 翻訳メモリー。null は与えないでください。
	 * @param source ソース文字列。
	 * @return 適切な完全一致訳が無ければ null。複数 100% マッチする場合も null。1件だけ適切な完全一致訳がある場合にのみ該当する完全一致訳。
	 */
	public static String searchExactMatch(final BentenTmEngine tm, final String source) {
		if (tm == null) {
			throw new IllegalArgumentException("BentenTmUtil#searchExactMatch: argument tm must not be null."); //$NON-NLS-1$
		}
		if (source == null) {
			// source が null では、検索する意味がないのですぐに抜けます。
			return null;
		}

		return searchExactMatch(tm.fuzzySearch(source), source);
	}

	/**
	 * 完全一致訳を検索します。
	 *
	 * <UL>
	 * <LI>最初は、単純に文字列が完全一致するかどうかをもとに検索します。
	 * <LI>単純な完全一致では検索があたらなかった場合には、前後の空白を無視して検索を行います。
	 * <LI>source が完全に一致する翻訳が翻訳メモリーにあるかどうか検索します。
	 * <LI>source が完全一致する翻訳のうち、target が異なる複数の翻訳が無いことを確認します。
	 * <LI>TM エンジンが戻す翻訳品質については、この処理では参考にしていません。自力で内容を比較して、一致するかどうか判断しています。
	 * </UL>
	 *
	 * @param matchList 一致リスト。
	 * @param source ソース文字列。
	 * @return 適切な完全一致訳が無ければ null。複数の完全一致が存在する場合も null。1件だけ適切な完全一致訳がある場合にのみ該当する完全一致訳。
	 */
	static String searchExactMatch(final List<BentenTmSearchResult> matchList, final String source) {
		// 最初は、単純に文字列が完全一致するかどうかをもとに検索します。
		String result = searchExactMatch(matchList, source, false);
		if (result == null) {
			// 単純な完全一致では検索があたらなかった場合には、前後の空白を無視して検索を行います。
			result = searchExactMatch(matchList, source, true);
		}
		return result;
	}

	/**
	 * 完全一致訳を検索します。
	 *
	 * <UL>
	 * <LI>この処理では、フラグにより文字列の前後の空白を無視して処理を行います。
	 * <LI>source が完全に一致する翻訳が翻訳メモリーにあるかどうか検索します。
	 * <LI>source が完全一致する翻訳のうち、target が異なる複数の翻訳が無いことを確認します。
	 * <LI>TM エンジンが戻す翻訳品質については、この処理では参考にしていません。自力で内容を比較して、一致するかどうか判断しています。
	 * </UL>
	 *
	 * @param matchList 一致リスト。
	 * @param source ソース文字列。
	 * @param isUseUntrim 前後の空白を無視するかどうか。
	 * @return 適切な完全一致訳が無ければ null。複数の完全一致が存在する場合も null。1件だけ適切な完全一致訳がある場合にのみ該当する完全一致訳。
	 */
	static String searchExactMatch(final List<BentenTmSearchResult> matchList, final String source,
			final boolean isUseUntrim) {
		if (source == null) {
			// source が null では、検索する意味がないのですぐに抜けます。
			return null;
		}

		// 完全一致した翻訳を記憶するためのリスト。
		final List<BentenTmSearchResult> exactMatchList = new ArrayList<BentenTmSearchResult>();

		for (final BentenTmSearchResult entry : matchList) {
			String target = null;
			if (isUseUntrim == false) {
				if (source.equals(entry.getSource())) {
					target = entry.getTarget();
				}
			} else {
				// trim の逆の処理をおこなうことにより、多少のゆとりを持った完全一致検索を行います。
				target = adjustForUntrim(source, entry.getSource(), entry.getTarget());
			}
			if (target == null) {
				// ソースが完全一致していないので、この翻訳は対象外とします。
				// 注意: TM から戻される翻訳品質という情報については、判断材料から除外しています。
				continue;
			}

			// 入力値とソースが一致しています。これを完全一致した翻訳リストへ記憶します。
			exactMatchList.add(entry);
		}

		if (exactMatchList.size() == 0) {
			// 完全一致訳が 1 件もありませんでした。
			// この source では完全一致訳は 1 件も存在しないため、処理中断します。
			return null;
		}

		// 完全一致訳の 1 件目について正候補として扱います。
		final BentenTmSearchResult candidate = exactMatchList.get(0);

		// 複数件ヒットした場合の内容確認を実施します。
		for (final BentenTmSearchResult entryOther : exactMatchList) {
			String target = entryOther.getTarget();
			if (isUseUntrim) {
				target = adjustForUntrim(source, entryOther.getSource(), entryOther.getTarget());
			}
			String targetCandidate = candidate.getTarget();
			if (isUseUntrim) {
				targetCandidate = adjustForUntrim(source, candidate.getSource(), candidate.getTarget());
			}
			if (targetCandidate.equals(target) == false) {
				// 複数の完全一致翻訳のなかで翻訳結果が一致しません。
				// このため、この翻訳結果は完全一致訳としては利用できません。
				// 処理を中断します。
				return null;
			}
		}

		// ここまで通過したので、翻訳は 1 件のみです。
		if (isUseUntrim == false) {
			return candidate.getTarget();
		} else {
			return adjustForUntrim(source, candidate.getSource(), candidate.getTarget());
		}
	}

	/**
	 * trim の逆の効果を使って、source と翻訳候補の source, target から target を生成。
	 *
	 * @param source ソース
	 * @param sourceCandidate ソース候補
	 * @param targetCandidate ターゲット候補
	 * @return target ターゲット
	 */
	public static String adjustForUntrim(final String source, final String sourceCandidate, final String targetCandidate) {
		// ソースが一致
		if (source.equals(sourceCandidate)) {
			return targetCandidate;
		}

		final JapaneseTranslationString sourceTS = new JapaneseTranslationString(source);
		final JapaneseTranslationString tmSourceTS = new JapaneseTranslationString(sourceCandidate);
		final JapaneseTranslationString tmTargetTS = new JapaneseTranslationString(targetCandidate);

		// ソースがトリム後に一致
		if (sourceTS.trim().equals(tmSourceTS.trim())) {
			return sourceTS.revert(tmTargetTS.trim());
		}
		return null;
	}
}
