/**
    SDL_wx - using wxWidgets in SDL
    Copyright (C) 2003-2006 kr2.

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    kr2
    ee12054g@hotmail.co.jp
*/
#include "SDL_wx.hpp"

#include <wx/wxprec.h>

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

#ifndef WX_PRECOMP
    #include <wx/wx.h>
#endif

#include <wx/listimpl.cpp>

#include <SDL.h>
#include <SDL_thread.h>

#ifdef _MSC_VER
#pragma comment(lib, "SDL.lib")
#pragma comment(lib, "SDLmain.lib")
#endif

#include "SDLWX_CArgs.hpp"
#include "wxSDLApp.hpp"

/* 必ずNO_MAINにすること */
IMPLEMENT_APP_NO_MAIN(wxSDLApp)

/* リストの定義  */
WX_DEFINE_LIST(SDLWXList);

static bool gs_bWXInit = false ;

enum {
	ID_WXAPP = wxID_HIGHEST + 1
} ;

BEGIN_DECLARE_EVENT_TYPES()
	DECLARE_LOCAL_EVENT_TYPE(SDL_EVT_REGISTEROBJECT, 0)
	DECLARE_LOCAL_EVENT_TYPE(SDL_EVT_UNREGISTEROBJECT, 1)
	DECLARE_LOCAL_EVENT_TYPE(SDL_EVT_DATA, 2)
END_DECLARE_EVENT_TYPES()

DEFINE_LOCAL_EVENT_TYPE(SDL_EVT_REGISTEROBJECT)
DEFINE_LOCAL_EVENT_TYPE(SDL_EVT_UNREGISTEROBJECT)
DEFINE_LOCAL_EVENT_TYPE(SDL_EVT_DATA)

bool wxSDLApp::OnInit()
{
	/*
		ダミーフレームを生成
		これが無いと、wxApp::WakeUpIdleを呼び出しても
		TopWindowが無い場合はIdleイベントが呼び出されず、
		wxWidgetsのスレッドが終了しないので注意。
	*/
	wxFrame* DummyFrame = new wxFrame(NULL,wxID_ANY,_T(""));
	SetTopWindow(DummyFrame);
	gs_bWXInit = true ;
	return TRUE;
}

/*
	イベントテーブルの実装
*/
BEGIN_EVENT_TABLE(wxSDLApp, wxApp)
EVT_COMMAND(ID_WXAPP,SDL_EVT_REGISTEROBJECT,wxSDLApp::OnRegisterObject)
EVT_COMMAND(ID_WXAPP,SDL_EVT_UNREGISTEROBJECT,wxSDLApp::OnUnregisterObject)
EVT_COMMAND(ID_WXAPP,SDL_EVT_DATA,wxSDLApp::OnData)
END_EVENT_TABLE()

/*
	WX_Objectのリストへの登録
*/
void wxSDLApp::OnRegisterObject(wxCommandEvent& evt)
{
	WX_Object* obj = (WX_Object*)evt.GetClientData() ;
	/* Objectの初期化 */
	if ( !obj->OnInit() ) {
		obj->OnExit () ;
		delete obj ;
		wxExit () ;
	}else{
		m_SDLWXObjList.Append(obj) ;
	}
}

void wxSDLApp::OnUnregisterObject(wxCommandEvent& evt)
{
	WX_Object* obj = (WX_Object*)evt.GetClientData() ;
	m_SDLWXObjList.DeleteObject(obj) ;
	/* Objectの破棄 */
	obj->OnUnregistered() ;
	obj->OnExit () ;
	delete obj ;
}

void wxSDLApp::OnData (wxCommandEvent& evt)
{
	SDLWXList::Node* node ;
	WX_Data* data = (WX_Data*)evt.GetClientData() ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		obj->OnData (data) ;
	}
}

void wxSDLApp::OnAssert(const wxChar *file, int line, const wxChar *cond, const wxChar *msg) 
{
	SDLWXList::Node* node ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		obj->OnAssert (file,line,cond, msg) ;
	}
}

int wxSDLApp::OnExit ()
{
	SDLWXList::Node* node ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		obj->OnUnregistered () ;
		obj->OnExit () ;
		delete obj ;
	}
	m_SDLWXObjList.Clear () ;
	return 0 ;
}

bool wxSDLApp::OnCmdLineError(wxCmdLineParser& parser) 
{
	SDLWXList::Node* node ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		if ( !obj->OnCmdLineError (parser) ) return false ;
	}
	return true ;
}

bool wxSDLApp::OnCmdLineHelp(wxCmdLineParser& parser) 
{
	SDLWXList::Node* node ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		if ( !obj->OnCmdLineHelp (parser) ) return false ;
	}
	return true ;
}

bool wxSDLApp::OnCmdLineParsed(wxCmdLineParser& parser) 
{
	SDLWXList::Node* node ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		if ( !obj->OnCmdLineParsed (parser) ) return false ;
	}
	return true ;
}

void wxSDLApp::OnFatalException() 
{
	SDLWXList::Node* node ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		obj->OnFatalException () ;
	}
}

void wxSDLApp::OnInitCmdLine(wxCmdLineParser& parser) 
{
	SDLWXList::Node* node ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		obj->OnInitCmdLine (parser) ;
	}
}

void wxSDLApp::OnQueryEndSession(wxCloseEvent& event) 
{
	SDLWXList::Node* node ;
	for ( node=m_SDLWXObjList.GetFirst(); node!=NULL; node = node->GetNext() ) {
		WX_Object* obj = node->GetData() ;
		obj->OnQueryEndSession (event) ;
	}
}

int WX_ThreadFunc ( void* data )
{
	if ( data == NULL ) return 0 ;
	// main関数の引数を取得
	SDLWX_CArgs* args = (SDLWX_CArgs*)data ;
	int argc = args->Argc() ;
	wxEntry(argc,args->Argv());
	delete args ;
	return 0 ;
}

static SDL_Thread* gs_pWX_Thread = NULL ;

int WX_Init(int argc, char* argv[])
{
	if ( WX_WasInit() ) return 0 ;
	// wxWidgets用のスレッドを生成する同時に、メイン関数の引数も渡す
	gs_pWX_Thread = SDL_CreateThread(
		WX_ThreadFunc,
		new SDLWX_CArgs(argc,argv)
	);
	if ( gs_pWX_Thread == NULL ) {
		return -1 ;
	}
	return 0 ;
}

int WX_WasInit ()
{
	return gs_bWXInit?1:0 ;
}

int WX_Quit()
{
	if ( !WX_WasInit() ) return -1 ;
	if ( (&wxGetApp())!=NULL ) {
		wxExit() ;
		wxWakeUpIdle() ;
	}
	int status ;
	SDL_WaitThread(gs_pWX_Thread, &status) ;
	// 初期化リセット 
	gs_bWXInit = false ;
	return status ;
}

WX_ObjectID WX_RegisterObjectInner(WX_Object* obj)
{
	if ( !WX_WasInit() ) return NULL ;
	if ( obj == NULL ) return NULL ;
	/* イベントとWX_Objectを同時に渡す */
	wxCommandEvent CmdEvt(SDL_EVT_REGISTEROBJECT,ID_WXAPP) ;
	CmdEvt.SetClientData(obj) ;
	wxGetApp().AddPendingEvent(CmdEvt);
	return obj ;
}

/* wxWidgetsからWX_Objectの登録を抹消 */
void WX_UnregisterObject(WX_ObjectID id) {
	if ( !WX_WasInit() ) return ;
	if ( id == NULL ) return ;
	/* イベントとWX_Objectを同時に渡す */
	wxCommandEvent CmdEvt(SDL_EVT_UNREGISTEROBJECT,ID_WXAPP) ;
	CmdEvt.SetClientData(id) ;
	wxGetApp().AddPendingEvent(CmdEvt);
}

void WX_SendData(WX_Data* obj)
{
	if ( !WX_WasInit() ) return ;
	/* イベントとWX_Objectを同時に渡す */
	wxCommandEvent CmdEvt(SDL_EVT_DATA,ID_WXAPP) ;
	CmdEvt.SetClientData(obj) ;
	wxGetApp().AddPendingEvent(CmdEvt);
}

void WX_SendData(::Uint8 type, int code, WX_Data* obj)
{
	SDL_Event event ;
	event.type = type ;
	event.user.type = type ;
	event.user.code = code ;
	event.user.data1 = obj ;
	event.user.data2 = NULL ;
	SDL_PushEvent ( &event ) ;
}

/* rcg06192001 get linked library's version. */
const SDL_version *WX_Linked_Version(void)
{
	static SDL_version linked_version;
	SDL_WX_VERSION(&linked_version);
	return(&linked_version);
}
