/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh.file;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShFileSystem;
import net.morilib.sh.ShProcess;

public class ShOSProcess implements ShProcess {

	private static final int BUFSIZE = 1024;

	static class BException extends RuntimeException {

		IOException ioex;

		public BException(IOException t) {
			ioex = t;
		}

	}

	static abstract class InThread0 extends Thread {

		InputStream src;
		OutputStream dest;
		boolean live = true;

		InThread0(InputStream src, OutputStream dest) {
			this.src  = src;
			this.dest = dest;
		}

		synchronized void setStop() {
			live = false;
		}

	}

	static class InThread1 extends InThread0 {

		InThread1(InputStream src, OutputStream dest) {
			super(src, dest);
		}

		public void run() {
			byte[] b = new byte[BUFSIZE];
			int l;

			try {
				while((l = src.read(b)) >= 0) {
					dest.write(b, 0, l);
					dest.flush();
				}
			} catch(IOException e) {
				throw new BException(e);
			}
		}

	};

	static class InThread3 extends InThread0 {

		InThread3(InputStream src, OutputStream dest) {
			super(src, dest);
		}

		public void run() {
			int c;

			try {
				while(live) {
					while(live && src.available() == 0);
					if(!live || (c = src.read()) < 0)  return;
					dest.write(c);
					dest.flush();
				}
			} catch (IOException e) {
				throw new BException(e);
			}
		}

	};

	private ShRealFile command;

	ShOSProcess(ShRealFile command) {
		this.command = command;
	}

	public int main(ShEnvironment env,  ShFileSystem fs,
			InputStream stdin, PrintStream stdout,
			PrintStream stderr, String... args) throws IOException {
		InThread0 ot = null, er = null, in = null;
		ProcessBuilder b;
		List<String> l;
		Process p;

		try {
			l = new ArrayList<String>(Arrays.asList(args));
			l.set(0, command.file.toString());
			b = new ProcessBuilder(l);
			p = b.start();
			in = new InThread3(stdin, p.getOutputStream());
			ot = new InThread1(p.getInputStream(), stdout);
			er = new InThread1(p.getErrorStream(), stderr);
			in.start();  ot.start();  er.start();
			return p.waitFor();
		} catch(BException ex) {
			throw ex.ioex;
		} catch(InterruptedException e) {
			return 255;
		} finally {
			try {
				if(in != null) { in.setStop();  in.join(); }
				if(ot != null)  ot.join();
				if(er != null)  er.join();
				stdout.flush();
				stderr.flush();
			} catch (InterruptedException e) {
				return 255;
			}
		}
	}

}
