/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sed.line;

import net.morilib.sed.SedLineMatcher;
import net.morilib.sed.SedPatternEngine;

/**
 * パターンにマッチするマッチ器です。
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public class SedPatternLineMatcher implements SedLineMatcher {

	//
	private boolean caseInsensitive, allmatch;
	private String pattern;

	/**
	 * マッチ器を生成します。
	 * 
	 * @param pat パターン
	 */
	public SedPatternLineMatcher(String pat) {
		this(pat, false, false);
	}

	/**
	 * マッチ器を生成します。
	 * 
	 * @param pat パターン
	 * @param caseInsensive 大文字小文字を区別しないときtrue
	 * @param allmatch 全体マッチのときtrue
	 */
	public SedPatternLineMatcher(String pat, boolean caseInsensitive,
			boolean allmatch) {
		if(pat == null)  throw new NullPointerException();
		pattern = pat;
		this.caseInsensitive = caseInsensitive;
		this.allmatch = allmatch;
	}

	public boolean matches(SedPatternEngine pa, String s, long lineno,
			long relno, boolean lastno) {
		return pa.matches(pattern, s, caseInsensitive, allmatch);
	}

	/**
	 * 大文字小文字を区別しないときtrueを得ます。
	 * 
	 * @return 大文字小文字を区別しないときtrue
	 */
	public boolean isCaseInsensitive() {
		return caseInsensitive;
	}

	/**
	 * 全体マッチのときtrueを得ます。
	 * 
	 * @return 全体マッチのときtrue
	 */
	public boolean isAllmatch() {
		return allmatch;
	}

	/**
	 * パターンを得ます。
	 * 
	 * @return パターン
	 */
	public String getPattern() {
		return pattern;
	}

	public int hashCode() {
		return pattern.hashCode();
	}

	public boolean equals(Object o) {
		if(o instanceof SedPatternLineMatcher) {
			return pattern.equals(((SedPatternLineMatcher)o).pattern);
		}
		return false;
	}

	public String toString() {
		return "/" + pattern + "/";
	}

}
