/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.misc;

import java.math.BigInteger;

import net.morilib.awk.value.AwkComplex;
import net.morilib.awk.value.AwkFloat;
import net.morilib.awk.value.AwkInteger;
import net.morilib.awk.value.AwkValue;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public final class AwkComplexMath {

	//
	private static final AwkValue I =
			AwkComplex.valueOf(0.0, 1.0);
	private static final AwkValue TWO_I =
			AwkComplex.valueOf(0.0, 2.0);
	private static final Double P_ZERO = new Double( 0.0);
	private static final Double M_ZERO = new Double(-0.0);

	/**
	 * 
	 */
	public static final AwkValue INEXACT_PI =
			AwkFloat.valueOf(Math.PI);

	//
	private AwkComplexMath() {
		// do nothing
	}

	//
	private static AwkValue exptC(AwkValue nm1, AwkValue nm2) {
		double a1 = nm1.toFloat();
		double b1 = nm1.toImaginary();
		double r1 = Math.hypot(a1, b1);
		double t1 = Math.atan2(b1, a1);
		double a2 = nm2.toFloat();
		double b2 = nm2.toImaginary();
		double rr = a2 * Math.log(r1) - t1 * b2;
		double tr = b2 * Math.log(r1) + t1 * a2;

		return AwkComplex.valueOf(
				Math.exp(rr) * Math.cos(tr),
				Math.exp(rr) * Math.sin(tr));
	}

	/**
	 * 
	 * @param r
	 * @return
	 */
	public static boolean isPlusZero(double r) {
		return (r == 0.0) && P_ZERO.equals(new Double(r));
	}

	/**
	 * 
	 * @param r
	 * @return
	 */
	public static boolean isMinusZero(double r) {
		return (r == 0.0) && M_ZERO.equals(new Double(r));
	}

	/**
	 * 
	 * @param nm1
	 * @param nm2
	 * @return
	 */
	public static AwkValue expt(AwkValue nm1, AwkValue nm2) {
		if(nm1.isNaN() || nm2.isNaN()) {
			return AwkFloat.NaN;
		} else if(nm2.isZeroValue()) {
			if(nm1.isInteger() && nm2.isInteger()) {
				return AwkInteger.valueOf(1);
			} else {
				return AwkFloat.valueOf(1);
			}
		} else if(nm1.isZeroValue()) {
			if(nm1.isInteger() && nm2.isInteger()) {
				return AwkInteger.valueOf(0);
			} else {
				return AwkFloat.valueOf(0);
			}
		} else if(nm1.isOne()) {
			if(nm1.isInteger() && nm2.isInteger()) {
				return AwkInteger.valueOf(1);
			} else {
				return AwkFloat.valueOf(1);
			}
		} else if(nm2.isOne()) {
			if(nm1.isInteger() && nm2.isInteger()) {
				return nm1;
			} else {
				return AwkFloat.valueOf(nm1.toFloat());
			}
		} else if(nm2.isComplex()) {
			return exptC(nm1, nm2);
		} else if(nm1.isComplex()) {
			return expt(nm1, nm2.toFloat());
		} else if(nm1.toFloat() > 0) {
			return AwkFloat.valueOf(Math.pow(
					nm1.toFloat(), nm2.toFloat()));
		} else {
			return expt(nm1, nm2.toFloat());
		}
	}

	/**
	 * 
	 * @param nm1
	 * @param r
	 * @return
	 */
	public static AwkValue expt(AwkValue nm1, double r) {
		double a1 = nm1.toFloat();
		double b1 = nm1.toImaginary();

		if(nm1.isNaN() || Double.isNaN(r)) {
			return AwkFloat.NaN;
		} else if(b1 == 0.0 && a1 == 1.0) {
			return AwkFloat.valueOf(1);
		} else if(r == Double.POSITIVE_INFINITY) {
			double r1 = Math.hypot(a1, b1);

			if(r1 < 1.0 && r1 > -1.0) {
				return AwkFloat.valueOf(0);
			} else if(b1 == 0.0 && a1 > 1.0) {
				return AwkFloat.valueOf(Double.POSITIVE_INFINITY);
			} else {
				return AwkFloat.NaN;
			}
		} else if(r == Double.NEGATIVE_INFINITY) {
			double r1 = Math.hypot(a1, b1);

			if(r1 > 1.0 || r1 < -1.0) {
				return AwkFloat.valueOf(0);
			} else if(b1 == 0.0 && a1 < 1.0 && a1 > 0.0) {
				return AwkFloat.valueOf(Double.POSITIVE_INFINITY);
			} else {
				return AwkFloat.NaN;
			}
		} else if(b1 == 0.0 && a1 > 0.0) {
			return AwkFloat.valueOf(Math.pow(a1, r));
		} else {
			double r1 = Math.hypot(a1, b1);
			double t1 = Math.atan2(b1, a1);
			double a2 = r;
			double rr = a2 * Math.log(r1);
			double tr = t1 * a2;

			return AwkComplex.valueOf(
					Math.exp(rr) * Math.cos(tr),
					Math.exp(rr) * Math.sin(tr));
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue sqrt(AwkValue n) {
		BigInteger b;
		AwkValue x;

		if(n.isZeroValue()) {
			return n;
		} else if(n.isOne()) {
			return n;
		} else if(n.isInteger()) {
			b = Math2.sqrtExact(n.toInteger().abs());
			if(b.signum() > 0) {
				x = AwkInteger.valueOf(b);
			} else {
				x = AwkFloat.valueOf(Math.sqrt(
						Math.abs(n.toFloat())));
			}
			return n.toFloat() > 0 ?
					x : AwkComplex.valueOf(0, x.toFloat());
		} else if(n.isComplex()) {
			return expt(n, 0.5);
		} else if(n.toFloat() > 0.0) {
			return AwkFloat.valueOf(Math.sqrt(n.toFloat()));
		} else {
			return AwkComplex.valueOf(
					0.0, Math.sqrt(-n.toFloat()));
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue exp(AwkValue n) {
		if(n.isComplex()) {
			double a = n.toFloat();
			double b = n.toImaginary();

			return AwkComplex.valueOf(
					Math.exp(a) * Math.cos(b),
					Math.exp(a) * Math.sin(b));
		} else {
			return AwkFloat.valueOf(Math.exp(n.toFloat()));
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue log(AwkValue n) {
		if(n.isReal() && n.toFloat() > 0) {
			return AwkFloat.valueOf(Math.log(n.toFloat()));
		} else {
			double a = n.toFloat();
			double b = n.toImaginary();
			double r = Math.hypot(a, b);
			double t = Math.atan2(b, a);

			return AwkComplex.valueOf(Math.log(r), t);
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue sin(AwkValue n) {
		if(n.isReal()) {
			return AwkFloat.valueOf(Math.sin(n.toFloat()));
		} else if(n.toFloat() == 0) {
			double y = n.toImaginary();
			double b = Math.sinh(y);

			return AwkComplex.valueOf(0.0, b);
		} else {
			double x = n.toFloat();
			double y = n.toImaginary();
			double a = Math.sin(x) * Math.cosh(y);
			double b = Math.cos(x) * Math.sinh(y);

			return AwkComplex.valueOf(a, b);
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue cos(AwkValue n) {
		if(n.isReal()) {
			return AwkFloat.valueOf(Math.cos(n.toFloat()));
		} else if(n.toFloat() == 0) {
			double y = n.toImaginary();
			double a = Math.cosh(y);

			return AwkComplex.valueOf(a, 0.0);
		} else {
			double x = n.toFloat();
			double y = n.toImaginary();
			double a = Math.cos(x) * Math.cosh(y);
			double b = -(Math.sin(x) * Math.sinh(y));

			return AwkComplex.valueOf(a, b);
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue tan(AwkValue n) {
		if(n.isReal()) {
			return AwkFloat.valueOf(Math.tan(n.toFloat()));
		} else if(n.toFloat() == 0) {
			double y = n.toImaginary();

			return AwkComplex.valueOf(0.0, Math.tanh(y));
		} else if(n.toImaginary() == Double.POSITIVE_INFINITY) {
			double x = n.toFloat();
			AwkValue nm, dn;

			nm = AwkComplex.valueOf(Math.tan(x), 1);
			dn = AwkComplex.valueOf(1, Math.tan(x));
			return AwkValue.div(nm, dn);
		} else if(n.toImaginary() == Double.NEGATIVE_INFINITY) {
			double x = n.toFloat();
			AwkValue nm, dn;

			nm = AwkComplex.valueOf(Math.tan(x), -1);
			dn = AwkComplex.valueOf(1, -Math.tan(x));
			return AwkValue.div(nm, dn);
		} else {
			return AwkValue.div(sin(n), cos(n));
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue asin(AwkValue n) {
		double x = n.toFloat();
		double y = n.toImaginary();

		if(n.isReal() && x >= -1.0 && x <= 1.0) {
			return AwkFloat.valueOf(Math.asin(n.toFloat()));
		} else if(Double.isInfinite(x) && Double.isInfinite(y)) {
			return AwkFloat.NaN;
		} else if(x == Double.POSITIVE_INFINITY) {
			if(y <= 0) {
				x = -x;
			}
			return AwkComplex.valueOf(Math.PI / 2, x);
		} else if(x == Double.NEGATIVE_INFINITY) {
			if(y >= 0) {
				x = -x;
			}
			return AwkComplex.valueOf(-Math.PI / 2, x);
		} else if(y == Double.POSITIVE_INFINITY) {
			return AwkComplex.valueOf(0, y);
		} else if(y == Double.NEGATIVE_INFINITY) {
			return AwkComplex.valueOf(0, y);
		} else {
			AwkValue z1 = AwkValue.mul(I, n);
			AwkValue z2 = sqrt(AwkValue.sub(AwkFloat.valueOf(1),
					AwkValue.mul(n, n)));
			AwkValue z3 = log(AwkValue.add(z1, z2));

			//return LispComplex.MINUS_I.mul(z3);
			return AwkComplex.valueOf(
					z3.toImaginary(), -z3.toFloat());
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue acos(AwkValue n) {
		double x = n.toFloat();

		if(n.isReal() && x >= -1.0 && x <= 1.0) {
			return AwkFloat.valueOf(Math.acos(n.toFloat()));
		} else {
			AwkValue pi2 = AwkFloat.valueOf(Math.PI / 2);

			return AwkValue.sub(pi2, asin(n));
		}
	}

	/**
	 * 
	 * @param n
	 * @return
	 */
	public static AwkValue atan(AwkValue n) {
		if(n.isReal()) {
			return AwkFloat.valueOf(Math.atan(n.toFloat()));
		} else if(Double.isInfinite(n.toImaginary())) {
			double im = n.toImaginary();

			if(Double.isInfinite(n.toFloat())) {
				return AwkFloat.NaN;
			} else if(im > 0) {
				return AwkFloat.valueOf(Math.PI / 2);
			} else {
				return AwkFloat.valueOf(-Math.PI / 2);
			}
		} else if(n.toFloat() == 0.0 &&
				n.toImaginary() == 1.0) {
			return AwkComplex.valueOf(0, Double.POSITIVE_INFINITY);
		} else if(n.toFloat() == 0.0 &&
				n.toImaginary() == -1.0) {
			return AwkComplex.valueOf(0, Double.NEGATIVE_INFINITY);
		} else {
			AwkValue z1 = AwkValue.add(AwkFloat.valueOf(1),
					AwkValue.mul(I, n));
			AwkValue z2 = AwkValue.sub(AwkFloat.valueOf(1),
					AwkValue.mul(I, n));
			AwkValue z3 = AwkValue.sub(log(z1), log(z2));

			return AwkValue.div(z3, TWO_I);
		}
	}

}
