/***************************************************************************

	Apache 3											ATF-011
	Round Up 5											ATC-011
	Cycle Warriors										ABA-011

	Incredibly complex hardware!  These are all different boards, but share
	a similar sprite chip (TZB215 on Apache 3, TZB315 on others).  Other
	graphics (road, sky, bg/fg layers) all differ between games.

	Todo:
		Sprite rotation
		Finish road layer (Round Up 5)
		Implement road layer (Apache 3, Cycle Warriors)
		BG layer(s) (Cycle Warriors)
		BG layer (Round Up 5) - May be driven by missing VRAM data
		Round Up 5 always boots with a coin inserted
		Round Up 5 doesn't survive a reset
		Dip switches
		Various other things..

	Emulation by Bryan McPhail, mish@tendril.co.uk


	Cycle Warriors Board Layout

	ABA-011


			6296             CW24A                   5864
								CW25A                   5864
			YM2151                  50MHz

								TZ8315                 CW26A
													5864
		TC51821  TC51832                               D780C-1
		TC51821  TC51832
		TC51821  TC51832                     16MHz
		TC51821  TC51832

		CW00A   CW08A
		CW01A   CW09A
		CW02A   CW10A
		CW03A   CW11A            68000-12              81C78
		CW04A   CW12A                                  81C78
		CW05A   CW13A                CW16B  CW18B      65256
		CW06A   CW14A                CW17A  CW19A      65256
		CW07A   CW15A                       CW20A
											CW21       65256
								68000-12   CW22A      65256
											CW23

	ABA-012

							HD6445


							51832
							51832
							51832
							51832

							CW28
							CW29
							CW30

	CW27

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "tatsumi.h"

static data16_t *cyclwarr_cpub_ram;
data16_t *tatsumi_c_ram, *apache3_g_ram;
data16_t *roundup5_d0000_ram, *roundup5_e0000_ram;
data8_t *tatsumi_rom_sprite_lookup1, *tatsumi_rom_sprite_lookup2;
data8_t *tatsumi_rom_clut0, *tatsumi_rom_clut1;
data8_t *roundup5_unknown0, *roundup5_unknown1, *roundup5_unknown2;
data8_t *apache3_bg_ram;

/***************************************************************************/

static READ16_HANDLER(cyclwarr_cpu_b_r) { return cyclwarr_cpub_ram[offset+0x800]; }
static WRITE16_HANDLER(cyclwarr_cpu_b_w){ COMBINE_DATA(&cyclwarr_cpub_ram[offset+0x800]); }
static READ16_HANDLER(cyclwarr_cpu_bb_r){ return cyclwarr_cpub_ram[offset]; }
static WRITE16_HANDLER(cyclwarr_cpu_bb_w) { COMBINE_DATA(&cyclwarr_cpub_ram[offset]); }
static READ16_HANDLER(cyclwarr_palette_r) { return paletteram16[offset]; }
static READ16_HANDLER(cyclwarr_sprite_r) { return spriteram16[offset]; }
static WRITE16_HANDLER(cyclwarr_sprite_w) { COMBINE_DATA(&spriteram16[offset]); }
static READ16_HANDLER(cyclwarr_input_r) { return readinputport(offset); }
static READ16_HANDLER(cyclwarr_input2_r) { return readinputport(offset+3); }

/***************************************************************************/

static MEMORY_READ_START( readmem_apache3 )
	{ 0x00000, 0x07fff, MRA_RAM },
	{ 0x08000, 0x08fff, MRA_RAM },
	{ 0x0c000, 0x0dfff, MRA_RAM },
//	{ 0x0e800, 0x0e803, MRA_NOP }, // CRT
	{ 0x0f000, 0x0f000, input_port_3_r }, // Dip 1
	{ 0x0f001, 0x0f001, input_port_4_r }, // Dip 2
	{ 0x0f800, 0x0f801, apache3_bank_r },
	{ 0x10000, 0x1ffff, apache3_v30_v20_r },
	{ 0x20000, 0x2ffff, tatsumi_v30_68000_r },
	{ 0xa0000, 0xfffff, MRA_ROM },
MEMORY_END

static MEMORY_WRITE_START( writemem_apache3 )
	{ 0x00000, 0x07fff, MWA_RAM },
	{ 0x08000, 0x08fff, apache3_palette_w, &paletteram },
	{ 0x0c000, 0x0dfff, roundup5_text_w, &videoram },
	{ 0x0e800, 0x0e803, MWA_NOP }, // CRT
	{ 0x0f000, 0x0f001, MWA_NOP }, // todo
	{ 0x0f800, 0x0f801, apache3_bank_w },
	{ 0x10000, 0x1ffff, apache3_v30_v20_w },
	{ 0x20000, 0x23fff, tatsumi_v30_68000_w },
	{ 0xa0000, 0xfffff, MWA_ROM },
MEMORY_END

static MEMORY_READ16_START( sub_readmem_apache3 )
	{ 0x00000, 0x7ffff, MRA16_ROM },
	{ 0x80000, 0x83fff, MRA16_RAM },
	{ 0x90000, 0x93fff, MRA16_RAM },
	{ 0xd0000, 0xdffff, MRA16_RAM },
	{ 0xe0000, 0xe7fff, apache3_z80_r },
MEMORY_END

static MEMORY_WRITE16_START( sub_writemem_apache3 )
	{ 0x00000, 0x7ffff, MWA16_ROM },
	{ 0x80000, 0x83fff, MWA16_RAM, &tatsumi_68k_ram },
	{ 0x90000, 0x93fff, MWA16_RAM, &spriteram16 },
	{ 0x9a000, 0x9a1ff, tatsumi_sprite_control_w, &tatsumi_sprite_control_ram },
	{ 0xa0000, 0xa0001, apache3_a0000_w },
	{ 0xb0000, 0xb0001, apache3_irq_ack_w }, //todo - z80 reset?
	{ 0xc0000, 0xc0001, MWA16_RAM, &tatsumi_c_ram },
	{ 0xd0000, 0xdffff, MWA16_RAM, &apache3_g_ram },
	{ 0xe0000, 0xe7fff, apache3_z80_w },
MEMORY_END

static MEMORY_READ_START( readmem_iop_apache3 )
	{ 0x00000, 0x01fff, MRA_RAM },
	{ 0x04000, 0x04003, MRA_NOP }, // piu select .. ?
	{ 0x06000, 0x06000, input_port_0_r }, // esw
	{ 0x08001, 0x08001, tatsumi_hack_ym2151_r }, //YM2151_status_port_0_r },
	{ 0x0a000, 0x0a000, tatsumi_hack_oki_r }, //OKIM6295_status_0_r },
	{ 0x0e000, 0x0e000, apache3_adc_r }, // adc
	{ 0xf0000, 0xfffff, MRA_ROM },
MEMORY_END

static MEMORY_WRITE_START( writemem_iop_apache3 )
	{ 0x00000, 0x01fff, MWA_RAM },
	{ 0x04000, 0x04003, MWA_NOP }, // piu select .. ?
	{ 0x08000, 0x08000, YM2151_register_port_0_w },
	{ 0x08001, 0x08001, YM2151_data_port_0_w },
	{ 0x0a000, 0x0a000, OKIM6295_data_0_w },
	{ 0x0e000, 0x0e007, apache3_adc_w }, //adc select
	{ 0xf0000, 0xfffff, MWA_ROM },
MEMORY_END

static MEMORY_READ_START( z80_readmem_apache3 )
	{ 0x00000, 0x1fff, MRA_RAM },
	{ 0x08000, 0x83ff, MRA_RAM },
MEMORY_END

static MEMORY_WRITE_START( z80_writemem_apache3 )
	{ 0x00000, 0x1fff, MWA_RAM, &apache3_z80_ram },
	{ 0x08000, 0x83ff, MWA_RAM, &apache3_bg_ram },
MEMORY_END

/*****************************************************************/

static MEMORY_READ_START( readmem_roundup5 )
	{ 0x00000, 0x07fff, MRA_RAM },
	{ 0x08000, 0x0bfff, MRA_RAM },
	{ 0x0d000, 0x0d000, input_port_3_r }, /* Dip 1 */
	{ 0x0d001, 0x0d001, input_port_4_r }, /* Dip 2 */
	{ 0x0f000, 0x0ffff, MRA_RAM },
	{ 0x10000, 0x1ffff, roundup_v30_z80_r },
	{ 0x20000, 0x2ffff, tatsumi_v30_68000_r },
	{ 0x30000, 0x3ffff, roundup5_vram_r },
	{ 0x80000, 0xfffff, MRA_ROM },
MEMORY_END

static MEMORY_WRITE_START( writemem_roundup5 )
	{ 0x00000, 0x07fff, MWA_RAM },
	{ 0x08000, 0x0bfff, roundup5_text_w, &videoram },
	{ 0x0c000, 0x0c003, roundup5_crt_w },
	{ 0x0d400, 0x0d40f, MWA_RAM, &roundup5_unknown0 },
	{ 0x0d800, 0x0d801, MWA_RAM, &roundup5_unknown1 }, // VRAM2 X scroll (todo },
	{ 0x0dc00, 0x0dc01, MWA_RAM, &roundup5_unknown2 }, // VRAM2 Y scroll (todo },
	{ 0x0e000, 0x0e001, roundup5_control_w },
	{ 0x0f000, 0x0ffff, roundup5_palette_w, &paletteram },
	{ 0x10000, 0x1ffff, roundup_v30_z80_w },
	{ 0x20000, 0x23fff, tatsumi_v30_68000_w },
	{ 0x30000, 0x3ffff, roundup5_vram_w },
	{ 0x80000, 0xfffff, MWA_ROM },
MEMORY_END

static MEMORY_READ16_START( readmem_roundup5_sub )
	{ 0x00000, 0x7ffff, MRA16_ROM },
	{ 0x80000, 0x83fff, MRA16_RAM },
	{ 0x90000, 0x93fff, MRA16_RAM },
	{ 0xa0000, 0xa0fff, MRA16_RAM },
	{ 0xb0000, 0xb0fff, MRA16_RAM },
	{ 0xc0000, 0xc0fff, MRA16_RAM },
MEMORY_END

static MEMORY_WRITE16_START( writemem_roundup5_sub )
	{ 0x00000, 0x7ffff, MWA16_ROM },
	{ 0x80000, 0x83fff, MWA16_RAM, &tatsumi_68k_ram },
	{ 0x90000, 0x93fff, MWA16_RAM, &spriteram16 },
	{ 0x9a000, 0x9a1ff, tatsumi_sprite_control_w, &tatsumi_sprite_control_ram },
	{ 0xa0000, 0xa0fff, MWA16_RAM, &roundup_r_ram }, // Road control data
	{ 0xb0000, 0xb0fff, MWA16_RAM, &roundup_p_ram }, // Road pixel data
	{ 0xc0000, 0xc0fff, MWA16_RAM, &roundup_l_ram }, // Road colour data
	{ 0xd0002, 0xd0003, roundup5_d0000_w, &roundup5_d0000_ram },
	{ 0xe0000, 0xe0001, roundup5_e0000_w, &roundup5_e0000_ram },
MEMORY_END

static MEMORY_READ_START( readmem_roundup5_sound )
	{ 0x0000, 0xdfff, MRA_ROM },
	{ 0xe000, 0xffef, MRA_RAM }, // maybe less than this...
	{ 0xfff1, 0xfff1, tatsumi_hack_ym2151_r }, //YM2151_status_port_0_r },
	{ 0xfff4, 0xfff4, tatsumi_hack_oki_r }, //OKIM6295_status_0_r },
	{ 0xfff8, 0xfff8, input_port_0_r },
	{ 0xfff9, 0xfff9, input_port_1_r },
	{ 0xfffc, 0xfffc, input_port_2_r },
MEMORY_END

static MEMORY_WRITE_START( writemem_roundup5_sound )
	{ 0x0000, 0xdfff, MWA_ROM },
	{ 0xe000, 0xffef, MWA_RAM },
	{ 0xfff0, 0xfff0, YM2151_register_port_0_w },
	{ 0xfff1, 0xfff1, YM2151_data_port_0_w },
	{ 0xfff4, 0xfff4, OKIM6295_data_0_w },

	{ 0xfff9, 0xfff9, MWA_NOP }, //irq ack?
	{ 0xfffa, 0xfffa, MWA_NOP }, //irq ack?
MEMORY_END

/******************************************************************************/

static MEMORY_READ16_START( readmem_cyclwarr_a )
	{ 0x000000, 0x00ffff, MRA16_RAM },
	{ 0x03e000, 0x03efff, MRA16_RAM },
	{ 0x040000, 0x043fff, cyclwarr_cpu_bb_r },
	{ 0x080000, 0x08ffff, cyclwarr_videoram2_r },
	{ 0x090000, 0x09ffff, cyclwarr_videoram_r },
	{ 0x0b9002, 0x0b9009, cyclwarr_input_r }, //b9008 - dips
	// ba000 + ba002 - dips
	{ 0x0ba000, 0x0ba003, cyclwarr_input2_r }, //temp
	{ 0x0ba004, 0x0ba007, cyclwarr_input2_r },
	{ 0x0ba008, 0x0ba009, cyclwarr_control_r },
	{ 0x0c0000, 0x0c3fff, cyclwarr_sprite_r },
	{ 0x0d0000, 0x0d3fff, cyclwarr_palette_r },
	{ 0x140000, 0x1bffff, MRA16_BANK2 }, /* CPU B ROM */
	{ 0x2c0000, 0x33ffff, MRA16_BANK1 }, /* CPU A ROM */
MEMORY_END

static MEMORY_WRITE16_START( writemem_cyclwarr_a )
	{ 0x000000, 0x00dfff, MWA16_RAM },
	{ 0x00e000, 0x00ffff, MWA16_RAM, &videoram16 },
	{ 0x03e000, 0x03efff, MWA16_RAM },
	{ 0x040000, 0x043fff, cyclwarr_cpu_bb_w },
	{ 0x080000, 0x08ffff, cyclwarr_videoram2_w, &cyclwarr_videoram2 },
	{ 0x090000, 0x09ffff, cyclwarr_videoram_w, &cyclwarr_videoram },
	{ 0x0ba008, 0x0ba009, cyclwarr_control_w },
	{ 0x0c0000, 0x0c3fff, cyclwarr_sprite_w, &spriteram16 },
	{ 0x0ca000, 0x0ca1ff, tatsumi_sprite_control_w, &tatsumi_sprite_control_ram },
	{ 0x0d0000, 0x0d3fff, paletteram16_xRRRRRGGGGGBBBBB_word_w, &paletteram16 },
MEMORY_END

static MEMORY_READ16_START( readmem_cyclwarr_b )
	{ 0x000000, 0x00ffff, MRA16_RAM },
	{ 0x080000, 0x08ffff, cyclwarr_videoram2_r },
	{ 0x090000, 0x09ffff, cyclwarr_videoram_r },
	{ 0x0c0000, 0x0c3fff, cyclwarr_sprite_r },
	{ 0x0d0000, 0x0d3fff, cyclwarr_palette_r },
	{ 0x140000, 0x1bffff, MRA16_BANK2 },
	{ 0x2c0000, 0x33ffff, MRA16_BANK1 },
MEMORY_END

static MEMORY_WRITE16_START( writemem_cyclwarr_b )
	{ 0x000000, 0x00dfff, MWA16_RAM, &cyclwarr_cpub_ram },
	{ 0x00e000, 0x00ffff, MWA16_RAM },
	{ 0x080000, 0x08ffff, cyclwarr_videoram2_w },
	{ 0x090000, 0x09ffff, cyclwarr_videoram_w },
	{ 0x0c0000, 0x0c3fff, cyclwarr_sprite_w },
	{ 0x0ca000, 0x0ca1ff, tatsumi_sprite_control_w },
	{ 0x0d0000, 0x0d3fff, paletteram16_xRRRRRGGGGGBBBBB_word_w },
MEMORY_END

static MEMORY_READ_START( readmem_cyclwarr_c )
	{ 0x0000, 0xdfff, MRA_ROM },
	{ 0xe000, 0xffef, MRA_RAM }, // maybe less than this...
	{ 0xfff1, 0xfff1, tatsumi_hack_ym2151_r }, //YM2151_status_port_0_r },
	{ 0xfff4, 0xfff4, tatsumi_hack_oki_r }, // OKIM6295_status_0_r },
	{ 0xfff8, 0xfff8, input_port_0_r },
	{ 0xfff9, 0xfff9, input_port_1_r },
//	{ 0xfffa, 0xfffa, input_port_0_r },// MRA_NOP }, //irq ack???
	{ 0xfffc, 0xfffc, input_port_2_r },// MRA_NOP },
MEMORY_END

static MEMORY_WRITE_START( writemem_cyclwarr_c )
	{ 0x0000, 0xdfff, MWA_ROM },
	{ 0xe000, 0xffef, MWA_RAM },
	{ 0xfff0, 0xfff0, YM2151_register_port_0_w },
	{ 0xfff1, 0xfff1, YM2151_data_port_0_w },
	{ 0xfff4, 0xfff4, OKIM6295_data_0_w },
	{ 0xfff9, 0xfff9, MWA_NOP }, //irq ack?
	{ 0xfffa, 0xfffa, MWA_NOP }, //irq ack?
MEMORY_END

/******************************************************************************/

INPUT_PORTS_START( apache3 )
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )

	PORT_START
	PORT_ANALOG( 0xff, 0x7f, IPT_AD_STICK_X | IPF_PLAYER1 | IPF_CENTER , 25, 15, 0x00, 0xff )

	PORT_START
	PORT_ANALOG( 0xff, 0x7f, IPT_AD_STICK_Y | IPF_PLAYER1 | IPF_CENTER , 25, 15, 0x00, 0xff )

	PORT_START
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Test" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END

INPUT_PORTS_START( roundup5 )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )

	PORT_START
	PORT_ANALOG( 0xff, 0x7f, IPT_AD_STICK_X | IPF_PLAYER1 | IPF_CENTER , 25, 15, 0x00, 0xff )

	PORT_START
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Language" )
	PORT_DIPSETTING(    0x20, "Japanese" )
	PORT_DIPSETTING(    0x00, "English" )
	PORT_DIPNAME( 0x40, 0x00, "Stage 5 Continue" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, "Output Mode" )
	PORT_DIPSETTING(    0x00, "A" )
	PORT_DIPSETTING(    0x80, "B" )

	PORT_START
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Test" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END

INPUT_PORTS_START( cyclwarr )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SERVICE2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE4 )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER1 )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER2 )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER3 )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START4 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER4 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER4 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER4 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_PLAYER4 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_PLAYER4 )

	PORT_START
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_BUTTON4  )

	PORT_START
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_BUTTON4  )

	PORT_START
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_BUTTON4  )
INPUT_PORTS_END

/******************************************************************************/

static struct GfxLayout roundup5_charlayout =
{
	8,8,	/* 16*16 sprites */
	RGN_FRAC(1,1),	/* 4096 sprites */
	4,	/* 4 bits per pixel */
	{ 0, 1, 2, 3 },
	{ 8,12,0,4, 24,28, 16,20},
	{ 0*32, 1*32, 2*32, 3*32, 4*32, 5*32, 6*32, 7*32},
	32*8	/* every sprite takes 32 consecutive bytes */
};

static struct GfxLayout cyclwarr_charlayout =
{
	8,8,
	RGN_FRAC(1,3),
	3,
	{ RGN_FRAC(0,3), RGN_FRAC(1,3), RGN_FRAC(2,3)},
	{ 0, 1, 2, 3, 4, 5, 6, 7},
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8},
	8*8
};
static struct GfxLayout cyclwarr_charlayout2 =
{
	8,8,
	RGN_FRAC(1,3),
	3,
	{ RGN_FRAC(2,3), RGN_FRAC(1,3), RGN_FRAC(0,3)},
	{ 0, 1, 2, 3, 4, 5, 6, 7},
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8},
	8*8
};
static struct GfxLayout roundup5_vramlayout =
{
	8,8,
	4096 + 2048,
	3,
	{ 0x30000 * 8, 0x18000 * 8, 0 },
	{ 0, 1, 2, 3, 4, 5, 6, 7},
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16},
	8*16
};

static struct GfxDecodeInfo gfxdecodeinfo_apache3[] =
{
	{ REGION_GFX1, 0, &roundup5_charlayout,    1024, 128},
	{ REGION_GFX4, 0, &cyclwarr_charlayout,     768, 16},
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo roundup5_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &roundup5_charlayout,     1024, 256},
	{ 0, 0, &roundup5_vramlayout,					0, 16},
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo cyclwarr_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &roundup5_charlayout,    8192, 512},
	{ REGION_GFX5, 0, &cyclwarr_charlayout,    0, 512},
	{ REGION_GFX5, 0, &cyclwarr_charlayout2,   0, 512},
	{ -1 } /* end of array */
};

/******************************************************************************/

static void sound_irq(int state)
{
	cpu_set_irq_line(2, 0, state);
}

static struct YM2151interface ym2151_interface =
{
	1,
	16000000/4, // ?
	{ YM3012_VOL(45,MIXER_PAN_LEFT,45,MIXER_PAN_RIGHT) },
	{ sound_irq }
};

static struct OKIM6295interface okim6295_interface =
{
	1,
	{ 20000000/8/132 }, /* Frequency */ // ?
	{ REGION_SOUND1 },
	{ 75 }
};

static INTERRUPT_GEN( roundup5_interrupt )
{
	cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xc8/4);	/* VBL */
}

static MACHINE_DRIVER_START( apache3 )

	/* basic machine hardware */
	MDRV_CPU_ADD(V30,20000000 / 2) /* NEC V30 CPU, 20MHz / 2 */
	MDRV_CPU_MEMORY(readmem_apache3,writemem_apache3)
	MDRV_CPU_VBLANK_INT(roundup5_interrupt,1)

	MDRV_CPU_ADD(M68000,20000000 / 2) /* 68000 CPU, 20MHz / 2 */
	MDRV_CPU_MEMORY(sub_readmem_apache3,sub_writemem_apache3)

	MDRV_CPU_ADD(V20, 8000000) //???
	MDRV_CPU_MEMORY(readmem_iop_apache3,writemem_iop_apache3)

	MDRV_CPU_ADD(Z80, 8000000) //???
	MDRV_CPU_MEMORY(z80_readmem_apache3,z80_writemem_apache3)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)
	MDRV_INTERLEAVE(100)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_RGB_DIRECT)
	MDRV_SCREEN_SIZE(40*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 40*8-1, 0*8, 30*8-1)
	MDRV_GFXDECODE(gfxdecodeinfo_apache3)
	MDRV_PALETTE_LENGTH(1024 + 4096) /* 1024 real colours, and 4096 arranged as series of cluts */

	MDRV_VIDEO_START(apache3)
	MDRV_VIDEO_UPDATE(apache3)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM2151, ym2151_interface)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( roundup5 )

	/* basic machine hardware */
	MDRV_CPU_ADD(V30,20000000 / 2) /* NEC V30 CPU, 20MHz / 2 */
	MDRV_CPU_MEMORY(readmem_roundup5,writemem_roundup5)
	MDRV_CPU_VBLANK_INT(roundup5_interrupt,1)

	MDRV_CPU_ADD(M68000,20000000 / 2) /* 68000 CPU, 20MHz / 2 */
	MDRV_CPU_MEMORY(readmem_roundup5_sub,writemem_roundup5_sub)

	MDRV_CPU_ADD(Z80, 4000000) //???
	MDRV_CPU_MEMORY(readmem_roundup5_sound,writemem_roundup5_sound)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)
	MDRV_INTERLEAVE(100)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_RGB_DIRECT)
	MDRV_SCREEN_SIZE(40*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 40*8-1, 0*8, 30*8-1)
	MDRV_GFXDECODE(roundup5_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(1024 + 4096) /* 1024 real colours, and 4096 arranged as series of cluts */

	MDRV_VIDEO_START(roundup5)
	MDRV_VIDEO_UPDATE(roundup5)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM2151, ym2151_interface)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( cyclwarr )

	/* basic machine hardware */
	MDRV_CPU_ADD(M68000,20000000 / 2) /* NEC V30 CPU, 20MHz / 2 */
	MDRV_CPU_MEMORY(readmem_cyclwarr_a,writemem_cyclwarr_a )
	MDRV_CPU_VBLANK_INT(irq5_line_hold,1)

	MDRV_CPU_ADD(M68000,20000000 / 2) /* 68000 CPU, 20MHz / 2 */
	MDRV_CPU_MEMORY(readmem_cyclwarr_b,writemem_cyclwarr_b)
	MDRV_CPU_VBLANK_INT(irq5_line_hold,1)

	MDRV_CPU_ADD(Z80, 4000000) //???
	MDRV_CPU_MEMORY(readmem_cyclwarr_c,writemem_cyclwarr_c)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)
	MDRV_INTERLEAVE(200)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_RGB_DIRECT)
	MDRV_SCREEN_SIZE(40*8, 32*8)
	MDRV_VISIBLE_AREA(0*8, 40*8-1, 2*8, 30*8-1)
	MDRV_GFXDECODE(cyclwarr_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(8192 + 8192) //todo

	MDRV_VIDEO_START(cyclwarr)
	MDRV_VIDEO_UPDATE(cyclwarr)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM2151, ym2151_interface)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END

/***************************************************************************/

ROM_START( apache3 )
	ROM_REGION( 0x100000, REGION_CPU1, 0 ) /* v30 main cpu */
	ROM_LOAD16_BYTE( "ap-25f.125",   0x0a0001, 0x10000, 0x3c7530f4 )
	ROM_LOAD16_BYTE( "ap-26f.133",   0x0a0000, 0x10000, 0x2955997f )
	ROM_LOAD16_BYTE( "ap-23f.110",   0x0e0001, 0x10000, 0xd7077149 )
	ROM_LOAD16_BYTE( "ap-24f.118",   0x0e0000, 0x10000, 0x0bdef11b )

	ROM_REGION( 0x80000, REGION_CPU2, 0 ) /* 68000 sub cpu */
	ROM_LOAD16_BYTE( "ap-19c.80",   0x000001, 0x10000, 0x0908e468 )
	ROM_LOAD16_BYTE( "ap-21c.97",   0x000000, 0x10000, 0x38a056fb )
	ROM_LOAD16_BYTE( "ap-20a.89",   0x040001, 0x20000, 0x92d24b5e )
	ROM_LOAD16_BYTE( "ap-22a.105",  0x040000, 0x20000, 0xa8458a92 )

	ROM_REGION( 0x100000, REGION_CPU3, 0 ) /* 64k code for sound V20 */
	ROM_LOAD( "ap-27d.151",   0x0f0000, 0x10000, 0x294b4d79 )

	ROM_REGION( 0x10000, REGION_CPU4, 0 ) /* 64k code for z80 */
	/* Filled in by main cpu */

	ROM_REGION( 0x200000, REGION_GFX1, ROMREGION_DISPOSE )
	/* Filled in by both regions below */

	ROM_REGION( 0x100000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD32_BYTE( "ap-00c.15",   0x000000, 0x20000, 0xad1ddc2b )
	ROM_LOAD32_BYTE( "ap-01c.22",   0x000001, 0x20000, 0x6286ff00 )
	ROM_LOAD32_BYTE( "ap-04c.58",   0x000002, 0x20000, 0xdc6d55e4 )
	ROM_LOAD32_BYTE( "ap-05c.65",   0x000003, 0x20000, 0x2e6e495f )
	ROM_LOAD32_BYTE( "ap-02c.34",   0x080000, 0x20000, 0xaf4ee7cb )
	ROM_LOAD32_BYTE( "ap-03c.46",   0x080001, 0x20000, 0x60ab495c )
	ROM_LOAD32_BYTE( "ap-06c.71",   0x080002, 0x20000, 0x0ea90e55 )
	ROM_LOAD32_BYTE( "ap-07c.75",   0x080003, 0x20000, 0xba685543 )

	ROM_REGION( 0x100000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD32_BYTE( "ap-08c.14",   0x000000, 0x20000, 0x6437b580 )
	ROM_LOAD32_BYTE( "ap-09c.21",   0x000001, 0x20000, 0x54d18ef9 )
	ROM_LOAD32_BYTE( "ap-12c.57",   0x000002, 0x20000, 0xf95cf5cf )
	ROM_LOAD32_BYTE( "ap-13c.64",   0x000003, 0x20000, 0x67a248c3 )
	ROM_LOAD32_BYTE( "ap-10c.33",   0x080000, 0x20000, 0x74418df4 )
	ROM_LOAD32_BYTE( "ap-11c.45",   0x080001, 0x20000, 0x195bf78e )
	ROM_LOAD32_BYTE( "ap-14c.70",   0x080002, 0x20000, 0x58f7fe16 )
	ROM_LOAD32_BYTE( "ap-15c.74",   0x080003, 0x20000, 0x1ffd5496 )

	ROM_REGION( 0x18000, REGION_GFX4, ROMREGION_DISPOSE )
	ROM_LOAD( "ap-18d.73",   0x000000, 0x8000, 0x55e664bf )
	ROM_LOAD( "ap-17d.68",   0x008000, 0x8000, 0x6199afe4 )
	ROM_LOAD( "ap-16d.63",   0x010000, 0x8000, 0xf115656d )

	ROM_REGION( 0x30000, REGION_SOUND1, 0 )	 /* ADPCM samples */
	ROM_LOAD( "ap-28c.171",   0x000000, 0x20000, 0xb349f0c2 )
	ROM_LOAD( "ap-29c.176",   0x020000, 0x10000, 0xb38fced3 )
ROM_END

ROM_START( roundup5 )
	ROM_REGION( 0x100000, REGION_CPU1, 0 ) /* v30 main cpu */
	ROM_LOAD16_BYTE( "ru-23s",   0x080000, 0x20000, 0x2dc8c521 )
	ROM_LOAD16_BYTE( "ru-26s",   0x080001, 0x20000, 0x1e16b531 )
	ROM_LOAD16_BYTE( "ru-22t",   0x0c0000, 0x20000, 0x9611382e )
	ROM_LOAD16_BYTE( "ru-25t",   0x0c0001, 0x20000, 0xb6cd0f2d )

	ROM_REGION( 0x80000, REGION_CPU2, 0 ) /* 68000 sub cpu */
	ROM_LOAD16_BYTE( "ru-20s",   0x000000, 0x20000, 0xc5524558 )
	ROM_LOAD16_BYTE( "ru-18s",   0x000001, 0x20000, 0x163ef03d )
	ROM_LOAD16_BYTE( "ru-24s",   0x040000, 0x20000, 0xb9f91b70 )
	ROM_LOAD16_BYTE( "ru-19s",   0x040001, 0x20000, 0xe3953800 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 ) /* 64k code for sound Z80 */
	ROM_LOAD( "ru-28d",   0x000000, 0x10000, 0xdf36c6c5 )

	ROM_REGION( 0x180000, REGION_GFX1, ROMREGION_DISPOSE )
	/* Filled in by both regions below */

	ROM_REGION( 0x0c0000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD32_BYTE( "ru-00b",   0x000000, 0x20000, 0x388a0647 )
	ROM_LOAD32_BYTE( "ru-02b",   0x000001, 0x20000, 0xeff33945 )
	ROM_LOAD32_BYTE( "ru-04b",   0x000002, 0x20000, 0x40fda247 )
	ROM_LOAD32_BYTE( "ru-06b",   0x000003, 0x20000, 0xcd2484f3 )
	ROM_LOAD32_BYTE( "ru-01b",   0x080000, 0x10000, 0x5e91f401 )
	ROM_LOAD32_BYTE( "ru-03b",   0x080001, 0x10000, 0x2fb109de )
	ROM_LOAD32_BYTE( "ru-05b",   0x080002, 0x10000, 0x23dd10e1 )
	ROM_LOAD32_BYTE( "ru-07b",   0x080003, 0x10000, 0xbb40f46e )

	ROM_REGION( 0x0c0000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD32_BYTE( "ru-08b",   0x000000, 0x20000, 0x01729e3c )
	ROM_LOAD32_BYTE( "ru-10b",   0x000001, 0x20000, 0xcd2357a7 )
	ROM_LOAD32_BYTE( "ru-12b",   0x000002, 0x20000, 0xca63b1f8 )
	ROM_LOAD32_BYTE( "ru-14b",   0x000003, 0x20000, 0xdde79bfc )
	ROM_LOAD32_BYTE( "ru-09b",   0x080000, 0x10000, 0x629ac0a6 )
	ROM_LOAD32_BYTE( "ru-11b",   0x080001, 0x10000, 0xfe3fbf53 )
	ROM_LOAD32_BYTE( "ru-13b",   0x080002, 0x10000, 0xd0f6e747 )
	ROM_LOAD32_BYTE( "ru-15b",   0x080003, 0x10000, 0x6ee6b22e )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	 /* ADPCM samples */
	ROM_LOAD( "ru-17b",   0x000000, 0x20000, 0x82391b47 )
	ROM_LOAD( "ru-16e",   0x020000, 0x10000, 0x374fe170 )
ROM_END

ROM_START( cyclwarr )
	ROM_REGION( 0x180000, REGION_CPU1, 0 ) /* 68000 main cpu */
	ROM_LOAD16_BYTE( "cw16b",   0x100000, 0x20000, 0xcb1a737a )
	ROM_LOAD16_BYTE( "cw18b",   0x100001, 0x20000, 0x0633ddcb )
	ROM_LOAD16_BYTE( "cw17a",   0x140000, 0x20000, 0x2ad6f836 )
	ROM_LOAD16_BYTE( "cw19a",   0x140001, 0x20000, 0xd3853658 )

	ROM_REGION( 0x180000, REGION_CPU2, 0 ) /* 68000 sub cpu */
	ROM_LOAD16_BYTE( "cw20a",   0x100000, 0x20000, 0xc3578ac1 )
	ROM_LOAD16_BYTE( "cw22a",   0x100001, 0x20000, 0x5339ed24 )
	ROM_LOAD16_BYTE( "cw21",    0x140000, 0x20000, 0xed90d956 )
	ROM_LOAD16_BYTE( "cw23",    0x140001, 0x20000, 0x009cdc78 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 ) /* 64k code for sound Z80 */
	ROM_LOAD( "cw26a",   0x000000, 0x10000, 0xf7a70e3a )

	ROM_REGION( 0x200000, REGION_GFX1, ROMREGION_DISPOSE )
	/* Filled in by both regions below */

	ROM_REGION( 0x100000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD32_BYTE( "cw00a",   0x000000, 0x20000, 0x058a77f1 )
	ROM_LOAD32_BYTE( "cw08a",   0x000001, 0x20000, 0xf53993e7 )
	ROM_LOAD32_BYTE( "cw02a",   0x000002, 0x20000, 0x4dadf3cb )
	ROM_LOAD32_BYTE( "cw10a",   0x000003, 0x20000, 0x3b7cd251 )
	ROM_LOAD32_BYTE( "cw01a",   0x080000, 0x20000, 0x7c639948 )
	ROM_LOAD32_BYTE( "cw09a",   0x080001, 0x20000, 0x4ba24af5 )
	ROM_LOAD32_BYTE( "cw03a",   0x080002, 0x20000, 0x3ca6f98e )
	ROM_LOAD32_BYTE( "cw11a",   0x080003, 0x20000, 0x5d760392 )

	ROM_REGION( 0x100000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD32_BYTE( "cw04a",   0x000000, 0x20000, 0xf05f594d )
	ROM_LOAD32_BYTE( "cw12a",   0x000001, 0x20000, 0x4ac07e8b )
	ROM_LOAD32_BYTE( "cw06a",   0x000002, 0x20000, 0xf628edc9 )
	ROM_LOAD32_BYTE( "cw14a",   0x000003, 0x20000, 0xa9131f5f	)
	ROM_LOAD32_BYTE( "cw05a",   0x080000, 0x20000, 0xc8f5faa9 )
	ROM_LOAD32_BYTE( "cw13a",   0x080001, 0x20000, 0x8091d381 )
	ROM_LOAD32_BYTE( "cw07a",   0x080002, 0x20000, 0x314579b5 )
	ROM_LOAD32_BYTE( "cw15a",   0x080003, 0x20000, 0x7ed4b721 )

	ROM_REGION( 0x20000, REGION_GFX4, 0 )
	ROM_LOAD( "cw27",   0x000000, 0x20000, 0x2db48a9e )

	ROM_REGION( 0x60000, REGION_GFX5, ROMREGION_DISPOSE )
	ROM_LOAD( "cw28",   0x000000, 0x20000, 0x3fc568ed )
	ROM_LOAD( "cw29",   0x020000, 0x20000, 0x64dd519c )
	ROM_LOAD( "cw30",   0x040000, 0x20000, 0x331d0711 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	 /* ADPCM samples */
	ROM_LOAD( "cw24a",   0x000000, 0x20000, 0x22600cba )
	ROM_LOAD( "cw25a",   0x020000, 0x20000, 0x372c6bc8 )
ROM_END

/***************************************************************************/

static DRIVER_INIT( apache3 )
{
	UINT8 *dst = memory_region(REGION_GFX1);
	UINT8 *src1 = memory_region(REGION_GFX2);
	UINT8 *src2 = memory_region(REGION_GFX3);
	int i;

	cpu_set_irq_line(3, 34, ASSERT_LINE); // ?

	for (i=0; i<0x100000; i+=32) {
		memcpy(dst,src1,32);
		src1+=32;
		dst+=32;
		memcpy(dst,src2,32);
		dst+=32;
		src2+=32;
	}

	// Copy sprite & palette data out of GFX rom area
	tatsumi_rom_sprite_lookup1=auto_malloc(0x4000);
	tatsumi_rom_sprite_lookup2=auto_malloc(0x4000);
	tatsumi_rom_clut0=auto_malloc(0x1000);
	tatsumi_rom_clut1=auto_malloc(0x1000);
	memcpy(tatsumi_rom_sprite_lookup1, memory_region(REGION_GFX2),0x4000);
	memcpy(tatsumi_rom_sprite_lookup2, memory_region(REGION_GFX3),0x4000);
	memcpy(tatsumi_rom_clut0, memory_region(REGION_GFX2)+0x100000-0x800,0x800);
	memcpy(tatsumi_rom_clut1, memory_region(REGION_GFX3)+0x100000-0x800,0x800);

	tatsumi_reset();
}

static DRIVER_INIT( roundup5 )
{
	UINT8 *dst = memory_region(REGION_GFX1);
	UINT8 *src1 = memory_region(REGION_GFX2);
	UINT8 *src2 = memory_region(REGION_GFX3);
	int i;

	for (i=0; i<0xc0000; i+=32) {
		memcpy(dst,src1,32);
		src1+=32;
		dst+=32;
		memcpy(dst,src2,32);
		dst+=32;
		src2+=32;
	}

	// Copy sprite & palette data out of GFX rom area
	tatsumi_rom_sprite_lookup1=auto_malloc(0x4000);
	tatsumi_rom_sprite_lookup2=auto_malloc(0x4000);
	tatsumi_rom_clut0=auto_malloc(0x800);
	tatsumi_rom_clut1=auto_malloc(0x800);
	memcpy(tatsumi_rom_sprite_lookup1, memory_region(REGION_GFX2),0x4000);
	memcpy(tatsumi_rom_sprite_lookup2, memory_region(REGION_GFX3),0x4000);
	memcpy(tatsumi_rom_clut0, memory_region(REGION_GFX2)+0xc0000-0x800,0x800);
	memcpy(tatsumi_rom_clut1, memory_region(REGION_GFX3)+0xc0000-0x800,0x800);

	tatsumi_reset();
}

static DRIVER_INIT( cyclwarr )
{
	UINT8 *dst = memory_region(REGION_GFX1);
	UINT8 *src1 = memory_region(REGION_GFX2);
	UINT8 *src2 = memory_region(REGION_GFX3);
	int i;
	for (i=0; i<0x100000; i+=32) {
		memcpy(dst,src1,32);
		src1+=32;
		dst+=32;
		memcpy(dst,src2,32);
		dst+=32;
		src2+=32;
	}

	dst = memory_region(REGION_CPU1);
	memcpy(dst,dst+0x100000,8);
	cpu_setbank(1, memory_region(REGION_CPU1) + 0x100000);

	dst = memory_region(REGION_CPU2);
	memcpy(dst,dst+0x100000,8);
	cpu_setbank(2, memory_region(REGION_CPU2) + 0x100000);

	// Copy sprite & palette data out of GFX rom area
	tatsumi_rom_sprite_lookup1=auto_malloc(0x4000);
	tatsumi_rom_sprite_lookup2=auto_malloc(0x4000);
	tatsumi_rom_clut0=auto_malloc(0x1000);
	tatsumi_rom_clut1=auto_malloc(0x1000);
	memcpy(tatsumi_rom_sprite_lookup1, memory_region(REGION_GFX2),0x4000);
	memcpy(tatsumi_rom_sprite_lookup2, memory_region(REGION_GFX3),0x4000);
	memcpy(tatsumi_rom_clut0, memory_region(REGION_GFX2)+0x100000-0x1000,0x1000);
	memcpy(tatsumi_rom_clut1, memory_region(REGION_GFX3)+0x100000-0x1000,0x1000);

	tatsumi_reset();
}

/***************************************************************************/

/* http://www.tatsu-mi.co.jp/game/HTML/history.html */

/* 1986 Lock On */
/* 1987 Gray Out */
GAMEX( 1988, apache3,  0, apache3,   apache3,  apache3,  ROT0, "Tatsumi", "Apache 3", GAME_IMPERFECT_GRAPHICS )
GAMEX( 1989, roundup5, 0, roundup5,  roundup5, roundup5, ROT0, "Tatsumi", "Round Up 5 - Super Delta Force", GAME_IMPERFECT_GRAPHICS )
GAMEX( 1991, cyclwarr, 0, cyclwarr,  cyclwarr, cyclwarr, ROT0, "Tatsumi", "Cycle Warriors", GAME_IMPERFECT_GRAPHICS )
/* 1992 Big Fight */
