/***************************************************************************

	Atari Skull & Crossbones hardware

	driver by Aaron Giles

	Games supported:
		* Skull & Crossbones (1989) [5 sets]

	Known bugs:
		* none at this time

****************************************************************************

	Memory map (TBA)

***************************************************************************/


#include "driver.h"
#include "machine/atarigen.h"
#include "sndhrdw/atarijsa.h"
#include "skullxbo.h"



/*************************************
 *
 *	Initialization & interrupts
 *
 *************************************/

static void update_interrupts(void)
{
	int newstate = 0;

	if (atarigen_scanline_int_state)
		newstate = 1;
	if (atarigen_video_int_state)
		newstate = 2;
	if (atarigen_sound_int_state)
		newstate = 4;

	if (newstate)
		cpu_set_irq_line(0, newstate, ASSERT_LINE);
	else
		cpu_set_irq_line(0, 7, CLEAR_LINE);
}


static void irq_gen(int param)
{
	(void)param;
	atarigen_scanline_int_gen();
}


static void alpha_row_update(int scanline)
{
	data16_t *check = &atarigen_alpha[(scanline / 8) * 64 + 42];

	/* check for interrupts in the alpha ram */
	/* the interrupt occurs on the HBLANK of the 6th scanline following */
	if (check < &atarigen_alpha[0x7c0] && (*check & 0x8000))
		timer_set(cpu_getscanlineperiod() * 6.9, 0, irq_gen);

	/* update the playfield and motion objects */
	skullxbo_scanline_update(scanline);
}


static MACHINE_INIT( skullxbo )
{
	atarigen_eeprom_reset();
	atarigen_interrupt_reset(update_interrupts);
	atarigen_scanline_timer_reset(alpha_row_update, 8);
	atarijsa_reset();
}



/*************************************
 *
 *	I/O read dispatch.
 *
 *************************************/

static READ16_HANDLER( special_port1_r )
{
	int temp = readinputport(1);
	if (atarigen_cpu_to_sound_ready) temp ^= 0x0040;
	if (atarigen_get_hblank()) temp ^= 0x0010;
	return temp;
}



/*************************************
 *
 *	Who knows what this is?
 *
 *************************************/

static WRITE16_HANDLER( skullxbo_mobwr_w )
{
	logerror("MOBWR[%02X] = %04X\n", offset, data);
}



/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

static MEMORY_READ16_START( main_readmem )
	{ 0x000000, 0x07ffff, MRA16_ROM },
	{ 0xff2000, 0xff2fff, MRA16_RAM },
	{ 0xff5000, 0xff5001, atarigen_sound_r },
	{ 0xff5800, 0xff5801, input_port_0_word_r },
	{ 0xff5802, 0xff5803, special_port1_r },
	{ 0xff6000, 0xff6fff, atarigen_eeprom_r },
	{ 0xff8000, 0xffffff, MRA16_RAM },
MEMORY_END


static MEMORY_WRITE16_START( main_writemem )
	{ 0x000000, 0x07ffff, MWA16_ROM },
	{ 0xff0000, 0xff07ff, skullxbo_mobmsb_w },
	{ 0xff0800, 0xff0bff, atarigen_halt_until_hblank_0_w },
	{ 0xff0c00, 0xff0fff, atarigen_eeprom_enable_w },
	{ 0xff1000, 0xff13ff, atarigen_video_int_ack_w },
	{ 0xff1400, 0xff17ff, atarigen_sound_w },
	{ 0xff1800, 0xff1bff, atarigen_sound_reset_w },
	{ 0xff1c00, 0xff1c7f, skullxbo_playfieldlatch_w },
	{ 0xff1c80, 0xff1cff, skullxbo_xscroll_w, &atarigen_xscroll },
	{ 0xff1d00, 0xff1d7f, atarigen_scanline_int_ack_w },
	{ 0xff1d80, 0xff1dff, watchdog_reset16_w },
	{ 0xff1e00, 0xff1e7f, skullxbo_playfieldlatch_w },
	{ 0xff1e80, 0xff1eff, skullxbo_xscroll_w },
	{ 0xff1f00, 0xff1f7f, atarigen_scanline_int_ack_w },
	{ 0xff1f80, 0xff1fff, watchdog_reset16_w },
	{ 0xff2000, 0xff2fff, atarigen_666_paletteram_w, &paletteram16 },
	{ 0xff4000, 0xff47ff, skullxbo_yscroll_w, &atarigen_yscroll },
	{ 0xff4800, 0xff4fff, skullxbo_mobwr_w },
	{ 0xff6000, 0xff6fff, atarigen_eeprom_w, &atarigen_eeprom, &atarigen_eeprom_size },
	{ 0xff8000, 0xff9fff, atarigen_playfield_latched_lsb_w, &atarigen_playfield },
	{ 0xffa000, 0xffbfff, atarigen_playfield_upper_w, &atarigen_playfield_upper },
	{ 0xffc000, 0xffcf7f, atarigen_alpha_w, &atarigen_alpha },
	{ 0xffcf80, 0xffcfff, atarimo_0_slipram_w, &atarimo_0_slipram },
	{ 0xffd000, 0xffdfff, atarimo_0_spriteram_w, &atarimo_0_spriteram },
	{ 0xffe000, 0xffffff, MWA16_RAM },
MEMORY_END



/*************************************
 *
 *	Port definitions
 *
 *************************************/

INPUT_PORTS_START( skullxbo )
	PORT_START      /* ff5800 */
	PORT_BIT( 0x00ff, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER1 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER1 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER1 )

	PORT_START      /* ff5802 */
	PORT_BIT( 0x000f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, IPT_UNUSED )	/* HBLANK */
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, IPT_VBLANK )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_UNUSED )	/* /AUDBUSY */
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_PLAYER2 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_PLAYER2 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_PLAYER2 )

	JSA_II_PORT		/* audio board port */
INPUT_PORTS_END



/*************************************
 *
 *	Graphics definitions
 *
 *************************************/

static struct GfxLayout pflayout =
{
	16,8,
	RGN_FRAC(1,2),
	4,
	{ 0, 1, 2, 3 },
	{ RGN_FRAC(1,2)+0,RGN_FRAC(1,2)+0, RGN_FRAC(1,2)+4,RGN_FRAC(1,2)+4, 0,0, 4,4,
	  RGN_FRAC(1,2)+8,RGN_FRAC(1,2)+8, RGN_FRAC(1,2)+12,RGN_FRAC(1,2)+12, 8,8, 12,12 },
	{ 0*8, 2*8, 4*8, 6*8, 8*8, 10*8, 12*8, 14*8 },
	16*8
};


static struct GfxLayout anlayout =
{
	16,8,
	RGN_FRAC(1,1),
	2,
	{ 0, 1 },
	{ 0,0, 2,2, 4,4, 6,6, 8,8, 10,10, 12,12, 14,14 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16 },
	8*16
};


static struct GfxLayout molayout =
{
	16,8,
	RGN_FRAC(1,5),
	5,
	{ RGN_FRAC(4,5), RGN_FRAC(3,5), RGN_FRAC(2,5), RGN_FRAC(1,5), RGN_FRAC(0,5) },
	{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	{ 0*8, 2*8, 4*8, 6*8, 8*8, 10*8, 12*8, 14*8 },
	16*8
};


static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &molayout, 0x000, 32 },
	{ REGION_GFX2, 0, &pflayout, 0x200, 16 },
	{ REGION_GFX3, 0, &anlayout, 0x300, 16 },
	{ -1 }
};



/*************************************
 *
 *	Machine driver
 *
 *************************************/

static MACHINE_DRIVER_START( skullxbo )

	/* basic machine hardware */
	MDRV_CPU_ADD(M68000, ATARI_CLOCK_14MHz/2)
	MDRV_CPU_MEMORY(main_readmem,main_writemem)
	MDRV_CPU_VBLANK_INT(atarigen_video_int_gen,1)
	
	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)
	
	MDRV_MACHINE_INIT(skullxbo)
	MDRV_NVRAM_HANDLER(atarigen)
	
	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_UPDATE_BEFORE_VBLANK | VIDEO_PIXEL_ASPECT_RATIO_1_2)
	MDRV_SCREEN_SIZE(42*16, 30*8)
	MDRV_VISIBLE_AREA(0*8, 42*16-1, 0*8, 30*8-1)
	MDRV_GFXDECODE(gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(2048)
	
	MDRV_VIDEO_START(skullxbo)
	MDRV_VIDEO_UPDATE(skullxbo)
	
	/* sound hardware */
	MDRV_IMPORT_FROM(jsa_ii_mono)
MACHINE_DRIVER_END



/*************************************
 *
 *	ROM definition(s)
 *
 *************************************/

ROM_START( skullxbo )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "5150228a.072", 0x000000, 0x010000, 0x9546d88b )
	ROM_LOAD16_BYTE( "5151228c.072", 0x000001, 0x010000, 0xb9ed8bd4 )
	ROM_LOAD16_BYTE( "5152213a.072", 0x020000, 0x010000, 0xc07e44fc )
	ROM_LOAD16_BYTE( "5153213c.072", 0x020001, 0x010000, 0xfef8297f )
	ROM_LOAD16_BYTE( "1154200a.072", 0x040000, 0x010000, 0xde4101a3 )
	ROM_LOAD16_BYTE( "1155200c.072", 0x040001, 0x010000, 0x78c0f6ad )
	ROM_LOAD16_BYTE( "1156185a.072", 0x070000, 0x008000, 0xcde16b55 )
	ROM_LOAD16_BYTE( "1157185c.072", 0x070001, 0x008000, 0x31c77376 )

	ROM_REGION( 0x14000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1149-1b.072",  0x010000, 0x004000, 0x8d730e7a )
	ROM_CONTINUE(             0x004000, 0x00c000 )

	ROM_REGION( 0x190000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1102-13r.072", 0x000000, 0x010000, 0x90becdfa )
	ROM_LOAD( "1104-28r.072", 0x010000, 0x010000, 0x33609071 )
	ROM_LOAD( "1106-41r.072", 0x020000, 0x010000, 0x71962e9f )
	ROM_LOAD( "1101-13p.072", 0x030000, 0x010000, 0x4d41701e )
	ROM_LOAD( "1103-28p.072", 0x040000, 0x010000, 0x3011da3b )
	ROM_LOAD( "1108-53r.072", 0x050000, 0x010000, 0x386c7edc )
	ROM_LOAD( "1110-67r.072", 0x060000, 0x010000, 0xa54d16e6 )
	ROM_LOAD( "1112-81r.072", 0x070000, 0x010000, 0x669411f6 )
	ROM_LOAD( "1107-53p.072", 0x080000, 0x010000, 0xcaaeb57a )
	ROM_LOAD( "1109-67p.072", 0x090000, 0x010000, 0x61cb4e28 )
	ROM_LOAD( "1114-95r.072", 0x0a0000, 0x010000, 0xe340d5a1 )
	ROM_LOAD( "1116109r.072", 0x0b0000, 0x010000, 0xf25b8aca )
	ROM_LOAD( "1118123r.072", 0x0c0000, 0x010000, 0x8cf73585 )
	ROM_LOAD( "1113-95p.072", 0x0d0000, 0x010000, 0x899b59af )
	ROM_LOAD( "1115109p.072", 0x0e0000, 0x010000, 0xcf4fd19a )
	ROM_LOAD( "1120137r.072", 0x0f0000, 0x010000, 0xfde7c03d )
	ROM_LOAD( "1122151r.072", 0x100000, 0x010000, 0x6ff6a9f2 )
	ROM_LOAD( "1124165r.072", 0x110000, 0x010000, 0xf11909f1 )
	ROM_LOAD( "1119137p.072", 0x120000, 0x010000, 0x6f8003a1 )
	ROM_LOAD( "1121151p.072", 0x130000, 0x010000, 0x8ff0a1ec )
	ROM_LOAD( "1125123n.072", 0x140000, 0x010000, 0x3aa7c756 )
	ROM_LOAD( "1126137n.072", 0x150000, 0x010000, 0xcb82c9aa )
	ROM_LOAD( "1128151n.072", 0x160000, 0x010000, 0xdce32863 )
	/* 170000-18ffff empty */

	ROM_REGION( 0x0a0000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "2129180p.072", 0x000000, 0x010000, 0x36b1a578 )
	ROM_LOAD( "2131193p.072", 0x010000, 0x010000, 0x7b7c04a1 )
	ROM_LOAD( "2133208p.072", 0x020000, 0x010000, 0xe03fe4d9 )
	ROM_LOAD( "2135221p.072", 0x030000, 0x010000, 0x7d497110 )
	ROM_LOAD( "2137235p.072", 0x040000, 0x010000, 0xf91e7872 )
	ROM_LOAD( "2130180r.072", 0x050000, 0x010000, 0xb25368cc )
	ROM_LOAD( "2132193r.072", 0x060000, 0x010000, 0x112f2d20 )
	ROM_LOAD( "2134208r.072", 0x070000, 0x010000, 0x84884ed6 )
	ROM_LOAD( "2136221r.072", 0x080000, 0x010000, 0xbc028690 )
	ROM_LOAD( "2138235r.072", 0x090000, 0x010000, 0x60cec955 )

	ROM_REGION( 0x008000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "2141250k.072", 0x000000, 0x008000, 0x60d6d6df )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* 256k for ADPCM samples */
	ROM_LOAD( "1145-7k.072",  0x000000, 0x010000, 0xd9475d58 )
	ROM_LOAD( "1146-7j.072",  0x010000, 0x010000, 0x133e6aef )
	ROM_LOAD( "1147-7e.072",  0x020000, 0x010000, 0xba4d556e )
	ROM_LOAD( "1148-7d.072",  0x030000, 0x010000, 0xc48df49a )
ROM_END


ROM_START( skullxb4 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "4150228a.072", 0x000000, 0x010000, 0x607fc73b )
	ROM_LOAD16_BYTE( "4151228c.072", 0x000001, 0x010000, 0x76bbf619 )
	ROM_LOAD16_BYTE( "4152213a.072", 0x020000, 0x010000, 0x095206f5 )
	ROM_LOAD16_BYTE( "4153213c.072", 0x020001, 0x010000, 0xe44be9d3 )
	ROM_LOAD16_BYTE( "1154200a.072", 0x040000, 0x010000, 0xde4101a3 )
	ROM_LOAD16_BYTE( "1155200c.072", 0x040001, 0x010000, 0x78c0f6ad )
	ROM_LOAD16_BYTE( "1156185a.072", 0x070000, 0x008000, 0xcde16b55 )
	ROM_LOAD16_BYTE( "1157185c.072", 0x070001, 0x008000, 0x31c77376 )

	ROM_REGION( 0x14000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1149-1b.072",  0x010000, 0x004000, 0x8d730e7a )
	ROM_CONTINUE(             0x004000, 0x00c000 )

	ROM_REGION( 0x190000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1102-13r.072", 0x000000, 0x010000, 0x90becdfa )
	ROM_LOAD( "1104-28r.072", 0x010000, 0x010000, 0x33609071 )
	ROM_LOAD( "1106-41r.072", 0x020000, 0x010000, 0x71962e9f )
	ROM_LOAD( "1101-13p.072", 0x030000, 0x010000, 0x4d41701e )
	ROM_LOAD( "1103-28p.072", 0x040000, 0x010000, 0x3011da3b )
	ROM_LOAD( "1108-53r.072", 0x050000, 0x010000, 0x386c7edc )
	ROM_LOAD( "1110-67r.072", 0x060000, 0x010000, 0xa54d16e6 )
	ROM_LOAD( "1112-81r.072", 0x070000, 0x010000, 0x669411f6 )
	ROM_LOAD( "1107-53p.072", 0x080000, 0x010000, 0xcaaeb57a )
	ROM_LOAD( "1109-67p.072", 0x090000, 0x010000, 0x61cb4e28 )
	ROM_LOAD( "1114-95r.072", 0x0a0000, 0x010000, 0xe340d5a1 )
	ROM_LOAD( "1116109r.072", 0x0b0000, 0x010000, 0xf25b8aca )
	ROM_LOAD( "1118123r.072", 0x0c0000, 0x010000, 0x8cf73585 )
	ROM_LOAD( "1113-95p.072", 0x0d0000, 0x010000, 0x899b59af )
	ROM_LOAD( "1115109p.072", 0x0e0000, 0x010000, 0xcf4fd19a )
	ROM_LOAD( "1120137r.072", 0x0f0000, 0x010000, 0xfde7c03d )
	ROM_LOAD( "1122151r.072", 0x100000, 0x010000, 0x6ff6a9f2 )
	ROM_LOAD( "1124165r.072", 0x110000, 0x010000, 0xf11909f1 )
	ROM_LOAD( "1119137p.072", 0x120000, 0x010000, 0x6f8003a1 )
	ROM_LOAD( "1121151p.072", 0x130000, 0x010000, 0x8ff0a1ec )
	ROM_LOAD( "1125123n.072", 0x140000, 0x010000, 0x3aa7c756 )
	ROM_LOAD( "1126137n.072", 0x150000, 0x010000, 0xcb82c9aa )
	ROM_LOAD( "1128151n.072", 0x160000, 0x010000, 0xdce32863 )
	/* 170000-18ffff empty */

	ROM_REGION( 0x0a0000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "2129180p.072", 0x000000, 0x010000, 0x36b1a578 )
	ROM_LOAD( "2131193p.072", 0x010000, 0x010000, 0x7b7c04a1 )
	ROM_LOAD( "2133208p.072", 0x020000, 0x010000, 0xe03fe4d9 )
	ROM_LOAD( "2135221p.072", 0x030000, 0x010000, 0x7d497110 )
	ROM_LOAD( "2137235p.072", 0x040000, 0x010000, 0xf91e7872 )
	ROM_LOAD( "2130180r.072", 0x050000, 0x010000, 0xb25368cc )
	ROM_LOAD( "2132193r.072", 0x060000, 0x010000, 0x112f2d20 )
	ROM_LOAD( "2134208r.072", 0x070000, 0x010000, 0x84884ed6 )
	ROM_LOAD( "2136221r.072", 0x080000, 0x010000, 0xbc028690 )
	ROM_LOAD( "2138235r.072", 0x090000, 0x010000, 0x60cec955 )

	ROM_REGION( 0x008000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "2141250k.072", 0x000000, 0x008000, 0x60d6d6df )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* 256k for ADPCM samples */
	ROM_LOAD( "1145-7k.072",  0x000000, 0x010000, 0xd9475d58 )
	ROM_LOAD( "1146-7j.072",  0x010000, 0x010000, 0x133e6aef )
	ROM_LOAD( "1147-7e.072",  0x020000, 0x010000, 0xba4d556e )
	ROM_LOAD( "1148-7d.072",  0x030000, 0x010000, 0xc48df49a )
ROM_END


ROM_START( skullxb3 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "3150228a.072", 0x000000, 0x010000, 0x47083d59 )
	ROM_LOAD16_BYTE( "3151228c.072", 0x000001, 0x010000, 0x2c03feaf )
	ROM_LOAD16_BYTE( "3152213a.072", 0x020000, 0x010000, 0xaa0471de )
	ROM_LOAD16_BYTE( "3153213c.072", 0x020001, 0x010000, 0xa65386f9 )
	ROM_LOAD16_BYTE( "1154200a.072", 0x040000, 0x010000, 0xde4101a3 )
	ROM_LOAD16_BYTE( "1155200c.072", 0x040001, 0x010000, 0x78c0f6ad )
	ROM_LOAD16_BYTE( "1156185a.072", 0x070000, 0x008000, 0xcde16b55 )
	ROM_LOAD16_BYTE( "1157185c.072", 0x070001, 0x008000, 0x31c77376 )

	ROM_REGION( 0x14000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1149-1b.072",  0x010000, 0x004000, 0x8d730e7a )
	ROM_CONTINUE(             0x004000, 0x00c000 )

	ROM_REGION( 0x190000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1102-13r.072", 0x000000, 0x010000, 0x90becdfa )
	ROM_LOAD( "1104-28r.072", 0x010000, 0x010000, 0x33609071 )
	ROM_LOAD( "1106-41r.072", 0x020000, 0x010000, 0x71962e9f )
	ROM_LOAD( "1101-13p.072", 0x030000, 0x010000, 0x4d41701e )
	ROM_LOAD( "1103-28p.072", 0x040000, 0x010000, 0x3011da3b )
	ROM_LOAD( "1108-53r.072", 0x050000, 0x010000, 0x386c7edc )
	ROM_LOAD( "1110-67r.072", 0x060000, 0x010000, 0xa54d16e6 )
	ROM_LOAD( "1112-81r.072", 0x070000, 0x010000, 0x669411f6 )
	ROM_LOAD( "1107-53p.072", 0x080000, 0x010000, 0xcaaeb57a )
	ROM_LOAD( "1109-67p.072", 0x090000, 0x010000, 0x61cb4e28 )
	ROM_LOAD( "1114-95r.072", 0x0a0000, 0x010000, 0xe340d5a1 )
	ROM_LOAD( "1116109r.072", 0x0b0000, 0x010000, 0xf25b8aca )
	ROM_LOAD( "1118123r.072", 0x0c0000, 0x010000, 0x8cf73585 )
	ROM_LOAD( "1113-95p.072", 0x0d0000, 0x010000, 0x899b59af )
	ROM_LOAD( "1115109p.072", 0x0e0000, 0x010000, 0xcf4fd19a )
	ROM_LOAD( "1120137r.072", 0x0f0000, 0x010000, 0xfde7c03d )
	ROM_LOAD( "1122151r.072", 0x100000, 0x010000, 0x6ff6a9f2 )
	ROM_LOAD( "1124165r.072", 0x110000, 0x010000, 0xf11909f1 )
	ROM_LOAD( "1119137p.072", 0x120000, 0x010000, 0x6f8003a1 )
	ROM_LOAD( "1121151p.072", 0x130000, 0x010000, 0x8ff0a1ec )
	ROM_LOAD( "1125123n.072", 0x140000, 0x010000, 0x3aa7c756 )
	ROM_LOAD( "1126137n.072", 0x150000, 0x010000, 0xcb82c9aa )
	ROM_LOAD( "1128151n.072", 0x160000, 0x010000, 0xdce32863 )
	/* 170000-18ffff empty */

	ROM_REGION( 0x0a0000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "2129180p.072", 0x000000, 0x010000, 0x36b1a578 )
	ROM_LOAD( "2131193p.072", 0x010000, 0x010000, 0x7b7c04a1 )
	ROM_LOAD( "2133208p.072", 0x020000, 0x010000, 0xe03fe4d9 )
	ROM_LOAD( "2135221p.072", 0x030000, 0x010000, 0x7d497110 )
	ROM_LOAD( "2137235p.072", 0x040000, 0x010000, 0xf91e7872 )
	ROM_LOAD( "2130180r.072", 0x050000, 0x010000, 0xb25368cc )
	ROM_LOAD( "2132193r.072", 0x060000, 0x010000, 0x112f2d20 )
	ROM_LOAD( "2134208r.072", 0x070000, 0x010000, 0x84884ed6 )
	ROM_LOAD( "2136221r.072", 0x080000, 0x010000, 0xbc028690 )
	ROM_LOAD( "2138235r.072", 0x090000, 0x010000, 0x60cec955 )

	ROM_REGION( 0x008000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "2141250k.072", 0x000000, 0x008000, 0x60d6d6df )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* 256k for ADPCM samples */
	ROM_LOAD( "1145-7k.072",  0x000000, 0x010000, 0xd9475d58 )
	ROM_LOAD( "1146-7j.072",  0x010000, 0x010000, 0x133e6aef )
	ROM_LOAD( "1147-7e.072",  0x020000, 0x010000, 0xba4d556e )
	ROM_LOAD( "1148-7d.072",  0x030000, 0x010000, 0xc48df49a )
ROM_END


ROM_START( skullxb2 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "2150228a.072", 0x000000, 0x010000, 0x8614f9ef )
	ROM_LOAD16_BYTE( "2151228c.072", 0x000001, 0x010000, 0x47090acb )
	ROM_LOAD16_BYTE( "2152213a.072", 0x020000, 0x010000, 0x02f6a944 )
	ROM_LOAD16_BYTE( "2153213c.072", 0x020001, 0x010000, 0x589ce449 )
	ROM_LOAD16_BYTE( "1154200a.072", 0x040000, 0x010000, 0xde4101a3 )
	ROM_LOAD16_BYTE( "1155200c.072", 0x040001, 0x010000, 0x78c0f6ad )
	ROM_LOAD16_BYTE( "1156185a.072", 0x070000, 0x008000, 0xcde16b55 )
	ROM_LOAD16_BYTE( "1157185c.072", 0x070001, 0x008000, 0x31c77376 )

	ROM_REGION( 0x14000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1149-1b.072",  0x010000, 0x004000, 0x8d730e7a )
	ROM_CONTINUE(             0x004000, 0x00c000 )

	ROM_REGION( 0x190000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1102-13r.072", 0x000000, 0x010000, 0x90becdfa )
	ROM_LOAD( "1104-28r.072", 0x010000, 0x010000, 0x33609071 )
	ROM_LOAD( "1106-41r.072", 0x020000, 0x010000, 0x71962e9f )
	ROM_LOAD( "1101-13p.072", 0x030000, 0x010000, 0x4d41701e )
	ROM_LOAD( "1103-28p.072", 0x040000, 0x010000, 0x3011da3b )
	ROM_LOAD( "1108-53r.072", 0x050000, 0x010000, 0x386c7edc )
	ROM_LOAD( "1110-67r.072", 0x060000, 0x010000, 0xa54d16e6 )
	ROM_LOAD( "1112-81r.072", 0x070000, 0x010000, 0x669411f6 )
	ROM_LOAD( "1107-53p.072", 0x080000, 0x010000, 0xcaaeb57a )
	ROM_LOAD( "1109-67p.072", 0x090000, 0x010000, 0x61cb4e28 )
	ROM_LOAD( "1114-95r.072", 0x0a0000, 0x010000, 0xe340d5a1 )
	ROM_LOAD( "1116109r.072", 0x0b0000, 0x010000, 0xf25b8aca )
	ROM_LOAD( "1118123r.072", 0x0c0000, 0x010000, 0x8cf73585 )
	ROM_LOAD( "1113-95p.072", 0x0d0000, 0x010000, 0x899b59af )
	ROM_LOAD( "1115109p.072", 0x0e0000, 0x010000, 0xcf4fd19a )
	ROM_LOAD( "1120137r.072", 0x0f0000, 0x010000, 0xfde7c03d )
	ROM_LOAD( "1122151r.072", 0x100000, 0x010000, 0x6ff6a9f2 )
	ROM_LOAD( "1124165r.072", 0x110000, 0x010000, 0xf11909f1 )
	ROM_LOAD( "1119137p.072", 0x120000, 0x010000, 0x6f8003a1 )
	ROM_LOAD( "1121151p.072", 0x130000, 0x010000, 0x8ff0a1ec )
	ROM_LOAD( "1125123n.072", 0x140000, 0x010000, 0x3aa7c756 )
	ROM_LOAD( "1126137n.072", 0x150000, 0x010000, 0xcb82c9aa )
	ROM_LOAD( "1128151n.072", 0x160000, 0x010000, 0xdce32863 )
	/* 170000-18ffff empty */

	ROM_REGION( 0x0a0000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "2129180p.072", 0x000000, 0x010000, 0x36b1a578 )
	ROM_LOAD( "2131193p.072", 0x010000, 0x010000, 0x7b7c04a1 )
	ROM_LOAD( "2133208p.072", 0x020000, 0x010000, 0xe03fe4d9 )
	ROM_LOAD( "2135221p.072", 0x030000, 0x010000, 0x7d497110 )
	ROM_LOAD( "2137235p.072", 0x040000, 0x010000, 0xf91e7872 )
	ROM_LOAD( "2130180r.072", 0x050000, 0x010000, 0xb25368cc )
	ROM_LOAD( "2132193r.072", 0x060000, 0x010000, 0x112f2d20 )
	ROM_LOAD( "2134208r.072", 0x070000, 0x010000, 0x84884ed6 )
	ROM_LOAD( "2136221r.072", 0x080000, 0x010000, 0xbc028690 )
	ROM_LOAD( "2138235r.072", 0x090000, 0x010000, 0x60cec955 )

	ROM_REGION( 0x008000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "2141250k.072", 0x000000, 0x008000, 0x60d6d6df )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* 256k for ADPCM samples */
	ROM_LOAD( "1145-7k.072",  0x000000, 0x010000, 0xd9475d58 )
	ROM_LOAD( "1146-7j.072",  0x010000, 0x010000, 0x133e6aef )
	ROM_LOAD( "1147-7e.072",  0x020000, 0x010000, 0xba4d556e )
	ROM_LOAD( "1148-7d.072",  0x030000, 0x010000, 0xc48df49a )
ROM_END


ROM_START( skullxb1 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1150228a.072", 0x000000, 0x010000, 0x376bb0c7 )
	ROM_LOAD16_BYTE( "1151228c.072", 0x000001, 0x010000, 0x858382f7 )
	ROM_LOAD16_BYTE( "1152213a.072", 0x020000, 0x010000, 0xdc5b2008 )
	ROM_LOAD16_BYTE( "1153213c.072", 0x020001, 0x010000, 0xe5266c7c )
	ROM_LOAD16_BYTE( "1154200a.072", 0x040000, 0x010000, 0xde4101a3 )
	ROM_LOAD16_BYTE( "1155200c.072", 0x040001, 0x010000, 0x78c0f6ad )
	ROM_LOAD16_BYTE( "1156185a.072", 0x070000, 0x008000, 0xcde16b55 )
	ROM_LOAD16_BYTE( "1157185c.072", 0x070001, 0x008000, 0x31c77376 )

	ROM_REGION( 0x14000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1149-1b.072",  0x010000, 0x004000, 0x8d730e7a )
	ROM_CONTINUE(             0x004000, 0x00c000 )

	ROM_REGION( 0x190000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1102-13r.072", 0x000000, 0x010000, 0x90becdfa )
	ROM_LOAD( "1104-28r.072", 0x010000, 0x010000, 0x33609071 )
	ROM_LOAD( "1106-41r.072", 0x020000, 0x010000, 0x71962e9f )
	ROM_LOAD( "1101-13p.072", 0x030000, 0x010000, 0x4d41701e )
	ROM_LOAD( "1103-28p.072", 0x040000, 0x010000, 0x3011da3b )
	ROM_LOAD( "1108-53r.072", 0x050000, 0x010000, 0x386c7edc )
	ROM_LOAD( "1110-67r.072", 0x060000, 0x010000, 0xa54d16e6 )
	ROM_LOAD( "1112-81r.072", 0x070000, 0x010000, 0x669411f6 )
	ROM_LOAD( "1107-53p.072", 0x080000, 0x010000, 0xcaaeb57a )
	ROM_LOAD( "1109-67p.072", 0x090000, 0x010000, 0x61cb4e28 )
	ROM_LOAD( "1114-95r.072", 0x0a0000, 0x010000, 0xe340d5a1 )
	ROM_LOAD( "1116109r.072", 0x0b0000, 0x010000, 0xf25b8aca )
	ROM_LOAD( "1118123r.072", 0x0c0000, 0x010000, 0x8cf73585 )
	ROM_LOAD( "1113-95p.072", 0x0d0000, 0x010000, 0x899b59af )
	ROM_LOAD( "1115109p.072", 0x0e0000, 0x010000, 0xcf4fd19a )
	ROM_LOAD( "1120137r.072", 0x0f0000, 0x010000, 0xfde7c03d )
	ROM_LOAD( "1122151r.072", 0x100000, 0x010000, 0x6ff6a9f2 )
	ROM_LOAD( "1124165r.072", 0x110000, 0x010000, 0xf11909f1 )
	ROM_LOAD( "1119137p.072", 0x120000, 0x010000, 0x6f8003a1 )
	ROM_LOAD( "1121151p.072", 0x130000, 0x010000, 0x8ff0a1ec )
	ROM_LOAD( "1125123n.072", 0x140000, 0x010000, 0x3aa7c756 )
	ROM_LOAD( "1126137n.072", 0x150000, 0x010000, 0xcb82c9aa )
	ROM_LOAD( "1128151n.072", 0x160000, 0x010000, 0xdce32863 )
	/* 170000-18ffff empty */

	ROM_REGION( 0x0a0000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "2129180p.072", 0x000000, 0x010000, 0x36b1a578 )
	ROM_LOAD( "2131193p.072", 0x010000, 0x010000, 0x7b7c04a1 )
	ROM_LOAD( "2133208p.072", 0x020000, 0x010000, 0xe03fe4d9 )
	ROM_LOAD( "2135221p.072", 0x030000, 0x010000, 0x7d497110 )
	ROM_LOAD( "2137235p.072", 0x040000, 0x010000, 0xf91e7872 )
	ROM_LOAD( "2130180r.072", 0x050000, 0x010000, 0xb25368cc )
	ROM_LOAD( "2132193r.072", 0x060000, 0x010000, 0x112f2d20 )
	ROM_LOAD( "2134208r.072", 0x070000, 0x010000, 0x84884ed6 )
	ROM_LOAD( "2136221r.072", 0x080000, 0x010000, 0xbc028690 )
	ROM_LOAD( "2138235r.072", 0x090000, 0x010000, 0x60cec955 )

	ROM_REGION( 0x008000, REGION_GFX3, ROMREGION_DISPOSE )
	ROM_LOAD( "2141250k.072", 0x000000, 0x008000, 0x60d6d6df )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* 256k for ADPCM samples */
	ROM_LOAD( "1145-7k.072",  0x000000, 0x010000, 0xd9475d58 )
	ROM_LOAD( "1146-7j.072",  0x010000, 0x010000, 0x133e6aef )
	ROM_LOAD( "1147-7e.072",  0x020000, 0x010000, 0xba4d556e )
	ROM_LOAD( "1148-7d.072",  0x030000, 0x010000, 0xc48df49a )
ROM_END



/*************************************
 *
 *	ROM decoding
 *
 *************************************/

static DRIVER_INIT( skullxbo )
{
	atarigen_eeprom_default = NULL;
	atarijsa_init(1, 2, 1, 0x0080);
	memset(memory_region(REGION_GFX1) + 0x170000, 0, 0x20000);
}



/*************************************
 *
 *	Game driver(s)
 *
 *************************************/

GAME( 1989, skullxbo, 0,        skullxbo, skullxbo, skullxbo, ROT0, "Atari Games", "Skull & Crossbones (rev 5)" )
GAME( 1989, skullxb4, skullxbo, skullxbo, skullxbo, skullxbo, ROT0, "Atari Games", "Skull & Crossbones (rev 4)" )
GAME( 1989, skullxb3, skullxbo, skullxbo, skullxbo, skullxbo, ROT0, "Atari Games", "Skull & Crossbones (rev 3)" )
GAME( 1989, skullxb2, skullxbo, skullxbo, skullxbo, skullxbo, ROT0, "Atari Games", "Skull & Crossbones (rev 2)" )
GAME( 1989, skullxb1, skullxbo, skullxbo, skullxbo, skullxbo, ROT0, "Atari Games", "Skull & Crossbones (rev 1)" )
