/***************************************************************************

	Atari Runaway hardware

	Games supported:
		* Qwak (prototype)
		* Runaway (prototype)

	original Qwak driver written by Mike Balfour

****************************************************************************/

#include "driver.h"
#include "machine/atari_vg.h"

extern VIDEO_START( runaway );
extern VIDEO_START( qwak );
extern VIDEO_UPDATE( runaway );
extern VIDEO_UPDATE( qwak );

extern UINT8* runaway_video_ram;
extern UINT8* runaway_sprite_ram;

extern WRITE_HANDLER( runaway_paletteram_w );
extern WRITE_HANDLER( runaway_video_ram_w );
extern WRITE_HANDLER( runaway_tile_bank_w );


static void interrupt_callback(int scanline)
{
	/* assume Centipede-style interrupt timing */

	cpu_set_irq_line(0, 0, (scanline & 32) ? ASSERT_LINE : CLEAR_LINE);

	scanline += 32;

	if (scanline >= 263)
	{
		scanline = 16;
	}

	timer_set(cpu_getscanlinetime(scanline), scanline, interrupt_callback);
}


static MACHINE_INIT( runaway )
{
	timer_set(cpu_getscanlinetime(16), 16, interrupt_callback);
}


static READ_HANDLER( runaway_input_r )
{
	UINT8 val = 0;

	if (readinputport(0) & (1 << offset))
	{
		val |= 0x80;
	}
	if (readinputport(1) & (1 << offset))
	{
		val |= 0x40;
	}

	return val;
}


static READ_HANDLER( runaway_pot_r )
{
	return (readinputport(4) << (7 - offset)) & 0x80;
}


static WRITE_HANDLER( runaway_led_w )
{
	set_led_status(offset, ~data & 1);
}


static WRITE_HANDLER( runaway_irq_ack_w )
{
	cpu_set_irq_line(0, 0, CLEAR_LINE);
}


static MEMORY_READ_START( readmem )
	{ 0x0000, 0x03ff, MRA_RAM },
	{ 0x0400, 0x07ff, MRA_RAM },
	{ 0x3000, 0x3007, runaway_input_r },
	{ 0x4000, 0x4000, input_port_2_r },
	{ 0x5000, 0x5000, atari_vg_earom_r },
	{ 0x6000, 0x600f, pokey1_r },
	{ 0x7000, 0x700f, pokey2_r },
	{ 0x8000, 0xcfff, MRA_ROM },
	{ 0xf000, 0xffff, MRA_ROM },	/* for the interrupt vectors */
MEMORY_END


static MEMORY_WRITE_START( writemem )
	{ 0x0000, 0x03ff, MWA_RAM },
	{ 0x0400, 0x07bf, runaway_video_ram_w, &runaway_video_ram },
	{ 0x07c0, 0x07ff, MWA_RAM, &runaway_sprite_ram },
	{ 0x1000, 0x1000, runaway_irq_ack_w },
	{ 0x1400, 0x143F, atari_vg_earom_w },
	{ 0x1800, 0x1800, atari_vg_earom_ctrl_w },
	{ 0x1c00, 0x1c0f, runaway_paletteram_w },
	{ 0x2000, 0x2000, MWA_NOP }, /* coin counter? */
	{ 0x2001, 0x2001, MWA_NOP }, /* coin counter? */
	{ 0x2003, 0x2004, runaway_led_w },
	{ 0x2005, 0x2005, runaway_tile_bank_w },
	{ 0x6000, 0x600f, pokey1_w },
	{ 0x7000, 0x700f, pokey2_w },
	{ 0x8000, 0xcfff, MWA_ROM },
MEMORY_END


INPUT_PORTS_START( qwak )
	PORT_START	/* 3000 D7 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_VBLANK )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN3 )

	PORT_START	/* 3000 D6 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* 4000 */
	PORT_DIPNAME( 0x01, 0x00, "DIP 1" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x02, 0x00, "DIP 2" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x04, 0x00, "DIP 3" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x08, 0x00, "DIP 4" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x10, 0x00, "DIP 5" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x20, 0x00, "DIP 6" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x40, 0x00, "DIP 7" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x80, 0x00, "DIP 8" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))

	PORT_START /* 6008 not used */
	PORT_START /* 7000 not used */
INPUT_PORTS_END


INPUT_PORTS_START( runaway )
	PORT_START /* 3000 D7 */
	PORT_BIT ( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT ( 0x02, IP_ACTIVE_HIGH, IPT_VBLANK )
	PORT_BIT ( 0x04, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT ( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_SERVICE( 0x10, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN3 )

	PORT_START /* 3000 D6 */
	PORT_BIT ( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT ( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT ( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT ( 0x08, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT ( 0x10, IP_ACTIVE_LOW, IPT_START1 )	/* also level skip if invincibility is on */
	PORT_BIT ( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT ( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT ( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START /* 4000 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Coinage ))
	PORT_DIPSETTING(    0x00, DEF_STR( 2C_1C ))
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_1C ))
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ))
	PORT_DIPSETTING(    0x03, DEF_STR( Free_Play ))
	PORT_DIPNAME( 0x0c, 0x04, "Coin 3 Multiplier" )
	PORT_DIPSETTING(    0x0c, "1" )
	PORT_DIPSETTING(    0x08, "4" )
	PORT_DIPSETTING(    0x04, "5" )
	PORT_DIPSETTING(    0x00, "6" )
	PORT_DIPNAME( 0x10, 0x10, "Coin 2 Multiplier" )
	PORT_DIPSETTING(    0x10, "1" )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unknown ))
	PORT_DIPSETTING(    0x20, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unknown ))
	PORT_DIPSETTING(    0x40, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown ))
	PORT_DIPSETTING(    0x80, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))

	PORT_START /* 6008 */
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Lives ))
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPSETTING(    0x01, "3" )
	PORT_DIPSETTING(    0x02, "4" )
	PORT_DIPSETTING(    0x03, "5" )
	PORT_DIPNAME( 0x0c, 0x08, DEF_STR( Bonus_Life ))
	PORT_DIPSETTING(    0x00, "Never" )
	PORT_DIPSETTING(    0x04, "Every 5000" )
	PORT_DIPSETTING(    0x08, "Every 10000" )
	PORT_DIPSETTING(    0x0c, "Every 15000" )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x10, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x20, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Difficulty ))
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x40, "Hard" )
	PORT_DIPNAME( 0x80, 0x00, "Invincibility" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ))
	PORT_DIPSETTING(    0x80, DEF_STR( On ))

	PORT_START /* 7000 */
	PORT_BIT ( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_TOGGLE )
	PORT_BIT ( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT ( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT ( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT ( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x60, 0x00, "Language" )
	PORT_DIPSETTING(    0x00, "English" )
	PORT_DIPSETTING(    0x20, "German" )
	PORT_DIPSETTING(    0x40, "French" )
	PORT_DIPSETTING(    0x60, "Spanish" )
	PORT_BIT ( 0x80, IP_ACTIVE_HIGH, IPT_UNKNOWN )
INPUT_PORTS_END


static struct GfxLayout runaway_tile_layout =
{
	8, 8,
	256,
	3,
	{
		RGN_FRAC(2,3), RGN_FRAC(1,3), RGN_FRAC(0,3)
	},
	{
		0, 1, 2, 3, 4, 5, 6, 7
	},
	{
		0x00, 0x08, 0x10, 0x18, 0x20, 0x28, 0x30, 0x38
	},
	0x40
};


static struct GfxLayout qwak_tile_layout =
{
	8, 8,
	256,
	4,
	{
		RGN_FRAC(3,4), RGN_FRAC(2,4), RGN_FRAC(1,4), RGN_FRAC(0,4)
	},
	{
		0, 1, 2, 3, 4, 5, 6, 7
	},
	{
		0x00, 0x08, 0x10, 0x18, 0x20, 0x28, 0x30, 0x38
	},
	0x40
};


static struct GfxLayout runaway_sprite_layout =
{
	8, 16,
	384,
	3,
	{
		RGN_FRAC(2,3), RGN_FRAC(1,3), RGN_FRAC(0,3)
	},
	{
		0, 1, 2, 3, 4, 5, 6, 7
	},
	{
		0x00, 0x08, 0x10, 0x18, 0x20, 0x28, 0x30, 0x38,
		0x40, 0x48, 0x50, 0x58, 0x60, 0x68, 0x70, 0x78
	},
	0x80
};


static struct GfxLayout qwak_sprite_layout =
{
	8, 16,
	128,
	4,
	{
		RGN_FRAC(3,4), RGN_FRAC(2,4), RGN_FRAC(1,4), RGN_FRAC(0,4)
	},
	{
		0, 1, 2, 3, 4, 5, 6, 7
	},
	{
		0x00, 0x08, 0x10, 0x18, 0x20, 0x28, 0x30, 0x38,
		0x40, 0x48, 0x50, 0x58, 0x60, 0x68, 0x70, 0x78
	},
	0x80
};


static struct GfxDecodeInfo runaway_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x000, &runaway_tile_layout,   0, 1 },
	{ REGION_GFX1, 0x800, &runaway_sprite_layout, 8, 1 },
	{ -1 }
};


static struct GfxDecodeInfo qwak_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x800, &qwak_tile_layout,   0, 1 },
	{ REGION_GFX1, 0x000, &qwak_sprite_layout, 0, 1 },
	{ -1 }
};


static struct POKEYinterface pokey_interface =
{
	2,
	12096000 / 8,
	{ 100, 100 },
	{ 0, runaway_pot_r },
	{ 0, runaway_pot_r },
	{ 0, runaway_pot_r },
	{ 0, runaway_pot_r },
	{ 0, runaway_pot_r },
	{ 0, runaway_pot_r },
	{ 0, runaway_pot_r },
	{ 0, runaway_pot_r },
	{ input_port_3_r, 0 },
};


static MACHINE_DRIVER_START( runaway )

	/* basic machine hardware */
	MDRV_CPU_ADD(M6502, 12096000 / 8) /* ? */
	MDRV_CPU_MEMORY(readmem, writemem)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(23. * 1000000 / 15750)

	MDRV_MACHINE_INIT(runaway)
	MDRV_NVRAM_HANDLER(atari_vg)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(256, 240)
	MDRV_VISIBLE_AREA(0, 255, 0, 239)
	MDRV_GFXDECODE(runaway_gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(16)

	MDRV_VIDEO_START(runaway)
	MDRV_VIDEO_UPDATE(runaway)

	/* sound hardware */
	MDRV_SOUND_ADD(POKEY, pokey_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( qwak )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(runaway)

	/* video hardware */
	MDRV_GFXDECODE(qwak_gfxdecodeinfo)

	MDRV_VIDEO_START(qwak)
	MDRV_VIDEO_UPDATE(qwak)

MACHINE_DRIVER_END


ROM_START( runaway )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "choo8000.d1", 0x8000, 0x1000, 0x35794abe )
	ROM_LOAD( "choo9000.d1", 0x9000, 0x1000, 0x0d63756d )
	ROM_LOAD( "chooa000.e1", 0xa000, 0x1000, 0xe6806b6b )
	ROM_LOAD( "choob000.f1", 0xb000, 0x1000, 0x6aa52bc4 )
	ROM_LOAD( "chooc000",    0xc000, 0x1000, 0x452ddea2 )
	ROM_RELOAD(              0xf000, 0x1000 )

	ROM_REGION( 0x6000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "chop0", 0x0000, 0x2000, 0x225a8c5e )
	ROM_LOAD( "chop1", 0x2000, 0x2000, 0x70389c0f )
	ROM_LOAD( "chop2", 0x4000, 0x2000, 0x63655f1c )
ROM_END


ROM_START( qwak )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "qwak8000.bin", 0x8000, 0x1000, 0x4d002d8a )
	ROM_LOAD( "qwak9000.bin", 0x9000, 0x1000, 0xe0c78fd7 )
	ROM_LOAD( "qwaka000.bin", 0xa000, 0x1000, 0xe5770fc9 )
	ROM_LOAD( "qwakb000.bin", 0xb000, 0x1000, 0x90771cc0 )
	ROM_RELOAD(               0xf000, 0x1000 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "qwakgfx0.bin", 0x0000, 0x1000, 0xbed2c067 )
	ROM_LOAD( "qwakgfx1.bin", 0x1000, 0x1000, 0x73a31d28 )
	ROM_LOAD( "qwakgfx2.bin", 0x2000, 0x1000, 0x07fd9e80 )
	ROM_LOAD( "qwakgfx3.bin", 0x3000, 0x1000, 0xe8416f2b )
ROM_END


GAME( 1982, qwak,    0, qwak,    qwak,    0, ROT270, "Atari", "Qwak (prototype)" )
GAME( 1982, runaway, 0, runaway, runaway, 0, ROT0,   "Atari", "Runaway (prototype)" )
