/***************************************************************************

  Nintendo 8080 hardware

	- Space Fever
	- Space Fever High Splitter
	- Space Launcher
	- Sheriff / Bandido
	- Helifire

***************************************************************************/

#include "driver.h"
#include "cpu/i8039/i8039.h"
#include <math.h>

#define HARDWARE_IS_SPACE_FEVER     ( n8080_hardware == 1 )
#define HARDWARE_IS_SHERIFF         ( n8080_hardware == 2 )
#define HARDWARE_IS_HELIFIRE        ( n8080_hardware == 3 )

static int n8080_hardware;

static void *sound_timer[3];

static int mono_flop[3];

static int sheriff_color_mode;
static int sheriff_color_data;

static int helifire_scroll;
static int helifire_decay;

static UINT8 helifire_LSFR[63];

static UINT8* n8080_videoram;
static UINT8* n8080_colorram;

static int spacefev_ufo_frame;
static int spacefev_ufo_cycle;
static int spacefev_red_screen;
static int spacefev_red_cannon;

static void *spacefev_red_cannon_timer;

static unsigned shift_data;
static unsigned shift_bits;

static UINT16 prev_sound_pins;
static UINT16 curr_sound_pins;

static int flip_screen;


/* following data is based on screen shots */

static const UINT8 sheriff_color_PROM[] =
{
	0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8,
	0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8,
	0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8,
	0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xf,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xf,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xf,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0x9, 0x9, 0xb,
	0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xe, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb,
	0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xe, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xe, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0x9, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0x9, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0x9, 0xb,
	0xe, 0xe, 0xe, 0xa, 0xa, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x2, 0x2, 0x6, 0xe, 0xe, 0xd, 0xf, 0x9, 0xb,
	0xe, 0xe, 0xe, 0xa, 0xa, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x2, 0x2, 0x6, 0xe, 0xe, 0xd, 0xf, 0x9, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0x9, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0x9, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0x9, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xa, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0x3, 0x3, 0x3, 0x3, 0x3, 0x3,
	0x3, 0x3, 0x3, 0x3, 0x3, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xa, 0xb,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb,
	0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xa, 0xf,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0x9, 0x9, 0xb,
	0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0xb, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xf,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xf,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xf,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe,
	0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0xe, 0x6, 0x6, 0x6, 0xe, 0xe, 0xd, 0xf, 0xf, 0xf,
	0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8,
	0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8,
	0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8,
	0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x0, 0x0, 0x0, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8,
};


static struct DACinterface n8080_dac_interface =
{
	1, { 30 }
};


struct SN76477interface sheriff_sn76477_interface =
{
	1,
	{ 35 },
	{ RES_K(36)  },  /* 04 */
	{ RES_K(100) },  /* 05 */
	{ CAP_N(1)   },  /* 06 */
	{ RES_K(620) },  /* 07 */
	{ CAP_U(1)   },  /* 08 */
	{ RES_K(20)  },  /* 10 */
	{ RES_K(150) },  /* 11 */
	{ RES_K(47)  },  /* 12 */
	{ 0          },  /* 16 */
	{ CAP_N(1)   },  /* 17 */
	{ RES_M(1.5) },  /* 18 */
	{ 0          },  /* 19 */
	{ RES_M(1.5) },  /* 20 */
	{ CAP_N(47)  },  /* 21 */
	{ CAP_N(47)  },  /* 23 */
	{ RES_K(560) },  /* 24 */
};


struct SN76477interface spacefev_sn76477_interface =
{
	1,
	{ 35 },
	{ RES_K(36)  },  /* 04 */
	{ RES_K(150) },  /* 05 */
	{ CAP_N(1)   },  /* 06 */
	{ RES_M(1)   },  /* 07 */
	{ CAP_U(1)   },  /* 08 */
	{ RES_K(20)  },  /* 10 */
	{ RES_K(150) },  /* 11 */
	{ RES_K(47)  },  /* 12 */
	{ 0          },  /* 16 */
	{ CAP_N(1)   },  /* 17 */
	{ RES_M(1.5) },  /* 18 */
	{ 0          },  /* 19 */
	{ RES_M(1)   },  /* 20 */
	{ CAP_N(47)  },  /* 21 */
	{ CAP_N(47)  },  /* 23 */
	{ RES_K(820) },  /* 24 */
};


static WRITE_HANDLER( n8080_shift_bits_w )
{
	shift_bits = data & 7;
}
static WRITE_HANDLER( n8080_shift_data_w )
{
	shift_data = (shift_data >> 8) | (data << 8);
}


static READ_HANDLER( n8080_shift_r )
{
	return shift_data >> (8 - shift_bits);
}


static PALETTE_INIT( n8080 )
{
	int i;

	for (i = 0; i < 8; i++)
	{
		palette_set_color(i,
			(i & 1) ? 255 : 0,
			(i & 2) ? 255 : 0,
			(i & 4) ? 255 : 0);
	}
}

static PALETTE_INIT( helifire )
{
	int i;

	palette_init_n8080(NULL, NULL);

	for (i = 0; i < 0x100; i++)
	{
		int level = 0xff * exp(-3 * i / 255.); /* capacitor discharge */

		palette_set_color(0x000 + 8 + i, 0x00, 0x00, level);   /* shades of blue */
		palette_set_color(0x100 + 8 + i, 0x00, 0xC0, level);   /* shades of blue w/ green star */

		palette_set_color(0x200 + 8 + i, level, 0x00, 0x00);   /* shades of red */
		palette_set_color(0x300 + 8 + i, level, 0xC0, 0x00);   /* shades of red w/ green star */
	}
}


static VIDEO_START( spacefev )
{
	spacefev_ufo_frame = 0;
	spacefev_ufo_cycle = 0;

	spacefev_red_screen = 0;
	spacefev_red_cannon = 0;

	return 0;
}


static VIDEO_START( helifire )
{
	UINT8 data = 0;

	int i;

	for (i = 0; i < 63; i++)
	{
		int bit =
			(data >> 6) ^
			(data >> 7) ^ 1;

		data = (data << 1) | (bit & 1);

		helifire_LSFR[i] = data;
	}

	helifire_scroll = 0;

	return 0;
}


static VIDEO_UPDATE( spacefev )
{
	UINT8 mask = flip_screen ? 0xff : 0x00;

	int x;
	int y;

	const UINT8* pRAM = n8080_videoram;

	for (y = 0; y < 256; y++)
	{
		UINT16* pLine = bitmap->line[y ^ mask];

		for (x = 0; x < 256; x += 8)
		{
			int n;

			UINT8 color = 0;

			if (spacefev_red_screen)
			{
				color = 1;
			}
			else
			{
				UINT8 val = memory_region(REGION_PROMS)[x >> 3];

				if ((x >> 3) == 0x06)
				{
					color = spacefev_red_cannon ? 1 : 7;
				}

				if ((x >> 3) == 0x1b)
				{
					static const UINT8 ufo_color[] =
					{
						1, /* red     */
						2, /* green   */
						7, /* white   */
						3, /* yellow  */
						5, /* magenta */
						6, /* cyan    */
					};

					color = ufo_color[spacefev_ufo_cycle];
				}

				for (n = color + 1; n < 8; n++)
				{
					if (~val & (1 << n))
					{
						color = n;
					}
				}
			}

			for (n = 0; n < 8; n++)
			{
				pLine[(x + n) ^ mask] = (pRAM[x >> 3] & (1 << n)) ? color : 0;
			}
		}

		pRAM += 32;
	}
}


static VIDEO_UPDATE( sheriff )
{
	UINT8 mask = flip_screen ? 0xff : 0x00;

	int x;
	int y;

	const UINT8* pRAM = n8080_videoram;

	for (y = 0; y < 256; y++)
	{
		UINT16* pLine = bitmap->line[y ^ mask];

		for (x = 0; x < 256; x += 8)
		{
			int n;

			UINT8 color = sheriff_color_PROM[32 * (y >> 3) + (x >> 3)];

			if (sheriff_color_mode == 1 && !(color & 8))
			{
				color = sheriff_color_data ^ 7;
			}

			if (sheriff_color_mode == 2)
			{
				color = sheriff_color_data ^ 7;
			}

			if (sheriff_color_mode == 3)
			{
				color = 7;
			}

			for (n = 0; n < 8; n++)
			{
				pLine[(x + n) ^ mask] = (pRAM[x >> 3] & (1 << n)) ? (color & 7) : 0;
			}
		}

		pRAM += 32;
	}
}


static VIDEO_UPDATE( helifire )
{
	UINT8 mask = flip_screen ? 0xff : 0x00;

	int x;
	int y;

	const UINT8* pRAM = n8080_videoram;

	int SUN_BRIGHTNESS = readinputport(4);
	int SEA_BRIGHTNESS = readinputport(5);

	static const int wave[8] = { 0, 1, 2, 2, 2, 1, 0, 0 };

	int level;

	int counter = helifire_scroll;

	for (y = 0; y < 256; y++)
	{
		UINT16* pLine = bitmap->line[y ^ mask];

		counter = (counter + 1) % 257;

		level = 120 + wave[counter & 7];

		/* draw sky */

		for (x = level; x < 256; x++)
		{
			pLine[x] = 0x200 + 8 + SUN_BRIGHTNESS + x - level;
		}

		/* draw stars */

		if (counter % 8 == 4)
		{
			int step = (320 * counter) % sizeof helifire_LSFR;

			int data =
				((helifire_LSFR[step] & 1) << 6) |
				((helifire_LSFR[step] & 2) << 4) |
				((helifire_LSFR[step] & 4) << 2) |
				((helifire_LSFR[step] & 8) << 0);

			pLine[0x80 + data] |= 0x100;
		}

		/* draw sea */

		for (x = 0; x < level; x++)
		{
			pLine[x] = 8 + SEA_BRIGHTNESS + x;
		}

		/* draw foreground */

		for (x = 0; x < 256; x += 8)
		{
			int n;

			int offset = 32 * y + (x >> 3);

			for (n = 0; n < 8; n++)
			{
				if (pRAM[offset] & (1 << n))
				{
					pLine[(x + n) ^ mask] = n8080_colorram[offset] & 7;
				}
			}
		}
	}
}


static VIDEO_EOF( spacefev )
{
	spacefev_ufo_frame = (spacefev_ufo_frame + 1) % 32;

	if (spacefev_ufo_frame == 0)
	{
		spacefev_ufo_cycle = (spacefev_ufo_cycle + 1) % 6;
	}
}


static VIDEO_EOF( helifire )
{
	helifire_scroll = (helifire_scroll + 256) % 257;
}


static INTERRUPT_GEN( interrupt )
{
	if (cpu_getvblank())
	{
		cpu_set_irq_line_and_vector(0, 0, PULSE_LINE, 0xcf);  /* RST $08 */
	}
	else
	{
		cpu_set_irq_line_and_vector(0, 0, PULSE_LINE, 0xd7);  /* RST $10 */
	}
}


static void spacefev_vco_voltage_timer(int dummy)
{
	double voltage = 0;

	if (mono_flop[2])
	{
		voltage = 5 * (1 - exp(- timer_timeelapsed(sound_timer[2]) / 0.22));
	}

	SN76477_set_vco_voltage(0, voltage);
}


static void helifire_decay_timer(int dummy)
{
	/* ... */
}


static void spacefev_update_SN76477_status(void)
{
	double dblR0 = RES_M(1.0);
	double dblR1 = RES_M(1.5);

	if (!mono_flop[0])
	{
		dblR0 = 1 / (1 / RES_K(150) + 1 / dblR0); /* ? */
	}
	if (!mono_flop[1])
	{
		dblR1 = 1 / (1 / RES_K(620) + 1 / dblR1); /* ? */
	}

	SN76477_set_decay_res(0, dblR0);

	SN76477_set_vco_res(0, dblR1);

	SN76477_enable_w(0,
		!mono_flop[0] &&
		!mono_flop[1] &&
		!mono_flop[2]);

	SN76477_vco_w(0, mono_flop[1]);

	SN76477_mixer_b_w(0, mono_flop[0]);
}


static void sheriff_update_SN76477_status(void)
{
	if (mono_flop[1])
	{
		SN76477_set_vco_voltage(0, 5);
	}
	else
	{
		SN76477_set_vco_voltage(0, 0);
	}

	SN76477_enable_w(0,
		!mono_flop[0] &&
		!mono_flop[1]);

	SN76477_vco_w(0, mono_flop[0]);

	SN76477_mixer_b_w(0, !mono_flop[0]);
}


static void update_SN76477_status(void)
{
	if (HARDWARE_IS_SPACE_FEVER)
	{
		spacefev_update_SN76477_status();
	}
	if (HARDWARE_IS_SHERIFF)
	{
		sheriff_update_SN76477_status();
	}
}


static void start_mono_flop(int n, double expire)
{
	mono_flop[n] = 1;

	update_SN76477_status();

	timer_adjust(sound_timer[n], expire, n, 0);
}


static void stop_mono_flop(int n)
{
	mono_flop[n] = 0;

	update_SN76477_status();

	timer_adjust(sound_timer[n], TIME_NEVER, n, 0);
}


static void start_red_cannon(double expire)
{
	spacefev_red_cannon = 1;

	timer_adjust(spacefev_red_cannon_timer, expire, 0, 0);
}


static void stop_red_cannon(int dummy)
{
	spacefev_red_cannon = 0;

	timer_adjust(spacefev_red_cannon_timer, TIME_NEVER, 0, 0);
}


static void spacefev_sound_pins_changed(void)
{
	UINT16 changes = ~curr_sound_pins & prev_sound_pins;

	if (changes & (1 << 0x3))
	{
		stop_mono_flop(1);
	}
	if (changes & ((1 << 0x3) | (1 << 0x6)))
	{
		stop_mono_flop(2);
	}
	if (changes & (1 << 0x3))
	{
		start_mono_flop(0, TIME_IN_MSEC(0.55 * 36 * 100));
	}
	if (changes & (1 << 0x6))
	{
		start_mono_flop(1, TIME_IN_MSEC(0.55 * 22 * 33));
	}
	if (changes & (1 << 0x4))
	{
		start_mono_flop(2, TIME_IN_MSEC(0.55 * 22 * 33));
	}
	if (changes & ((1 << 0x2) | (1 << 0x3) | (1 << 0x5)))
	{
		cpu_set_irq_line(1, 0, PULSE_LINE);
	}
}


static void sheriff_sound_pins_changed(void)
{
	UINT16 changes = ~curr_sound_pins & prev_sound_pins;

	if (changes & (1 << 0x6))
	{
		stop_mono_flop(1);
	}
	if (changes & (1 << 0x6))
	{
		start_mono_flop(0, TIME_IN_MSEC(0.55 * 33 * 33));
	}
	if (changes & (1 << 0x4))
	{
		start_mono_flop(1, TIME_IN_MSEC(0.55 * 33 * 33));
	}
	if (changes & ((1 << 0x2) | (1 << 0x3) | (1 << 0x5)))
	{
		cpu_set_irq_line(1, 0, PULSE_LINE);
	}
}


static void helifire_sound_pins_changed(void)
{
	UINT16 changes = ~curr_sound_pins & prev_sound_pins;

	/* lacking emulation of sound bits 10, 11, 12 and 4 */

	if (changes & (1 << 6))
	{
		cpu_set_irq_line(1, 0, PULSE_LINE);
	}
}


static void sound_pins_changed(void)
{
	if (HARDWARE_IS_SPACE_FEVER)
	{
		spacefev_sound_pins_changed();
	}
	if (HARDWARE_IS_SHERIFF)
	{
		sheriff_sound_pins_changed();
	}
	if (HARDWARE_IS_HELIFIRE)
	{
		helifire_sound_pins_changed();
	}

	prev_sound_pins = curr_sound_pins;
}


static void delayed_sound_1(int data)
{
	static UINT8 prev_data = 0;

	curr_sound_pins &= ~(
		(1 << 0x7) |
		(1 << 0x5) |
		(1 << 0x6) |
		(1 << 0x3) |
		(1 << 0x4) |
		(1 << 0x1));

	if (~data & 0x01) curr_sound_pins |= 1 << 0x7;
	if (~data & 0x02) curr_sound_pins |= 1 << 0x5; /* pulse */
	if (~data & 0x04) curr_sound_pins |= 1 << 0x6; /* pulse */
	if (~data & 0x08) curr_sound_pins |= 1 << 0x3; /* pulse (except in Helifire) */
	if (~data & 0x10) curr_sound_pins |= 1 << 0x4; /* pulse (except in Helifire) */
	if (~data & 0x20) curr_sound_pins |= 1 << 0x1;

	if (HARDWARE_IS_SPACE_FEVER)
	{
		if (data & ~prev_data & 0x10)
		{
			start_red_cannon(TIME_IN_MSEC(0.55 * 68 * 10));
		}

		spacefev_red_screen = data & 0x08;
	}

	sound_pins_changed();

	prev_data = data;
}


static void delayed_sound_2(int data)
{
	curr_sound_pins &= ~(
		(1 << 0x8) |
		(1 << 0x9) |
		(1 << 0xA) |
		(1 << 0xB) |
		(1 << 0x2) |
		(1 << 0xC));

	if (~data & 0x01) curr_sound_pins |= 1 << 0x8;
	if (~data & 0x02) curr_sound_pins |= 1 << 0x9;
	if (~data & 0x04) curr_sound_pins |= 1 << 0xA;
	if (~data & 0x08) curr_sound_pins |= 1 << 0xB;
	if (~data & 0x10) curr_sound_pins |= 1 << 0x2; /* pulse */
	if (~data & 0x20) curr_sound_pins |= 1 << 0xC;

	if (HARDWARE_IS_SPACE_FEVER)
	{
		flip_screen = data & 0x20;
	}

	sound_pins_changed();
}


static WRITE_HANDLER( n8080_sound_1_w )
{
	timer_set(TIME_NOW, data, delayed_sound_1); /* force CPUs to sync */
}
static WRITE_HANDLER( n8080_sound_2_w )
{
	timer_set(TIME_NOW, data, delayed_sound_2); /* force CPUs to sync */
}


static READ_HANDLER( n8080_8035_p1_r )
{
	UINT8 val = 0;

	if (curr_sound_pins & (1 << 0xB)) val |= 0x01;
	if (curr_sound_pins & (1 << 0xA)) val |= 0x02;
	if (curr_sound_pins & (1 << 0x9)) val |= 0x04;
	if (curr_sound_pins & (1 << 0x8)) val |= 0x08;
	if (curr_sound_pins & (1 << 0x5)) val |= 0x10;
	if (curr_sound_pins & (1 << 0x3)) val |= 0x20;
	if (curr_sound_pins & (1 << 0x2)) val |= 0x40;
	if (curr_sound_pins & (1 << 0x1)) val |= 0x80;

	return val;
}


static READ_HANDLER( helifire_8035_extended_ram_r )
{
	UINT8 val = 0;

	if (curr_sound_pins & (1 << 0x7)) val |= 0x01;
	if (curr_sound_pins & (1 << 0x8)) val |= 0x02;
	if (curr_sound_pins & (1 << 0x9)) val |= 0x04;
	if (curr_sound_pins & (1 << 0x1)) val |= 0x08;

	return val;
}


static READ_HANDLER( n8080_8035_t0_r )
{
	return (curr_sound_pins & (1 << 0x7)) ? 1 : 0;
}
static READ_HANDLER( n8080_8035_t1_r )
{
	return (curr_sound_pins & (1 << 0xC)) ? 1 : 0;
}


static READ_HANDLER( helifire_8035_t0_r )
{
	return (curr_sound_pins & (1 << 0x3)) ? 1 : 0;
}
static READ_HANDLER( helifire_8035_t1_r )
{
	return (curr_sound_pins & (1 << 0x4)) ? 1 : 0;
}


static WRITE_HANDLER( n8080_dac_w )
{
	DAC_data_w(0, data & 0x80);
}


static WRITE_HANDLER( helifire_dac_data_w )
{
	DAC_data_w(0, data);
}


static WRITE_HANDLER( helifire_dac_vref_w )
{
	helifire_decay = ~data & 0x80;
}


static WRITE_HANDLER( n8080_video_control_w )
{
	sheriff_color_mode = (data >> 3) & 3;
	sheriff_color_data = (data >> 0) & 7;

	flip_screen = data & 0x20;
}


static MEMORY_READ_START( sheriff_readmem )
	{ 0x0000, 0x3fff, MRA_ROM },
MEMORY_END

static MEMORY_WRITE_START( sheriff_writemem )
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x7fff, MWA_RAM, &n8080_videoram },
MEMORY_END

static MEMORY_READ_START( helifire_readmem )
	{ 0x0000, 0x3fff, MRA_ROM },
MEMORY_END

static MEMORY_WRITE_START( helifire_writemem )
	{ 0x0000, 0x3fff, MWA_ROM },
	{ 0x4000, 0x7fff, MWA_RAM, &n8080_videoram },
	{ 0xc000, 0xdfff, MWA_RAM, &n8080_colorram },
MEMORY_END

static PORT_READ_START( sheriff_readport )
	{ 0x00, 0x00, input_port_0_r },
	{ 0x01, 0x01, input_port_1_r },
	{ 0x02, 0x02, input_port_2_r },
	{ 0x03, 0x03, n8080_shift_r },
	{ 0x04, 0x04, input_port_3_r },
PORT_END

static PORT_WRITE_START( sheriff_writeport )
	{ 0x02, 0x02, n8080_shift_bits_w },
	{ 0x03, 0x03, n8080_shift_data_w },
	{ 0x04, 0x04, n8080_sound_1_w },
	{ 0x05, 0x05, n8080_sound_2_w },
	{ 0x06, 0x06, n8080_video_control_w },
PORT_END

static MEMORY_READ_START( sheriff_sound_readmem )
	{ 0x0000, 0x03ff, MRA_ROM },
MEMORY_END

static MEMORY_WRITE_START( sheriff_sound_writemem )
	{ 0x0000, 0x03ff, MWA_ROM },
MEMORY_END

static PORT_READ_START( sheriff_sound_readport )
	{ I8039_t0, I8039_t0, n8080_8035_t0_r },
	{ I8039_t1, I8039_t1, n8080_8035_t1_r },
	{ I8039_p1, I8039_p1, n8080_8035_p1_r },
PORT_END

static PORT_WRITE_START( sheriff_sound_writeport )
	{ I8039_p2, I8039_p2, n8080_dac_w },
PORT_END

static PORT_READ_START( helifire_sound_readport )
	{ I8039_t0, I8039_t0, helifire_8035_t0_r },
	{ I8039_t1, I8039_t1, helifire_8035_t1_r },
	{ 0x00, 0x7f, helifire_8035_extended_ram_r },
PORT_END

static PORT_WRITE_START( helifire_sound_writeport )
	{ I8039_p1, I8039_p1, helifire_dac_data_w },
	{ I8039_p2, I8039_p2, helifire_dac_vref_w },
PORT_END

static MACHINE_INIT( spacefev )
{
	n8080_hardware = 1;

	timer_pulse(TIME_IN_HZ(1000), 0, spacefev_vco_voltage_timer);

	sound_timer[0] = timer_alloc(stop_mono_flop);
	sound_timer[1] = timer_alloc(stop_mono_flop);
	sound_timer[2] = timer_alloc(stop_mono_flop);

	spacefev_red_cannon_timer = timer_alloc(stop_red_cannon);

	SN76477_envelope_1_w(0, 1);
	SN76477_envelope_2_w(0, 0);
	SN76477_mixer_a_w(0, 0);
	SN76477_mixer_b_w(0, 0);
	SN76477_mixer_c_w(0, 0);
	SN76477_noise_clock_w(0, 0);

	mono_flop[0] = 0;
	mono_flop[1] = 0;
	mono_flop[2] = 0;

	delayed_sound_1(0);
	delayed_sound_2(0);
}


static MACHINE_INIT( sheriff )
{
	n8080_hardware = 2;

	sound_timer[0] = timer_alloc(stop_mono_flop);
	sound_timer[1] = timer_alloc(stop_mono_flop);

	SN76477_envelope_1_w(0, 1);
	SN76477_envelope_2_w(0, 0);
	SN76477_mixer_a_w(0, 0);
	SN76477_mixer_b_w(0, 0);
	SN76477_mixer_c_w(0, 0);
	SN76477_noise_clock_w(0, 0);

	mono_flop[0] = 0;
	mono_flop[1] = 0;

	delayed_sound_1(0);
	delayed_sound_2(0);

	n8080_video_control_w(0, 0);
}


static MACHINE_INIT( helifire )
{
	n8080_hardware = 3;

	timer_pulse(TIME_IN_HZ(1000), 0, helifire_decay_timer);

	delayed_sound_1(0);
	delayed_sound_2(0);

	n8080_video_control_w(0, 0);

	helifire_decay = 0;
}


static MACHINE_DRIVER_START( n8080 )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", 8080, 20160000 / 10)
	MDRV_CPU_MEMORY(sheriff_readmem,sheriff_writemem)
	MDRV_CPU_PORTS(sheriff_readport,sheriff_writeport)
	MDRV_CPU_VBLANK_INT(interrupt, 2)

	MDRV_CPU_ADD_TAG("sound", I8035, 6000000 / I8039_CLOCK_DIVIDER)
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(sheriff_sound_readmem,sheriff_sound_writemem)
	MDRV_CPU_PORTS(sheriff_sound_readport,sheriff_sound_writeport)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(256, 256)
	MDRV_VISIBLE_AREA(0, 255, 16, 239)
	MDRV_PALETTE_LENGTH(8)
	MDRV_PALETTE_INIT(n8080)

	/* sound hardware */
	MDRV_SOUND_ADD(DAC, n8080_dac_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( spacefev )

	MDRV_IMPORT_FROM(n8080)

	/* basic machine hardware */
	MDRV_MACHINE_INIT(spacefev)

	/* video hardware */
	MDRV_VIDEO_START(spacefev)
	MDRV_VIDEO_UPDATE(spacefev)
	MDRV_VIDEO_EOF(spacefev)

	/* sound hardware */
	MDRV_SOUND_ADD(SN76477, spacefev_sn76477_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( sheriff )

	MDRV_IMPORT_FROM(n8080)

	/* basic machine hardware */
	MDRV_MACHINE_INIT(sheriff)

	/* video hardware */
	MDRV_VIDEO_UPDATE(sheriff)

	/* sound hardware */
	MDRV_SOUND_ADD(SN76477, sheriff_sn76477_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( helifire )

	MDRV_IMPORT_FROM(n8080)

	/* basic machine hardware */
	MDRV_MACHINE_INIT(helifire)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(helifire_readmem,helifire_writemem)
	MDRV_CPU_PORTS(sheriff_readport,sheriff_writeport)

	MDRV_CPU_MODIFY("sound")
	MDRV_CPU_MEMORY(sheriff_sound_readmem,sheriff_sound_writemem)
	MDRV_CPU_PORTS(helifire_sound_readport,helifire_sound_writeport)

	/* video hardware */
	MDRV_VIDEO_START(helifire)
	MDRV_VIDEO_UPDATE(helifire)
	MDRV_VIDEO_EOF(helifire)
	MDRV_PALETTE_LENGTH(0x400 + 8)
	MDRV_PALETTE_INIT(helifire)

MACHINE_DRIVER_END


INPUT_PORTS_START( spacefev )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_COIN1 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_PLAYER2 )
	PORT_BITX(0x08, 0x00, 0, "Game A", KEYCODE_Q, IP_JOY_NONE )
	PORT_BITX(0x10, 0x00, 0, "Game B", KEYCODE_W, IP_JOY_NONE )
	PORT_BITX(0x20, 0x00, 0, "Game C", KEYCODE_E, IP_JOY_NONE )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED ) /* enables diagnostic ROM at $1c00 */

	PORT_START
	PORT_DIPNAME( 0x03, 0x00, DEF_STR( Lives ))
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x03, "6" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x04, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x08, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x10, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x20, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x40, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x80, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))

	PORT_START

INPUT_PORTS_END


INPUT_PORTS_START( highsplt )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_COIN1 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_PLAYER2 )
	PORT_BITX(0x08, 0x00, 0, "Game A", KEYCODE_Q, IP_JOY_NONE )
	PORT_BITX(0x10, 0x00, 0, "Game B", KEYCODE_W, IP_JOY_NONE )
	PORT_BITX(0x20, 0x00, 0, "Game C", KEYCODE_E, IP_JOY_NONE )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED ) /* enables diagnostic ROM at $2000 */

	PORT_START
	PORT_DIPNAME( 0x03, 0x00, DEF_STR( Lives ))
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x03, "6" )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Bonus_Life ))
	PORT_DIPSETTING(    0x00, "1500" )
	PORT_DIPSETTING(    0x04, "2000" )
	PORT_DIPSETTING(    0x08, "3000" )
	PORT_DIPSETTING(    0x0c, "4000" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x10, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x20, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x40, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x80, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))

	PORT_START

INPUT_PORTS_END


INPUT_PORTS_START( spacelnc )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_2WAY )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_2WAY )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_COIN1 )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_2WAY | IPF_PLAYER2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_2WAY | IPF_PLAYER2 )
	PORT_BITX(0x08, 0x00, 0, "Game A", KEYCODE_Q, IP_JOY_NONE )
	PORT_BITX(0x10, 0x00, 0, "Game B", KEYCODE_W, IP_JOY_NONE )
	PORT_BITX(0x20, 0x00, 0, "Game C", KEYCODE_E, IP_JOY_NONE )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED ) /* enables diagnostic ROM at $2000 */

	PORT_START
	PORT_DIPNAME( 0x03, 0x00, DEF_STR( Lives ))
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x03, "6" )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Bonus_Life ))
	PORT_DIPSETTING(    0x00, "1000" )
	PORT_DIPSETTING(    0x04, "3000" )
	PORT_DIPSETTING(    0x08, "5000" )
	PORT_DIPSETTING(    0x0c, "8000" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x10, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x20, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x40, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x80, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))

	PORT_START

INPUT_PORTS_END


INPUT_PORTS_START( sheriff )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_RIGHT )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_LEFT )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_UP )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_DOWN )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_RIGHT )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_LEFT )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_UP )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_DOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_RIGHT  | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_LEFT   | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_UP     | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_DOWN   | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_RIGHT | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_LEFT  | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_UP    | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_DOWN  | IPF_COCKTAIL )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP1 */
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP2 */
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP3 enables diagnostic ROM at $2400 */
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_COIN1 )

	PORT_START
	PORT_DIPNAME( 0x03, 0x00, DEF_STR( Lives ))
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x03, "6" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x04, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x08, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x10, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x20, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x40, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))
INPUT_PORTS_END


INPUT_PORTS_START( bandido )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_RIGHT )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_LEFT )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_UP )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICKLEFT_DOWN )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_RIGHT )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_LEFT )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_UP )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_JOYSTICKRIGHT_DOWN )

	PORT_START

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP1 */
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP2 */
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP3 enables diagnostic ROM at $2400 */
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_COIN1 )

	PORT_START
	PORT_DIPNAME( 0x03, 0x00, DEF_STR( Lives ))
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x03, "6" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Coinage ))
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ))
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ))
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x08, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x10, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x20, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x20, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Unused ))
	PORT_DIPSETTING(    0x40, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Unknown )) /* don't know if this is used */
	PORT_DIPSETTING(    0x80, DEF_STR( Off ))
	PORT_DIPSETTING(    0x00, DEF_STR( On ))
INPUT_PORTS_END


INPUT_PORTS_START( helifire )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT  | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP    | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN  | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x02, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_BIT( 0x04, IP_ACTIVE_HIGH, IPT_START1 )
	PORT_BIT( 0x08, IP_ACTIVE_HIGH, IPT_START2 )
	PORT_BIT( 0x10, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP1 */
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP2 */
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_UNUSED ) /* EXP3 */
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_COIN1 )

	PORT_START
	PORT_DIPNAME( 0x03, 0x00, DEF_STR( Lives ))
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x03, "6" )
	PORT_DIPNAME( 0x0c, 0x00, DEF_STR( Bonus_Life ))
	PORT_DIPSETTING(    0x00, "5000" )
	PORT_DIPSETTING(    0x04, "6000" )
	PORT_DIPSETTING(    0x08, "8000" )
	PORT_DIPSETTING(    0x0c, "10000" )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Coinage ))
	PORT_DIPSETTING(    0x10, DEF_STR( 2C_1C ))
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ))
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Cabinet ))
	PORT_DIPSETTING(    0x80, DEF_STR( Upright ))
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ))

	/* potentiometers */

	PORT_START	/* 04 */
	PORT_DIPNAME( 0xff, 0x50, "VR1 sun brightness" )
	PORT_DIPSETTING(    0x00, "00" )
	PORT_DIPSETTING(    0x10, "10" )
	PORT_DIPSETTING(    0x20, "20" )
	PORT_DIPSETTING(    0x30, "30" )
	PORT_DIPSETTING(    0x40, "40" )
	PORT_DIPSETTING(    0x50, "50" )
	PORT_DIPSETTING(    0x60, "60" )
	PORT_DIPSETTING(    0x70, "70" )

	PORT_START	/* 05 */
	PORT_DIPNAME( 0xff, 0x00, "VR2 sea brightness" )
	PORT_DIPSETTING(    0x00, "00" )
	PORT_DIPSETTING(    0x10, "10" )
	PORT_DIPSETTING(    0x20, "20" )
	PORT_DIPSETTING(    0x30, "30" )
	PORT_DIPSETTING(    0x40, "40" )
	PORT_DIPSETTING(    0x50, "50" )
	PORT_DIPSETTING(    0x60, "60" )
	PORT_DIPSETTING(    0x70, "70" )

INPUT_PORTS_END


ROM_START( spacefev )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )
	ROM_LOAD( "sf.f1",    0x0000, 0x0400, 0x35f295bd )
	ROM_LOAD( "sf.f2",    0x0400, 0x0400, 0x0c633f4c )
	ROM_LOAD( "sf.g1",    0x0800, 0x0400, 0xf3d851cb )
	ROM_LOAD( "sf.g2",    0x0c00, 0x0400, 0x1faef63a )
	ROM_LOAD( "sf.h1",    0x1000, 0x0400, 0xb365389d )
	ROM_LOAD( "sf.h2",    0x1400, 0x0400, 0xa163e800 )
	ROM_LOAD( "sf.i1",    0x1800, 0x0400, 0x00027be2 )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "sf.snd",   0x0000, 0x0400, 0x95c2c1ee )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "sf.prm",   0x0000, 0x0020, 0xc5914ec1 )
ROM_END

ROM_START( spacefva )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )
	ROM_LOAD( "sf-a.f1",  0x0000, 0x0400, 0x7fa305e8 )
	ROM_LOAD( "sf-a.f2",  0x0400, 0x0400, 0x7c1429aa )
	ROM_LOAD( "sf-a.g1",  0x0800, 0x0400, 0x75f6efc1 )
	ROM_LOAD( "sf-a.g2",  0x0c00, 0x0400, 0xfb6bcf4a )
	ROM_LOAD( "sf-a.h1",  0x1000, 0x0400, 0x3beef037 )
	ROM_LOAD( "sf-a.h2",  0x1400, 0x0400, 0xbddbc94f )
	ROM_LOAD( "sf-a.i1",  0x1800, 0x0400, 0x437786c5 )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "sf.snd",   0x0000, 0x0400, 0x95c2c1ee )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "sf.prm",   0x0000, 0x0020, 0xc5914ec1 )
ROM_END

ROM_START( highsplt )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )
	ROM_LOAD( "hs.f1",    0x0000, 0x0400, 0xb8887351 )
	ROM_LOAD( "hs.f2",    0x0400, 0x0400, 0xcda933a7 )
	ROM_LOAD( "hs.g1",    0x0800, 0x0400, 0xde17578a )
	ROM_LOAD( "hs.g2",    0x0c00, 0x0400, 0xf1a90948 )
	ROM_LOAD( "hs.h1",    0x1000, 0x0400, 0xeefb4273 )
	ROM_LOAD( "hs.h2",    0x1400, 0x0400, 0xe91703e8 )
	ROM_LOAD( "hs.i1",    0x1800, 0x0400, 0x41e18df9 )
	ROM_LOAD( "hs.i2",    0x1c00, 0x0400, 0xeff9f82d )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "hs.snd",   0x0000, 0x0400, 0x939e01d4 )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "sf.prm",   0x0000, 0x0020, 0xc5914ec1 )
ROM_END

ROM_START( highspla )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "hs.f1",    0x0000, 0x0400, 0xb8887351 )
	ROM_LOAD( "hs.f2",    0x0400, 0x0400, 0xcda933a7 )
	ROM_LOAD( "hs.g1",    0x0800, 0x0400, 0xde17578a )
	ROM_LOAD( "hs.g2",    0x0c00, 0x0400, 0xf1a90948 )
	ROM_LOAD( "hs-a.h1",  0x1000, 0x0400, 0xb0505da3 )
	ROM_LOAD( "hs.h2",    0x1400, 0x0400, 0xe91703e8 )
	ROM_LOAD( "hs-a.i1",  0x1800, 0x0400, 0xaa36b25d )
	ROM_LOAD( "hs.i2",    0x1c00, 0x0400, 0xeff9f82d )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "hs.snd",   0x0000, 0x0400, 0x939e01d4 )

	ROM_REGION( 0x0020, REGION_PROMS, 0 )
	ROM_LOAD( "sf.prm",   0x0000, 0x0020, 0xc5914ec1 )
ROM_END

ROM_START( spacelnc )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )
	ROM_LOAD( "sl.f1",    0x0000, 0x0400, 0x6ad59e40 )
	ROM_LOAD( "sl.f2",    0x0400, 0x0400, 0x2de568e2 )
	ROM_LOAD( "sl.g1",    0x0800, 0x0400, 0x06d0ab36 )
	ROM_LOAD( "sl.g2",    0x0c00, 0x0400, 0x73ac4fe6 )
	ROM_LOAD( "sl.h1",    0x1000, 0x0400, 0x7f42a94b )
	ROM_LOAD( "sl.h2",    0x1400, 0x0400, 0x04b7a5f9 )
	ROM_LOAD( "sl.i1",    0x1800, 0x0400, 0xd30007a3 )
	ROM_LOAD( "sl.i2",    0x1c00, 0x0400, 0x640ffd2f )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "sl.snd",   0x0000, 0x0400, 0x8e1ff929 )

	ROM_REGION( 0x0040, REGION_PROMS, ROMREGION_ERASE00 )
	ROM_LOAD( "sl.prm",   0x0020, 0x0020, 0 )
ROM_END

ROM_START( sheriff )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )
	ROM_LOAD( "sh.f1",    0x0000, 0x0400, 0xe79df6e8 )
	ROM_LOAD( "sh.f2",    0x0400, 0x0400, 0xda67721a )
	ROM_LOAD( "sh.g1",    0x0800, 0x0400, 0x3fb7888e )
	ROM_LOAD( "sh.g2",    0x0c00, 0x0400, 0x585fcfee )
	ROM_LOAD( "sh.h1",    0x1000, 0x0400, 0xe59eab52 )
	ROM_LOAD( "sh.h2",    0x1400, 0x0400, 0x79e69a6a )
	ROM_LOAD( "sh.i1",    0x1800, 0x0400, 0xdda7d1e8 )
	ROM_LOAD( "sh.i2",    0x1c00, 0x0400, 0x5c5f3f86 )
	ROM_LOAD( "sh.j1",    0x2000, 0x0400, 0x0aa8b79a )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "sh.snd",   0x0000, 0x0400, 0x75731745 )

	ROM_REGION( 0x0400, REGION_PROMS, 0 )
	ROM_LOAD( "sh.prm",   0x0000, 0x0400, 0 )
ROM_END

ROM_START( bandido )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )
	ROM_LOAD( "sh-a.f1",  0x0000, 0x0400, 0xaec94829 )
	ROM_LOAD( "sh.f2",    0x0400, 0x0400, 0xda67721a )
	ROM_LOAD( "sh.g1",    0x0800, 0x0400, 0x3fb7888e )
	ROM_LOAD( "sh.g2",    0x0c00, 0x0400, 0x585fcfee )
	ROM_LOAD( "sh-a.h1",  0x1000, 0x0400, 0x5cb63677 )
	ROM_LOAD( "sh.h2",    0x1400, 0x0400, 0x79e69a6a )
	ROM_LOAD( "sh.i1",    0x1800, 0x0400, 0xdda7d1e8 )
	ROM_LOAD( "sh.i2",    0x1c00, 0x0400, 0x5c5f3f86 )
	ROM_LOAD( "sh.j1",    0x2000, 0x0400, 0x0aa8b79a )
	ROM_LOAD( "sh-a.j2",  0x2400, 0x0400, 0xa10b848a )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "sh.snd",   0x0000, 0x0400, 0x75731745 )

	ROM_REGION( 0x0400, REGION_PROMS, 0 )
	ROM_LOAD( "sh.prm",   0x0000, 0x0400, 0 )
ROM_END

ROM_START( helifire )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )
	ROM_LOAD( "hf.f1",    0x0000, 0x0400, 0x032f89ca )
	ROM_LOAD( "hf.f2",    0x0400, 0x0400, 0x2774e70f )
	ROM_LOAD( "hf.g1",    0x0800, 0x0400, 0xb5ad6e8a )
	ROM_LOAD( "hf.g2",    0x0c00, 0x0400, 0x5e015bf4 )
	ROM_LOAD( "hf.h1",    0x1000, 0x0400, 0x23bb4e5a )
	ROM_LOAD( "hf.h2",    0x1400, 0x0400, 0x358227c6 )
	ROM_LOAD( "hf.i1",    0x1800, 0x0400, 0x0c679f44 )
	ROM_LOAD( "hf.i2",    0x1c00, 0x0400, 0xd8b7a398 )
	ROM_LOAD( "hf.j1",    0x2000, 0x0400, 0x98ef24db )
	ROM_LOAD( "hf.j2",    0x2400, 0x0400, 0x5e2b5877 )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "hf.snd",   0x0000, 0x0400, 0x9d77a31f )
ROM_END

ROM_START( helifira )
	ROM_REGION( 0x8000, REGION_CPU1, 0 )
	ROM_LOAD( "hf-a.f1",  0x0000, 0x0400, 0x92c9d6c1 )
	ROM_LOAD( "hf-a.f2",  0x0400, 0x0400, 0xa264dde8 )
	ROM_LOAD( "hf.g1",    0x0800, 0x0400, 0xb5ad6e8a )
	ROM_LOAD( "hf-a.g2",  0x0c00, 0x0400, 0xa987ebcd )
	ROM_LOAD( "hf-a.h1",  0x1000, 0x0400, 0x25abcaf0 )
	ROM_LOAD( "hf.h2",    0x1400, 0x0400, 0x358227c6 )
	ROM_LOAD( "hf.i1",    0x1800, 0x0400, 0x0c679f44 )
	ROM_LOAD( "hf-a.i2",  0x1c00, 0x0400, 0x296610fd )
	ROM_LOAD( "hf.j1",    0x2000, 0x0400, 0x98ef24db )
	ROM_LOAD( "hf.j2",    0x2400, 0x0400, 0x5e2b5877 )

	ROM_REGION( 0x0400, REGION_CPU2, 0 )
	ROM_LOAD( "hf.snd",   0x0000, 0x0400, 0x9d77a31f )
ROM_END


GAME (1979, spacefev, 0,        spacefev, spacefev, 0, ROT0, "Nintendo", "Space Fever (set 1)" )
GAME (1979, spacefva, spacefev, spacefev, spacefev, 0, ROT0, "Nintendo", "Space Fever (set 2)" )
GAME (1979, highsplt, 0,        spacefev, highsplt, 0, ROT0, "Nintendo", "Space Fever High Splitter (set 1)" )
GAME (1979, highspla, highsplt, spacefev, highsplt, 0, ROT0, "Nintendo", "Space Fever High Splitter (set 2)" )
GAMEX(1979, spacelnc, 0,        spacefev, spacelnc, 0, ROT0, "Nintendo", "Space Launcher", GAME_WRONG_COLORS )
GAME (1979, sheriff,  0,        sheriff,  sheriff,  0, ROT0, "Nintendo", "Sheriff" )
GAME (1980, bandido,  sheriff,  sheriff,  bandido,  0, ROT0, "Exidy",    "Bandido" )
GAMEX(1980, helifire, 0,        helifire, helifire, 0, ROT0, "Nintendo", "HeliFire (set 1)", GAME_IMPERFECT_SOUND | GAME_IMPERFECT_GRAPHICS | GAME_NO_COCKTAIL )
GAMEX(1980, helifira, helifire, helifire, helifire, 0, ROT0, "Nintendo", "HeliFire (set 2)", GAME_IMPERFECT_SOUND | GAME_IMPERFECT_GRAPHICS | GAME_NO_COCKTAIL )
