/***************************************************************************

Some Dynax games using the second version of their blitter

driver by Luca Elia and Nicola Salmoria

CPU:    Z80
Sound:  various
VDP:    HD46505SP (6845) (CRT controller)
Custom: TC17G032AP-0246 (blitter)

----------------------------------------------------------------------------------------
Year + Game					Board(s)			Sound						Palette
----------------------------------------------------------------------------------------
88 Hana no Mai				D1610088L1			AY8912 YM2203        M5205	PROM
88 Hana Kochou				D201901L			AY8912 YM2203        M5205	PROM
89 Hana Oriduru				D2304268L			AY8912        YM2413 M5205	RAM
89 Dragon Punch				D24?		 		       YM2203				PROM
89 Mahjong Friday			D2607198L1			              YM2413		PROM
89 Jantouki					D2711078L-B			**TODO** (1)
89 Sports Match				D31?		 	           YM2203				PROM
90 Mahjong Campus Hunting	D3312108L1-1 + DXB?	AY8912        YM2413 M5205	RAM
90 7jigen no Youseitachi	D3707198L1   + DXB?	AY8912        YM2413 M5205	RAM
91 Mahjong Yarunara			D4508308L-2  + DXB	AY8912        YM2413 M5205	RAM
91 Mahjong Angels			D6107068L-1  + DXB	AY8912        YM2413 M5205	RAM
91 Mahjong Dial Q2			D5212298L-1		                  YM2413		PROM
92 Quiz TV Gassyuukoku Q&Q	D6410288L-1  + DXB	AY8912        YM2413 M5205	RAM
94 Maya											      YM2203				PROM
----------------------------------------------------------------------------------------
DXB = Dynax Motherboard System D5512068L1-1/2

(1) quite different from the others: it has a slave Z80 and *two* blitters

Notes:
- In some games (drgpunch etc) there's a more complete service mode. To enter
  it, set the service mode dip switch and reset keeping start1 pressed.
  In hnkochou, keep F2 pressed and reset.

- sprtmtch and drgpunch are "clones", but the gfx are very different; sprtmtch
  is a trimmed down version, without all animations between levels.

- according to the readme, mjfriday should have a M5205. However there don't seem to be
  accesses to it, and looking at the ROMs I don't see ADPCM data. Note that apart from a
  minor difference in the memory map mjfriday and mjdialq2 are identical, and mjdialq2
  doesn't have a 5205 either. Therefore, I think it's either a mistake in the readme or
  the chip is on the board but unused.

TODO:
- Inputs are grossly mapped, especially for the card games.

- mjdialq2: the title screen is corrupted by the scrolling logo. This would be
  fixed by not wrapping around when drawing bpast the bottom of the bitmap,
  but doing so would break the last picture of gal 6 (which is scrolled up so
  that the bottom of the bitmap is near the middle of the screen).

- In the interleaved games, "reverse write" test in service mode is wrong due to
  interleaving. Correct behaviour? None of these games has a "flip screen" dip
  switch, while mjfriday and mjdialq2, which aren't interleaved, have it.

- Palette banking is not correct, see quiztvqq cross hatch test.

- 7jigen: priority 0x30 is ok when used in the "gals check", but is wrong during
  attract mode, where the girl is hidden by the background. Another possible
  priority issue in attract mode is when the balls scroll over the devil.

- mjangels: the title screen is wrong. Is this related to scroll_high being 1?
  (it usually is either 0 or 3)

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "includes/dynax.h"


/***************************************************************************


								Interrupts


***************************************************************************/

/***************************************************************************
								Sports Match
***************************************************************************/

UINT8 dynax_blitter_irq;
UINT8 dynax_sound_irq;
UINT8 dynax_vblank_irq;

/* It runs in IM 0, thus needs an opcode on the data bus */
void sprtmtch_update_irq(void)
{
	int irq	=	((dynax_sound_irq)   ? 0x08 : 0) |
				((dynax_vblank_irq)  ? 0x10 : 0) |
				((dynax_blitter_irq) ? 0x20 : 0) ;
	cpu_set_irq_line_and_vector(0, 0, irq ? ASSERT_LINE : CLEAR_LINE, 0xc7 | irq); /* rst $xx */
}

WRITE_HANDLER( dynax_vblank_ack_w )
{
	dynax_vblank_irq = 0;
	sprtmtch_update_irq();
}

WRITE_HANDLER( dynax_blitter_ack_w )
{
	dynax_blitter_irq = 0;
	sprtmtch_update_irq();
}

INTERRUPT_GEN( sprtmtch_vblank_interrupt )
{
	dynax_vblank_irq = 1;
	sprtmtch_update_irq();
}

void sprtmtch_sound_callback(int state)
{
	dynax_sound_irq = state;
	sprtmtch_update_irq();
}

/***************************************************************************


								Memory Maps


***************************************************************************/

/***************************************************************************
								Sports Match
***************************************************************************/

static WRITE_HANDLER( dynax_coincounter_0_w )
{
	coin_counter_w(0, data & 1);
	if (data & ~1)
		logerror("CPU#0 PC %06X: Warning, coin counter 0 <- %02X\n", activecpu_get_pc(), data);
}
static WRITE_HANDLER( dynax_coincounter_1_w )
{
	coin_counter_w(1, data & 1);
	if (data & ~1)
		logerror("CPU#0 PC %06X: Warning, coin counter 1 <- %02X\n", activecpu_get_pc(), data);
}

static READ_HANDLER( ret_ff )	{	return 0xff;	}


static int keyb;

static READ_HANDLER( hanamai_keyboard_0_r )
{
	int res = 0x3f;

	/* the game reads all rows at once (keyb = 0) to check if a key is pressed */
	if (~keyb & 0x01) res &= readinputport(3);
	if (~keyb & 0x02) res &= readinputport(4);
	if (~keyb & 0x04) res &= readinputport(5);
	if (~keyb & 0x08) res &= readinputport(6);
	if (~keyb & 0x10) res &= readinputport(7);

	return res;
}

static READ_HANDLER( hanamai_keyboard_1_r )
{
	return 0x3f;
}

static WRITE_HANDLER( hanamai_keyboard_w )
{
	keyb = data;
}


static WRITE_HANDLER( dynax_rombank_w )
{
	data8_t *ROM = memory_region(REGION_CPU1);
	cpu_setbank(1,&ROM[0x08000+0x8000*(data & 0x0f)]);
}


static int hnoridur_bank;

static WRITE_HANDLER( hnoridur_rombank_w )
{
	data8_t *ROM = memory_region(REGION_CPU1) + 0x10000 + 0x8000*data;
//logerror("%04x: rom bank = %02x\n",activecpu_get_pc(),data);
	cpu_setbank(1,ROM);
	hnoridur_bank = data;
}

static data8_t palette_ram[16*256*2];
static int palbank;

static WRITE_HANDLER( hnoridur_palbank_w )
{
	palbank = data & 0x0f;
	dynax_blit_palbank_w(0,data);
}

static WRITE_HANDLER( hnoridur_palette_w )
{
	switch (hnoridur_bank)
	{
		case 0x10:
			if (offset >= 0x100) return;
			palette_ram[256*palbank + offset + 16*256] = data;
			break;

		case 0x14:
			if (offset >= 0x100) return;
			palette_ram[256*palbank + offset] = data;
			break;

		// hnoridur: R/W RAM
		case 0x18:
		{
			data8_t *RAM = memory_region(REGION_CPU1) + 0x10000 + hnoridur_bank * 0x8000;
			RAM[offset] = data;
			return;
		}

		default:
			usrintf_showmessage("palette_w with bank = %02x",hnoridur_bank);
			break;
	}

	{
		int x =	(palette_ram[256*palbank + offset]<<8) + palette_ram[256*palbank + offset + 16*256];
		/* The bits are in reverse order! */
		int r = BITSWAP8((x >>  0) & 0x1f, 7,6,5, 0,1,2,3,4 );
		int g = BITSWAP8((x >>  5) & 0x1f, 7,6,5, 0,1,2,3,4 );
		int b = BITSWAP8((x >> 10) & 0x1f, 7,6,5, 0,1,2,3,4 );
		r =  (r << 3) | (r >> 2);
		g =  (g << 3) | (g >> 2);
		b =  (b << 3) | (b >> 2);
		palette_set_color(256*palbank + offset,r,g,b);
	}
}

static WRITE_HANDLER( yarunara_palette_w )
{
	int addr = 512*palbank + offset;

	switch (hnoridur_bank)
	{
		case 0x10:
			palette_ram[addr] = data;
			break;

		case 0x1c:	// RTC
			return;

		default:
			usrintf_showmessage("palette_w with bank = %02x",hnoridur_bank);
			return;
	}

	{
		int br = palette_ram[addr & ~0x10];		// bbbrrrrr
		int bg = palette_ram[addr | 0x10];		// bb0ggggg
		int r = br & 0x1f;
		int g = bg & 0x1f;
		int b = ((bg & 0xc0)>>3) | ((br & 0xe0)>>5);
		r =  (r << 3) | (r >> 2);
		g =  (g << 3) | (g >> 2);
		b =  (b << 3) | (b >> 2);
		palette_set_color( 256*palbank + ((offset&0xf)|((offset&0x1e0)>>1)) ,r,g,b);
	}
}

static WRITE_HANDLER( nanajign_palette_w )
{
	switch (hnoridur_bank)
	{
		case 0x10:
			palette_ram[256*palbank + offset + 16*256] = data;
			break;

		case 0x14:
			palette_ram[256*palbank + offset] = data;
			break;

		default:
			usrintf_showmessage("palette_w with bank = %02x",hnoridur_bank);
			break;
	}

	{
		int bg = palette_ram[256*palbank + offset];
		int br = palette_ram[256*palbank + offset + 16*256];
		int r = br & 0x1f;
		int g = bg & 0x1f;
		int b = ((bg & 0xc0)>>3) | ((br & 0xe0)>>5);
		r =  (r << 3) | (r >> 2);
		g =  (g << 3) | (g >> 2);
		b =  (b << 3) | (b >> 2);
		palette_set_color(256*palbank + offset,r,g,b);
	}
}

static int msm5205next;
static int resetkludge;

static void adpcm_int(int data)
{
	static int toggle;

	MSM5205_data_w(0,msm5205next >> 4);
	msm5205next<<=4;

	toggle = 1 - toggle;
	if (toggle)
	{
		if (resetkludge)	// don't know what's wrong, but NMIs when the 5205 is reset make the game crash
		cpu_set_nmi_line(0,PULSE_LINE);
	}
}

static WRITE_HANDLER( adpcm_data_w )
{
	msm5205next = data;
}

static WRITE_HANDLER( adpcm_reset_w )
{
	resetkludge = data & 1;
	MSM5205_reset_w(0,~data & 1);
}

static MACHINE_INIT( adpcm )
{
	/* start with the MSM5205 reset */
	resetkludge = 0;
	MSM5205_reset_w(0,1);
}

static WRITE_HANDLER( yarunara_layer_half_w )
{
	hanamai_layer_half_w(0,data >> 1);
}
static WRITE_HANDLER( yarunara_layer_half2_w )
{
	hnoridur_layer_half2_w(0,data >> 1);
}
WRITE_HANDLER( nanajign_layer_half_w )
{
	hanamai_layer_half_w(0,~data);
}


static MEMORY_READ_START( sprtmtch_readmem )
	{ 0x0000, 0x6fff, MRA_ROM },
	{ 0x7000, 0x7fff, MRA_RAM },
	{ 0x8000, 0xffff, MRA_BANK1 },
MEMORY_END
static MEMORY_WRITE_START( sprtmtch_writemem )
	{ 0x0000, 0x6fff, MWA_ROM },
	{ 0x7000, 0x7fff, MWA_RAM },
	{ 0x7000, 0x7fff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x8000, 0xffff, MWA_ROM },
MEMORY_END


static MEMORY_READ_START( hnoridur_readmem )
	{ 0x0000, 0x6fff, MRA_ROM },
	{ 0x7000, 0x7fff, MRA_RAM },
	{ 0x8000, 0xffff, MRA_BANK1 },
MEMORY_END
static MEMORY_WRITE_START( hnoridur_writemem )
	{ 0x0000, 0x6fff, MWA_ROM },
	{ 0x7000, 0x7fff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x8000, 0xffff, hnoridur_palette_w },
MEMORY_END


static MEMORY_READ_START( mcnpshnt_readmem )
	{ 0x0000, 0x5fff, MRA_ROM },
	{ 0x6000, 0x6fff, MRA_RAM },
	{ 0x7000, 0x7fff, MRA_RAM },
	{ 0x8000, 0xffff, MRA_BANK1 },
MEMORY_END
static MEMORY_WRITE_START( mcnpshnt_writemem )
	{ 0x0000, 0x5fff, MWA_ROM },
	{ 0x6000, 0x6fff, MWA_RAM },
	{ 0x7000, 0x7fff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x8000, 0x8fff, hnoridur_palette_w },
MEMORY_END


static MEMORY_READ_START( mjdialq2_readmem )
	{ 0x0000, 0x07ff, MRA_ROM },
	{ 0x0800, 0x1fff, MRA_RAM },
	{ 0x2000, 0x7fff, MRA_ROM },
	{ 0x8000, 0xffff, MRA_BANK1 },
MEMORY_END
static MEMORY_WRITE_START( mjdialq2_writemem )
	{ 0x0000, 0x07ff, MWA_ROM },
	{ 0x0800, 0x0fff, MWA_RAM },
	{ 0x1000, 0x1fff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x2000, 0x7fff, MWA_ROM },
	{ 0x8000, 0xffff, MWA_ROM },
MEMORY_END


static MEMORY_READ_START( yarunara_readmem )
	{ 0x0000, 0x5fff, MRA_ROM		},	// ROM
	{ 0x6000, 0x6fff, MRA_RAM		},	// RAM
	{ 0x7000, 0x7fff, MRA_RAM		},	// NVRAM
	{ 0x8000, 0xffff, MRA_BANK1		},	// ROM (Banked)
MEMORY_END
static MEMORY_WRITE_START( yarunara_writemem )
	{ 0x0000, 0x5fff, MWA_ROM					},	// ROM
	{ 0x6000, 0x6fff, MWA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x8000, 0x81ff, yarunara_palette_w		},	// Palette or RTC
	{ 0x8000, 0xffff, MWA_ROM					},	// ROM (Banked)
MEMORY_END


static MEMORY_READ_START( nanajign_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},	// ROM
	{ 0x6000, 0x7fff, MRA_RAM					},	// RAM
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM (Banked)
MEMORY_END
static MEMORY_WRITE_START( nanajign_writemem )
	{ 0x0000, 0x5fff, MWA_ROM					},	// ROM
	{ 0x6000, 0x7fff, MWA_RAM					},	// RAM
	{ 0x8000, 0x80ff, nanajign_palette_w		},
	{ 0x8000, 0xffff, MWA_ROM					},	// ROM (Banked)
MEMORY_END





static PORT_READ_START( hanamai_readport )
	{ 0x60, 0x60, hanamai_keyboard_0_r		},	// P1
	{ 0x61, 0x61, hanamai_keyboard_1_r		},	// P2
	{ 0x62, 0x62, input_port_2_r			},	// Coins
	{ 0x63, 0x63, ret_ff					},	// ?
	{ 0x78, 0x78, YM2203_status_port_0_r	},	// YM2203
	{ 0x79, 0x79, YM2203_read_port_0_r		},	// 2 x DSW
PORT_END
static PORT_WRITE_START( hanamai_writeport )
	{ 0x00, 0x00, dynax_extra_scrollx_w		},	// screen scroll X
	{ 0x20, 0x20, dynax_extra_scrolly_w		},	// screen scroll Y
	{ 0x41, 0x47, dynax_blitter_rev2_w		},	// Blitter
	{ 0x50, 0x50, dynax_rombank_w			},	// BANK ROM Select	hnkochou only
	{ 0x64, 0x64, hanamai_keyboard_w		},	// keyboard row select
	{ 0x65, 0x65, dynax_rombank_w			},	// BANK ROM Select  hanamai only
	{ 0x66, 0x66, dynax_vblank_ack_w		},	// VBlank IRQ Ack
	{ 0x67, 0x67, adpcm_data_w				},	// MSM5205 data
	{ 0x68, 0x68, dynax_layer_enable_w		},	// Layers Enable
	{ 0x69, 0x69, hanamai_priority_w		},	// layer priority
	{ 0x6a, 0x6a, dynax_blit_dest_w			},	// Destination Layer
	{ 0x6b, 0x6b, dynax_blit_pen_w			},	// Destination Pen
	{ 0x6c, 0x6c, dynax_blit_palette01_w	},	// Layers Palettes (Low Bits)
	{ 0x6d, 0x6d, dynax_blit_palette2_w		},	//
	{ 0x6e, 0x6e, dynax_blit_backpen_w		},	// Background Color
	{ 0x70, 0x70, adpcm_reset_w				},	// MSM5205 reset
	{ 0x71, 0x71, dynax_flipscreen_w		},	// Flip Screen
	{ 0x72, 0x72, dynax_coincounter_0_w		},	// Coin Counters
	{ 0x73, 0x73, dynax_coincounter_1_w		},	//
	{ 0x74, 0x74, dynax_blitter_ack_w		},	// Blitter IRQ Ack
	{ 0x76, 0x76, dynax_blit_palbank_w		},	// Layers Palettes (High Bit)
	{ 0x77, 0x77, hanamai_layer_half_w		},	// half of the interleaved layer to write to
	{ 0x78, 0x78, YM2203_control_port_0_w	},	// YM2203
	{ 0x79, 0x79, YM2203_write_port_0_w		},	//
	{ 0x7a, 0x7a, AY8910_control_port_0_w	},	// AY8910
	{ 0x7b, 0x7b, AY8910_write_port_0_w		},	//
//	{ 0x7c, 0x7c, IOWP_NOP					},	// CRT Controller
//	{ 0x7d, 0x7d, IOWP_NOP					},	// CRT Controller
PORT_END


static PORT_READ_START( hnoridur_readport )
	{ 0x21, 0x21, input_port_2_r			},	// Coins
	{ 0x22, 0x22, hanamai_keyboard_1_r		},	// P2
	{ 0x23, 0x23, hanamai_keyboard_0_r		},	// P1
	{ 0x24, 0x24, input_port_1_r			},	// DSW2
	{ 0x25, 0x25, input_port_9_r			},	// DSW4
	{ 0x26, 0x26, input_port_8_r			},	// DSW3
	{ 0x36, 0x36, AY8910_read_port_0_r		},	// AY8910, DSW1
	{ 0x57, 0x57, ret_ff					},	// ?
PORT_END
static PORT_WRITE_START( hnoridur_writeport )
	{ 0x01, 0x07, dynax_blitter_rev2_w		},	// Blitter
	{ 0x20, 0x20, hanamai_keyboard_w		},	// keyboard row select
	{ 0x30, 0x30, adpcm_reset_w				},	// MSM5205 reset
	{ 0x32, 0x32, adpcm_data_w				},	// MSM5205 data
	{ 0x34, 0x34, YM2413_register_port_0_w	},	// YM2413
	{ 0x35, 0x35, YM2413_data_port_0_w		},	//
	{ 0x38, 0x38, AY8910_write_port_0_w		},	// AY8910
	{ 0x3a, 0x3a, AY8910_control_port_0_w	},	//
//	{ 0x10, 0x10, IOWP_NOP					},	// CRT Controller
//	{ 0x11, 0x11, IOWP_NOP					},	// CRT Controller
	{ 0x40, 0x40, dynax_blit_pen_w			},	// Destination Pen
	{ 0x41, 0x41, dynax_blit_dest_w			},	// Destination Layer
	{ 0x42, 0x42, dynax_blit_palette01_w	},	// Layers Palettes
	{ 0x43, 0x43, dynax_blit_palette2_w		},	//
	{ 0x44, 0x44, hanamai_priority_w		},	// layer priority and enable
	{ 0x45, 0x45, dynax_blit_backpen_w		},	// Background Color
	{ 0x47, 0x47, hnoridur_palbank_w		},
	{ 0x50, 0x50, dynax_extra_scrollx_w		},	// screen scroll X
	{ 0x51, 0x51, dynax_extra_scrolly_w		},	// screen scroll Y
	{ 0x54, 0x54, hnoridur_rombank_w		},	// BANK ROM Select
	{ 0x56, 0x56, dynax_vblank_ack_w		},	// VBlank IRQ Ack
	{ 0x60, 0x60, dynax_flipscreen_w		},	// Flip Screen
	{ 0x61, 0x61, hanamai_layer_half_w		},	// half of the interleaved layer to write to
	{ 0x62, 0x62, hnoridur_layer_half2_w		},	//
	{ 0x67, 0x67, dynax_blitter_ack_w		},	// Blitter IRQ Ack
	{ 0x70, 0x70, dynax_coincounter_0_w		},	// Coin Counters
	{ 0x71, 0x71, dynax_coincounter_1_w		},	//
PORT_END


// Almost identical to hnoridur
static PORT_READ_START( mcnpshnt_readport )
	{ 0x21, 0x21, input_port_2_r			},	// Coins
	{ 0x22, 0x22, hanamai_keyboard_1_r		},	// P2
	{ 0x23, 0x23, hanamai_keyboard_0_r		},	// P1
	{ 0x24, 0x24, input_port_0_r			},	// DSW2
	{ 0x26, 0x26, input_port_1_r			},	// DSW3
	{ 0x57, 0x57, ret_ff					},	// ?
PORT_END
static PORT_WRITE_START( mcnpshnt_writeport )
	{ 0x01, 0x07, dynax_blitter_rev2_w		},	// Blitter
//	{ 0x10, 0x10, IOWP_NOP					},	// CRT Controller
//	{ 0x11, 0x11, IOWP_NOP					},	// CRT Controller
	{ 0x20, 0x20, hanamai_keyboard_w		},	// keyboard row select
	{ 0x30, 0x30, adpcm_reset_w				},	// MSM5205 reset
	{ 0x32, 0x32, adpcm_data_w				},	// MSM5205 data
	{ 0x34, 0x34, YM2413_register_port_0_w	},	// YM2413
	{ 0x35, 0x35, YM2413_data_port_0_w		},	//
	{ 0x38, 0x38, AY8910_write_port_0_w		},	// AY8910
	{ 0x3a, 0x3a, AY8910_control_port_0_w	},	//
	{ 0x40, 0x40, dynax_blit_pen_w			},	// Destination Pen
	{ 0x41, 0x41, dynax_blit_dest_w			},	// Destination Layer
	{ 0x42, 0x42, dynax_blit_palette01_w	},	// Layers Palettes
	{ 0x43, 0x43, dynax_blit_palette2_w		},	//
	{ 0x44, 0x44, hanamai_priority_w		},	// layer priority and enable
	{ 0x45, 0x45, dynax_blit_backpen_w		},	// Background Color
	{ 0x47, 0x47, hnoridur_palbank_w		},
	{ 0x50, 0x50, dynax_extra_scrollx_w		},	// screen scroll X
	{ 0x51, 0x51, dynax_extra_scrolly_w		},	// screen scroll Y
	{ 0x54, 0x54, hnoridur_rombank_w		},	// BANK ROM Select
	{ 0x56, 0x56, dynax_vblank_ack_w		},	// VBlank IRQ Ack
	{ 0x60, 0x60, dynax_flipscreen_w		},	// Flip Screen
	{ 0x61, 0x61, nanajign_layer_half_w		},	// half of the interleaved layer to write to
	{ 0x62, 0x62, hnoridur_layer_half2_w		},	//
	{ 0x67, 0x67, dynax_blitter_ack_w		},	// Blitter IRQ Ack
	{ 0x70, 0x70, dynax_coincounter_0_w		},	// Coin Counters
	{ 0x71, 0x71, dynax_coincounter_1_w		},	//
PORT_END


static PORT_READ_START( sprtmtch_readport )
	{ 0x10, 0x10, YM2203_status_port_0_r	},	// YM2203
	{ 0x11, 0x11, YM2203_read_port_0_r		},	// 2 x DSW
	{ 0x20, 0x20, input_port_0_r			},	// P1
	{ 0x21, 0x21, input_port_1_r			},	// P2
	{ 0x22, 0x22, input_port_2_r			},	// Coins
	{ 0x23, 0x23, ret_ff					},	// ?
PORT_END
static PORT_WRITE_START( sprtmtch_writeport )
	{ 0x01, 0x07, dynax_blitter_rev2_w		},	// Blitter
	{ 0x10, 0x10, YM2203_control_port_0_w	},	// YM2203
	{ 0x11, 0x11, YM2203_write_port_0_w		},	//
//	{ 0x12, 0x12, IOWP_NOP					},	// CRT Controller
//	{ 0x13, 0x13, IOWP_NOP					},	// CRT Controller
	{ 0x30, 0x30, dynax_layer_enable_w		},	// Layers Enable
	{ 0x31, 0x31, dynax_rombank_w			},	// BANK ROM Select
	{ 0x32, 0x32, dynax_blit_dest_w			},	// Destination Layer
	{ 0x33, 0x33, dynax_blit_pen_w			},	// Destination Pen
	{ 0x34, 0x34, dynax_blit_palette01_w	},	// Layers Palettes (Low Bits)
	{ 0x35, 0x35, dynax_blit_palette2_w		},	//
	{ 0x36, 0x36, dynax_blit_backpen_w		},	// Background Color
	{ 0x37, 0x37, dynax_vblank_ack_w		},	// VBlank IRQ Ack
//	{ 0x40, 0x40, adpcm_reset_w				},	// MSM5205 reset
	{ 0x41, 0x41, dynax_flipscreen_w		},	// Flip Screen
	{ 0x42, 0x42, dynax_coincounter_0_w		},	// Coin Counters
	{ 0x43, 0x43, dynax_coincounter_1_w		},	//
	{ 0x44, 0x44, dynax_blitter_ack_w		},	// Blitter IRQ Ack
	{ 0x45, 0x45, dynax_blit_palbank_w		},	// Layers Palettes (High Bit)
PORT_END



static PORT_READ_START( mjfriday_readport )
	{ 0x63, 0x63, hanamai_keyboard_0_r		},	// P1
	{ 0x62, 0x62, hanamai_keyboard_1_r		},	// P2
	{ 0x61, 0x61, input_port_2_r			},	// Coins
	{ 0x64, 0x64, input_port_0_r			},	// DSW
	{ 0x67, 0x67, input_port_1_r			},	// DSW
PORT_END
static PORT_WRITE_START( mjfriday_writeport )
	{ 0x41, 0x47, dynax_blitter_rev2_w		},	// Blitter
//	{ 0x50, 0x50, IOWP_NOP					},	// CRT Controller
//	{ 0x51, 0x51, IOWP_NOP					},	// CRT Controller
	{ 0x60, 0x60, hanamai_keyboard_w		},	// keyboard row select
	{ 0x70, 0x70, YM2413_register_port_0_w	},	// YM2413
	{ 0x71, 0x71, YM2413_data_port_0_w		},	//
	{ 0x00, 0x00, dynax_blit_pen_w			},	// Destination Pen
	{ 0x01, 0x01, dynax_blit_palette01_w	},	// Layers Palettes (Low Bits)
	{ 0x02, 0x02, dynax_rombank_w			},	// BANK ROM Select
	{ 0x03, 0x03, dynax_blit_backpen_w		},	// Background Color
	{ 0x10, 0x11, mjdialq2_blit_dest_w		},	// Destination Layer
	{ 0x12, 0x12, dynax_blit_palbank_w		},	// Layers Palettes (High Bit)
	{ 0x13, 0x13, dynax_flipscreen_w		},	// Flip Screen
	{ 0x14, 0x14, dynax_coincounter_0_w		},	// Coin Counters
	{ 0x15, 0x15, dynax_coincounter_1_w		},	//
	{ 0x16, 0x17, mjdialq2_layer_enable_w	},	// Layers Enable
//	{ 0x80, 0x80, IOWP_NOP					},	// IRQ ack?
PORT_END


static PORT_READ_START( nanajign_readport )
	{ 0x11, 0x11, input_port_2_r			},	// Coins
	{ 0x12, 0x12, hanamai_keyboard_1_r		},	// P2
	{ 0x13, 0x13, hanamai_keyboard_0_r		},	// P1
	{ 0x14, 0x14, input_port_0_r			},	// DSW1
	{ 0x15, 0x15, input_port_1_r			},	// DSW2
	{ 0x16, 0x16, input_port_8_r			},	// DSW3
PORT_END
static PORT_WRITE_START( nanajign_writeport )
	{ 0x00, 0x00, adpcm_reset_w				},	// MSM5205 reset
	{ 0x02, 0x02, adpcm_data_w				},	// MSM5205 data
	{ 0x04, 0x04, YM2413_register_port_0_w	},	// YM2413
	{ 0x05, 0x05, YM2413_data_port_0_w		},	//
	{ 0x08, 0x08, AY8910_write_port_0_w		},	// AY8910
	{ 0x0a, 0x0a, AY8910_control_port_0_w	},	//
	{ 0x10, 0x10, hanamai_keyboard_w		},	// keyboard row select
//	{ 0x20, 0x21, IOWP_NOP					},	// CRT Controller
	{ 0x31, 0x37, dynax_blitter_rev2_w		},	// Blitter
	{ 0x40, 0x40, dynax_coincounter_0_w		},	// Coin Counter
	{ 0x50, 0x50, dynax_flipscreen_w		},	// Flip Screen
	{ 0x51, 0x51, hanamai_layer_half_w		},	// half of the interleaved layer to write to
	{ 0x52, 0x52, hnoridur_layer_half2_w		},	//
	{ 0x57, 0x57, dynax_blitter_ack_w		},	// Blitter IRQ Ack
	{ 0x60, 0x60, dynax_extra_scrollx_w		},	// screen scroll X
	{ 0x62, 0x62, dynax_extra_scrolly_w		},	// screen scroll Y
	{ 0x6a, 0x6a, hnoridur_rombank_w		},	// BANK ROM Select
	{ 0x6c, 0x6c, dynax_vblank_ack_w		},	// VBlank IRQ Ack
	{ 0x70, 0x70, dynax_blit_pen_w			},	// Destination Pen
	{ 0x71, 0x71, dynax_blit_dest_w			},	// Destination Layer
	{ 0x72, 0x72, dynax_blit_palette01_w	},	// Layers Palettes
	{ 0x73, 0x73, dynax_blit_palette2_w		},	//
	{ 0x74, 0x74, hanamai_priority_w		},	// layer priority and enable
	{ 0x75, 0x75, dynax_blit_backpen_w		},	// Background Color
	{ 0x77, 0x77, hnoridur_palbank_w		},
PORT_END



/***************************************************************************
					Yarunara / Quiz TV Q&Q / Mahjong Angels
***************************************************************************/

static data8_t yarunara_select, yarunara_ip;
static WRITE_HANDLER( yarunara_input_w )
{
	switch (offset)
	{
		case 0:	yarunara_select = data;
				yarunara_ip = 0;
				break;

		case 1:	break;
	}

}

static READ_HANDLER( yarunara_input_r )
{
	switch (offset)
	{
		case 0:
		{
			switch( yarunara_select )
			{
				case 0x00:
					return readinputport(2);	// coins

				case 0x02:
					return 0xff;	// bit 7 must be 1. Bit 2?

				default:
					return 0xff;
			}
		}

		case 1:
		{
			switch( yarunara_select )
			{
				// player 2
				case 0x01:	//quiztvqq
				case 0x81:
					return readinputport(3 + 5 + yarunara_ip++);

				// player 1
				case 0x02:	//quiztvqq
				case 0x82:
					return readinputport(3 + yarunara_ip++);

				default:
					return 0xff;
			}
		}
	}
	return 0xff;
}

static WRITE_HANDLER( yarunara_rombank_w )
{
	UINT8 *rom = memory_region(REGION_CPU1) + 0x10000 + 0x8000 * data;
	cpu_setbank(1, rom);

	hnoridur_bank = data;
	if (data == 0x1c)
		rom[0x0d] = 0x00;	// RTC
}

WRITE_HANDLER( yarunara_flipscreen_w )
{
	dynax_flipscreen_w(0,(data&2)?1:0);
}

static PORT_READ_START( yarunara_readport )
	{ 0x02, 0x03, yarunara_input_r		},	// Controls
	{ 0x4c, 0x4c, input_port_0_r		},	// DSW 1
	{ 0x4f, 0x4f, input_port_1_r		},	// DSW 2
PORT_END

static PORT_WRITE_START( yarunara_writeport )
	{ 0x00, 0x01, yarunara_input_w			},	// Controls
	{ 0x11, 0x17, dynax_blitter_rev2_w		},	// Blitter
	{ 0x20, 0x20, adpcm_reset_w				},	// MSM5205 reset
	{ 0x22, 0x22, adpcm_data_w				},	// MSM5205 data
	{ 0x24, 0x24, YM2413_register_port_0_w	},	// YM2413
	{ 0x25, 0x25, YM2413_data_port_0_w		},	//
	{ 0x28, 0x28, AY8910_write_port_0_w		},	// AY8910
	{ 0x2a, 0x2a, AY8910_control_port_0_w	},	//
	{ 0x48, 0x48, dynax_extra_scrollx_w		},	// screen scroll X
	{ 0x49, 0x49, dynax_extra_scrolly_w		},	// screen scroll Y
	{ 0x4a, 0x4a, yarunara_rombank_w		},	// BANK ROM Select
	{ 0x4b, 0x4b, dynax_vblank_ack_w		},	// VBlank IRQ Ack

	{ 0x50, 0x50, yarunara_flipscreen_w		},
	{ 0x51, 0x51, yarunara_layer_half_w		},	// half of the interleaved layer to write to
	{ 0x52, 0x52, yarunara_layer_half2_w	},	//
	// 53 ?
	// 54 ?
	{ 0x57, 0x57, dynax_blitter_ack_w		},	// Blitter IRQ Ack
	{ 0x68, 0x68, dynax_blit_pen_w			},	// Destination Pen
	{ 0x69, 0x69, dynax_blit_dest_w			},	// Destination Layer
	{ 0x6a, 0x6a, dynax_blit_palette01_w	},	// Layers Palettes
	{ 0x6b, 0x6b, dynax_blit_palette2_w		},	//
	{ 0x6c, 0x6c, hanamai_priority_w		},	// layer priority and enable
	{ 0x6d, 0x6d, dynax_blit_backpen_w		},	// Background Color
	// 6e ?
PORT_END



/***************************************************************************

Lady Frog, or is it Dragon Punch, or is Lady Frog the name of a bootleg
Dragon Punch?

The program jumps straight away to an unmapped memory address. I don't know,
maybe there's a ROM missing.

roldfrog.001 contains
VIDEO COMPUTER SYSTEM  (C)1989 DYNAX INC  NAGOYA JAPAN  DRAGON PUNCH  VER. 1.30

***************************************************************************/

static MEMORY_READ16_START( roldfrog_readmem )
	{ 0x000000, 0x3fffff, MRA16_ROM },
	{ 0x881800, 0x881fff, MRA16_RAM },
	{ 0x840000, 0x840001, input_port_0_word_r },
	{ 0x840002, 0x840003, input_port_1_word_r },
	{ 0x840004, 0x840005, input_port_2_word_r },
	{ 0x840006, 0x840007, input_port_3_word_r },
	{ 0xffc000, 0xffffff, MRA16_RAM },
MEMORY_END

static MEMORY_WRITE16_START( roldfrog_writemem )
	{ 0x000000, 0x3fffff, MWA16_ROM },
	{ 0x800000, 0x83ffff, MWA16_RAM },
	{ 0x881800, 0x881fff, MWA16_RAM },
	{ 0xffc000, 0xffffff, MWA16_RAM },
MEMORY_END


static struct GfxLayout charlayout =
{
	8,8,
	RGN_FRAC(1,4),
	4,
	{ RGN_FRAC(0,4), RGN_FRAC(1,4), RGN_FRAC(2,4), RGN_FRAC(3,4) },
	{ STEP8(0,1) },
	{ STEP8(0,8) },
	8*8
};

static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX2, 0, &charlayout,  0, 1 },
	{ -1 } /* end of array */
};

/***************************************************************************


								Input Ports


***************************************************************************/

INPUT_PORTS_START( hanamai )
	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE )	// Test
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )	// Analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )	// Memory Reset
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "1",   KEYCODE_1_PAD,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "5",   KEYCODE_5_PAD,    IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Y",   KEYCODE_Y,        IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1                              )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "2",     KEYCODE_2_PAD,  IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "6",     KEYCODE_6_PAD,  IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,      IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "3",   KEYCODE_3_PAD,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "7",   KEYCODE_7_PAD,    IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "4",   KEYCODE_4_PAD,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "8",   KEYCODE_8_PAD,    IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Flip",   KEYCODE_X,    IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( hnkochou )
	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN2 )		// Pay
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )	// Test (there isn't a dip switch)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )	// Analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )	// Memory Reset
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN3 )		// Note
	PORT_BIT_IMPULSE( 0x40, IP_ACTIVE_LOW, IPT_COIN1, 2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1                              )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Flip",   KEYCODE_X,        IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( hnoridur )
	PORT_START	/* note that these are in reverse order wrt the others */
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )

	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE )	// Test
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )	// Analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )	// Memory Reset
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "1",   KEYCODE_1_PAD,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "5",   KEYCODE_5_PAD,    IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Y",   KEYCODE_Y,        IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1                              )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "2",     KEYCODE_2_PAD,  IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "6",     KEYCODE_6_PAD,  IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,      IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "3",   KEYCODE_3_PAD,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "7",   KEYCODE_7_PAD,    IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "4",   KEYCODE_4_PAD,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "8",   KEYCODE_8_PAD,    IP_JOY_NONE )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Flip",   KEYCODE_X,    IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( sprtmtch )
	PORT_START	// IN0 - Player 1
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	// IN1 - Player 2
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START	// IN2 - Coins
	PORT_BIT_IMPULSE(  0x01, IP_ACTIVE_LOW, IPT_COIN1, 10)
	PORT_BIT_IMPULSE(  0x02, IP_ACTIVE_LOW, IPT_COIN2, 10)
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_UNKNOWN  )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_UNKNOWN  )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_UNKNOWN  )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN  )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN  )

	PORT_START	// IN3 - DSW
	PORT_DIPNAME( 0x07, 0x07, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 5C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 1C_4C ) )
	PORT_DIPNAME( 0x38, 0x38, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 5C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x18, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x38, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x30, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x28, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x20, DEF_STR( 1C_4C ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	// IN4 - DSW
	PORT_DIPNAME( 0x07, 0x04, DEF_STR( Difficulty ) )	// Time
	PORT_DIPSETTING(    0x00, "1 (Easy)" )
	PORT_DIPSETTING(    0x01, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x03, "4" )
	PORT_DIPSETTING(    0x04, "5" )
	PORT_DIPSETTING(    0x05, "6" )
	PORT_DIPSETTING(    0x06, "7" )
	PORT_DIPSETTING(    0x07, "8 (Hard)" )
	PORT_DIPNAME( 0x18, 0x10, "Vs Time" )
	PORT_DIPSETTING(    0x18, "8 s" )
	PORT_DIPSETTING(    0x10, "10 s" )
	PORT_DIPSETTING(    0x08, "12 s" )
	PORT_DIPSETTING(    0x00, "14 s" )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-7" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 2-8" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END

INPUT_PORTS_START( mjfriday )
	PORT_START
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x04, 0x04, "PINFU with TSUMO" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x18, 0x18, "Player Strength" )
	PORT_DIPSETTING(    0x18, "Weak" )
	PORT_DIPSETTING(    0x10, "Normal" )
	PORT_DIPSETTING(    0x08, "Strong" )
	PORT_DIPSETTING(    0x00, "Strongest" )
	PORT_DIPNAME( 0x60, 0x60, "CPU Strength" )
	PORT_DIPSETTING(    0x60, "Weak" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Strong" )
	PORT_DIPSETTING(    0x00, "Strongest" )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_DIPNAME( 0x01, 0x01, "Auto TSUMO" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x02, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Gfx test" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )	// "17B"
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )	// "18B"
	PORT_SERVICE(0x04, IP_ACTIVE_LOW )	// Test (there isn't a dip switch)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )	// Analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )	// Memory Reset
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )	// "06B"
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )	// "18A"

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1                              )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Flip",   KEYCODE_X,        IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( mjdialq2 )
	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )	// "17B"
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )	// "18B"
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )	// Test (there isn't a dip switch)
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )	// Analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )	// Memory Reset
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )	// "06B"
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )	// "18A"

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1                              )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Flip",   KEYCODE_X,        IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( yarunara )
	PORT_START	// IN0 - DSW
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x08, "Easy" )		// 1,6
	PORT_DIPSETTING(    0x0c, "Normal" ) 	// 3,4
	PORT_DIPSETTING(    0x04, "Hard" )		// 5,2
	PORT_DIPSETTING(    0x00, "Hardest" )	// 7,0
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Scrambled comm.?" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Set Date" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN1 - DSW
	PORT_DIPNAME( 0x01, 0x01, "PINFU with TSUMO" )
	PORT_DIPSETTING(    0x01, DEF_STR( No ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 2-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3*" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BITX(    0x20, 0x20, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Choose Bonus", IP_KEY_NONE, IP_JOY_NONE )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	// IN2
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN  )	// "17B"
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN  )	// "18B"
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, "Test", KEYCODE_F1, IP_JOY_NONE )	// Test
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )	// Analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )	// Memory Reset
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN  )	// "06B"
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1    )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START	// IN3
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "A",     KEYCODE_A,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "E",     KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "I",     KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "M",     KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Kan",   KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1                                )

	PORT_START	// IN4
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )

	PORT_START	// IN5
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "C",     KEYCODE_C,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "G",     KEYCODE_G,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "K",     KEYCODE_K,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Chi",   KEYCODE_SPACE,    IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Ron",   KEYCODE_Z,        IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN6
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "D",     KEYCODE_D,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "H",     KEYCODE_H,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "L",     KEYCODE_L,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Pon",   KEYCODE_LALT,     IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN7
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "1",   KEYCODE_1_PAD,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "2",   KEYCODE_2_PAD,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "3",   KEYCODE_3_PAD,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "4",   KEYCODE_4_PAD,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "5",   KEYCODE_5_PAD,        IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "6",   KEYCODE_6_PAD,        IP_JOY_NONE )

	// Keyboard 2
	PORT_START	// IN8
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN9
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN10
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN11
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN12
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( quiztvqq )
	PORT_START	// IN0 - DSW
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x08, "Easy"    )
	PORT_DIPSETTING(    0x0c, "Normal"  )
	PORT_DIPSETTING(    0x04, "Hard"    )
	PORT_DIPSETTING(    0x00, "Hardest" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Voices" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x80, 0x80, "Set Date" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN1 - DSW
	PORT_DIPNAME( 0x03, 0x03, "Unknown 2-0&1" )
	PORT_DIPSETTING(    0x03, "0" )
	PORT_DIPSETTING(    0x02, "1" )
	PORT_DIPSETTING(    0x01, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3*" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4*" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	// IN2
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, "Test", KEYCODE_F1, IP_JOY_NONE )	// Test, during boot
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2   )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1   )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN3
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START	// IN4
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN5
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN6
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN7
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )

	// Keyboard 2
	PORT_START	// IN8
	PORT_BIT(0x01, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT(0x02, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT(0x04, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT(0x08, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
	PORT_BIT(0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(0x20, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START	// IN9
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN10
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN11
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_START	// IN12
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( mcnpshnt )
	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x06, 0x06, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x06, "0" )
	PORT_DIPSETTING(    0x04, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x18, DEF_STR( 1C_1C ) )
//	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x10, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )	//*
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START
	PORT_DIPNAME( 0x07, 0x07, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x07, "0" )
	PORT_DIPSETTING(    0x06, "1" )
	PORT_DIPSETTING(    0x05, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x03, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x01, "6" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BITX(    0x10, 0x10, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Choose Bonus", IP_KEY_NONE, IP_JOY_NONE )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )	//*
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )	//*
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE )	// Test
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )	// Analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )	// Memory Reset
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1                              )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( nanajign )
	PORT_START
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x38, 0x38, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x38, "0" )
	PORT_DIPSETTING(    0x30, "1" )
	PORT_DIPSETTING(    0x28, "2" )
	PORT_DIPSETTING(    0x20, "3" )
	PORT_DIPSETTING(    0x18, "4" )
	PORT_DIPSETTING(    0x10, "5" )
	PORT_DIPSETTING(    0x08, "6" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )		//?
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )	//*
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )	//*
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )	//*
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )	//*
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE )	// Test
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE2 )	// Analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE3 )	// Memory Reset
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SERVICE1 )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "A",   KEYCODE_A,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "E",   KEYCODE_E,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "I",   KEYCODE_I,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "M",   KEYCODE_M,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Kan", KEYCODE_LCONTROL, IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1                              )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "B",     KEYCODE_B,        IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "F",     KEYCODE_F,        IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "J",     KEYCODE_J,        IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "N",     KEYCODE_N,        IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Reach", KEYCODE_LSHIFT,   IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "Bet",   KEYCODE_RCONTROL, IP_JOY_NONE )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "C",   KEYCODE_C,     IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "G",   KEYCODE_G,     IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "K",   KEYCODE_K,     IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Chi", KEYCODE_SPACE, IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "Ron", KEYCODE_Z,     IP_JOY_NONE )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "D",   KEYCODE_D,    IP_JOY_NONE )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "H",   KEYCODE_H,    IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "L",   KEYCODE_L,    IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "Pon", KEYCODE_LALT, IP_JOY_NONE )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


/***************************************************************************


								Machine Drivers


***************************************************************************/

/***************************************************************************
								Hana no Mai
***************************************************************************/

static struct AY8910interface hanamai_ay8910_interface =
{
	1,			/* 1 chip */
	22000000 / 8,	/* 2.75MHz ??? */
	{ 20 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 }
};

static struct YM2203interface hanamai_ym2203_interface =
{
	1,
	22000000 / 8,					/* 2.75MHz */
	{ YM2203_VOL(50,20) },
	{ input_port_1_r },				/* Port A Read: DSW */
	{ input_port_0_r },				/* Port B Read: DSW */
	{ 0 },							/* Port A Write */
	{ 0 },							/* Port B Write */
	{ sprtmtch_sound_callback },	/* IRQ handler */
};

struct MSM5205interface hanamai_msm5205_interface =
{
	1,
	384000,
	{ adpcm_int },			/* IRQ handler */
	{ MSM5205_S48_4B },		/* 8 KHz, 4 Bits  */
	{ 100 }
};



static MACHINE_DRIVER_START( hanamai )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main",Z80,22000000 / 4)	/* 5.5MHz */
	MDRV_CPU_MEMORY(sprtmtch_readmem,sprtmtch_writemem)
	MDRV_CPU_PORTS(hanamai_readport,hanamai_writeport)
	MDRV_CPU_VBLANK_INT(sprtmtch_vblank_interrupt,1)	/* IM 0 needs an opcode on the data bus */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)
	MDRV_MACHINE_INIT(adpcm)

	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER|VIDEO_PIXEL_ASPECT_RATIO_1_2)
	MDRV_SCREEN_SIZE(512, 256)
	MDRV_VISIBLE_AREA(0, 512-1, 16, 256-1)
	MDRV_PALETTE_LENGTH(512)

	MDRV_PALETTE_INIT(sprtmtch)			// static palette
	MDRV_VIDEO_START(hanamai)
	MDRV_VIDEO_UPDATE(hanamai)

	/* sound hardware */
	MDRV_SOUND_ADD(AY8910, hanamai_ay8910_interface)
	MDRV_SOUND_ADD(YM2203, hanamai_ym2203_interface)
	MDRV_SOUND_ADD(MSM5205, hanamai_msm5205_interface)
MACHINE_DRIVER_END



/***************************************************************************
								Hana Oriduru
***************************************************************************/

static struct AY8910interface hnoridur_ay8910_interface =
{
	1,			/* 1 chip */
	22000000 / 8,	/* 2.75MHz ??? */
	{ 20 },
	{ input_port_0_r },		/* Port A Read: DSW */
	{ 0 },
	{ 0 },
	{ 0 }
};

static struct YM2413interface hnoridur_ym2413_interface=
{
	1,	/* 1 chip */
	3580000,	/* 3.58 MHz */
	{ YM2413_VOL(100,MIXER_PAN_CENTER,100,MIXER_PAN_CENTER) }
};

static MACHINE_DRIVER_START( hnoridur )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main",Z80,22000000 / 4)	/* 5.5MHz */
	MDRV_CPU_MEMORY(hnoridur_readmem,hnoridur_writemem)
	MDRV_CPU_PORTS(hnoridur_readport,hnoridur_writeport)
	MDRV_CPU_VBLANK_INT(sprtmtch_vblank_interrupt,1)	/* IM 0 needs an opcode on the data bus */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)
	MDRV_MACHINE_INIT(adpcm)

	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER|VIDEO_PIXEL_ASPECT_RATIO_1_2)
	MDRV_SCREEN_SIZE(512, 256)
	MDRV_VISIBLE_AREA(0, 512-1, 16, 256-1)
	MDRV_PALETTE_LENGTH(16*256)

	MDRV_VIDEO_START(hnoridur)
	MDRV_VIDEO_UPDATE(hnoridur)

	/* sound hardware */
	MDRV_SOUND_ADD(AY8910, hnoridur_ay8910_interface)
	MDRV_SOUND_ADD(YM2413, hnoridur_ym2413_interface)
	MDRV_SOUND_ADD(MSM5205, hanamai_msm5205_interface)
MACHINE_DRIVER_END


/***************************************************************************
								Sports Match
***************************************************************************/

static struct YM2203interface sprtmtch_ym2203_interface =
{
	1,
	22000000 / 8,					/* 2.75MHz */
	{ YM2203_VOL(100,20) },
	{ input_port_3_r },				/* Port A Read: DSW */
	{ input_port_4_r },				/* Port B Read: DSW */
	{ 0 },							/* Port A Write */
	{ 0 },							/* Port B Write */
	{ sprtmtch_sound_callback },	/* IRQ handler */
};

static MACHINE_DRIVER_START( sprtmtch )

	/* basic machine hardware */
	MDRV_CPU_ADD(Z80,22000000 / 4)	/* 5.5MHz */
	MDRV_CPU_MEMORY(sprtmtch_readmem,sprtmtch_writemem)
	MDRV_CPU_PORTS(sprtmtch_readport,sprtmtch_writeport)
	MDRV_CPU_VBLANK_INT(sprtmtch_vblank_interrupt,1)	/* IM 0 needs an opcode on the data bus */

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER|VIDEO_PIXEL_ASPECT_RATIO_1_2)
	MDRV_SCREEN_SIZE(512, 256)
	MDRV_VISIBLE_AREA(0, 512-1, 16, 256-1)
	MDRV_PALETTE_LENGTH(512)

	MDRV_PALETTE_INIT(sprtmtch)			// static palette
	MDRV_VIDEO_START(sprtmtch)
	MDRV_VIDEO_UPDATE(sprtmtch)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2203, sprtmtch_ym2203_interface)
MACHINE_DRIVER_END


/***************************************************************************
								Lady Frog
***************************************************************************/

static MACHINE_DRIVER_START( roldfrog )

	/* basic machine hardware */
	MDRV_CPU_ADD(Z80,22000000 / 4)	/* 5.5MHz */
	MDRV_CPU_MEMORY(sprtmtch_readmem,sprtmtch_writemem)
	MDRV_CPU_PORTS(sprtmtch_readport,sprtmtch_writeport)
	MDRV_CPU_VBLANK_INT(sprtmtch_vblank_interrupt,1)	/* IM 0 needs an opcode on the data bus */

// Until the protection on the 68000 is figured out (or it will crash)
#if 0
	MDRV_CPU_ADD(M68000, 10000000)	/* 10 MHz??? */
	MDRV_CPU_MEMORY(roldfrog_readmem,roldfrog_writemem)
	MDRV_CPU_VBLANK_INT(irq6_line_hold,1)
#endif

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER|VIDEO_PIXEL_ASPECT_RATIO_1_2)
	MDRV_SCREEN_SIZE(512, 256)
	MDRV_VISIBLE_AREA(0, 512-1, 16, 256-1)
	MDRV_PALETTE_LENGTH(512)
	MDRV_GFXDECODE(gfxdecodeinfo)	// has gfx

	// no static palette
	MDRV_VIDEO_START(sprtmtch)
	MDRV_VIDEO_UPDATE(sprtmtch)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2203, sprtmtch_ym2203_interface)
MACHINE_DRIVER_END



/***************************************************************************
							Mahjong Friday
***************************************************************************/

static struct YM2413interface mjfriday_ym2413_interface=
{
	1,	/* 1 chip */
	24000000/6,	/* ???? */
	{ YM2413_VOL(100,MIXER_PAN_CENTER,100,MIXER_PAN_CENTER) }
};

static MACHINE_DRIVER_START( mjfriday )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main",Z80,24000000/4)	/* 6 MHz? */
	MDRV_CPU_MEMORY(sprtmtch_readmem,sprtmtch_writemem)
	MDRV_CPU_PORTS(mjfriday_readport,mjfriday_writeport)
	MDRV_CPU_VBLANK_INT(irq0_line_hold,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(256, 256)
	MDRV_VISIBLE_AREA(0, 256-1, 16, 256-1)
	MDRV_PALETTE_LENGTH(512)

	MDRV_PALETTE_INIT(sprtmtch)			// static palette
	MDRV_VIDEO_START(mjdialq2)
	MDRV_VIDEO_UPDATE(mjdialq2)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2413, mjfriday_ym2413_interface)
MACHINE_DRIVER_END


/***************************************************************************
							Mahjong Dial Q2
***************************************************************************/

static MACHINE_DRIVER_START( mjdialq2 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM( mjfriday )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(mjdialq2_readmem,mjdialq2_writemem)
MACHINE_DRIVER_END


/***************************************************************************
					Yarunara / Quiz TV Q&Q / Mahjong Angels
***************************************************************************/

INTERRUPT_GEN( yarunara_clock_interrupt )
{
	dynax_sound_irq = 1;
	sprtmtch_update_irq();

	dynax_sound_irq = 0;
	sprtmtch_update_irq();
}

static MACHINE_DRIVER_START( yarunara )

	/* basic machine hardware */
	MDRV_IMPORT_FROM( hnoridur )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(yarunara_readmem,yarunara_writemem)
	MDRV_CPU_PORTS(yarunara_readport,yarunara_writeport)
	MDRV_CPU_PERIODIC_INT(yarunara_clock_interrupt,10)	// RTC

	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_VISIBLE_AREA(0, 336-1, 8, 256-1-8-1)
MACHINE_DRIVER_END


/***************************************************************************
							Mahjong Campus Hunting
***************************************************************************/

static MACHINE_DRIVER_START( mcnpshnt )

	MDRV_IMPORT_FROM( hnoridur )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(mcnpshnt_readmem,mcnpshnt_writemem)
	MDRV_CPU_PORTS(mcnpshnt_readport,mcnpshnt_writeport)

	MDRV_VIDEO_START(mcnpshnt)	// different priorities
MACHINE_DRIVER_END


/***************************************************************************
							7jigen
***************************************************************************/

static MACHINE_DRIVER_START( nanajign )

	MDRV_IMPORT_FROM( hnoridur )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(nanajign_readmem,nanajign_writemem)
	MDRV_CPU_PORTS(nanajign_readport,nanajign_writeport)
MACHINE_DRIVER_END


/***************************************************************************


								ROMs Loading


***************************************************************************/


/***************************************************************************

Hana no Mai
(c)1988 Dynax

D1610088L1

CPU:	Z80-A
Sound:	AY-3-8912A
	YM2203C
	M5205
OSC:	22.000MHz
Custom:	(TC17G032AP-0246)

***************************************************************************/

ROM_START( hanamai )
	ROM_REGION( 0x90000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "1611.13a", 0x00000, 0x10000, 0x5ca0b073 )
	ROM_LOAD( "1610.14a", 0x48000, 0x10000, 0xb20024aa )

	ROM_REGION( 0x140000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "1604.12e", 0x000000, 0x20000, 0x3b8362f7 )
	ROM_LOAD( "1609.12c", 0x020000, 0x20000, 0x91c5d211 )
	ROM_LOAD( "1603.13e", 0x040000, 0x20000, 0x16a2a680 )
	ROM_LOAD( "1608.13c", 0x060000, 0x20000, 0x793dd4f8 )
	ROM_LOAD( "1602.14e", 0x080000, 0x20000, 0x3189a45f )
	ROM_LOAD( "1607.14c", 0x0a0000, 0x20000, 0xa58edfd0 )
	ROM_LOAD( "1601.15e", 0x0c0000, 0x20000, 0x84ff07af )
	ROM_LOAD( "1606.15c", 0x0e0000, 0x10000, 0xce7146c1 )
	ROM_LOAD( "1605.10e", 0x100000, 0x10000, 0x0f4fd9e4 )
	ROM_LOAD( "1612.10c", 0x120000, 0x10000, 0x8d9fb6e1 )

	ROM_REGION( 0x400, REGION_PROMS, ROMREGION_DISPOSE )	/* Color PROMs */
	ROM_LOAD( "2.3j",  0x000, 0x200, 0x7b0618a5 )	// FIXED BITS (0xxxxxxx)
	ROM_LOAD( "1.4j",  0x200, 0x200, 0x9cfcdd2d )
ROM_END


/***************************************************************************

Hana Kochou (Hana no Mai BET ver.)
(c)1988 Dynax

D201901L2
D201901L1-0

CPU:	Z80-A
Sound:	AY-3-8912A
	YM2203C
	M5205
OSC:	22.000MHz
VDP:	HD46505SP
Custom:	(TC17G032AP-0246)

***************************************************************************/

ROM_START( hnkochou )
	ROM_REGION( 0x90000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "2009.s4a", 0x00000, 0x10000, 0xb3657123 )
	ROM_LOAD( "2008.s3a", 0x18000, 0x10000, 0x1c009be0 )

	ROM_REGION( 0x0e0000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "2004.12e", 0x000000, 0x20000, 0x178aa996 )
	ROM_LOAD( "2005.12c", 0x020000, 0x20000, 0xca57cac5 )
	ROM_LOAD( "2003.13e", 0x040000, 0x20000, 0x092edf8d )
	ROM_LOAD( "2006.13c", 0x060000, 0x20000, 0x610c22ec )
	ROM_LOAD( "2002.14e", 0x080000, 0x20000, 0x759b717d )
	ROM_LOAD( "2007.14c", 0x0a0000, 0x20000, 0xd0f22355 )
	ROM_LOAD( "2001.15e", 0x0c0000, 0x20000, 0x09ace2b5 )

	ROM_REGION( 0x400, REGION_PROMS, ROMREGION_DISPOSE )	/* Color PROMs */
	ROM_LOAD( "2.3j",  0x000, 0x200, 0x7b0618a5 )	// FIXED BITS (0xxxxxxx)
	ROM_LOAD( "1.4j",  0x200, 0x200, 0x9cfcdd2d )
ROM_END



/***************************************************************************

Hana Oriduru
(c)1989 Dynax
D2304268L

CPU  : Z80B
Sound: AY-3-8912A YM2413 M5205
OSC  : 22MHz (X1, near main CPU), 384KHz (X2, near M5205)
       3.58MHz (X3, Sound section)

CRT Controller: HD46505SP (6845)
Custom chip: DYNAX TC17G032AP-0246 JAPAN 8929EAI

***************************************************************************/

ROM_START( hnoridur )
	ROM_REGION( 0x10000 + 0x19*0x8000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "2309.12",  0x00000, 0x20000, 0x5517dd68 )
	ROM_RELOAD(           0x10000, 0x20000 )

	ROM_REGION( 0x100000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "2302.21",  0x000000, 0x20000, 0x9dde2d59 )
	ROM_LOAD( "2303.22",  0x020000, 0x20000, 0x1ac59443 )
	ROM_LOAD( "2301.20",  0x040000, 0x20000, 0x24391ddc )
	ROM_LOAD( "2304.1",   0x060000, 0x20000, 0x9792d9fa )
	ROM_LOAD( "2305.2",   0x080000, 0x20000, 0x249d360a )
	ROM_LOAD( "2306.3",   0x0a0000, 0x20000, 0x014a4945 )
	ROM_LOAD( "2307.4",   0x0c0000, 0x20000, 0x8b6f8a2d )
	ROM_LOAD( "2308.5",   0x0e0000, 0x20000, 0x6f996e6e )
ROM_END



/***************************************************************************

Sports Match
Dynax 1989

                     5563
                     3101
        SW2 SW1
                             3103
         YM2203              3102
                     16V8
                     Z80         DYNAX
         22MHz

           6845
                         53462
      17G                53462
      18G                53462
                         53462
                         53462
                         53462

***************************************************************************/

ROM_START( drgpunch )
	ROM_REGION( 0x90000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "2401.3d", 0x00000, 0x10000, 0xb310709c )
	ROM_LOAD( "2402.6d", 0x28000, 0x10000, 0xd21ed237 )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "2403.6c", 0x00000, 0x20000, 0xb936f202 )
	ROM_LOAD( "2404.5c", 0x20000, 0x20000, 0x2ee0683a )
	ROM_LOAD( "2405.3c", 0x40000, 0x20000, 0xaefbe192 )
	ROM_LOAD( "2406.1c", 0x60000, 0x20000, 0xe137f96e )
	ROM_LOAD( "2407.6a", 0x80000, 0x20000, 0xf3f1b065 )
	ROM_LOAD( "2408.5a", 0xa0000, 0x20000, 0x3a91e2b9 )

	ROM_REGION( 0x400, REGION_PROMS, ROMREGION_DISPOSE )	/* Color PROMs */
	ROM_LOAD( "2.18g", 0x000, 0x200, 0x9adccc33 )	// FIXED BITS (0xxxxxxx)
	ROM_LOAD( "1.17g", 0x200, 0x200, 0x324fa9cf )
ROM_END

ROM_START( sprtmtch )
	ROM_REGION( 0x90000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "3101.3d", 0x00000, 0x08000, 0xd8fa9638 )
	ROM_CONTINUE(        0x28000, 0x08000 )

	ROM_REGION( 0x40000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "3102.6c", 0x00000, 0x20000, 0x46f90e59 )
	ROM_LOAD( "3103.5c", 0x20000, 0x20000, 0xad29d7bd )

	ROM_REGION( 0x400, REGION_PROMS, ROMREGION_DISPOSE )	/* Color PROMs */
	ROM_LOAD( "18g", 0x000, 0x200, 0xdcc4e0dd )	// FIXED BITS (0xxxxxxx)
	ROM_LOAD( "17g", 0x200, 0x200, 0x5443ebfb )
ROM_END


/***************************************************************************

The Return of Lady Frog
Microhard, 1993

PCB Layout
----------


YM2203                            68000
YM3014    6116           **       2   6
          6116          6116      3   7
6264                              4   8
1  Z80           MACH130          5   9
                 681000        6264  6264


DSW2              2148                10
DSW1              2148  6264  30MHz   11
                  2148  6264  24MHz   12
                  2148                13

Notes:
      68000 Clock = >10MHz (my meter can only read up to 10.000MHz)
        Z80 Clock = 3MHz
               ** = possibly PLD (surface is scratched, type PLCC44)
    Vertical Sync = 60Hz
      Horiz. Sync = 15.56kHz


***************************************************************************/

ROM_START( roldfrog )
	ROM_REGION( 0x90000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "roldfrog.001", 0x00000, 0x20000, 0xba9eb1c6 )
	ROM_RELOAD(               0x20000, 0x20000 )

	ROM_REGION( 0x400000, REGION_CPU2, 0 )	/* 68000 code */
	ROM_LOAD16_BYTE( "roldfrog.002",	0x000000, 0x080000, 0x724cf022 )
	ROM_LOAD16_BYTE( "roldfrog.006",	0x000001, 0x080000, 0xe52a7ae2 )
	ROM_LOAD16_BYTE( "roldfrog.003",	0x100000, 0x080000, 0xa1d49967 )
	ROM_LOAD16_BYTE( "roldfrog.007",	0x100001, 0x080000, 0xe5805c4e )
	ROM_LOAD16_BYTE( "roldfrog.004",	0x200000, 0x080000, 0x709281f5 )
	ROM_LOAD16_BYTE( "roldfrog.008",	0x200001, 0x080000, 0x39adcba4 )
	ROM_LOAD16_BYTE( "roldfrog.005",	0x300000, 0x080000, 0xb683160c )
	ROM_LOAD16_BYTE( "roldfrog.009",	0x300001, 0x080000, 0xe475fb76 )

	ROM_REGION( 0x10000, REGION_GFX1, 0 )	/* blitter data ?? */

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "roldfrog.010",       0x00000, 0x20000, 0x51fd0e1a )
	ROM_LOAD( "roldfrog.011",       0x20000, 0x20000, 0x610bf6f3 )
	ROM_LOAD( "roldfrog.012",       0x40000, 0x20000, 0x466ede67 )
	ROM_LOAD( "roldfrog.013",       0x60000, 0x20000, 0xfad3e8be )
ROM_END

ROM_START( roldfrga )
	ROM_REGION( 0x90000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "roldfrog.001", 0x00000, 0x20000, 0xba9eb1c6 )
	ROM_RELOAD(               0x20000, 0x20000 )

	ROM_REGION( 0x400000, REGION_CPU2, 0 )	/* 68000 code */
	ROM_LOAD16_BYTE( "roldfrog.002",	0x000000, 0x080000, 0x724cf022 )
	ROM_LOAD16_BYTE( "roldfrog.006",	0x000001, 0x080000, 0xe52a7ae2 )
	ROM_LOAD16_BYTE( "roldfrog.003",	0x100000, 0x080000, 0xa1d49967 )
	ROM_LOAD16_BYTE( "roldfrog.007",	0x100001, 0x080000, 0xe5805c4e )
	ROM_LOAD16_BYTE( "roldfrog.004",	0x200000, 0x080000, 0x709281f5 )
	ROM_LOAD16_BYTE( "roldfrog.008",	0x200001, 0x080000, 0x39adcba4 )
	ROM_LOAD16_BYTE( "roldfrog.005",	0x300000, 0x080000, 0xb683160c )
	ROM_LOAD16_BYTE( "9",	            0x300001, 0x080000, 0xfd515b58 )	// differs with roldfrog.009 by 1 byte

	ROM_REGION( 0x10000, REGION_GFX1, 0 )	/* blitter data ?? */

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "roldfrog.010",       0x00000, 0x20000, 0x51fd0e1a )
	ROM_LOAD( "roldfrog.011",       0x20000, 0x20000, 0x610bf6f3 )
	ROM_LOAD( "roldfrog.012",       0x40000, 0x20000, 0x466ede67 )
	ROM_LOAD( "roldfrog.013",       0x60000, 0x20000, 0xfad3e8be )
ROM_END


/***************************************************************************

Mahjong Friday
(c)1989 Dynax
D2607198L1

CPU  : Zilog Z0840006PSC (Z80)
Sound: YM2413 M5205
OSC  : 24MHz (X1)

CRT Controller: HD46505SP (6845)
Custom chip: DYNAX TC17G032AP-0246 JAPAN 8828EAI

***************************************************************************/

ROM_START( mjfriday )
	ROM_REGION( 0x90000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "2606.2b",  0x00000, 0x10000, 0x00e0e0d3 )
	ROM_LOAD( "2605.2c",  0x28000, 0x10000, 0x5459ebda )

	ROM_REGION( 0x80000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "2601.2h",  0x000000, 0x20000, 0x70a01fc7 )
	ROM_LOAD( "2602.2g",  0x020000, 0x20000, 0xd9167c10 )
	ROM_LOAD( "2603.2f",  0x040000, 0x20000, 0x11892916 )
	ROM_LOAD( "2604.2e",  0x060000, 0x20000, 0x3cc1a65d )

	ROM_REGION( 0x400, REGION_PROMS, ROMREGION_DISPOSE )	/* Color PROMs */
	ROM_LOAD( "d26_2.9e", 0x000, 0x200, 0xd6db5c60 )	// FIXED BITS (0xxxxxxx)
	ROM_LOAD( "d26_1.8e", 0x200, 0x200, 0xaf5edf32 )
ROM_END


/***************************************************************************

Maya
Promat, 1994

PCB Layout
----------

    6845      6264   3
 DSW1  DSW2    1     4
   YM2203      2     5
   3014B

              Z80
  22.1184MHz

 PROM1  TPC1020  D41264
 PROM2            (x6)


Notes:
      Z80 Clock: 5.522MHz
          HSync: 15.925 kHz
          VSync: 60Hz


***************************************************************************/

ROM_START( maya )
	ROM_REGION( 0x90000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "1.17e", 0x00000, 0x10000, 0x5aaa015e )
	ROM_LOAD( "2.15e", 0x28000, 0x10000, 0x7ea5b49a )

	ROM_REGION( 0xc0000, REGION_USER1, ROMREGION_DISPOSE )	/* blitter data */
	ROM_LOAD( "3.18g", 0x00000, 0x40000, 0x8534af04 )
	ROM_LOAD( "4.17g", 0x40000, 0x40000, 0xab85ce5e )
	ROM_LOAD( "5.15g", 0x80000, 0x40000, 0xc4316dec )

	ROM_REGION( 0xc0000, REGION_GFX1, 0 )
	/* blitter data will be decrypted here*/

	ROM_REGION( 0x400, REGION_PROMS, ROMREGION_DISPOSE )	/* Color PROMs */
	ROM_LOAD( "prom2.5b",  0x000, 0x200, 0xd276bf61 )	// FIXED BITS (0xxxxxxx)
	ROM_LOAD( "prom1.6b",  0x200, 0x200, 0xe38eb360 )
ROM_END

static DRIVER_INIT( maya )
{
	/* Address lines scrambling on 1 z80 rom */
	int i;
	data8_t	*gfx = (data8_t *)memory_region(REGION_GFX1);
	data8_t	*rom = memory_region(REGION_CPU1) + 0x28000,
			*end = rom + 0x10000;
	for (;rom < end; rom+=8)
	{
		data8_t temp[8];
		temp[0] = rom[0];	temp[1] = rom[1];	temp[2] = rom[2];	temp[3] = rom[3];
		temp[4] = rom[4];	temp[5] = rom[5];	temp[6] = rom[6];	temp[7] = rom[7];

		rom[0] = temp[0];	rom[1] = temp[4];	rom[2] = temp[1];	rom[3] = temp[5];
		rom[4] = temp[2];	rom[5] = temp[6];	rom[6] = temp[3];	rom[7] = temp[7];
	}

	/* Address lines scrambling on the blitter data roms */
	rom = memory_region(REGION_USER1);

	for (i = 0; i < 0xc0000; i++)
		gfx[i] = rom[BITSWAP24(i,23,22,21,20,19,18,14,15, 16,17,13,12,11,10,9,8, 7,6,5,4,3,2,1,0)];
}


/***************************************************************************

Mahjong Dial Q2
(c)1991 Dynax
D5212298L-1

CPU  : Z80
Sound: YM2413
OSC  : (240-100 624R001 KSSOB)?
Other: TC17G032AP-0246
CRT Controller: HD46505SP (6845)

***************************************************************************/

ROM_START( mjdialq2 )
	ROM_REGION( 0x78000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "5201.2b", 0x00000, 0x10000, 0x5186c2df )
	ROM_RELOAD(          0x10000, 0x08000 )				// 1
	ROM_CONTINUE(        0x20000, 0x08000 )				// 3
	ROM_LOAD( "5202.2c", 0x30000, 0x08000, 0x8e8b0038 )	// 5
	ROM_CONTINUE(        0x70000, 0x08000 )				// d

	ROM_REGION( 0xa0000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "5207.2h", 0x00000, 0x20000, 0x7794cd62 )
	ROM_LOAD( "5206.2g", 0x20000, 0x20000, 0x9e810021 )
	ROM_LOAD( "5205.2f", 0x40000, 0x20000, 0x8c05572f )
	ROM_LOAD( "5204.2e", 0x60000, 0x20000, 0x958ef9ab )
	ROM_LOAD( "5203.2d", 0x80000, 0x20000, 0x706072d7 )

	ROM_REGION( 0x400, REGION_PROMS, ROMREGION_DISPOSE )	/* Color PROMs */
	ROM_LOAD( "d52-2.9e", 0x000, 0x200, 0x18585ce3 )	// FIXED BITS (0xxxxxxx)
	ROM_LOAD( "d52-1.8e", 0x200, 0x200, 0x8868247a )
ROM_END


/***************************************************************************

Mahjong Yarunara
(c)1991 Dynax
D5512068L1-1
D4508308L-2 (sub board)

CPU  : Z80B
Sound: AY-3-8912A YM2413 M5205
OSC  : 22.000MHz (near main CPU), 14.31818MHz (Sound section)
       YC-38 (X1), 384KHz (X2, M5205)

ROMs (all ROMs are 541000 = 27C010 compatible):
5501M.2D     [d86fade5]
5502M.4D     [1ef09ff0]
5503M.8C     [9276a10a]
5504M.9C     [6ac42304]
5505M.10C    [b2ca9838]
5506M.11C    [161058fd]
5507M.13C    [7de17b26]
5508M.16C    [ced3155b]
5509M.17C    [ca46ed48]

Subboard ROMs (5515M is 27C040, others are 541000):
5510M.2A     [bb9c71e1]
5511M.3A     [40ee77d8]
5512M.4A     [b4220316]
5513M.1B     [32b7bcbd]
5514M.2B     [ac714bb7]
5515M.4B     [ef130237]

PALs (not dumped):
D55A.4E
D55B.11F
D55C.16N
D55D.17D
D55EH.6A

CRT controller:
HD46505SP (6845)

Real time clock:
OKI M6242B

Custom chip:
DYNAX NL-001 WD10100

***************************************************************************/

ROM_START( yarunara )
	ROM_REGION( 0x10000 + 0x1d*0x8000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "5501m.2d",  0x00000, 0x20000, 0xd86fade5 )
	ROM_RELOAD(            0x10000, 0x20000 )
	ROM_LOAD( "5502m.4d",  0x30000, 0x20000, 0x1ef09ff0 )

	ROM_REGION( 0x3a0000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "5507m.13c", 0x000000, 0x80000, 0x7de17b26 )

	ROM_LOAD( "5508m.16c", 0x100000, 0x20000, 0xced3155b )
	ROM_LOAD( "5509m.17c", 0x120000, 0x20000, 0xca46ed48 )
	ROM_LOAD( "5506m.11c", 0x140000, 0x20000, 0x161058fd )
	ROM_LOAD( "5505m.10c", 0x160000, 0x20000, 0xb2ca9838 )
	ROM_LOAD( "5504m.9c",  0x180000, 0x20000, 0x6ac42304 )
	ROM_LOAD( "5503m.8c",  0x1a0000, 0x20000, 0x9276a10a )

	ROM_LOAD( "5515m.4b",  0x200000, 0x80000, 0xef130237 )

	ROM_LOAD( "5514m.2b",  0x300000, 0x20000, 0xac714bb7 )
	ROM_LOAD( "5513m.1b",  0x320000, 0x20000, 0x32b7bcbd )
	ROM_LOAD( "5512m.4a",  0x340000, 0x20000, 0xb4220316 )
	ROM_LOAD( "5511m.3a",  0x360000, 0x20000, 0x40ee77d8 )
	ROM_LOAD( "5510m.2a",  0x380000, 0x20000, 0xbb9c71e1 )
ROM_END


/***************************************************************************

Quiz TV Gassyuukoku Q&Q (JPN ver.)
(c)1992 Dynax

DX-BASE (Dynax Motherboard System) D5512068L1-2
D6410288L-1 (SUB)

6401.2D   prg. / samples
6402.4D
6403.5D

6404.S2A  chr.
6405.S2B
6406.S2C
6407.S2D
6408.S2E
6409.S2F
6410.S2G
6411.S2H
6412.S3A
6413.S3B

***************************************************************************/

ROM_START( quiztvqq )
	ROM_REGION( 0x10000 + 0x28*0x8000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "6401.2d",      0x000000, 0x020000, 0xce0e237c )
	// 14-17
	ROM_RELOAD(               0x0b0000, 0x020000 )
	// 04-07
	ROM_LOAD( "6402.4d",      0x030000, 0x020000, 0xcf7a9aa8 )
	// 24-27
	ROM_CONTINUE(             0x130000, 0x020000 )
	// 08-0b
	ROM_LOAD( "6403.5d",      0x050000, 0x020000, 0x62402ac9 )

	ROM_REGION( 0x1a0000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "6404.s2a",     0x000000, 0x080000, 0x996ebe0f )
	ROM_LOAD( "6405.s2b",     0x080000, 0x020000, 0x666bfb03 )
	ROM_LOAD( "6406.s2c",     0x0a0000, 0x020000, 0x006871ef )
	ROM_LOAD( "6407.s2d",     0x0c0000, 0x020000, 0x9cc61541 )
	ROM_LOAD( "6408.s2e",     0x0e0000, 0x020000, 0x65a98946 )
	ROM_LOAD( "6409.s2f",     0x100000, 0x020000, 0xd5d11061 )
	ROM_LOAD( "6410.s2g",     0x120000, 0x020000, 0xbd769d46 )
	ROM_LOAD( "6411.s2h",     0x140000, 0x020000, 0x7bd43065 )
	ROM_LOAD( "6412.s3a",     0x160000, 0x020000, 0x43e645f3 )
	ROM_LOAD( "6413.s3b",     0x180000, 0x020000, 0xf7b81238 )
ROM_END


/***************************************************************************

Mahjong Angels (Comic Theater Vol.2)
(c)1991 Dynax

DX-BASE (Dynax Motherboard System) D5512068L1-1
D6107068L-1 (SUB)

612-01.2D   prg. / samples
612-02.4D
612-03.5D

612-04.S1A  chr.
612-05.S2A
612-06.S1B
612-07.S2B
612-08.S3C
612-09.S4C
612-10.S5C
612-11.S6C

***************************************************************************/

ROM_START( mjangels )
	ROM_REGION( 0x10000 + 0x28*0x8000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "612-01.2d",    0x000000, 0x020000, 0xcd353ba9 )
	// 00-03
	ROM_RELOAD(               0x010000, 0x20000 )
	// 0c-0f
	ROM_RELOAD(               0x070000, 0x20000 )
	// 24-27
	ROM_RELOAD(               0x130000, 0x20000 )
	// 04-07
	ROM_LOAD( "612-02.4d",    0x030000, 0x020000, 0xc1be70f9 )
	// 08-0b
	ROM_LOAD( "612-03.5d",    0x050000, 0x020000, 0x62402ac9 )

	ROM_REGION( 0x1c0000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "612-04.s1a",   0x000000, 0x080000, 0xc9b568a0 )
	ROM_LOAD( "612-05.s2a",   0x080000, 0x040000, 0x2ed51c5d )

	ROM_LOAD( "612-06.s1b",   0x100000, 0x020000, 0x8612904d )
	ROM_LOAD( "612-07.s2b",   0x120000, 0x020000, 0x0828c59d )
	ROM_LOAD( "612-11.s6c",   0x140000, 0x020000, 0x473b6fcd )
	ROM_LOAD( "612-10.s5c",   0x160000, 0x020000, 0xbf1edb0e )
	ROM_LOAD( "612-09.s4c",   0x180000, 0x020000, 0x8345999e )
	ROM_LOAD( "612-08.s3c",   0x1a0000, 0x020000, 0xaad88516 )
ROM_END


/***************************************************************************

Mahjong Campus Hunting
(c)1990 Dynax
D3312108L1-1
D23SUB BOARD1 (sub board)

CPU  : Z80B
Sound: AY-3-8912A YM2413 M5205
OSC  : 22MHz (X1, near main CPU), 384KHz (X2, near M5205)
       3.58MHz (X3, Sound section)

ROMs:
3309.20      [0c7d72f0]	OKI M271000ZB
3310.21      [28f5f194]	 |
3311.22      [cddbf667]	 |
3312.1       [cf0afbb5]	 |
3313.2       [36e25beb]	 |
3314.3       [f1cf01bc]	 |
3315.4       [7cac01c7]	/

3316.10      [44006ee5]	M5M27C101P
3317.11      [4bb62bb4]	/
3318.12      [e3b457a8]	27C010

Subboard ROMs:
3301.1B      [8ec98d60]	OKI M271000ZB
3302.2B      [d7024f2d]	 |
3303.3B      [01548edc]	 |
3304.4B      [deef9a4e]	 |
3305.1A      [8a9ebab8]	 |
3306.2A      [86afcc80]	 |
3307.3A      [07dbaf8a]	 |
3308.4A      [a2cac53d]	/

PALs:
D33A.24 (16L8)
D33B.79 (16L8)
D33C.67 (16R8)

CRT Controller:
HD46505SP (6845)

Custom chip:
DYNAX TC17G032AP-0246 JAPAN 8951EAY

***************************************************************************/

ROM_START( mcnpshnt )
	ROM_REGION( 0x10000 + 0xc*0x8000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "3318.12", 0x000000, 0x020000, 0xe3b457a8 )
	ROM_RELOAD(          0x010000, 0x020000 )
	ROM_LOAD( "3317.11", 0x030000, 0x020000, 0x4bb62bb4 )
	ROM_LOAD( "3316.10", 0x050000, 0x020000, 0x44006ee5 )

	ROM_REGION( 0x300000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "3310.21", 0x000000, 0x020000, 0x28f5f194 )
	ROM_LOAD( "3311.22", 0x020000, 0x020000, 0xcddbf667 )
	ROM_LOAD( "3309.20", 0x040000, 0x020000, 0x0c7d72f0 )
	ROM_LOAD( "3312.1",  0x060000, 0x020000, 0xcf0afbb5 )
	ROM_LOAD( "3313.2",  0x080000, 0x020000, 0x36e25beb )
	ROM_LOAD( "3314.3",  0x0a0000, 0x020000, 0xf1cf01bc )
	ROM_LOAD( "3315.4",  0x0c0000, 0x020000, 0x7cac01c7 )

	ROM_LOAD( "3301.1b", 0x200000, 0x020000, 0x8ec98d60 )
	ROM_LOAD( "3302.2b", 0x220000, 0x020000, 0xd7024f2d )
	ROM_LOAD( "3303.3b", 0x240000, 0x020000, 0x01548edc )
	ROM_LOAD( "3304.4b", 0x260000, 0x020000, 0xdeef9a4e )
	ROM_LOAD( "3308.4a", 0x280000, 0x020000, 0xa2cac53d )
	ROM_LOAD( "3307.3a", 0x2a0000, 0x020000, 0x07dbaf8a )
	ROM_LOAD( "3306.2a", 0x2c0000, 0x020000, 0x86afcc80 )
	ROM_LOAD( "3305.1a", 0x2e0000, 0x020000, 0x8a9ebab8 )
ROM_END


/***************************************************************************

7jigen no Youseitachi (Mahjong 7 Dimensions)
(c)1990 Dynax

D3707198L1
D23SUB BOARD1

CPU:	Z80-B
Sound:	AY-3-8912A
	YM2413
	M5205
OSC:	22.000MHz
	3.58MHz
	384KHz
VDP:	HD46505SP
Custom:	(TC17G032AP-0246)


3701.1A   prg.

3702.3A   samples
3703.4A

3704.S1B  chr.
3705.S2B
3706.S3B
3707.S4B
3708.S1A
3709.S2A
3710.S3A
3711.S4A
3712.14A
3713.16A
3714.17A
3715.18A
3716.20A
3717.17B

***************************************************************************/

ROM_START( 7jigen )
	ROM_REGION( 0x10000 + 0xc*0x8000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "3701.1a",      0x000000, 0x020000, 0xee8ab3c4 )
	ROM_RELOAD(               0x010000, 0x020000 )
	ROM_LOAD( "3702.3a",      0x030000, 0x020000, 0x4e43a0bb )
	ROM_LOAD( "3703.4a",      0x050000, 0x020000, 0xec77b564 )

	ROM_REGION( 0x300000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "3713.16a",     0x000000, 0x020000, 0xf3a745d2 )
	ROM_LOAD( "3712.14a",     0x020000, 0x020000, 0x88786680 )
	ROM_LOAD( "3715.18a",     0x040000, 0x020000, 0x19f7ab13 )
	ROM_LOAD( "3716.20a",     0x060000, 0x020000, 0x4f0c7f06 )
	ROM_LOAD( "3717.17b",     0x080000, 0x020000, 0x960cfd62 )
	ROM_LOAD( "3714.17a",     0x0a0000, 0x020000, 0x44ba5e35 )

	ROM_LOAD( "3704.s1b",     0x200000, 0x020000, 0x26348ae4 )
	ROM_LOAD( "3705.s2b",     0x220000, 0x020000, 0x5b5ea036 )
	ROM_LOAD( "3706.s3b",     0x240000, 0x020000, 0x7fdfb600 )
	ROM_LOAD( "3707.s4b",     0x260000, 0x020000, 0x67fa83ea )
	ROM_LOAD( "3711.s4a",     0x280000, 0x020000, 0xf1d4399d )
	ROM_LOAD( "3710.s3a",     0x2a0000, 0x020000, 0x0a92af7c )
	ROM_LOAD( "3709.s2a",     0x2c0000, 0x020000, 0x86f27f1c )
	ROM_LOAD( "3708.s1a",     0x2e0000, 0x020000, 0x8082d0ac )
ROM_END


/***************************************************************************


								Game Drivers


***************************************************************************/

GAME ( 1988, hanamai,  0,        hanamai,  hanamai,  0,    ROT180, "Dynax",                  "Hana no Mai (Japan)"                                   )
GAME ( 1989, hnkochou, hanamai,  hanamai,  hnkochou, 0,    ROT180, "Dynax",                  "Hana Kochou [BET] (Japan)"                             )
GAMEX( 1989, hnoridur, 0,        hnoridur, hnoridur, 0,    ROT180, "Dynax",                  "Hana Oriduru (Japan)",                                 GAME_IMPERFECT_GRAPHICS ) // 1 rom is bad
GAME ( 1989, drgpunch, 0,        sprtmtch, sprtmtch, 0,    ROT0,   "Dynax",                  "Dragon Punch (Japan)"                                  )
GAME ( 1989, sprtmtch, drgpunch, sprtmtch, sprtmtch, 0,    ROT0,   "Dynax (Fabtek license)", "Sports Match"                                          )
GAME ( 1989, mjfriday, 0,        mjfriday, mjfriday, 0,    ROT180, "Dynax",                  "Mahjong Friday (Japan)"                                )
GAME ( 1990, mcnpshnt, 0,        mcnpshnt, mcnpshnt, 0,    ROT0,   "Dynax",                  "Mahjong Campus Hunting (Japan)"                        )
GAMEX( 1990, 7jigen,   0,        nanajign, nanajign, 0,    ROT180, "Dynax",                  "7jigen no Youseitachi - Mahjong 7 Dimensions (Japan)", GAME_IMPERFECT_GRAPHICS )
GAMEX( 1991, mjdialq2, 0,        mjdialq2, mjdialq2, 0,    ROT180, "Dynax",                  "Mahjong Dial Q2 (Japan)",                              GAME_IMPERFECT_GRAPHICS )
GAME ( 1991, yarunara, 0,        yarunara, yarunara, 0,    ROT180, "Dynax",                  "Mahjong Yarunara (Japan)"                              )
GAMEX( 1991, mjangels, 0,        yarunara, yarunara, 0,    ROT180, "Dynax",                  "Mahjong Angels - Comic Theater Vol.2 (Japan)",         GAME_IMPERFECT_GRAPHICS )
GAME ( 1992, quiztvqq, 0,        yarunara, quiztvqq, 0,    ROT180, "Dynax",                  "Quiz TV Gassyuukoku Q&Q (Japan)"                       )
GAME ( 1994, maya,     0,        sprtmtch, sprtmtch, maya, ROT0,   "Promat",                 "Maya"                                                  )


// should move these to splash.c, its a protected bootleg of splash, not dynax stuff */
GAMEX(1993, roldfrog, 0,        roldfrog, sprtmtch, 0,    ROT0,   "Microhard", "The Return of Lady Frog", GAME_NOT_WORKING )
GAMEX(1993, roldfrga, roldfrog, roldfrog, sprtmtch, 0,    ROT0,   "Microhard", "The Return of Lady Frog (set 2)", GAME_NOT_WORKING )
