/***************************************************************************

Some Dynax/Nakanihon games using the third version of their blitter

Driver by Nicola Salmoria, Luca Elia

Hardware:
CPU: Z80 or 68000
Sound: (AY-3-8910) + YM2413 + MSM6295
Other: Real Time Clock (Oki MSM6242B or 72421B)

-----------------------------------------------------------------------------------------------------------------------------
Year + Game                 Board             CPU     Sound                    Custom                            Notes
-----------------------------------------------------------------------------------------------------------------------------
92 Monkey Mole Panic                          2xZ80   AY8910 + YM2413 + M6295  NL-001 1108(x2)   1427(x2)         8251
93 Animalandia Jr.                            2xZ80   AY8910 + YM2413 + M6295  NL-001 NL-003(x2) NL-004(x2)      TMP82C51
93 Quiz Channel Question  N7311208L1-2        Z80              YM2413 + M6295  NL-002 1108F0405  1427F0071
93 First Funky Fighter    N7403208L-2         2xZ80   YM2149 + YM2413 + M6295  NL-001 NL-002     NL-005
94 Mysterious World       D7107058L1-1        Z80     YM2149 + YM2413 + M6295  NL-002 1108F0405  1427F0071 4L02?
94 Mysterious Universe    D7107058L1-1        Z80     YM2149 + YM2413 + M6295  NL-002 1108F0405  1427F0071
94 Mysterious Orient      D7107058L1-1        Z80     YM2149 + YM2413 + M6295  NL-002 1108F0405  1427F0071
94 Quiz 365                                   68000   AY8910 + YM2413 + M6295
94 Rong Rong (J)          N8010178L1          Z80              YM2413 + M6295  NL-002 1108F0405  1427F0071 4L02F2637
94 Hana Ginga             D8102048L1          Z80     YM2149 + YM2413 + M6295  NL-002 1108F0405  1427F0071 4L02?
94 Super Hana Paradise    N8010178L1+N73RSUB  Z80              YM2413 + M6295  NL-002 1108F0406  1427F0071 4L02F2637
95 Dai Chuuka Ken         D11107218L1         Z80     AY8910 + YM2413 + M6295  70C160F009
95 Hana Gokou             N83061581L1         Z80     AY8910 + YM2413 + M6295  NL-002 1108?      1427?     4L02?
95 Nettoh Quiz Champion                       68000   AY8910 + YM2413 + M6295
95 Don Den Lover (J)      D1120901L8          68000   YMZ284 + YM2413 + M6295  NL-005
96 Don Den Lover (HK)     D11309208L1         68000   YMZ284 + YM2413 + M6295  NL-005
96 Panel&Variety Akamaru                      68000   YMZ284 + YM2413 + M6295  NL-005
96 Fantasic Love          NS5000101+?         Z80     YMZ284 + YM2413 + M6295  NL-005
96 Hana Kanzashi                              Z80              YM2413 + M6295  70C160F011?
97 Hana Kagerou                               Z80              YM2413 + M6295  70C160F011
98 Chuukanejyo            D11107218L1         Z80     AY8910 + YM2413 + M6295  70C160F009
98 Reach Ippatsu                              Z80              YM2413 + M6295  70C160F011
-----------------------------------------------------------------------------------------------------------------------------

Notes:

- the zooming Dynax logo in ddenlovr would flicker because the palette is
  updated one frame after the bitmap. This is fixed using a framebuffer but I
  don't think it's correct.


TODO:

- NVRAM, RTC

- verify whether clip_width/height is actually clip_x_end/y_end
  (this also applies to rectangles drawing, command 1c):
  the girl in hanakanz divided in 3 chunks (during the intro when bet is on)
  is ok with the latter setting; scene 2 of gal 1 check in hkagerou (press 1 in scene 1)
  is maybe clipped too much this way and hints at the former setting being correct.
  There is an #if to switch between the two modes in do_plot.

- ddenlovr: understand the extra commands for the blitter compressed data,
  used only by this game.

- ddenlovr: sometimes the colors of the girl in the presentation before the
  beginning of a stage are wrong, and they correct themselves when the board
  is drawn.
- The palette problems mentioned above happen in other games as well, e.g.
  quizchq attract mode.

- the registers right after the palette bank selectors (e00048-e0004f in ddenlovr)
  are not understood. They are related to the layer enable register and to the
  unknown blitter register 05.
  ddenlovr has a function at 001798 to initialize these four registers. It uses
  a table with 7 possible different layouts:
  0f 0f 3f cf
  4f 0f af 1f
  0f 0f 6f 9f
  0f 0f 0f ff
  0f 0f 7f 8f
  0f 0f cf 3f
  0f 0f 8f 7f
  the table is copied to e00048-e0004f and is also used to create a 16-entry
  array used to feed blitter register 05. Every element of the array is the OR
  of the values in the table above corresponding to bits set in the layer enable
  register. Note that in the table above the top 4 bits are split among the four
  entries.

- The meaning of blitter commands 43 and 8c is not understood.

- quizchq: it locks up, some samples are played at the wrong pitch

- quiz365 protection

- ddenlvrj, akamaru: the elapsed time text in the "game information" screen
  is all wrong (RTC/interrupts related).

***************************************************************************/

#include "driver.h"
#include "cpu/z80/z80.h"
#include "machine/random.h"
#include "machine/msm6242.h"

static UINT8 *pixmap[8];
static struct mame_bitmap *framebuffer;
static int extra_layers;


/***************************************************************************

                        Blitter Data Format

The gfx data is a bitstream. Command size is always 3 bits, argument size
can be from 1 to 8 bits (up to 16 bits seem to be allowed, but not used).

Data starts with an 8 bit header:
7------- not used
-654---- size-1 of arguments indicating pen number (B)
----3210 size-1 of arguments indicating number of pixels (A)

The commands are:
000 Increment Y
001 Followed by A bits (N) and by B bits (P): draw N+1 pixels using pen P
010 Followed by A bits (N) and by (N+1)*B bits: copy N+1 pixels
011 Followed by A bits (N): skip N pixels
100 not used
101 Followed by 4 bits: change argument size
110 Followed by 3 bits: change pen size
111 Stop.

The drawing operation is verified (quiz365) to modify dynax_blit_y.

***************************************************************************/

static int dynax_dest_layer;
static int dynax_blit_flip;
static int dynax_blit_x;
static int dynax_blit_y;
static int dynax_blit_address;
static int dynax_blit_pen,dynax_blit_pen_mode;
static int dynax_blitter_irq_flag,dynax_blitter_irq_enable;
static int dynax_rect_width, dynax_rect_height;
static int dynax_clip_width, dynax_clip_height;
static int dynax_line_length;
static int dynax_clip_ctrl = 0xf,dynax_clip_x,dynax_clip_y;
static int dynax_scroll[8*2];
static int dynax_priority, dynax_priority2;
static int dynax_bgcolor, dynax_bgcolor2;
static int dynax_layer_enable=0x0f, dynax_layer_enable2=0x0f;
static int dynax_palette_base[8], dynax_palette_mask[8];
static int dynax_transparency_pen[8], dynax_transparency_mask[8];
static int dynax_blit_reg;
static int dynax_blit_pen_mask = 0xff;	// not implemented
static int dynax_blit_rom_bits;			// usually 8, 16 in hanakanz
static const int *dynax_blit_commands;

enum { BLIT_NEXT = 0, BLIT_LINE, BLIT_COPY, BLIT_SKIP, BLIT_CHANGE_NUM, BLIT_CHANGE_PEN, BLIT_UNKNOWN, BLIT_STOP };
//                                              0          1                2                   3               4               5                   6                   7
static const int ddenlovr_blit_commands[8]	= { BLIT_NEXT, BLIT_LINE,		BLIT_COPY,			BLIT_SKIP,		BLIT_UNKNOWN,	BLIT_CHANGE_NUM,	BLIT_CHANGE_PEN,	BLIT_STOP	};
static const int hanakanz_blit_commands[8]	= { BLIT_NEXT, BLIT_CHANGE_PEN,	BLIT_CHANGE_NUM,	BLIT_UNKNOWN,	BLIT_SKIP,		BLIT_COPY,			BLIT_LINE,			BLIT_STOP	};
static const int mjflove_blit_commands[8]	= { BLIT_STOP, BLIT_CHANGE_PEN,	BLIT_CHANGE_NUM,	BLIT_UNKNOWN,	BLIT_SKIP,		BLIT_COPY,			BLIT_LINE,			BLIT_NEXT	};

VIDEO_START(ddenlovr)
{
	int i;
	for (i = 0; i < 8; i++)
		if (!(pixmap[i] = auto_malloc(512*512)))	return 1;

	if (!(framebuffer = auto_bitmap_alloc(Machine->drv->screen_width,Machine->drv->screen_height))) return 1;

	extra_layers = 0;

	// older games do not set these !?
	dynax_clip_width = 0x400;
	dynax_clip_height = 0x400;

	dynax_blit_rom_bits = 8;
	dynax_blit_commands = ddenlovr_blit_commands;
	return 0;
}

VIDEO_START(mmpanic)
{
	video_start_ddenlovr();

	extra_layers = 1;
	return 0;
}

VIDEO_START(hanakanz)
{
	video_start_ddenlovr();

	dynax_blit_rom_bits = 16;
	dynax_blit_commands = hanakanz_blit_commands;
	return 0;
}

VIDEO_START(mjflove)
{
	video_start_ddenlovr();

	dynax_blit_commands = mjflove_blit_commands;
	return 0;
}

static void dynax_flipscreen_w( UINT8 data )
{
	logerror("flipscreen = %02x (%s)\n",data,(data&1)?"off":"on");
}

static void dynax_blit_flip_w( UINT8 data )
{
	if ((data ^ dynax_blit_flip) & 0xec)
	{
#ifdef MAME_DEBUG
		usrintf_showmessage("warning dynax_blit_flip = %02x",data);
#endif
		logerror("warning dynax_blit_flip = %02x\n",data);
	}

	dynax_blit_flip = data;
}

static WRITE8_HANDLER( dynax_bgcolor_w )
{
	dynax_bgcolor = data;
}

static WRITE8_HANDLER( dynax_bgcolor2_w )
{
	dynax_bgcolor2 = data;
}

static WRITE16_HANDLER( ddenlovr_bgcolor_w )
{
	if (ACCESSING_LSB)
		dynax_bgcolor_w(offset,data);
}


static WRITE8_HANDLER( dynax_priority_w )
{
	dynax_priority = data;
}

static WRITE8_HANDLER( dynax_priority2_w )
{
	dynax_priority2 = data;
}

static WRITE16_HANDLER( ddenlovr_priority_w )
{
	if (ACCESSING_LSB)
		dynax_priority_w(offset,data);
}


static WRITE8_HANDLER( dynax_layer_enable_w )
{
	dynax_layer_enable = data;
}

static WRITE8_HANDLER( dynax_layer_enable2_w )
{
	dynax_layer_enable2 = data;
}


static WRITE16_HANDLER( ddenlovr_layer_enable_w )
{
	if (ACCESSING_LSB)
		dynax_layer_enable_w(offset,data);
}




static void do_plot(int x,int y,int pen)
{
	int addr, temp;
	int xclip, yclip;

	y &= 0x1ff;
	x &= 0x1ff;

	// swap x & y (see hanakanz gal check)
	if (dynax_blit_flip & 0x10)	{	temp = x;	x = y;	y = temp;	}

	// clipping rectangle (see hanakanz / hkagerou gal check)
#if 0
	xclip	=	(x < dynax_clip_x) || (x > dynax_clip_x+dynax_clip_width);
	yclip	=	(y < dynax_clip_y) || (y > dynax_clip_y+dynax_clip_height);
#else
	xclip	=	(x < dynax_clip_x) || (x > dynax_clip_width);
	yclip	=	(y < dynax_clip_y) || (y > dynax_clip_height);
#endif

	if (!(dynax_clip_ctrl & 1) &&  xclip) return;
	if (!(dynax_clip_ctrl & 2) && !xclip) return;
	if (!(dynax_clip_ctrl & 4) &&  yclip) return;
	if (!(dynax_clip_ctrl & 8) && !yclip) return;

	addr = 512 * y + x;

	if (dynax_dest_layer & 0x0001) pixmap[0][addr] = pen;
	if (dynax_dest_layer & 0x0002) pixmap[1][addr] = pen;
	if (dynax_dest_layer & 0x0004) pixmap[2][addr] = pen;
	if (dynax_dest_layer & 0x0008) pixmap[3][addr] = pen;

	if (!extra_layers)	return;

	if (dynax_dest_layer & 0x0100) pixmap[4][addr] = pen;
	if (dynax_dest_layer & 0x0200) pixmap[5][addr] = pen;
	if (dynax_dest_layer & 0x0400) pixmap[6][addr] = pen;
	if (dynax_dest_layer & 0x0800) pixmap[7][addr] = pen;
}


INLINE int fetch_bit(UINT8 *src_data,int src_len,int *bit_addr)
{
	int baddr = *bit_addr;

	*bit_addr = ((*bit_addr) + 1) & 0x7ffffff;

	if (baddr/8 >= src_len)
	{
#ifdef MAME_DEBUG
		usrintf_showmessage("GFX ROM OVER %06x",baddr/8);
#endif
		return 1;
	}

	return (src_data[baddr / 8] >> (7 - (baddr & 7))) & 1;
}

INLINE int fetch_word(UINT8 *src_data,int src_len,int *bit_addr,int word_len)
{
	int res = 0;

	while (word_len--)
	{
		res = (res << 1) | fetch_bit(src_data,src_len,bit_addr);
	}
	return res;
}



INLINE void log_draw_error(int src, int cmd)
{
	usrintf_showmessage("%06x: warning unknown pixel command %02x",src,cmd);
	logerror("%06x: warning unknown pixel command %02x\n",src,cmd);
}

/*  Copy from ROM
    initialized arguments are
    0D/0E/0F source data pointer
    14 X
    02 Y
    00 dest layer
    05 unknown, related to layer
    04 blit_pen
    06 blit_pen_mode (replace values stored in ROM)
*/

static int blit_draw(int src,int sx)
{
	UINT8 *src_data = memory_region(REGION_GFX1);
	int src_len = memory_region_length(REGION_GFX1);
	int bit_addr = (src & 0xffffff) * dynax_blit_rom_bits;	/* convert to bit address */
	int pen_size, arg_size, cmd;
	int x;
	int xinc = (dynax_blit_flip & 1) ? -1 : 1;
	int yinc = (dynax_blit_flip & 2) ? -1 : 1;

	pen_size = fetch_word(src_data,src_len,&bit_addr,4) + 1;
	arg_size = fetch_word(src_data,src_len,&bit_addr,4) + 1;

#ifdef MAME_DEBUG
	if (pen_size > 4 || arg_size > 8)
		usrintf_showmessage("warning: pen_size %d arg_size %d",pen_size,arg_size);
#endif

	x = sx;

	for (;;)
	{
		cmd = fetch_word(src_data,src_len,&bit_addr,3);
		switch ( dynax_blit_commands[cmd] )
		{
			case BLIT_NEXT:
				/* next line */
				dynax_blit_y += yinc;
				x = sx;
				break;

			case BLIT_LINE:
				{
					int length = fetch_word(src_data,src_len,&bit_addr,arg_size);
					int pen = fetch_word(src_data,src_len,&bit_addr,pen_size);
					if (dynax_blit_pen_mode) pen = (dynax_blit_pen & 0x0f);
					pen |= dynax_blit_pen & 0xf0;
					while (length-- >= 0)
					{
						do_plot(x,dynax_blit_y,pen);
						x += xinc;
					}
				}
				break;

			case BLIT_COPY:
				{
					int length = fetch_word(src_data,src_len,&bit_addr,arg_size);
					while (length-- >= 0)
					{
						int pen = fetch_word(src_data,src_len,&bit_addr,pen_size);
						if (dynax_blit_pen_mode) pen = (dynax_blit_pen & 0x0f);
						pen |= dynax_blit_pen & 0xf0;
						do_plot(x,dynax_blit_y,pen);
						x += xinc;
					}
				}
				break;

			case BLIT_SKIP:
				x += xinc * fetch_word(src_data,src_len,&bit_addr,arg_size);
				break;

			case BLIT_CHANGE_NUM:
				arg_size = fetch_word(src_data,src_len,&bit_addr,4) + 1;
				break;

			case BLIT_CHANGE_PEN:
				pen_size = fetch_word(src_data,src_len,&bit_addr,3) + 1;
				break;

			default:
				log_draw_error(src,cmd);
			// fall through
			case BLIT_STOP:
				return ((bit_addr + dynax_blit_rom_bits - 1) / dynax_blit_rom_bits) & 0xffffff;
		}
	}
}



/*  Draw a simple rectangle
*/
static void blit_rect_xywh(void)
{
	int x,y;

#ifdef MAME_DEBUG
//  if (dynax_clip_ctrl != 0x0f)
//      usrintf_showmessage("RECT clipx=%03x clipy=%03x ctrl=%x",dynax_clip_x,dynax_clip_y,dynax_clip_ctrl);
#endif

	for (y = 0; y <= dynax_rect_height; y++)
		for (x = 0; x <= dynax_rect_width; x++)
			do_plot( x + dynax_blit_x, y + dynax_blit_y, dynax_blit_pen);
}



/*  Unknown. Initialized arguments are
    00 dest layer
    05 unknown, related to layer
    14 X - always 0?
    02 Y
    0a width - always 0?
    0b height
    04 blit_pen
    0c line_length - always 0?
*/
static void blit_rect_yh(void)
{
	int start = 512 * dynax_blit_y;
	int length = 512 * (dynax_rect_height+1);

#ifdef MAME_DEBUG
//  if (dynax_clip_ctrl != 0x0f)
//      usrintf_showmessage("UNK8C clipx=%03x clipy=%03x ctrl=%x",dynax_clip_x,dynax_clip_y,dynax_clip_ctrl);
#endif

	if (start < 512*512)
	{
		if (start + length > 512*512)
			length = 512*512 - start;

		if (dynax_dest_layer & 0x0001) memset(pixmap[0] + start,dynax_blit_pen,length);
		if (dynax_dest_layer & 0x0002) memset(pixmap[1] + start,dynax_blit_pen,length);
		if (dynax_dest_layer & 0x0004) memset(pixmap[2] + start,dynax_blit_pen,length);
		if (dynax_dest_layer & 0x0008) memset(pixmap[3] + start,dynax_blit_pen,length);

		if (!extra_layers)	return;

		if (dynax_dest_layer & 0x0100) memset(pixmap[4] + start,dynax_blit_pen,length);
		if (dynax_dest_layer & 0x0200) memset(pixmap[5] + start,dynax_blit_pen,length);
		if (dynax_dest_layer & 0x0400) memset(pixmap[6] + start,dynax_blit_pen,length);
		if (dynax_dest_layer & 0x0800) memset(pixmap[7] + start,dynax_blit_pen,length);
	}
}



/*  Fill from (X,Y) to end of pixmap
    initialized arguments are
    00 dest layer
    05 unknown, related to layer
    14 X
    02 Y
    04 blit_pen
*/
static void blit_fill_xy(int x, int y)
{
	int start = 512 * y + x;

#ifdef MAME_DEBUG
//  if (x || y)
//      usrintf_showmessage("FILL command X %03x Y %03x",x,y);
#endif

	if (dynax_dest_layer & 0x0001) memset(pixmap[0] + start,dynax_blit_pen,512*512 - start);
	if (dynax_dest_layer & 0x0002) memset(pixmap[1] + start,dynax_blit_pen,512*512 - start);
	if (dynax_dest_layer & 0x0004) memset(pixmap[2] + start,dynax_blit_pen,512*512 - start);
	if (dynax_dest_layer & 0x0008) memset(pixmap[3] + start,dynax_blit_pen,512*512 - start);

	if (!extra_layers)	return;

	if (dynax_dest_layer & 0x0100) memset(pixmap[4] + start,dynax_blit_pen,512*512 - start);
	if (dynax_dest_layer & 0x0200) memset(pixmap[5] + start,dynax_blit_pen,512*512 - start);
	if (dynax_dest_layer & 0x0400) memset(pixmap[6] + start,dynax_blit_pen,512*512 - start);
	if (dynax_dest_layer & 0x0800) memset(pixmap[7] + start,dynax_blit_pen,512*512 - start);
}



/*  Draw horizontal line
    initialized arguments are
    00 dest layer
    05 unknown, related to layer
    14 X
    02 Y
    0c line length
    04 blit_pen
    dynax_blit_x and dynax_blit_y are left pointing to the last pixel at the end of the command
*/
static void blit_horiz_line(void)
{
	int i;

#ifdef MAME_DEBUG
	usrintf_showmessage("LINE X");

	if (dynax_clip_ctrl != 0x0f)
		usrintf_showmessage("LINE X clipx=%03x clipy=%03x ctrl=%x",dynax_clip_x,dynax_clip_y,dynax_clip_ctrl);

	if (dynax_blit_flip)
		usrintf_showmessage("LINE X flip=%x",dynax_blit_flip);
#endif

	for (i = 0; i <= dynax_line_length; i++)
		do_plot(dynax_blit_x++,dynax_blit_y,dynax_blit_pen);
}



/*  Draw vertical line
    initialized arguments are
    00 dest layer
    05 unknown, related to layer
    14 X
    02 Y
    0c line length
    04 blit_pen
    dynax_blit_x and dynax_blit_y are left pointing to the last pixel at the end of the command
*/
static void blit_vert_line(void)
{
	int i;

#ifdef MAME_DEBUG
	usrintf_showmessage("LINE Y");

	if (dynax_clip_ctrl != 0x0f)
		usrintf_showmessage("LINE Y clipx=%03x clipy=%03x ctrl=%x",dynax_clip_x,dynax_clip_y,dynax_clip_ctrl);
#endif

	for (i = 0; i <= dynax_line_length; i++)
		do_plot(dynax_blit_x,dynax_blit_y++,dynax_blit_pen);
}




INLINE void log_blit(int data)
{
#if 1
	logerror("%06x: blit src %06x x %03x y %03x flags %02x layer %02x pen %02x penmode %02x w %03x h %03x linelen %03x clip: ctrl %x xy %03x %03x wh %03x %03x\n",
			activecpu_get_pc(),
			dynax_blit_address,dynax_blit_x,dynax_blit_y,data,
			dynax_dest_layer,dynax_blit_pen,dynax_blit_pen_mode,dynax_rect_width,dynax_rect_height,dynax_line_length,
			dynax_clip_ctrl,dynax_clip_x,dynax_clip_y, dynax_clip_width,dynax_clip_height		);
#endif
}

static void blitter_w(int blitter, offs_t offset,UINT8 data,int irq_vector)
{
	static int dynax_blit_reg[2];
	int hi_bits;

profiler_mark(PROFILER_VIDEO);

	switch(offset)
	{
	case 0:
		dynax_blit_reg[blitter] = data;
		break;

	case 1:
		hi_bits = (dynax_blit_reg[blitter] & 0xc0) << 2;

		switch(dynax_blit_reg[blitter] & 0x3f)
		{
		case 0x00:
			if (blitter)	dynax_dest_layer = (dynax_dest_layer & 0x00ff) | (data<<8);
			else			dynax_dest_layer = (dynax_dest_layer & 0xff00) | (data<<0);
			break;

		case 0x01:
			dynax_flipscreen_w(data);
			break;

		case 0x02:
			dynax_blit_y = data | hi_bits;
			break;

		case 0x03:
			dynax_blit_flip_w(data);
			break;

		case 0x04:
			dynax_blit_pen = data;
			break;

		case 0x05:
			dynax_blit_pen_mask = data;
			break;

		case 0x06:
			// related to pen, can be 0 or 1 for 0x10 blitter command
			// 0 = only bits 7-4 of dynax_blit_pen contain data
			// 1 = bits 3-0 contain data as well
			dynax_blit_pen_mode = data;
			break;

		case 0x0a:
			dynax_rect_width = data | hi_bits;
			break;

		case 0x0b:
			dynax_rect_height = data | hi_bits;
			break;

		case 0x0c:
			dynax_line_length = data | hi_bits;
			break;

		case 0x0d:
			dynax_blit_address = (dynax_blit_address & 0xffff00) | (data <<0);
			break;
		case 0x0e:
			dynax_blit_address = (dynax_blit_address & 0xff00ff) | (data <<8);
			break;
		case 0x0f:
			dynax_blit_address = (dynax_blit_address & 0x00ffff) | (data<<16);
			break;

		case 0x14:
			dynax_blit_x = data | hi_bits;
			break;

		case 0x16:
			dynax_clip_x = data | hi_bits;
			break;

		case 0x17:
			dynax_clip_y = data | hi_bits;
			break;

		case 0x18:
		case 0x19:
		case 0x1a:
		case 0x1b:
		case 0x1c:
		case 0x1d:
		case 0x1e:
		case 0x1f:
			dynax_scroll[blitter*8 + (dynax_blit_reg[blitter] & 7)] = data | hi_bits;
			break;

		case 0x20:
			dynax_clip_ctrl = data;
			break;

		case 0x24:

			log_blit(data);

			switch (data)
			{
				case 0x04:	blit_fill_xy(0, 0);
							break;
				case 0x14:	blit_fill_xy(dynax_blit_x, dynax_blit_y);
							break;

				case 0x10:	dynax_blit_address = blit_draw(dynax_blit_address,dynax_blit_x);
							break;

				case 0x13:	blit_horiz_line();
							break;
				case 0x1b:	blit_vert_line();
							break;

				case 0x1c:	blit_rect_xywh();
							break;

				// These two are issued one after the other (43 then 8c)
				// 8c is issued immediately after 43 has finished, without
				// changing any argument
				case 0x43:	break;
				case 0x8c:	blit_rect_yh();
							break;

				default:
							;
				#ifdef MAME_DEBUG
					usrintf_showmessage("unknown blitter command %02x",data);
					logerror("%06x: unknown blitter command %02x\n", activecpu_get_pc(), data);
				#endif
			}

			if (irq_vector)
			{
				/* quizchq */
				cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, irq_vector);
			}
			else
			{
				/* ddenlovr */
				if (dynax_blitter_irq_enable)
				{
					dynax_blitter_irq_flag = 1;
					cpu_set_irq_line(0,1,HOLD_LINE);
				}
			}
			break;

		default:
			logerror("%06x: Blitter %d reg %02x = %02x\n", activecpu_get_pc(), blitter, dynax_blit_reg[blitter], data);
			break;
		}
	}

profiler_mark(PROFILER_END);
}




// differences wrt blitter_data_w: slightly different blitter commands
static void blitter_w_funkyfig(int blitter, offs_t offset,UINT8 data,int irq_vector)
{
	static int dynax_blit_reg[2];
	int hi_bits;

profiler_mark(PROFILER_VIDEO);

	switch(offset)
	{
	case 0:
		dynax_blit_reg[blitter] = data;
		break;

	case 1:
		hi_bits = (dynax_blit_reg[blitter] & 0xc0) << 2;

		switch(dynax_blit_reg[blitter] & 0x3f)
		{
		case 0x00:
			if (blitter)	dynax_dest_layer = (dynax_dest_layer & 0x00ff) | (data<<8);
			else			dynax_dest_layer = (dynax_dest_layer & 0xff00) | (data<<0);
			break;

		case 0x01:
			dynax_flipscreen_w(data);
			break;

		case 0x02:
			dynax_blit_y = data | hi_bits;
			break;

		case 0x03:
			dynax_blit_flip_w(data);
			break;

		case 0x04:
			dynax_blit_pen = data;
			break;

		case 0x05:
			dynax_blit_pen_mask = data;
			break;

		case 0x06:
			// related to pen, can be 0 or 1 for 0x10 blitter command
			// 0 = only bits 7-4 of dynax_blit_pen contain data
			// 1 = bits 3-0 contain data as well
			dynax_blit_pen_mode = data;
			break;

		case 0x0a:
			dynax_rect_width = data | hi_bits;
			break;

		case 0x0b:
			dynax_rect_height = data | hi_bits;
			break;

		case 0x0c:
			dynax_line_length = data | hi_bits;
			break;

		case 0x0d:
			dynax_blit_address = (dynax_blit_address & 0xffff00) | (data <<0);
			break;
		case 0x0e:
			dynax_blit_address = (dynax_blit_address & 0xff00ff) | (data <<8);
			break;
		case 0x0f:
			dynax_blit_address = (dynax_blit_address & 0x00ffff) | (data<<16);
			break;

		case 0x14:
			dynax_blit_x = data | hi_bits;
			break;

		case 0x16:
			dynax_clip_x = data | hi_bits;
			break;

		case 0x17:
			dynax_clip_y = data | hi_bits;
			break;

		case 0x18:
		case 0x19:
		case 0x1a:
		case 0x1b:
		case 0x1c:
		case 0x1d:
		case 0x1e:
		case 0x1f:
			dynax_scroll[blitter*8 + (dynax_blit_reg[blitter] & 7)] = data | hi_bits;
			break;

		case 0x20:
			dynax_clip_ctrl = data;
			break;

		case 0x24:

			log_blit(data);

			switch (data)
			{

				case 0x84:	// same as 04?
				case 0x04:	blit_fill_xy(0, 0);
							break;

//              unused?
//              case 0x14:  blit_fill_xy(dynax_blit_x, dynax_blit_y);
//                          break;

				case 0x00/*0x10*/:	dynax_blit_address = blit_draw(dynax_blit_address,dynax_blit_x);
							break;

				case 0x0b:	// same as 03? see the drawing of the R in "cRoss hatch" (key test)
				case 0x03/*0x13*/:	blit_horiz_line();
							break;
//              unused?
//              case 0x1b:  blit_vert_line();
//                          break;

				case 0x0c/*0x1c*/:	blit_rect_xywh();
							break;

				// These two are issued one after the other (43 then 8c)
				// 8c is issued immediately after 43 has finished, without
				// changing any argument
				case 0x43:	break;
				case 0x8c:	blit_rect_yh();
							break;

				default:
							;
				#ifdef MAME_DEBUG
					usrintf_showmessage("unknown blitter command %02x",data);
					logerror("%06x: unknown blitter command %02x\n", activecpu_get_pc(), data);
				#endif
			}

			cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, irq_vector);
			break;

		default:
			logerror("%06x: Blitter %d reg %02x = %02x\n", activecpu_get_pc(), blitter, dynax_blit_reg[blitter], data);
			break;
		}
	}

profiler_mark(PROFILER_END);
}




static WRITE8_HANDLER( hanakanz_blitter_reg_w )
{
	dynax_blit_reg = data;
}

// differences wrt blitter_data_w: registers are shuffled around, hi_bits in the low bits, clip_w/h, includes layers registers
static WRITE8_HANDLER( hanakanz_blitter_data_w )
{
	int hi_bits;

profiler_mark(PROFILER_VIDEO);

	hi_bits = (dynax_blit_reg & 0x03) << 8;

	switch(dynax_blit_reg & 0xfe)
	{
		case 0x00:
			dynax_dest_layer = data;
			break;

		case 0x04:
			dynax_flipscreen_w(data);
			break;

		case 0x08:
			dynax_blit_y = data | hi_bits;
			break;

		case 0x0c:
			dynax_blit_flip_w(data);
			break;

		case 0x10:
			dynax_blit_pen = data;
			break;

		case 0x14:
			dynax_blit_pen_mask = data;
			break;

		case 0x18:
			// related to pen, can be 0 or 1 for 0x10 blitter command
			// 0 = only bits 7-4 of dynax_blit_pen contain data
			// 1 = bits 3-0 contain data as well
			dynax_blit_pen_mode = data;
			break;

		case 0x28:
			dynax_rect_width = data | hi_bits;
			break;

		case 0x2c:
			dynax_rect_height = data | hi_bits;
			break;

		case 0x30:
			dynax_line_length = data | hi_bits;
			break;

		case 0x34:
			dynax_blit_address = (dynax_blit_address & 0xffff00) | (data <<0);
			break;
		case 0x38:
			dynax_blit_address = (dynax_blit_address & 0xff00ff) | (data <<8);
			break;
		case 0x3c:
			dynax_blit_address = (dynax_blit_address & 0x00ffff) | (data<<16);
			break;

		case 0x50:
			dynax_blit_x = data | hi_bits;
			break;

		case 0x58:
			dynax_clip_x = data | hi_bits;
			break;

		case 0x5c:
			dynax_clip_y = data | hi_bits;
			break;

		case 0x60:
		case 0x64:
		case 0x68:
		case 0x6c:
		case 0x70:
		case 0x74:
		case 0x78:
		case 0x7c:
			dynax_scroll[(dynax_blit_reg & 0x1c) >> 2] = data | hi_bits;
			break;

		case 0x80:
			dynax_clip_ctrl = data;
			break;

		case 0x88:
		case 0x8a:	// can be 3ff
			dynax_clip_height = data | hi_bits;
			break;

		case 0x8c:
		case 0x8e:	// can be 3ff
			dynax_clip_width = data | hi_bits;
			break;

		case 0xc0:
		case 0xc2:
		case 0xc4:
		case 0xc6:
			dynax_palette_base[(dynax_blit_reg >> 1) & 3] = data | (hi_bits & 0x100);
			break;

		case 0xc8:
		case 0xca:
		case 0xcc:
		case 0xce:
			dynax_palette_mask[(dynax_blit_reg >> 1) & 3] = data;
			break;

		case 0xd0:
		case 0xd2:
		case 0xd4:
		case 0xd6:
			dynax_transparency_pen[(dynax_blit_reg >> 1) & 3] = data;
			break;

		case 0xd8:
		case 0xda:
		case 0xdc:
		case 0xde:
			dynax_transparency_mask[(dynax_blit_reg >> 1) & 3] = data;
			break;

		case 0xe4:
			dynax_priority_w(0,data);
			break;

		case 0xe6:
			dynax_layer_enable_w(0,data);
			break;

		case 0xe8:
			dynax_bgcolor = data | hi_bits;
			break;

		case 0x90:

			log_blit(data);

			switch (data)
			{
				case 0x04:	blit_fill_xy(0, 0);
							break;
				case 0x14:	blit_fill_xy(dynax_blit_x, dynax_blit_y);
							break;

				case 0x10:	dynax_blit_address = blit_draw(dynax_blit_address,dynax_blit_x);
							break;

				case 0x13:	blit_horiz_line();
							break;
				case 0x1b:	blit_vert_line();
							break;

				case 0x1c:	blit_rect_xywh();
							break;

				// These two are issued one after the other (43 then 8c)
				// 8c is issued immediately after 43 has finished, without
				// changing any argument
				case 0x43:	break;
				case 0x8c:	blit_rect_yh();
							break;

				default:
							;
				#ifdef MAME_DEBUG
					usrintf_showmessage("unknown blitter command %02x",data);
					logerror("%06x: unknown blitter command %02x\n", activecpu_get_pc(), data);
				#endif
			}

			// NO IRQ !?

			break;

		default:
			logerror("%06x: Blitter 0 reg %02x = %02x\n", activecpu_get_pc(), dynax_blit_reg, data);
			break;
	}

profiler_mark(PROFILER_END);
}


static WRITE8_HANDLER( rongrong_blitter_w )
{
	blitter_w(0,offset,data,0xf8);
}

static WRITE16_HANDLER( ddenlovr_blitter_w )
{
	if (ACCESSING_LSB)
		blitter_w(0,offset,data & 0xff,0);
}


static WRITE16_HANDLER( ddenlovr_blitter_irq_ack_w )
{
	if (ACCESSING_LSB)
	{
		if (data & 1)
		{
			dynax_blitter_irq_enable = 1;
		}
		else
		{
			dynax_blitter_irq_enable = 0;
			dynax_blitter_irq_flag = 0;
		}
	}
}


static READ8_HANDLER( rongrong_gfxrom_r )
{
	UINT8 *rom	=	memory_region( REGION_GFX1 );
	size_t size	=	memory_region_length( REGION_GFX1 );
	int address	=	dynax_blit_address;

	if (address >= size)
	{
		logerror("CPU#0 PC %06X: Error, Blitter address %06X out of range\n", activecpu_get_pc(), address);
		address %= size;
	}

	dynax_blit_address = (dynax_blit_address + 1) & 0xffffff;

	return rom[address];
}

static READ16_HANDLER( ddenlovr_gfxrom_r )
{
	return rongrong_gfxrom_r(offset);
}



static void copylayer(struct mame_bitmap *bitmap,const struct rectangle *cliprect,int layer)
{
	int x,y;
	int scrollx = dynax_scroll[layer/4*8 + (layer%4) + 0];
	int scrolly = dynax_scroll[layer/4*8 + (layer%4) + 4];

	int palbase = dynax_palette_base[layer];
	int penmask = dynax_palette_mask[layer];

	int transpen = dynax_transparency_pen[layer];
	int transmask = dynax_transparency_mask[layer];

	palbase		&=	~penmask;
	transpen	&=	transmask;

	if (((dynax_layer_enable2 << 4) | dynax_layer_enable) & (1 << layer))
	{
		for (y = cliprect->min_y;y <= cliprect->max_y;y++)
		{
			for (x = cliprect->min_x;x <= cliprect->max_x;x++)
			{
				int pen = pixmap[layer][512 * ((y + scrolly) & 0x1ff) + ((x + scrollx) & 0x1ff)];
				if ( (pen & transmask) != transpen )
				{
					pen &= penmask;
					pen |= palbase;
					((UINT16 *)bitmap->line[y])[x] = pen;
				}
			}
		}
	}
}

VIDEO_UPDATE(ddenlovr)
{
	copybitmap(bitmap,framebuffer,0,0,0,0,cliprect,TRANSPARENCY_NONE,0);
}

/*
    I do the following in a eof handler, to avoid  palette/gfx synchronization
    issues with frameskipping
*/
VIDEO_EOF(ddenlovr)
{
	static const int order[24][4] =
	{
		{ 3,2,1,0 }, { 2,3,1,0 }, { 3,1,2,0 }, { 1,3,2,0 }, { 2,1,3,0 }, { 1,2,3,0 },
		{ 3,2,0,1 }, { 2,3,0,1 }, { 3,0,2,1 }, { 0,3,2,1 }, { 2,0,3,1 }, { 0,2,3,1 },
		{ 3,1,0,2 }, { 1,3,0,2 }, { 3,0,1,2 }, { 0,3,1,2 }, { 1,0,3,2 }, { 0,1,3,2 },
		{ 2,1,0,3 }, { 1,2,0,3 }, { 2,0,1,3 }, { 0,2,1,3 }, { 1,0,2,3 }, { 0,1,2,3 }
	};

	int pri;

	int enab = dynax_layer_enable;
	int enab2 = dynax_layer_enable2;

#if 0
	static int base = 0x0;

	int next;
	memset(pixmap[0],0,512*512);
	memset(pixmap[1],0,512*512);
	memset(pixmap[2],0,512*512);
	memset(pixmap[3],0,512*512);
	dynax_dest_layer = 8;
	dynax_blit_pen = 0;
	dynax_blit_pen_mode = 0;
	dynax_blit_y = 5;
	dynax_clip_ctrl = 0x0f;
	next = blit_draw(base,0);
	usrintf_showmessage("GFX %06x",base);
	if (input_code_pressed(KEYCODE_S)) base = next;
	if (input_code_pressed_once(KEYCODE_X)) base = next;
	if (input_code_pressed(KEYCODE_C)) { base--; while ((memory_region(REGION_GFX1)[base] & 0xf0) != 0x30) base--; }
	if (input_code_pressed(KEYCODE_V)) { base++; while ((memory_region(REGION_GFX1)[base] & 0xf0) != 0x30) base++; }
	if (input_code_pressed_once(KEYCODE_D)) { base--; while ((memory_region(REGION_GFX1)[base] & 0xf0) != 0x30) base--; }
	if (input_code_pressed_once(KEYCODE_F)) { base++; while ((memory_region(REGION_GFX1)[base] & 0xf0) != 0x30) base++; }
#endif

	fillbitmap(framebuffer,dynax_bgcolor,&Machine->visible_area);

#ifdef MAME_DEBUG
	if (input_code_pressed(KEYCODE_Z))
	{
		int mask,mask2;

		mask = 0;

		if (input_code_pressed(KEYCODE_Q))	mask |= 1;
		if (input_code_pressed(KEYCODE_W))	mask |= 2;
		if (input_code_pressed(KEYCODE_E))	mask |= 4;
		if (input_code_pressed(KEYCODE_R))	mask |= 8;

		mask2 = 0;

		if (extra_layers)
		{
			if (input_code_pressed(KEYCODE_A))	mask2 |= 1;
			if (input_code_pressed(KEYCODE_S))	mask2 |= 2;
			if (input_code_pressed(KEYCODE_D))	mask2 |= 4;
			if (input_code_pressed(KEYCODE_F))	mask2 |= 8;
		}

		if (mask || mask2)
		{
			dynax_layer_enable &= mask;
			dynax_layer_enable2 &= mask2;
		}
	}
#endif

	pri = dynax_priority;

		if (pri >= 24)
		{
			usrintf_showmessage("priority = %02x",pri);
			pri = 0;
		}

		copylayer(framebuffer,&Machine->visible_area,order[pri][0]);
		copylayer(framebuffer,&Machine->visible_area,order[pri][1]);
		copylayer(framebuffer,&Machine->visible_area,order[pri][2]);
		copylayer(framebuffer,&Machine->visible_area,order[pri][3]);

	if (extra_layers)
	{
		pri = dynax_priority2;

		if (pri >= 24)
		{
			usrintf_showmessage("priority2 = %02x",pri);
			pri = 0;
		}

		copylayer(framebuffer,&Machine->visible_area,order[pri][0]+4);
		copylayer(framebuffer,&Machine->visible_area,order[pri][1]+4);
		copylayer(framebuffer,&Machine->visible_area,order[pri][2]+4);
		copylayer(framebuffer,&Machine->visible_area,order[pri][3]+4);
	}

	dynax_layer_enable = enab;
	dynax_layer_enable2 = enab2;
}

READ16_HANDLER( ddenlovr_special_r )
{
	int res = readinputport(2) | (dynax_blitter_irq_flag << 6);

	return res;
}

static WRITE16_HANDLER( ddenlovr_coincounter_0_w )
{
	if (ACCESSING_LSB)
		coin_counter_w(0, data & 1);
}
static WRITE16_HANDLER( ddenlovr_coincounter_1_w )
{
	if (ACCESSING_LSB)
		coin_counter_w(1, data & 1);
}


static UINT8 palram[0x200];

static WRITE8_HANDLER( rongrong_palette_w )
{
	int r,g,b,d1,d2,indx;

	palram[offset] = data;

	indx = ((offset & 0x1e0) >> 1) | (offset & 0x00f);
	d1 = palram[offset & ~0x10];
	d2 = palram[offset |  0x10];

	r = d1 & 0x1f;
	g = d2 & 0x1f;
	/* what were they smoking??? */
	b = ((d1 & 0xe0) >> 5) | (d2 & 0xc0) >> 3;

	palette_set_color(indx,pal5bit(r),pal5bit(g),pal5bit(b));
}

static WRITE16_HANDLER( ddenlovr_palette_w )
{
	if (ACCESSING_LSB)
		rongrong_palette_w(offset,data & 0xff);
}


static WRITE8_HANDLER( dynax_palette_base_w )
{
	dynax_palette_base[offset] = data;
}
static WRITE8_HANDLER( dynax_palette_base2_w )
{
	dynax_palette_base[offset+4] = data;
}

static WRITE8_HANDLER( dynax_palette_mask_w )
{
	dynax_palette_mask[offset] = data;
}
static WRITE8_HANDLER( dynax_palette_mask2_w )
{
	dynax_palette_mask[offset+4] = data;
}

static WRITE8_HANDLER( dynax_transparency_pen_w )
{
	dynax_transparency_pen[offset] = data;
}
static WRITE8_HANDLER( dynax_transparency_pen2_w )
{
	dynax_transparency_pen[offset+4] = data;
}

static WRITE8_HANDLER( dynax_transparency_mask_w )
{
	dynax_transparency_mask[offset] = data;
}
static WRITE8_HANDLER( dynax_transparency_mask2_w )
{
	dynax_transparency_mask[offset+4] = data;
}


static WRITE16_HANDLER( ddenlovr_palette_base_w )
{
	if (ACCESSING_LSB)
		dynax_palette_base[offset] = data & 0xff;
}

static WRITE16_HANDLER( ddenlovr_palette_mask_w )
{
	if (ACCESSING_LSB)
		dynax_palette_mask[offset] = data & 0xff;
}

static WRITE16_HANDLER( ddenlovr_transparency_pen_w )
{
	if (ACCESSING_LSB)
		dynax_transparency_pen[offset] = data & 0xff;
}

static WRITE16_HANDLER( ddenlovr_transparency_mask_w )
{
	if (ACCESSING_LSB)
		dynax_transparency_mask[offset] = data & 0xff;
}


static WRITE8_HANDLER( quizchq_oki_bank_w )
{
	OKIM6295_set_bank_base(0, (data & 1) * 0x40000);
}

static WRITE16_HANDLER( ddenlovr_oki_bank_w )
{
	if (ACCESSING_LSB)
		OKIM6295_set_bank_base(0, (data & 7) * 0x40000);
}


static int okibank;

static WRITE16_HANDLER( quiz365_oki_bank1_w )
{
	if (ACCESSING_LSB)
	{
		okibank = (okibank & 2) | (data & 1);
		OKIM6295_set_bank_base(0, okibank * 0x40000);
	}
}

static WRITE16_HANDLER( quiz365_oki_bank2_w )
{
	if (ACCESSING_LSB)
	{
		okibank = (okibank & 1) | ((data & 1) << 1);
		OKIM6295_set_bank_base(0, okibank * 0x40000);
	}
}



static READ8_HANDLER( unk_r )
{
	return 0x78;
}

static READ16_HANDLER( unk16_r )
{
	return unk_r(offset);
}


static UINT8 dynax_select, dynax_select2;

WRITE8_HANDLER( dynax_select_w )
{
	dynax_select = data;
}

WRITE16_HANDLER( dynax_select_16_w )
{
	if (ACCESSING_LSB)
		dynax_select = data;
}

WRITE8_HANDLER( dynax_select2_w )
{
	dynax_select2 = data;
}

WRITE16_HANDLER( dynax_select2_16_w )
{
	if (ACCESSING_LSB)
		dynax_select2 = data;
}

READ8_HANDLER( rongrong_input2_r )
{
//  logerror("%04x: input2_r offset %d select %x\n",activecpu_get_pc(),offset,dynax_select2 );
	/* 0 and 1 are read from offset 1, 2 from offset 0... */
	switch( dynax_select2 )
	{
		case 0x00:	return readinputport(0);
		case 0x01:	return readinputport(1);
		case 0x02:	return readinputport(2);
	}
	return 0xff;
}


READ8_HANDLER( quiz365_input_r )
{
	if (!(dynax_select & 0x01))	return readinputport(3);
	if (!(dynax_select & 0x02))	return readinputport(4);
	if (!(dynax_select & 0x04))	return readinputport(5);
	if (!(dynax_select & 0x08))	return 0xff;//mame_rand(Machine);
	if (!(dynax_select & 0x10))	return 0xff;//mame_rand(Machine);
	return 0xff;
}

READ16_HANDLER( quiz365_input2_r )
{
//  logerror("%04x: input2_r offset %d select %x\n",activecpu_get_pc(),offset,dynax_select2 );
	/* 0 and 1 are read from offset 1, 2 from offset 0... */
	switch( dynax_select2 )
	{
		case 0x10:	return readinputport(0);
		case 0x11:	return readinputport(1);
		case 0x12:	return readinputport(2) | (dynax_blitter_irq_flag << 6);
	}
	return 0xff;
}

static UINT8 rongrong_blitter_busy_select;

WRITE8_HANDLER( rongrong_blitter_busy_w )
{
	rongrong_blitter_busy_select = data;
	if (data != 0x18)
		logerror("%04x: rongrong_blitter_busy_w data = %02x\n",activecpu_get_pc(),data);
}
READ8_HANDLER( rongrong_blitter_busy_r )
{
	switch( rongrong_blitter_busy_select )
	{
		case 0x18:	return 0;	// bit 5 = blitter busy

		default:
			logerror("%04x: rongrong_blitter_busy_r with select = %02x\n",activecpu_get_pc(),rongrong_blitter_busy_select);
	}
	return 0xff;
}


static WRITE16_HANDLER( quiz365_coincounter_w )
{
	if (ACCESSING_LSB)
	{
		if (dynax_select2 == 0x1c)
		{
			coin_counter_w(0, ~data & 1);
			coin_counter_w(1, ~data & 4);
		}
	}
}

/*
37,28,12    11      ->      88
67,4c,3a    ??      ->      51
*/
static UINT16 quiz365_protection[2];
static READ16_HANDLER( quiz365_protection_r )
{
	switch(quiz365_protection[0])
	{
		case 0x3a:
			return 0x0051;
		default:
			return 0x0088;
	}
}
static WRITE16_HANDLER( quiz365_protection_w )
{
	COMBINE_DATA(quiz365_protection + offset);
}

static MEMORY_READ16_START( quiz365_readmem )
	{ 0x000000, 0x17ffff, MRA16_ROM						},	// ROM
	{ 0x200c02, 0x200c03, quiz365_protection_r			},	// Protection
	{ 0x300204, 0x300207, quiz365_input2_r				},	//
	{ 0x300270, 0x300271, unk16_r						},	// ? must be 78 on startup (not necessary in ddlover)
	{ 0x300286, 0x300287, ddenlovr_gfxrom_r				},	// Video Chip
	{ 0x3002c0, 0x3002c1, OKIM6295_status_0_lsb_r		},	// Sound
	{ 0x300340, 0x30035f, msm6242_lsb_r					},	// 6242RTC
	{ 0x300384, 0x300385, AY8910_read_port_0_lsb_r		},
	{ 0xff0000, 0xffffff, MRA16_RAM						},	// RAM
MEMORY_END

static MEMORY_WRITE16_START( quiz365_writemem )
	{ 0x000000, 0x17ffff, MWA16_ROM						},	// ROM
	{ 0x200000, 0x2003ff, ddenlovr_palette_w			},	// Palette
	{ 0x200e0a, 0x200e0d, quiz365_protection_w			},	// Protection
//	{ 0x201000, 0x2017ff, MWA16_RAM 					},	// ?
	{ 0x300200, 0x300201, dynax_select2_16_w			},	//
	{ 0x300202, 0x300203, quiz365_coincounter_w			},	// Coin Counters + more stuff written on startup
	{ 0x300240, 0x300247, ddenlovr_palette_base_w		},
	{ 0x300248, 0x30024f, ddenlovr_palette_mask_w		},
	{ 0x300250, 0x300257, ddenlovr_transparency_pen_w	},
	{ 0x300258, 0x30025f, ddenlovr_transparency_mask_w	},
	{ 0x300268, 0x300269, ddenlovr_bgcolor_w			},
	{ 0x30026a, 0x30026b, ddenlovr_priority_w			},
	{ 0x30026c, 0x30026d, ddenlovr_layer_enable_w		},
	{ 0x300280, 0x300283, ddenlovr_blitter_w			},
	{ 0x300300, 0x300301, YM2413_register_port_0_lsb_w	},	// Sound
	{ 0x300302, 0x300303, YM2413_data_port_0_lsb_w		},	//
	{ 0x300340, 0x30035f, msm6242_lsb_w					},	// 6242RTC
	{ 0x3003ca, 0x3003cb, ddenlovr_blitter_irq_ack_w	},	// Blitter irq acknowledge
	{ 0x300380, 0x300381, AY8910_control_port_0_lsb_w	},
	{ 0x300382, 0x300383, AY8910_write_port_0_lsb_w		},
	{ 0x3002c0, 0x3002c1, OKIM6295_data_0_lsb_w 		},	//
	{ 0x3003c2, 0x3003c3, quiz365_oki_bank1_w			},
	{ 0x3003cc, 0x3003cd, quiz365_oki_bank2_w			},
	{ 0xff0000, 0xffffff, MWA16_RAM						},	// RAM
MEMORY_END


static UINT16 *ddenlvrj_dsw_sel;

static READ16_HANDLER( ddenlvrj_dsw_r )
{
	UINT16 dsw = 0;
	if ((~*ddenlvrj_dsw_sel) & 0x01)	dsw |= readinputport(3);	// dsw 1
	if ((~*ddenlvrj_dsw_sel) & 0x02)	dsw |= readinputport(4);	// dsw 2
	if ((~*ddenlvrj_dsw_sel) & 0x04)	dsw |= readinputport(5);	// ?
	return dsw;
}

static WRITE16_HANDLER( ddenlvrj_coincounter_w )
{
	if (ACCESSING_LSB)
	{
		coin_counter_w(0, data & 0x01);
		coin_counter_w(1, data & 0x04);
		//                data & 0x80 ?
	}
}

static READ16_HANDLER( ddenlvrj_blitter_r )
{
	return readinputport(2) | (dynax_blitter_irq_flag ? 0x60 : 0x00 );	// bit 4 = 1 -> blitter busy
}

static MEMORY_READ16_START( ddenlvrj_readmem )
	{ 0x000000, 0x07ffff, MRA16_ROM						},	// ROM
	{ 0x300070, 0x300071, unk16_r						},	// ? must be 78 on startup (not necessary in ddlover)
	{ 0x300086, 0x300087, ddenlovr_gfxrom_r				},	// Video Chip
	{ 0x300100, 0x30011f, msm6242_lsb_r					},	// 6242RTC
	{ 0x300180, 0x300181, input_port_0_word_r			},	// P1
	{ 0x300182, 0x300183, input_port_1_word_r			},	// P2
	{ 0x300184, 0x300185, ddenlvrj_blitter_r			},	// Coins + ?
	{ 0x300186, 0x300187, ddenlvrj_dsw_r				},	// DSW
	{ 0x300240, 0x300241, OKIM6295_status_0_lsb_r		},	// Sound
	{ 0xff0000, 0xffffff, MRA16_RAM						},	// RAM
MEMORY_END

static MEMORY_WRITE16_START( ddenlvrj_writemem )
	{ 0x000000, 0x07ffff, MWA16_ROM						},	// ROM
	{ 0x200000, 0x2003ff, ddenlovr_palette_w			},	// Palette
//  { 0x201000, 0x2017ff, MWA16_RAM                     },   // ? B0 on startup, then 00
	{ 0x300040, 0x300047, ddenlovr_palette_base_w		},
	{ 0x300048, 0x30004f, ddenlovr_palette_mask_w		},
	{ 0x300050, 0x300057, ddenlovr_transparency_pen_w	},
	{ 0x300058, 0x30005f, ddenlovr_transparency_mask_w	},
	{ 0x300068, 0x300069, ddenlovr_bgcolor_w			},
	{ 0x30006a, 0x30006b, ddenlovr_priority_w			},
	{ 0x30006c, 0x30006d, ddenlovr_layer_enable_w		},
	{ 0x300080, 0x300083, ddenlovr_blitter_w			},
	{ 0x3000c0, 0x3000c1, YM2413_register_port_0_lsb_w	},
	{ 0x3000c2, 0x3000c3, YM2413_data_port_0_lsb_w		},
	{ 0x300100, 0x30011f, msm6242_lsb_w					},	// 6242RTC
	{ 0x300140, 0x300141, AY8910_control_port_0_lsb_w	},
	{ 0x300142, 0x300143, AY8910_write_port_0_lsb_w		},
	{ 0x300188, 0x300189, ddenlvrj_coincounter_w		},	// Coin Counters
	{ 0x30018a, 0x30018b, MWA16_RAM, &ddenlvrj_dsw_sel	},	// DSW select
	{ 0x30018c, 0x30018d, ddenlovr_oki_bank_w			},
	{ 0x3001ca, 0x3001cb, ddenlovr_blitter_irq_ack_w	},	// Blitter irq acknowledge
	{ 0x300240, 0x300241, OKIM6295_data_0_lsb_w 		},
	{ 0xff0000, 0xffffff, MWA16_RAM						},	// RAM
MEMORY_END

static MEMORY_READ16_START( ddenlovr_readmem )
	{ 0x000000, 0x07ffff, MRA16_ROM						},	// ROM
	{ 0xe00086, 0xe00087, ddenlovr_gfxrom_r				},	// Video Chip
	{ 0xe00070, 0xe00071, unk16_r						},	// ? must be 78 on startup (not necessary in ddlover)
	{ 0xe00100, 0xe00101, input_port_0_word_r			},	// P1?
	{ 0xe00102, 0xe00103, input_port_1_word_r			},	// P2?
	{ 0xe00104, 0xe00105, ddenlovr_special_r			},	// Coins + ?
	{ 0xe00200, 0xe00201, input_port_3_word_r			},	// DSW
	{ 0xe00500, 0xe0051f, msm6242_lsb_r					},	// 6242RTC
	{ 0xe00604, 0xe00605, AY8910_read_port_0_lsb_r		},
	{ 0xe00700, 0xe00701, OKIM6295_status_0_lsb_r		},	// Sound
	{ 0xff0000, 0xffffff, MRA16_RAM						},	// RAM
MEMORY_END

static MEMORY_WRITE16_START( ddenlovr_writemem )
	{ 0x000000, 0x07ffff, MWA16_ROM						},	// ROM
	{ 0x300000, 0x300001, ddenlovr_oki_bank_w			},
	{ 0xd00000, 0xd003ff, ddenlovr_palette_w			},	// Palette
//	{ 0xd01000, 0xd017ff, MWA16_RAM 					},	// ? B0 on startup, then 00
	{ 0xe00040, 0xe00047, ddenlovr_palette_base_w		},
	{ 0xe00048, 0xe0004f, ddenlovr_palette_mask_w		},
	{ 0xe00050, 0xe00057, ddenlovr_transparency_pen_w	},
	{ 0xe00058, 0xe0005f, ddenlovr_transparency_mask_w	},
	{ 0xe00068, 0xe00069, ddenlovr_bgcolor_w			},
	{ 0xe0006a, 0xe0006b, ddenlovr_priority_w			},
	{ 0xe0006c, 0xe0006d, ddenlovr_layer_enable_w		},
	{ 0xe00080, 0xe00083, ddenlovr_blitter_w			},
	{ 0xe00302, 0xe00303, ddenlovr_blitter_irq_ack_w	},	// Blitter irq acknowledge
	{ 0xe00308, 0xe00309, ddenlovr_coincounter_0_w		},	// Coin Counters
	{ 0xe0030c, 0xe0030d, ddenlovr_coincounter_1_w		},	//
	{ 0xe00400, 0xe00401, YM2413_register_port_0_lsb_w	},	// Sound
	{ 0xe00402, 0xe00403, YM2413_data_port_0_lsb_w		},	//
	{ 0xe00500, 0xe0051f, msm6242_lsb_w					},	// 6242RTC
//	{ 0xe00302, 0xe00303, MWA16_NOP						},	// ?
	{ 0xe00600, 0xe00601, AY8910_control_port_0_lsb_w	},
	{ 0xe00602, 0xe00603, AY8910_write_port_0_lsb_w		},
	{ 0xe00700, 0xe00701, OKIM6295_data_0_lsb_w 		},	//
	{ 0xff0000, 0xffffff, MWA16_RAM						},	// RAM
MEMORY_END


static READ16_HANDLER( nettoqc_special_r )
{
	return readinputport(2) | (dynax_blitter_irq_flag ? 0x60 : 0x00 );
}

static READ16_HANDLER( nettoqc_input_r )
{
	if (!(dynax_select & 0x01))	return readinputport(3);
	if (!(dynax_select & 0x02))	return readinputport(4);
	if (!(dynax_select & 0x04))	return readinputport(5);
	return 0xffff;
}

/*
    Protection:

    Writes 37 28 12 to 200e0b then 11 to 200e0d. Expects to read 88 from 200c03
    Writes 67 4c 3a to 200e0b then 19 to 200e0d. Expects to read 51 from 200c03
*/

static UINT16 *nettoqc_protection_val;

static READ16_HANDLER( nettoqc_protection_r )
{
	switch( nettoqc_protection_val[0] & 0xff )
	{
		case 0x3a:	return 0x0051;
		default:	return 0x0088;
	}
}

static WRITE16_HANDLER( nettoqc_coincounter_w )
{
	if (ACCESSING_LSB)
	{
		coin_counter_w(0, data & 0x01);
		coin_counter_w(1, data & 0x04);
		//                data & 0x80 ?
	}
}

static WRITE16_HANDLER( nettoqc_oki_bank_w )
{
	if (ACCESSING_LSB)
		OKIM6295_set_bank_base(0, (data & 3) * 0x40000);
}

static MEMORY_READ16_START( nettoqc_readmem )
	{ 0x000000, 0x17ffff, MRA16_ROM								},	// ROM
	{ 0x200c02, 0x200c03, nettoqc_protection_r					},	//
	{ 0x300070, 0x300071, unk16_r								},	// ? must be 78 on startup (not necessary in ddlover)
	{ 0x300086, 0x300087, ddenlovr_gfxrom_r						},	// Video Chip
	{ 0x300100, 0x30011f, msm6242_lsb_r							},	// 6242RTC
	{ 0x300180, 0x300181, input_port_0_word_r					},	//
	{ 0x300182, 0x300183, input_port_1_word_r					},	//
	{ 0x300184, 0x300185, nettoqc_special_r						},	// Coins + ?
	{ 0x300186, 0x300187, nettoqc_input_r						},	// DSW's
	{ 0x300240, 0x300241, OKIM6295_status_0_lsb_r				},	// Sound
	{ 0xff0000, 0xffffff, MRA16_RAM								},	// RAM
MEMORY_END

static MEMORY_WRITE16_START( nettoqc_writemem )
	{ 0x000000, 0x17ffff, MWA16_ROM								},	// ROM
	{ 0x200000, 0x2003ff, ddenlovr_palette_w					},	// Palette
	{ 0x200e0a, 0x200e0d, MWA16_RAM, &nettoqc_protection_val	},	//
	{ 0x201000, 0x2017ff, MWA16_RAM 							},	// ?
	{ 0x300040, 0x300047, ddenlovr_palette_base_w				},	// palette base for the 4 layers
	{ 0x300048, 0x30004f, ddenlovr_palette_mask_w				},
	{ 0x300050, 0x300057, ddenlovr_transparency_pen_w			},
	{ 0x300058, 0x30005f, ddenlovr_transparency_mask_w			},
	{ 0x300068, 0x300069, ddenlovr_bgcolor_w					},
	{ 0x30006a, 0x30006b, ddenlovr_priority_w					},
	{ 0x30006c, 0x30006d, ddenlovr_layer_enable_w				},
	{ 0x300080, 0x300083, ddenlovr_blitter_w					},
	{ 0x3000c0, 0x3000c1, YM2413_register_port_0_lsb_w			},	// Sound
	{ 0x3000c2, 0x3000c3, YM2413_data_port_0_lsb_w				},	//
	{ 0x300100, 0x30011f, msm6242_lsb_w							},	// 6242RTC
	{ 0x300140, 0x300141, AY8910_control_port_0_lsb_w			},	//
	{ 0x300142, 0x300143, AY8910_write_port_0_lsb_w				},	//
	{ 0x300188, 0x300189, nettoqc_coincounter_w					},	// Coin Counters
	{ 0x30018a, 0x30018b, dynax_select_16_w						},	//
	{ 0x30018c, 0x30018d, nettoqc_oki_bank_w					},
	{ 0x3001ca, 0x3001cb, ddenlovr_blitter_irq_ack_w			},	// Blitter irq acknowledge
	{ 0x300240, 0x300241, OKIM6295_data_0_lsb_w 				},	//
	{ 0xff0000, 0xffffff, MWA16_RAM								},	// RAM
MEMORY_END


/***************************************************************************
                                Rong Rong
***************************************************************************/

READ8_HANDLER( rongrong_input_r )
{
	if (!(dynax_select & 0x01))	return readinputport(3);
	if (!(dynax_select & 0x02))	return readinputport(4);
	if (!(dynax_select & 0x04))	return 0xff;//mame_rand(Machine);
	if (!(dynax_select & 0x08))	return 0xff;//mame_rand(Machine);
	if (!(dynax_select & 0x10))	return readinputport(5);
	return 0xff;
}

WRITE8_HANDLER( rongrong_select_w )
{
	UINT8 *rom = memory_region(REGION_CPU1);

//logerror("%04x: rongrong_select_w %02x\n",activecpu_get_pc(),data);
	/* bits 0-4 = **both** ROM bank **AND** input select */
	memory_set_bankptr(1, &rom[0x10000 + 0x8000 * (data & 0x1f)]);
	dynax_select = data;

	/* bits 5-7 = RAM bank */
	memory_set_bankptr(2, &rom[0x110000 + 0x1000 * ((data & 0xe0) >> 5)]);
}



static MEMORY_READ_START( quizchq_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},	// ROM
	{ 0x6000, 0x6fff, MRA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM (Banked)
MEMORY_END

static MEMORY_WRITE_START( quizchq_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0x81ff, rongrong_palette_w		},	// ROM (Banked)
MEMORY_END

static PORT_READ_START( quizchq_readport )
	{ 0x03, 0x03, rongrong_gfxrom_r				},	// Video Chip
	{ 0x1b, 0x1b, rongrong_blitter_busy_r		},
	{ 0x1c, 0x1c, rongrong_input_r				},	//
	{ 0x22, 0x23, rongrong_input2_r				},	//
	{ 0x40, 0x40, OKIM6295_status_0_r			},	//
	{ 0x98, 0x98, unk_r							},	// ? must be 78 on startup
	{ 0xa0, 0xaf, msm6242_r						},	// 6242RTC
PORT_END

static PORT_WRITE_START( quizchq_writeport )
	{ 0x00, 0x01, rongrong_blitter_w			},
	{ 0x1b, 0x1b, rongrong_blitter_busy_w		},
	{ 0x1e, 0x1e, rongrong_select_w				},	//
	{ 0x20, 0x20, dynax_select2_w				},	//
	{ 0x40, 0x40, OKIM6295_data_0_w				},	//
	{ 0x60, 0x60, YM2413_register_port_0_w		},	// Sound
	{ 0x61, 0x61, YM2413_data_port_0_w			},	//
	{ 0x80, 0x83, dynax_palette_base_w			},
	{ 0x84, 0x87, dynax_palette_mask_w			},
	{ 0x88, 0x8b, dynax_transparency_pen_w		},
	{ 0x8c, 0x8f, dynax_transparency_mask_w		},
	{ 0x94, 0x94, dynax_bgcolor_w				},
	{ 0x95, 0x95, dynax_priority_w				},
	{ 0x96, 0x96, dynax_layer_enable_w			},
	{ 0xa0, 0xaf, msm6242_w						},	// 6242RTC
	{ 0xc0, 0xc0, quizchq_oki_bank_w			},
	{ 0xc2, 0xc2, IOWP_NOP						},	// enables palette RAM at f000
PORT_END



static MEMORY_READ_START( rongrong_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},	// ROM
	{ 0x6000, 0x6fff, MRA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM (Banked)
MEMORY_END

static MEMORY_WRITE_START( rongrong_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
	{ 0xf000, 0xf1ff, rongrong_palette_w		},
MEMORY_END

static PORT_READ_START( rongrong_readport )
	{ 0x03, 0x03, rongrong_gfxrom_r				},	// Video Chip
	{ 0x1b, 0x1b, rongrong_blitter_busy_r		},
	{ 0x1c, 0x1c, rongrong_input_r				},	//
	{ 0x20, 0x2f, msm6242_r 					},	// 6242RTC
	{ 0x40, 0x40, OKIM6295_status_0_r			},	//
	{ 0x98, 0x98, unk_r							},	// ? must be 78 on startup
	{ 0xa2, 0xa3, rongrong_input2_r				},	//
PORT_END

static PORT_WRITE_START( rongrong_writeport )
	{ 0x00, 0x01, rongrong_blitter_w			},
	{ 0x1e, 0x1e, rongrong_select_w				},	//
	{ 0x20, 0x2f, msm6242_w						},	// 6242RTC
	{ 0x40, 0x40, OKIM6295_data_0_w				},	//
	{ 0x60, 0x60, YM2413_register_port_0_w		},	// Sound
	{ 0x61, 0x61, YM2413_data_port_0_w			},	//
	{ 0x80, 0x83, dynax_palette_base_w			},	// palette base for the 4 layers
	{ 0x84, 0x87, dynax_palette_mask_w			},
	{ 0x88, 0x8b, dynax_transparency_pen_w		},
	{ 0x8c, 0x8f, dynax_transparency_mask_w		},
	{ 0x94, 0x94, dynax_bgcolor_w				},
	{ 0x95, 0x95, dynax_priority_w				},
	{ 0x96, 0x96, dynax_layer_enable_w			},
	{ 0xa0, 0xa0, dynax_select2_w				},	//
	{ 0xc2, 0xc2, IOWP_NOP						},	// enables palette RAM at f000, and protection device at f705/f706/f601
PORT_END
/*
1e input select,1c input read
    3e=dsw1 3d=dsw2
a0 input select,a2 input read (protection?)
    0=? 1=? 2=coins(from a3)
*/


/***************************************************************************
                                Monkey Mole Panic
***************************************************************************/


static READ8_HANDLER( magic_r )
{
	return 0x01;
}

static WRITE8_HANDLER( mmpanic_rombank_w )
{
	UINT8 *rom = memory_region(REGION_CPU1);
	memory_set_bankptr(1, &rom[0x10000 + 0x8000 * (data & 0x7)]);
	/* Bit 4? */
}

static WRITE8_HANDLER( mmpanic_soundlatch_w )
{
	soundlatch_w(0,data);
	cpu_set_irq_line(1, IRQ_LINE_NMI, PULSE_LINE);
}

static WRITE8_HANDLER( mmpanic_blitter_w )
{
	blitter_w(0,offset,data,0xdf);	// RST 18
}
static WRITE8_HANDLER( mmpanic_blitter2_w )
{
	blitter_w(1,offset,data,0xdf);	// RST 18
}

/* A led for each of the 9 buttons */
static UINT16 mmpanic_leds;

static void mmpanic_update_leds(void)
{
	set_led_status(0,mmpanic_leds);
}

/* leds 1-8 */
static WRITE8_HANDLER( mmpanic_leds_w )
{
	mmpanic_leds = (mmpanic_leds & 0xff00) | data;
	mmpanic_update_leds();
}
/* led 9 */
static WRITE8_HANDLER( mmpanic_leds2_w )
{
	mmpanic_leds = (mmpanic_leds & 0xfeff) | (data ? 0x0100 : 0);
	mmpanic_update_leds();
}


static WRITE8_HANDLER( mmpanic_lockout_w )
{
	if (dynax_select == 0x0c)
	{
		coin_counter_w(0,(~data) & 0x01);
		coin_lockout_w(0,(~data) & 0x02);
		set_led_status(1,(~data) & 0x04);
	}
}

static READ8_HANDLER( mmpanic_link_r )	{ return 0xff; }

/* Main CPU */

static MEMORY_READ_START( mmpanic_readmem )
	{ 0x0051, 0x0051, magic_r					},	// ?
	{ 0x0000, 0x5fff, MRA_ROM					},	// ROM
	{ 0x6000, 0x6fff, MRA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM (Banked)
MEMORY_END

static MEMORY_WRITE_START( mmpanic_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0x81ff, rongrong_palette_w		},	// ROM (Banked)
MEMORY_END

static PORT_READ_START( mmpanic_readport )
	{ 0x00, 0x0f, msm6242_r						},	// 6242RTC
	{ 0x38, 0x38, unk_r							},	// ? must be 78 on startup
	{ 0x58, 0x58, unk_r							},	// ? must be 78 on startup
	{ 0x63, 0x63, rongrong_gfxrom_r				},	// Video Chip
	{ 0x6a, 0x6a, input_port_0_r				},
	{ 0x6b, 0x6b, input_port_1_r				},
	{ 0x6c, 0x6d, mmpanic_link_r				},	// Other cabinets?
	{ 0x7c, 0x7c, OKIM6295_status_0_r			},	// Sound
	{ 0x94, 0x94, input_port_2_r				},	// DSW 1
	{ 0x98, 0x98, input_port_3_r				},	// DSW 2
	{ 0x9c, 0x9c, input_port_4_r				},	// DSW 1&2 high bits
PORT_END

static PORT_WRITE_START( mmpanic_writeport )
	{ 0x00, 0x0f, msm6242_w						},	// 6242RTC
	// Layers 0-3:
	{ 0x20, 0x23, dynax_palette_base_w			},
	{ 0x24, 0x27, dynax_palette_mask_w			},
	{ 0x28, 0x2b, dynax_transparency_pen_w		},
	{ 0x2c, 0x2f, dynax_transparency_mask_w		},
	{ 0x34, 0x34, dynax_bgcolor_w				},
	{ 0x35, 0x35, dynax_priority_w				},
	{ 0x36, 0x36, dynax_layer_enable_w			},
	// Layers 4-7:
	{ 0x40, 0x43, dynax_palette_base2_w			},
	{ 0x44, 0x47, dynax_palette_mask2_w			},
	{ 0x48, 0x4b, dynax_transparency_pen2_w		},
	{ 0x4c, 0x4f, dynax_transparency_mask2_w	},
	{ 0x54, 0x54, dynax_bgcolor2_w				},
	{ 0x55, 0x55, dynax_priority2_w				},
	{ 0x56, 0x56, dynax_layer_enable2_w			},

	{ 0x60, 0x61, mmpanic_blitter_w				},
	{ 0x64, 0x65, mmpanic_blitter2_w			},
	{ 0x68, 0x68, dynax_select_w				},
	{ 0x69, 0x69, mmpanic_lockout_w				},
	{ 0x74, 0x74, mmpanic_rombank_w				},

	{ 0x78, 0x78, IOWP_NOP						},	// 0, during RST 08 (irq acknowledge?)

	{ 0x7c, 0x7c, OKIM6295_data_0_w				},	// Sound
	{ 0x8c, 0x8c, mmpanic_soundlatch_w			},	//
	{ 0x88, 0x88, mmpanic_leds_w				},	// Leds
	{ 0x90, 0x90, IOWP_NOP						},	// written just before port 8c
	{ 0xa6, 0xa6, mmpanic_leds2_w				},	//
PORT_END

/* Sound CPU */

static MEMORY_READ_START( mmpanic_sound_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},	// ROM
	{ 0x6000, 0x66ff, MRA_RAM					},	// RAM
	{ 0x8000, 0xffff, MRA_ROM					},	// ROM
MEMORY_END

static MEMORY_WRITE_START( mmpanic_sound_writemem )
	{ 0x6000, 0x66ff, MWA_RAM					},	// RAM
MEMORY_END

static PORT_READ_START( mmpanic_sound_readport )
	{ 0x00, 0x00, soundlatch_r					},
	{ 0x02, 0x02, IORP_NOP						},	// read just before port 00
	{ 0x04, 0x04, IORP_NOP						},	// read only once at the start
PORT_END

static PORT_WRITE_START( mmpanic_sound_writeport )
	{ 0x04, 0x04, IOWP_NOP						},	// 0, during NMI
	{ 0x06, 0x06, IOWP_NOP						},	// almost always 1, sometimes 0
	{ 0x08, 0x08, YM2413_register_port_0_w		},
	{ 0x09, 0x09, YM2413_data_port_0_w			},
	{ 0x0c, 0x0c, AY8910_write_port_0_w			},
	{ 0x0e, 0x0e, AY8910_control_port_0_w		},
PORT_END



/***************************************************************************
                            The First Funky Fighter
***************************************************************************/

/* Main CPU */

static MEMORY_READ_START( funkyfig_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},
	{ 0x6000, 0x6fff, MRA_RAM					},
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM (Banked)
MEMORY_END

static MEMORY_WRITE_START( funkyfig_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0x81ff, rongrong_palette_w		},
	{ 0x8400, 0x87ff, MWA_NOP					},
MEMORY_END


static READ8_HANDLER( funkyfig_busy_r )
{
					// bit 0 ?
	return 0x00;	// bit 7 = blitter busy
}

static WRITE8_HANDLER( funkyfig_blitter_w )
{
	blitter_w_funkyfig(0,offset,data,0xe0);
}

WRITE8_HANDLER( funkyfig_rombank_w )
{
	UINT8 *rom = memory_region(REGION_CPU1);

	dynax_select = data;

	memory_set_bankptr(1, &rom[0x10000 + 0x8000 * (data & 0x0f)]);
	// bit 4 selects palette ram at 8000?
	memory_set_bankptr(2, &rom[0x90000 + 0x1000 * ((data & 0xe0) >> 5)]);
}

READ8_HANDLER( funkyfig_dsw_r )
{
	if (!(dynax_select & 0x01))	return readinputport(3);
	if (!(dynax_select & 0x02))	return readinputport(4);
	if (!(dynax_select & 0x04))	return readinputport(5);
	logerror("%06x: warning, unknown bits read, dynax_select = %02x\n", activecpu_get_pc(), dynax_select);
	return 0xff;
}

static UINT8 funkyfig_lockout;

READ8_HANDLER( funkyfig_coin_r )
{
	switch( dynax_select2 )
	{
		case 0x22:	return readinputport(2);
		case 0x23:	return funkyfig_lockout;
	}
	logerror("%06x: warning, unknown bits read, dynax_select2 = %02x\n", activecpu_get_pc(), dynax_select2);
	return 0xff;
}

READ8_HANDLER( funkyfig_key_r )
{
	switch( dynax_select2 )
	{
		case 0x20:	return readinputport(0);
		case 0x21:	return readinputport(1);
	}
	logerror("%06x: warning, unknown bits read, dynax_select2 = %02x\n", activecpu_get_pc(), dynax_select2);
	return 0xff;
}

static WRITE8_HANDLER( funkyfig_lockout_w )
{
	switch( dynax_select2 )
	{
		case 0x2c:
			funkyfig_lockout = data;
			coin_counter_w(0,  data  & 0x01);
			coin_lockout_w(0,(~data) & 0x02);
			if (data & ~0x03)
				logerror("%06x: warning, unknown bits written, lockout = %02x\n", activecpu_get_pc(), data);
			break;

//      case 0xef:  16 bytes on startup

		default:
			logerror("%06x: warning, unknown bits written, dynax_select2 = %02x, data = %02x\n", activecpu_get_pc(), dynax_select2, data);
	}
}

static PORT_READ_START( funkyfig_readport )
	{ 0x00, 0x00, OKIM6295_status_0_r			},	// Sound
	{ 0x04, 0x04, funkyfig_busy_r				},
	{ 0x1c, 0x1c, funkyfig_dsw_r				},
	{ 0x23, 0x23, rongrong_gfxrom_r				},	// Video Chip
	{ 0x40, 0x4f, msm6242_r						},	// 6242RTC
	{ 0x78, 0x78, unk_r							},	// ? must be 78 on startup
	{ 0x82, 0x82, funkyfig_coin_r				},
	{ 0x83, 0x83, funkyfig_key_r				},
//  Other cabinets?
PORT_END

static PORT_WRITE_START( funkyfig_writeport )
	{ 0x00, 0x00, OKIM6295_data_0_w				},	// Sound
	{ 0x01, 0x01, mmpanic_leds_w				},	// Leds
	{ 0x02, 0x02, mmpanic_soundlatch_w			},	//
	{ 0x1e, 0x1e, funkyfig_rombank_w			},
	{ 0x20, 0x21, funkyfig_blitter_w			},
	{ 0x40, 0x4f, msm6242_w						},	// 6242RTC

	// Layers 0-3:
	{ 0x60, 0x63, dynax_palette_base_w			},
	{ 0x64, 0x67, dynax_palette_mask_w			},
	{ 0x68, 0x6b, dynax_transparency_pen_w		},
	{ 0x6c, 0x6f, dynax_transparency_mask_w		},
	{ 0x74, 0x74, dynax_bgcolor_w				},
	{ 0x75, 0x75, dynax_priority_w				},
	{ 0x76, 0x76, dynax_layer_enable_w			},

	{ 0x80, 0x80, dynax_select2_w				},
	{ 0x81, 0x81, funkyfig_lockout_w			},
	{ 0xa2, 0xa2, mmpanic_leds2_w				},
PORT_END


/* Sound CPU */

static PORT_READ_START( funkyfig_sound_readport )
	{ 0x02, 0x02, soundlatch_r					},
	{ 0x04, 0x04, IORP_NOP						},	// read only once at the start
PORT_END



/***************************************************************************

    Hana Kanzashi

***************************************************************************/

WRITE8_HANDLER( hanakanz_rombank_w )
{
	UINT8 *rom = memory_region(REGION_CPU1);

	memory_set_bankptr(1, &rom[0x10000 + 0x8000 * (data & 0x0f)]);

	memory_set_bankptr(2, &rom[0x90000 + 0x1000 * ((data & 0xf0) >> 4)]);
}

static MEMORY_READ_START( hanakanz_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},
	{ 0x6000, 0x6fff, MRA_RAM					},
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM (Banked)
MEMORY_END

static MEMORY_WRITE_START( hanakanz_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
MEMORY_END


static UINT8 keyb,dsw;
static WRITE8_HANDLER( hanakanz_keyb_w )
{
	keyb = data;
}
static WRITE8_HANDLER( hanakanz_dsw_w )
{
	dsw = data;
}

static READ8_HANDLER( hanakanz_keyb_r )
{
	UINT8 val = 0xff;

	if      (!(keyb & 0x01))	val = readinputport(offset * 5 + 1);
	else if (!(keyb & 0x02))	val = readinputport(offset * 5 + 2);
	else if (!(keyb & 0x04))	val = readinputport(offset * 5 + 3);
	else if (!(keyb & 0x08))	val = readinputport(offset * 5 + 4);
	else if (!(keyb & 0x10))	val = readinputport(offset * 5 + 5);

	val |= readinputport(16 + offset);
	return val;
}

static READ8_HANDLER( hanakanz_dsw_r )
{
	if (!(dsw & 0x01))	return readinputport(11);
	if (!(dsw & 0x02))	return readinputport(12);
	if (!(dsw & 0x04))	return readinputport(13);
	if (!(dsw & 0x08))	return readinputport(14);
	if (!(dsw & 0x10))	return readinputport(15);
	return 0xff;
}

static READ8_HANDLER( hanakanz_busy_r )
{
	return 0x80;	// bit 7 == 0 -> blitter busy
}

static READ8_HANDLER( hanakanz_gfxrom_r )
{
	UINT8 *rom	=	memory_region( REGION_GFX1 );
	size_t size		=	memory_region_length( REGION_GFX1 );
	int address		=	(dynax_blit_address & 0xffffff) * 2;

	static UINT8 romdata[2];

	if (address >= size)
	{
		logerror("CPU#0 PC %06X: Error, Blitter address %06X out of range\n", activecpu_get_pc(), address);
		address %= size;
	}

	if (offset == 0)
	{
		romdata[0] = rom[address + 0];
		romdata[1] = rom[address + 1];

		dynax_blit_address = (dynax_blit_address + 1) & 0xffffff;

		return romdata[0];
	}
	else
	{
		return romdata[1];
	}
}


static WRITE8_HANDLER( hanakanz_coincounter_w )
{
	// bit 0 = coin counter
	// bit 1 = out counter
	// bit 2 = 1 if bet on
	// bit 3 = 1 if bet off

	coin_counter_w(0, data & 1);

	if (data & 0xf0)
		logerror("%04x: warning, coin counter = %02x\n", activecpu_get_pc(), data);

#ifdef MAME_DEBUG
//      usrintf_showmessage("93 = %02x",data);
#endif
}

static WRITE8_HANDLER( hanakanz_palette_w )
{
	static int palette_index;

	if (dynax_blit_reg & 0x80)
	{
		palette_index = data | ((dynax_blit_reg & 1) << 8);
	}
	else
	{
		// 0bbggggg bbbrrrrr
		// 04343210 21043210

		int g = dynax_blit_reg & 0x1f;
		int r = data & 0x1f;
		int b = ((data & 0xe0) >> 5) | ((dynax_blit_reg & 0x60) >> 2);
		palette_set_color((palette_index++)&0x1ff,pal5bit(r),pal5bit(g),pal5bit(b));
	}
}

static WRITE8_HANDLER( hanakanz_oki_bank_w )
{
	OKIM6295_set_bank_base(0, (data & 0x40) ? 0x40000 : 0);
}

static READ8_HANDLER( hanakanz_rand_r )
{
	return mame_rand();
}

static PORT_READ_START( hanakanz_readport )
	{ 0x2c, 0x2c, hanakanz_busy_r				},
	{ 0x32, 0x32, hanakanz_dsw_r				},
	{ 0x83, 0x84, hanakanz_gfxrom_r				},
	{ 0x90, 0x90, input_port_0_r				},
	{ 0x91, 0x92, hanakanz_keyb_r				},
	{ 0x96, 0x96, hanakanz_rand_r				},
	{ 0xc0, 0xc0, OKIM6295_status_0_r			},
	{ 0xe0, 0xef, msm6242_r						},	// 6242RTC
PORT_END

static PORT_WRITE_START( hanakanz_writeport )
	{ 0x2c, 0x2c, hanakanz_oki_bank_w			},
	{ 0x2e, 0x2e, hanakanz_blitter_reg_w		},
	{ 0x30, 0x30, hanakanz_rombank_w			},
	{ 0x31, 0x31, hanakanz_dsw_w				},
	{ 0x80, 0x80, hanakanz_blitter_data_w		},
	{ 0x81, 0x81, hanakanz_palette_w			},
	{ 0x93, 0x93, hanakanz_coincounter_w		},
	{ 0x94, 0x94, hanakanz_keyb_w				},
	{ 0xa0, 0xa0, YM2413_register_port_0_w		},
	{ 0xa1, 0xa1, YM2413_data_port_0_w			},
	{ 0xc0, 0xc0, OKIM6295_data_0_w				},
	{ 0xe0, 0xef, msm6242_w						},	// 6242RTC
PORT_END


static PORT_READ_START( hkagerou_readport )
	{ 0x2c, 0x2c, hanakanz_busy_r				},
	{ 0x32, 0x32, hanakanz_dsw_r				},
	{ 0x83, 0x84, hanakanz_gfxrom_r				},
	{ 0xb0, 0xb0, input_port_0_r				},
	{ 0xb1, 0xb2, hanakanz_keyb_r				},
	{ 0xb6, 0xb6, hanakanz_rand_r				},
	{ 0xc0, 0xc0, OKIM6295_status_0_r			},
	{ 0xe0, 0xef, msm6242_r						},	// 6242RTC
PORT_END

static PORT_WRITE_START( hkagerou_writeport )
	{ 0x2c, 0x2c, hanakanz_oki_bank_w			},
	{ 0x2e, 0x2e, hanakanz_blitter_reg_w		},
	{ 0x30, 0x30, hanakanz_rombank_w			},
	{ 0x31, 0x31, hanakanz_dsw_w				},
	{ 0x80, 0x80, hanakanz_blitter_data_w		},
	{ 0x81, 0x81, hanakanz_palette_w			},
	{ 0xa0, 0xa0, YM2413_register_port_0_w		},
	{ 0xa1, 0xa1, YM2413_data_port_0_w			},
	{ 0xb3, 0xb3, hanakanz_coincounter_w		},
	{ 0xb4, 0xb4, hanakanz_keyb_w				},
	{ 0xc0, 0xc0, OKIM6295_data_0_w				},
	{ 0xe0, 0xef, msm6242_w						},	// 6242RTC
PORT_END


static UINT8 mjreach1_protection_val;

static WRITE8_HANDLER( mjreach1_protection_w )
{
	mjreach1_protection_val = data;
}

static READ8_HANDLER( mjreach1_protection_r )
{
	return mjreach1_protection_val;
}

static PORT_READ_START( mjreach1_readport )
	{ 0x2c, 0x2c, hanakanz_busy_r				},
	{ 0x32, 0x32, hanakanz_dsw_r				},
	{ 0x83, 0x84, hanakanz_gfxrom_r				},
	{ 0x92, 0x92, hanakanz_rand_r				},
	{ 0x93, 0x93, mjreach1_protection_r			},
	{ 0x94, 0x94, input_port_0_r				},
	{ 0x95, 0x96, hanakanz_keyb_r				},
	{ 0xc0, 0xc0, OKIM6295_status_0_r			},
	{ 0xe0, 0xef, msm6242_r						},	// 6242RTC
PORT_END

static PORT_WRITE_START( mjreach1_writeport )
	{ 0x2c, 0x2c, hanakanz_oki_bank_w			},
	{ 0x2e, 0x2e, hanakanz_blitter_reg_w		},
	{ 0x30, 0x30, hanakanz_rombank_w			},
	{ 0x31, 0x31, hanakanz_dsw_w				},
	{ 0x80, 0x80, hanakanz_blitter_data_w		},
	{ 0x81, 0x81, hanakanz_palette_w			},
	{ 0x90, 0x90, hanakanz_keyb_w				},
	{ 0x93, 0x93, mjreach1_protection_w			},
	{ 0x97, 0x97, hanakanz_coincounter_w		},
	{ 0xa0, 0xa0, YM2413_register_port_0_w		},
	{ 0xa1, 0xa1, YM2413_data_port_0_w			},
	{ 0xc0, 0xc0, OKIM6295_data_0_w				},
	{ 0xe0, 0xef, msm6242_w						},	// 6242RTC
PORT_END


/***************************************************************************
     Mahjong Chuukanejyo
***************************************************************************/

static READ8_HANDLER( mjchuuka_keyb_r )
{
	UINT8 val = 0xff;

	if      (!(keyb & 0x01))	val = readinputport(offset * 5 + 1);
	else if (!(keyb & 0x02))	val = readinputport(offset * 5 + 2);
	else if (!(keyb & 0x04))	val = readinputport(offset * 5 + 3);
	else if (!(keyb & 0x08))	val = readinputport(offset * 5 + 4);
	else if (!(keyb & 0x10))	val = readinputport(offset * 5 + 5);

	val |= readinputport(16 + offset);
	if (offset)	val |= 0x80;	// blitter busy
	return val;
}

static WRITE8_HANDLER( mjchuuka_blitter_w )
{
	hanakanz_blitter_reg_w(0,offset >> 8);
	hanakanz_blitter_data_w(0,data);
}

static UINT8 mjchuuka_romdata[2];

static void mjchuuka_get_romdata(void)
{
	UINT8 *rom	=	memory_region( REGION_GFX1 );
	size_t size		=	memory_region_length( REGION_GFX1 );
	int address		=	(dynax_blit_address & 0xffffff) * 2;

	if (address >= size)
	{
		logerror("CPU#0 PC %06X: Error, Blitter address %06X out of range\n", activecpu_get_pc(), address);
		address %= size;
	}

	mjchuuka_romdata[0] = rom[address + 0];
	mjchuuka_romdata[1] = rom[address + 1];
}

static READ8_HANDLER( mjchuuka_gfxrom_0_r )
{
	mjchuuka_get_romdata();
	dynax_blit_address++;
	return mjchuuka_romdata[0];
}
static READ8_HANDLER( mjchuuka_gfxrom_1_r )
{
	return mjchuuka_romdata[1];
}

static WRITE8_HANDLER( mjchuuka_palette_w )
{
	static int palette_index;
	UINT16 rgb = (offset & 0xff00) | data;

	if (rgb & 0x8000)
	{
		palette_index = rgb & 0x1ff;
	}
	else
	{
		// 0bbggggg bbbrrrrr
		// 04343210 21043210

		int r = (rgb >> 0) & 0x1f;
		int g = (rgb >> 8) & 0x1f;
		int b = ((rgb >> 5) & 0x07) | ((rgb & 0x6000) >> 10);
		palette_set_color((palette_index++)&0x1ff,pal5bit(r),pal5bit(g),pal5bit(b));
	}
}

static WRITE8_HANDLER( mjchuuka_coincounter_w )
{
	// bit 0 = in counter
	// bit 1 = out counter
	// bit 3 = lockout
	// bit 8?

	coin_counter_w(0, data   & 0x01);
	coin_lockout_w(0,(~data) & 0x08);

	if (data & 0x74)
		logerror("%04x: warning, coin counter = %02x\n", activecpu_get_pc(), data);

#ifdef MAME_DEBUG
//    usrintf_showmessage("40 = %02x",data);
#endif
}

static WRITE8_HANDLER( mjchuuka_oki_bank_w )
{
	// data & 0x08 ?
	OKIM6295_set_bank_base(0, (data & 0x01) ? 0x40000 : 0);

#ifdef MAME_DEBUG
//    usrintf_showmessage("1e = %02x",data);
#endif
}

static PORT_READ_START( mjchuuka_readport )
	{ 0x13, 0x13, hanakanz_rand_r				},
	{ 0x23, 0x23, mjchuuka_gfxrom_0_r			},
	{ 0x42, 0x42, input_port_0_r				},
	{ 0x43, 0x44, mjchuuka_keyb_r				},
	{ 0x45, 0x45, mjchuuka_gfxrom_1_r			},
	{ 0x60, 0x60, input_port_11_r				},	// DSW 1
	{ 0x61, 0x61, input_port_12_r				},	// DSW 2
	{ 0x62, 0x62, input_port_13_r				},	// DSW 3
	{ 0x63, 0x63, input_port_14_r				},	// DSW 4
	{ 0x64, 0x64, input_port_15_r				},	// DSW 1-4 high bits
	{ 0x80, 0x80, OKIM6295_status_0_r			},
	{ 0xc0, 0xcf, msm6242_r						},	// 6242RTC
PORT_END

static PORT_WRITE_START( mjchuuka_writeport )
	{ 0x1c, 0x1c, hanakanz_rombank_w			},
	{ 0x20, 0x20, mjchuuka_blitter_w			},
	{ 0x21, 0x21, mjchuuka_palette_w			},
	{ 0x40, 0x40, mjchuuka_coincounter_w		},
	{ 0x41, 0x41, hanakanz_keyb_w				},
	{ 0x80, 0x80, OKIM6295_data_0_w				},
	{ 0xc0, 0xcf, msm6242_w						},	// 6242RTC
	{ 0xa0, 0xa0, YM2413_register_port_0_w		},
	{ 0xa1, 0xa1, YM2413_data_port_0_w			},
	{ 0xe0, 0xe0, AY8910_control_port_0_w		},
	{ 0xe1, 0xe1, AY8910_write_port_0_w			},
	{ 0x1e, 0x1e, mjchuuka_oki_bank_w			},
PORT_END


/***************************************************************************
                        Mahjong The Mysterious World
***************************************************************************/

static MEMORY_READ_START( mjmyster_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},
	{ 0x6000, 0x6fff, MRA_RAM					},
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM/RAM (Banked)
MEMORY_END

static MEMORY_WRITE_START( mjmyster_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
	{ 0xf000, 0xf1ff, rongrong_palette_w		},	// RAM enabled by bit 4 of rombank
	{ 0xf200, 0xffff, MWA_NOP					},	// ""
MEMORY_END

static WRITE8_HANDLER( mjmyster_rambank_w )
{
	UINT8 *rom = memory_region(REGION_CPU1);
	memory_set_bankptr(2, &rom[0x90000 + 0x1000 * (data & 0x07)]);
//  logerror("%04x: rambank = %02x\n", activecpu_get_pc(), data);
}

static WRITE8_HANDLER( mjmyster_select2_w )
{
	dynax_select2 = data;

	if (data & 0x80)	keyb = 1;
}

static READ8_HANDLER( mjmyster_coins_r )
{
	switch( dynax_select2 )
	{
		case 0x00:	return readinputport(0);
		case 0x01:	return 0xff;
		case 0x02:	return 0xff;	// bit 7 = 0 -> blitter busy, + hopper switch
		case 0x03:	return 0xff;
	}

	logerror("%06x: warning, unknown bits read, dynax_select2 = %02x\n", activecpu_get_pc(), dynax_select2);

	return 0xff;
}

static READ8_HANDLER( mjmyster_keyb_r )
{
	UINT8 ret = 0xff;

	if		(keyb & 0x01)	ret = readinputport(1);
	else if	(keyb & 0x02)	ret = readinputport(2);
	else if	(keyb & 0x04)	ret = readinputport(3);
	else if	(keyb & 0x08)	ret = readinputport(4);
	else if	(keyb & 0x10)	ret = readinputport(5);
	else	logerror("%06x: warning, unknown bits read, keyb = %02x\n", activecpu_get_pc(), keyb);

	keyb <<= 1;

	return ret;
}

static READ8_HANDLER( mjmyster_dsw_r )
{
	if (!(dynax_select & 0x01))	return readinputport(9);
	if (!(dynax_select & 0x02))	return readinputport(8);
	if (!(dynax_select & 0x04))	return readinputport(7);
	if (!(dynax_select & 0x08))	return readinputport(6);
	if (!(dynax_select & 0x10))	return readinputport(10);
	logerror("%06x: warning, unknown bits read, dynax_select = %02x\n", activecpu_get_pc(), dynax_select);
	return 0xff;
}

static WRITE8_HANDLER( mjmyster_coincounter_w )
{
	switch( dynax_select2 )
	{
		case 0x0c:
			coin_counter_w(0, (~data) & 0x01);	// coin in
			coin_counter_w(0, (~data) & 0x02);	// coin out actually
			#ifdef MAME_DEBUG
//              usrintf_showmessage("cc: %02x",data);
			#endif

			break;

		default:
			logerror("%06x: warning, unknown bits written, dynax_select2 = %02x, data = %02x\n", activecpu_get_pc(), dynax_select2, data);
	}
}

static WRITE8_HANDLER( mjmyster_blitter_w )
{
	blitter_w(0,offset,data,0xfc);
}

static PORT_READ_START( mjmyster_readport )
	{ 0x03, 0x03, rongrong_gfxrom_r				},
	{ 0x22, 0x22, mjmyster_coins_r				},
	{ 0x23, 0x23, mjmyster_keyb_r				},
	{ 0x40, 0x40, OKIM6295_status_0_r			},
	{ 0x44, 0x44, AY8910_read_port_0_r			},
	{ 0x60, 0x6f, msm6242_r						},	// 6242RTC
	{ 0x98, 0x98, unk_r							},	// ? must be 78 on startup
	{ 0xc2, 0xc2, hanakanz_rand_r				},
	{ 0xc3, 0xc3, mjmyster_dsw_r				},
PORT_END

static PORT_WRITE_START( mjmyster_writeport )
	{ 0x00, 0x01, mjmyster_blitter_w			},
	{ 0x1c, 0x1c, mjmyster_rambank_w			},
	{ 0x1e, 0x1e, mmpanic_rombank_w				},
	{ 0x20, 0x20, mjmyster_select2_w			},
	{ 0x21, 0x21, mjmyster_coincounter_w		},
	{ 0x40, 0x40, OKIM6295_data_0_w				},
	{ 0x42, 0x42, YM2413_register_port_0_w		},
	{ 0x43, 0x43, YM2413_data_port_0_w			},
	{ 0x46, 0x46, AY8910_write_port_0_w			},
	{ 0x48, 0x48, AY8910_control_port_0_w		},
	{ 0x60, 0x6f, msm6242_w						},	// 6242RTC
	{ 0x80, 0x83, dynax_palette_base_w			},
	{ 0x84, 0x87, dynax_palette_mask_w			},
	{ 0x88, 0x8b, dynax_transparency_pen_w		},
	{ 0x8c, 0x8f, dynax_transparency_mask_w		},
	{ 0x94, 0x94, dynax_bgcolor_w				},
	{ 0x95, 0x95, dynax_priority_w				},
	{ 0x96, 0x96, dynax_layer_enable_w			},
PORT_END

/***************************************************************************
                            Hanafuda Hana Ginga
***************************************************************************/

UINT8 hginga_rombank;
static WRITE8_HANDLER( hginga_rombank_w )
{
	UINT8 *rom = memory_region(REGION_CPU1);
	memory_set_bankptr(1, &rom[0x10000 + 0x8000 * (data & 0x7)]);
	hginga_rombank = data;
}

// similar to rongrong
static READ8_HANDLER( hginga_protection_r )
{
	UINT8 *rom = memory_region(REGION_CPU1);
	if (hginga_rombank & 0x10)
		return hanakanz_rand_r(0);
	return rom[0x10000 + 0x8000 * (hginga_rombank & 0x7) + 0xf601 - 0x8000];
}

static MEMORY_READ_START( hginga_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},
	{ 0x6000, 0x6fff, MRA_RAM					},
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0xf601, 0xf601, hginga_protection_r		},
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM/RAM (Banked)
MEMORY_END

static MEMORY_WRITE_START( hginga_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
	{ 0xf000, 0xf1ff, rongrong_palette_w		},	// RAM enabled by bit 4 of rombank
	{ 0xf200, 0xffff, MWA_NOP					},	// ""
MEMORY_END

static READ8_HANDLER( hginga_dsw_r )
{
	if (!(dynax_select & 0x01))	return readinputport(11);
	if (!(dynax_select & 0x02))	return readinputport(12);
	if (!(dynax_select & 0x04))	return readinputport(13);
	if (!(dynax_select & 0x08))	return readinputport(14);
	if (!(dynax_select & 0x10))	return readinputport(15);
	logerror("%06x: warning, unknown bits read, dynax_select = %02x\n", activecpu_get_pc(), dynax_select);
	return 0xff;
}

static UINT8 hginga_select, hginga_ip, hginga_coins;

static WRITE8_HANDLER( hginga_input_w )
{
	hginga_select = data;
	hginga_ip = 0;
}

static READ8_HANDLER( hginga_coins_r )
{
	switch( hginga_select )
	{
		case 0x20:	return readinputport(0);
		case 0x21:	return readinputport(16);
		case 0x22:	return 0x7f;	// bit 7 = blitter busy, bit 6 = hopper
		case 0x23:	return hginga_coins;
	}
	logerror("%04x: coins_r with select = %02x\n",activecpu_get_pc(),hginga_select);
	return 0xff;
}

static WRITE8_HANDLER( hginga_80_w )
{
//  usrintf_showmessage("port 80 = %02x",data);
}

static WRITE8_HANDLER( hginga_coins_w )
{
	switch( hginga_select )
	{
		case 0x2d:
			break;
		case 0x2c:
			// bit 0 = coin counter
			// bit 1 = out counter
			// bit 2 = 1 if bet on
			// bit 3 = 1 if bet on
			// bit 7?
			coin_counter_w(0, data & 1);
#ifdef MAME_DEBUG
//          usrintf_showmessage("COINS %02x",data);
#endif
			hginga_coins = data;
			break;
		default:
			logerror("%04x: coins_w with select = %02x, data = %02x\n",activecpu_get_pc(),hginga_select,data);
	}
}

static READ8_HANDLER( hginga_input_r )
{
	switch( hginga_select )
	{
		case 0x2d:
			return 0xff;

		// player 1
		case 0xa1:
			return readinputport(1 + hginga_ip++);

		// player 2
		case 0xa2:
			return readinputport(1 + 5 + hginga_ip++);
	}
	logerror("%04x: input_r with select = %02x\n",activecpu_get_pc(),hginga_select);
	return 0xff;
}

static WRITE8_HANDLER( hginga_blitter_w )
{
	if (offset == 0)
	{
		dynax_blit_reg = data;
	}
	else
	{
		switch (dynax_blit_reg & 0x3f)
		{
			case 0x00:
				switch (data & 0xf)
				{
					case 0x03:
					case 0x06:
					case 0x0a:
						data = data & ~2;	// do not mirror writes of other layers to layer 1? (see code at 38d)
						break;
				}
				break;

			case 0x24:
				if (data == 0x1b)
					data = 0x13;			// vertical lines -> horizontal lines (see numbers drawn on cards on "first chance")
				break;
		}
	}
	blitter_w(0,offset,data,0xfc);
}

static PORT_READ_START( hginga_readport )
	{ 0x03, 0x03, rongrong_gfxrom_r				},
	{ 0x1c, 0x1c, IORP_NOP						},
	{ 0x24, 0x24, AY8910_read_port_0_r			},
	{ 0x60, 0x6f, msm6242_r						},	// 6242RTC
	{ 0x42, 0x42, hginga_coins_r				},
	{ 0x43, 0x43, hginga_input_r				},
	{ 0xb8, 0xb8, unk_r							},	// ? must be 78 on startup
PORT_END

static PORT_WRITE_START( hginga_writeport )
	{ 0x00, 0x01, hginga_blitter_w				},
	{ 0x1c, 0x1c, mjmyster_rambank_w			},
	{ 0x1e, 0x1e, hginga_rombank_w				},
	{ 0x22, 0x22, YM2413_register_port_0_w		},
	{ 0x23, 0x23, YM2413_data_port_0_w			},
	{ 0x26, 0x26, AY8910_write_port_0_w			},
	{ 0x28, 0x28, AY8910_control_port_0_w		},
	{ 0x40, 0x40, hginga_input_w				},
	{ 0x41, 0x41, hginga_coins_w				},
	{ 0x60, 0x6f, msm6242_w						},	// 6242RTC
	{ 0x80, 0x80, hginga_80_w					},
	{ 0xa0, 0xa3, dynax_palette_base_w			},
	{ 0xa4, 0xa7, dynax_palette_mask_w			},
	{ 0xa8, 0xab, dynax_transparency_pen_w		},
	{ 0xac, 0xaf, dynax_transparency_mask_w		},
	{ 0xb4, 0xb4, dynax_bgcolor_w				},
	{ 0xb5, 0xb5, dynax_priority_w				},
	{ 0xb6, 0xb6, dynax_layer_enable_w			},
PORT_END


/***************************************************************************
                             Hanafuda Hana Gokou
***************************************************************************/

UINT8 hgokou_hopper;

static UINT8 hgokou_player_r(int player)
{
	UINT8 hopper_bit = ((hgokou_hopper && !(cpu_getcurrentframe()%10)) ? 0 : (1<<6));

	if (!(dynax_select2 & 0x01))	return readinputport(player * 5 + 1) | hopper_bit;
	if (!(dynax_select2 & 0x02))	return readinputport(player * 5 + 2) | hopper_bit;
	if (!(dynax_select2 & 0x04))	return readinputport(player * 5 + 3) | hopper_bit;
	if (!(dynax_select2 & 0x08))	return readinputport(player * 5 + 4) | hopper_bit;
	if (!(dynax_select2 & 0x10))	return readinputport(player * 5 + 5) | hopper_bit;

	return 0x7f;	// bit 7 = blitter busy, bit 6 = hopper
}

static READ8_HANDLER( hgokou_input_r )
{
	switch (hginga_select)
	{
		case 0x20:	return readinputport(0);
		case 0x21:	return hgokou_player_r(1);
		case 0x22:	return hgokou_player_r(0);
		case 0x23:	return hginga_coins;
	}
	logerror("%06x: warning, unknown bits read, hginga_select = %02x\n", activecpu_get_pc(), hginga_select);
	return 0xff;
}

static WRITE8_HANDLER( hgokou_input_w )
{
	switch (hginga_select)
	{
		case 0x2c:
			// bit 0 = coin counter
			// bit 1 = out counter
			// bit 2 = hopper
			coin_counter_w(0, data & 1);
			hgokou_hopper = data & 0x04;
#ifdef MAME_DEBUG
//          usrintf_showmessage("COINS %02x",data);
#endif
			hginga_coins = data;
			break;

		case 0x2d:	dynax_select2 = data;	break;

		case 0x2f:	break;	// ? written with 2f

		default:
			logerror("%04x: input_w with select = %02x, data = %02x\n",activecpu_get_pc(),hginga_select,data);
	}
}

// similar to rongrong
static READ8_HANDLER( hgokou_protection_r )
{
	UINT8 *rom = memory_region(REGION_CPU1);
	if (hginga_rombank == 0)
		return hanakanz_rand_r(0);
	return rom[0x10000 + 0x8000 * (hginga_rombank & 0x7) + 0xe601 - 0x8000];
}

static MEMORY_READ_START( hgokou_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},	// ROM
	{ 0x6000, 0x6fff, MRA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0xe601, 0xe601, hgokou_protection_r		},
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM (Banked)
MEMORY_END
static MEMORY_WRITE_START( hgokou_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
	{ 0xe000, 0xe1ff, rongrong_palette_w		},
	{ 0xe700, 0xe706, MWA_NOP					},
MEMORY_END


static PORT_READ_START( hgokou_readport )
	{ 0x03, 0x03, rongrong_gfxrom_r				},
	{ 0x1c, 0x1c, IORP_NOP						},	// ? ack on RTC int
	{ 0x20, 0x2f, msm6242_r						},	// 6242RTC
	{ 0x58, 0x58, unk_r							},	// ? must be 78 on startup
	{ 0x62, 0x62, hgokou_input_r				},
	{ 0x80, 0x80, OKIM6295_status_0_r			},
	{ 0x84, 0x84, AY8910_read_port_0_r			},
	{ 0xb0, 0xb0, hanakanz_rand_r				},
PORT_END
static PORT_WRITE_START( hgokou_writeport )
	{ 0x00, 0x01, hginga_blitter_w				},
	{ 0x1c, 0x1c, mjmyster_rambank_w			},
	{ 0x1e, 0x1e, hginga_rombank_w				},
	{ 0x20, 0x2f, msm6242_w						},	// 6242RTC
	{ 0x40, 0x43, dynax_palette_base_w			},
	{ 0x44, 0x47, dynax_palette_mask_w			},
	{ 0x48, 0x4b, dynax_transparency_pen_w		},
	{ 0x4c, 0x4f, dynax_transparency_mask_w		},
	{ 0x54, 0x54, dynax_bgcolor_w				},
	{ 0x55, 0x55, dynax_priority_w				},
	{ 0x56, 0x56, dynax_layer_enable_w			},
	{ 0x60, 0x60, hginga_input_w				},
	{ 0x61, 0x61, hgokou_input_w				},
	{ 0x80, 0x80, OKIM6295_data_0_w				},
	{ 0x82, 0x82, YM2413_register_port_0_w		},
	{ 0x83, 0x83, YM2413_data_port_0_w			},
	{ 0x86, 0x86, AY8910_write_port_0_w			},
	{ 0x88, 0x88, AY8910_control_port_0_w		},
PORT_END


/***************************************************************************
                            Super Hana Paradise
***************************************************************************/

static WRITE8_HANDLER( hparadis_select_w )
{
	UINT8 *rom = memory_region(REGION_CPU1);

	dynax_select = data;
	hginga_ip = 0;

	memory_set_bankptr(1, &rom[0x10000 + 0x8000 * (data & 0x07)]);
	memory_set_bankptr(2, &rom[0x50000 + 0x1000 * ((data & 0xe0) >> 5)]);
}


static READ8_HANDLER( hparadis_input_r )
{
	switch (hginga_select)
	{
		case 0x00:	return readinputport(0);				// P1 (Joy)
		case 0x01:	return readinputport(1);				// P2 (Joy)
		case 0x02:	return readinputport(2);				// Coins
		case 0x0d:	return 0x00;
		case 0x80:	return readinputport(3 + hginga_ip++);	// P1 (Keys)
		case 0x81:	return readinputport(8 + hginga_ip++);	// P2 (Keys)
	}
	logerror("%06x: warning, unknown bits read, hginga_select = %02x\n", activecpu_get_pc(), hginga_select);
	return 0xff;
}

static READ8_HANDLER( hparadis_dsw_r )
{
	if (!(dynax_select & 0x01))	return readinputport(13);
	if (!(dynax_select & 0x02))	return readinputport(14);
	if (!(dynax_select & 0x04))	return 0xff;
	if (!(dynax_select & 0x08))	return 0xff;
	if (!(dynax_select & 0x10))	return readinputport(15);
	return 0xff;
}

static WRITE8_HANDLER( hparadis_coin_w )
{
	switch ( hginga_select )
	{
		case 0x0c:	coin_counter_w(0, data & 1);	break;
		case 0x0d:	break;
		default:
			logerror("%04x: coins_w with select = %02x, data = %02x\n",activecpu_get_pc(),hginga_select,data);
	}
}

static MEMORY_READ_START( hparadis_readmem )
	{ 0x0000, 0x5fff, MRA_ROM					},	// ROM
	{ 0x6000, 0x6fff, MRA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MRA_BANK2					},	// RAM (Banked)
	{ 0x8000, 0xffff, MRA_BANK1					},	// ROM (Banked)
MEMORY_END
static MEMORY_WRITE_START( hparadis_writemem )
	{ 0x6000, 0x6fff, MWA_RAM					},	// RAM
	{ 0x7000, 0x7fff, MWA_BANK2					},	// RAM (Banked)
	{ 0xc000, 0xc1ff, rongrong_palette_w		},
MEMORY_END

// the RTC seems unused
static PORT_READ_START( hparadis_readport )
	{ 0x03, 0x03, rongrong_gfxrom_r				},
	{ 0x1b, 0x1b, rongrong_blitter_busy_r		},
	{ 0x1c, 0x1c, hparadis_dsw_r				},
	{ 0x40, 0x40, OKIM6295_status_0_r			},
	{ 0x98, 0x98, unk_r							},	// ? must be 78 on startup
	{ 0xa2, 0xa3, hparadis_input_r				},
PORT_END
static PORT_WRITE_START( hparadis_writeport )
	{ 0x00, 0x01, rongrong_blitter_w			},
	{ 0x1b, 0x1b, rongrong_blitter_busy_w		},
	{ 0x1e, 0x1e, hparadis_select_w				},
	{ 0x40, 0x40, OKIM6295_data_0_w				},
	{ 0x60, 0x60, YM2413_register_port_0_w		},
	{ 0x61, 0x61, YM2413_data_port_0_w			},
	{ 0x80, 0x83, dynax_palette_base_w			},
	{ 0x84, 0x87, dynax_palette_mask_w			},
	{ 0x88, 0x8b, dynax_transparency_pen_w		},
	{ 0x8c, 0x8f, dynax_transparency_mask_w		},
	{ 0x94, 0x94, dynax_bgcolor_w				},
	{ 0x95, 0x95, dynax_priority_w				},
	{ 0x96, 0x96, dynax_layer_enable_w			},
	{ 0xa0, 0xa0, hginga_input_w				},
	{ 0xa1, 0xa1, hparadis_coin_w				},
	{ 0xc2, 0xc2, IOWP_NOP						},	// enables palette RAM at c000
PORT_END


/***************************************************************************
                          Mahjong Mysterious World
***************************************************************************/

static READ8_HANDLER( mjmywrld_coins_r )
{
	switch( dynax_select2 )
	{
		case 0x80:	return readinputport(0);
		case 0x81:	return 0x00;
		case 0x82:	return 0xff;	// bit 7 = 0 -> blitter busy, + hopper switch
		case 0x83:	return 0x00;
	}

	logerror("%06x: warning, unknown bits read, dynax_select2 = %02x\n", activecpu_get_pc(), dynax_select2);

	return 0xff;
}

static PORT_READ_START( mjmywrld_readport )
	{ 0x03, 0x03, rongrong_gfxrom_r				},
	{ 0x22, 0x22, mjmywrld_coins_r				},
	{ 0x23, 0x23, mjmyster_keyb_r				},
	{ 0x40, 0x40, OKIM6295_status_0_r			},
	{ 0x44, 0x44, AY8910_read_port_0_r			},
	{ 0x60, 0x6f, msm6242_r						},	// 6242RTC
	{ 0x98, 0x98, unk_r							},	// ? must be 78 on startup
	{ 0xc0, 0xc0, hanakanz_rand_r				},
	{ 0xe0, 0xe0, mjmyster_dsw_r				},
PORT_END

static PORT_WRITE_START( mjmywrld_writeport )
	{ 0x00, 0x01, mjmyster_blitter_w			},
	{ 0x1c, 0x1c, mjmyster_rambank_w			},
	{ 0x1e, 0x1e, hginga_rombank_w				},
	{ 0x20, 0x20, mjmyster_select2_w			},
	{ 0x21, 0x21, mjmyster_coincounter_w		},
	{ 0x40, 0x40, OKIM6295_data_0_w				},
	{ 0x42, 0x42, YM2413_register_port_0_w		},
	{ 0x43, 0x43, YM2413_data_port_0_w			},
	{ 0x46, 0x46, AY8910_write_port_0_w			},
	{ 0x48, 0x48, AY8910_control_port_0_w		},
	{ 0x60, 0x6f, msm6242_w						},	// 6242RTC
	{ 0x80, 0x83, dynax_palette_base_w			},
	{ 0x84, 0x87, dynax_palette_mask_w			},
	{ 0x88, 0x8b, dynax_transparency_pen_w		},
	{ 0x8c, 0x8f, dynax_transparency_mask_w		},
	{ 0x94, 0x94, dynax_bgcolor_w				},
	{ 0x95, 0x95, dynax_priority_w				},
	{ 0x96, 0x96, dynax_layer_enable_w			},
PORT_END


/***************************************************************************
                  Panel & Variety Akamaru Q Jousyou Dont-R
***************************************************************************/

static UINT16 akamaru_protection1;
static READ16_HANDLER( akamaru_protection1_r )
{
	return (akamaru_protection1 & 0x0008) ? 0x0001 : 0x0000;
}
static WRITE16_HANDLER( akamaru_protection1_w )
{
	int bank;
	COMBINE_DATA( &akamaru_protection1 );
	// BCD number?
	bank = (((akamaru_protection1>>4)&0xf)%10) * 10 + ((akamaru_protection1&0xf)%10);
	OKIM6295_set_bank_base(0, bank * 0x40000);

//  usrintf_showmessage("bank $%0x (%d)",akamaru_protection1, bank);
}


static UINT16 *akamaru_protection2;
static READ16_HANDLER( akamaru_protection2_r )
{
	return 0x55;
}

static UINT16 *akamaru_dsw_sel;
static READ16_HANDLER( akamaru_dsw_r )
{
	UINT16 dsw = 0;
	if (akamaru_dsw_sel[1] == 0xff)	dsw |= readinputport(3);
	if (akamaru_dsw_sel[0] == 0xff)	dsw |= readinputport(4);
	return dsw;
}

static READ16_HANDLER( akamaru_blitter_r )
{
	return dynax_blitter_irq_flag << 6;	// bit 7 = 1 -> blitter busy
}

static READ16_HANDLER( akamaru_e0010d_r )
{
	return 0xffff;	// read but not used, it seems
}

static MEMORY_READ16_START( akamaru_readmem )
	{ 0x000000, 0x1fffff, MRA16_ROM						},	// ROM
	{ 0x624680, 0x624681, akamaru_protection1_r			},
	{ 0xe00086, 0xe00087, ddenlovr_gfxrom_r				},	// Video Chip
	{ 0xe00070, 0xe00071, unk16_r						},	// ? must be 78 on startup (not necessary in ddlover)
	{ 0xe00100, 0xe00101, input_port_0_word_r			},	// P1?
	{ 0xe00102, 0xe00103, input_port_1_word_r			},	// P2?
	{ 0xe00104, 0xe00105, input_port_2_word_r			},	// Coins
	{ 0xe00106, 0xe00107, akamaru_protection2_r			},
	{ 0xe0010c, 0xe0010d, akamaru_e0010d_r				},
	{ 0xe00200, 0xe00201, akamaru_dsw_r					},	// DSW
	{ 0xe00204, 0xe00205, akamaru_blitter_r				},	// Blitter Busy & IRQ
	{ 0xe00500, 0xe0051f, msm6242_lsb_r					},	// 6242RTC
	{ 0xe00604, 0xe00605, AY8910_read_port_0_lsb_r		},
	{ 0xe00700, 0xe00701, OKIM6295_status_0_lsb_r		},	// Sound
	{ 0xff0000, 0xffffff, MRA16_RAM						},	// RAM
MEMORY_END

static MEMORY_WRITE16_START( akamaru_writemem )
	{ 0x000000, 0x1fffff, MWA16_ROM						},	// ROM
	{ 0x213570, 0x213571, akamaru_protection1_w			},	// OKI bank
	{ 0xd00000, 0xd003ff, ddenlovr_palette_w			},	// Palette
//  { 0xd01000, 0xd017ff, MWA16_RAM                     },   // 0
	{ 0xe00040, 0xe00047, ddenlovr_palette_base_w		},
	{ 0xe00048, 0xe0004f, ddenlovr_palette_mask_w		},
	{ 0xe00050, 0xe00057, ddenlovr_transparency_pen_w	},
	{ 0xe00058, 0xe0005f, ddenlovr_transparency_mask_w	},
	{ 0xe00068, 0xe00069, ddenlovr_bgcolor_w			},
	{ 0xe0006a, 0xe0006b, ddenlovr_priority_w			},
	{ 0xe0006c, 0xe0006d, ddenlovr_layer_enable_w		},
	{ 0xe00080, 0xe00083, ddenlovr_blitter_w			},
	{ 0xe00108, 0xe0010b, MWA16_RAM, &akamaru_protection2	},
	{ 0xe00302, 0xe00303, ddenlovr_blitter_irq_ack_w	},	// Blitter irq acknowledge
	{ 0xe00304, 0xe00307, MWA16_RAM, &akamaru_dsw_sel	},	// DSW select
	{ 0xe00308, 0xe00309, ddenlovr_coincounter_0_w		},	// Coin Counters
	{ 0xe0030c, 0xe0030d, ddenlovr_coincounter_1_w		},	//
	{ 0xe00400, 0xe00401, YM2413_register_port_0_lsb_w	},
	{ 0xe00402, 0xe00403, YM2413_data_port_0_lsb_w		},
	{ 0xe00500, 0xe0051f, msm6242_lsb_w					},	// 6242RTC
	{ 0xe00600, 0xe00601, AY8910_control_port_0_lsb_w	},
	{ 0xe00602, 0xe00603, AY8910_write_port_0_lsb_w		},
	{ 0xe00700, 0xe00701, OKIM6295_data_0_lsb_w 		},
	{ 0xff0000, 0xffffff, MWA16_RAM						},	// RAM
MEMORY_END


/***************************************************************************
                          Mahjong Fantasic Love
***************************************************************************/

static WRITE8_HANDLER( mjflove_rombank_w )
{
	UINT8 *rom = memory_region(REGION_CPU1);
	memory_set_bankptr(1, &rom[0x10000 + 0x8000 * (data & 0xf)]);
}

static WRITE8_HANDLER( mjflove_okibank_w )
{
	OKIM6295_set_bank_base(0, (data & 0x07) * 0x40000);
//  usrintf_showmessage("SOUND = %02x",data);
}

static READ8_HANDLER( mjflove_protection_r )
{
	return 0x27;
}

static READ8_HANDLER( mjflove_keyb_r )
{
	UINT8 val = 0xff;

	if      (!(keyb & 0x01))	val = readinputport(offset * 5 + 1);
	else if (!(keyb & 0x02))	val = readinputport(offset * 5 + 2);
	else if (!(keyb & 0x04))	val = readinputport(offset * 5 + 3);
	else if (!(keyb & 0x08))	val = readinputport(offset * 5 + 4);
	else if (!(keyb & 0x10))	val = readinputport(offset * 5 + 5);

	return val;
}

UINT8 mjflove_irq_cause;

static READ8_HANDLER( mjflove_blitter_r )
{
	// bit 7 = 1 -> blitter busy
	// bit 6 = 0 -> VBLANK?
	// bit 5 = 0 -> RTC?
	return readinputport(0) | mjflove_irq_cause;
}

static WRITE8_HANDLER( mjflove_blitter_w )
{
	blitter_w(0,offset,data,0);
}

static WRITE8_HANDLER( mjflove_coincounter_w )
{
	// bit 0 = in counter
	coin_counter_w(0, data   & 0x01);

	if (data & 0xfe)
	{
		logerror("%04x: warning, coin counter = %02x\n", activecpu_get_pc(), data);
//      usrintf_showmessage("COIN = %02x",data);
	}
}

static PORT_READ_START( mjflove_readport )
	{ 0x0010, 0x0010, hanakanz_rand_r			},
	{ 0x001c, 0x001c, input_port_12_r			},
	{ 0x0038, 0x0038, IORP_NOP					},	// ? ack or watchdog
	{ 0x0043, 0x0043, rongrong_gfxrom_r			},
	{ 0x0080, 0x0081, mjflove_keyb_r			},
	{ 0x0082, 0x0082, mjflove_blitter_r			},
	{ 0x00da, 0x00da, mjflove_protection_r		},
	{ 0x0100, 0x0100, input_port_11_r			},
	{ 0x0280, 0x028f, msm6242_r					},	// 6242RTC
	{ 0x0380, 0x0380, OKIM6295_status_0_r		},
PORT_END

static PORT_WRITE_START( mjflove_writeport )
	{ 0x001e, 0x001e, hanakanz_keyb_w			},
	{ 0x0020, 0x0023, dynax_palette_base_w		},
	{ 0x0024, 0x0027, dynax_palette_mask_w		},
	{ 0x0028, 0x002b, dynax_transparency_pen_w	},
	{ 0x002c, 0x002f, dynax_transparency_mask_w	},
	{ 0x0034, 0x0034, dynax_bgcolor_w			},
	{ 0x0035, 0x0035, dynax_priority_w			},
	{ 0x0036, 0x0036, dynax_layer_enable_w		},
	{ 0x0040, 0x0041, mjflove_blitter_w			},
	{ 0x00f2, 0x00f2, mjmyster_rambank_w		},
	{ 0x00f8, 0x00f8, mjflove_rombank_w			},
	{ 0x00fa, 0x00fa, mjflove_okibank_w			},
	{ 0x0181, 0x0181, IOWP_NOP					},	// ? int. enable
	{ 0x0184, 0x0184, mjflove_coincounter_w		},
	{ 0x0200, 0x0200, YM2413_register_port_0_w	},
	{ 0x0201, 0x0201, YM2413_data_port_0_w		},
	{ 0x0280, 0x028f, msm6242_w					},	// 6242RTC
	{ 0x0300, 0x0300, AY8910_control_port_0_w	},
	{ 0x0301, 0x0301, AY8910_write_port_0_w		},
	{ 0x0380, 0x0380, OKIM6295_data_0_w			},
PORT_END


INPUT_PORTS_START( ddenlvrj )
	PORT_START	// IN0 - Player 1
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN1 - Player 2
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2 - Coins + ?
	PORT_BIT(  0x01, IP_ACTIVE_LOW,  IPT_COIN1    )
	PORT_BIT(  0x02, IP_ACTIVE_LOW,  IPT_COIN2    )
	PORT_BIT(  0x04, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BITX( 0x08, IP_ACTIVE_LOW,  IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT(  0x10, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter busy flag
	PORT_BIT(  0x20, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter irq flag?
	PORT_BIT(  0x40, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// RTC
	PORT_BIT(  0x80, IP_ACTIVE_HIGH, IPT_SPECIAL  )

	PORT_START	// IN3 - DSW1
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Helps" )
	PORT_DIPSETTING(    0x20, "1" )
	PORT_DIPSETTING(    0x00, "2" )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 1-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 1-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN4 - DSW2
	PORT_DIPNAME( 0x07, 0x07, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x06, "1 (Easy)" )
	PORT_DIPSETTING(    0x05, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x07, "4 (Normal)" )
	PORT_DIPSETTING(    0x03, "5" )
	PORT_DIPSETTING(    0x02, "6" )
	PORT_DIPSETTING(    0x01, "7" )
	PORT_DIPSETTING(    0x00, "8 (Hard)" )
	PORT_DIPNAME( 0x18, 0x18, "Timer Speed" )
	PORT_DIPSETTING(    0x10, "Slow" )
	PORT_DIPSETTING(    0x18, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x08, "Fast" )
	PORT_DIPSETTING(    0x00, "Fastest" )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 2-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN5 - DSW1 & 2 top bits
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-8" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-9" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 3-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 3-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, "Force Test Mode?" )	// shows all dsw's as off
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Yes ) )
INPUT_PORTS_END


INPUT_PORTS_START( ddenlovr )
	PORT_START	// IN0 - Player 1
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN1 - Player 2
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2 - Coins + ?
	PORT_BIT(  0x01, IP_ACTIVE_LOW,  IPT_COIN1    )
	PORT_BIT(  0x02, IP_ACTIVE_LOW,  IPT_COIN2    )
	PORT_BIT(  0x04, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BITX( 0x08, IP_ACTIVE_LOW,  IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT(  0x10, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
	PORT_BIT(  0x20, IP_ACTIVE_LOW,  IPT_SPECIAL  )	// ? quiz365
	PORT_BIT(  0x40, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter irq flag
	PORT_BIT(  0x80, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter busy flag

	PORT_START	// IN3 - DSW
	PORT_SERVICE( 0x01, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 1-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 1-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 1-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 1-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Comments / Help" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Yes ) )
	PORT_DIPNAME( 0xc0, 0xc0, "Show Girl" )
	PORT_DIPSETTING(    0x80, DEF_STR( No ) )
	PORT_DIPSETTING(    0x40, "Yes, Clothed" )
	PORT_DIPSETTING(    0xc0, "Yes, Bikini" )
	PORT_DIPSETTING(    0x00, "Yes, Topless" )
INPUT_PORTS_END


INPUT_PORTS_START( nettoqc )
	PORT_START	// IN0 - Player 1
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_BUTTON1  | IPF_PLAYER1 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_BUTTON2  | IPF_PLAYER1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_BUTTON3  | IPF_PLAYER1 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_BUTTON4  | IPF_PLAYER1 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN1 - Player 2
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_BUTTON1  | IPF_PLAYER2 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_BUTTON2  | IPF_PLAYER2 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_BUTTON3  | IPF_PLAYER2 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_BUTTON4  | IPF_PLAYER2 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2 - Coins + ?
	PORT_BIT(  0x01, IP_ACTIVE_LOW,  IPT_COIN1    )
	PORT_BIT(  0x02, IP_ACTIVE_LOW,  IPT_COIN2    )
	PORT_BIT(  0x04, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BITX( 0x08, IP_ACTIVE_LOW,  IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT(  0x10, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter busy flag
	PORT_BIT(  0x20, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// ?
	PORT_BIT(  0x40, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter irq flag
	PORT_BIT(  0x80, IP_ACTIVE_HIGH, IPT_SPECIAL  )

	PORT_START	// IN3 - DSW
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x0c, "1" )
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 1-6*" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 1-7*" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN4 - DSW
	PORT_DIPNAME( 0x03, 0x03, "Unknown 2-0&1*" )
	PORT_DIPSETTING(    0x02, "0" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x01, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3*" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4*" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5*" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6*" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 2-7*" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN5 - DSW
	PORT_DIPNAME( 0x01, 0x01, "Unknown 1-8*" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-8*" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-9*" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x80, 0x00, "Detailed Tests" )	// menu "8 OPTION" in service mode
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( quiz365 )
	PORT_START	// IN0 - Player 1
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_BUTTON1  | IPF_PLAYER1 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_BUTTON2  | IPF_PLAYER1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_BUTTON3  | IPF_PLAYER1 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_BUTTON4  | IPF_PLAYER1 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN1 - Player 2
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_BUTTON1  | IPF_PLAYER2 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_BUTTON2  | IPF_PLAYER2 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_BUTTON3  | IPF_PLAYER2 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_BUTTON4  | IPF_PLAYER2 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2 - Coins + ?
	PORT_BIT(  0x01, IP_ACTIVE_LOW,  IPT_COIN1    )
	PORT_BIT(  0x02, IP_ACTIVE_LOW,  IPT_COIN2    )
	PORT_BIT(  0x04, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BITX( 0x08, IP_ACTIVE_LOW,  IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT(  0x10, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
	PORT_BIT(  0x20, IP_ACTIVE_LOW,  IPT_SPECIAL  )	// ? quiz365
	PORT_BIT(  0x40, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter irq flag
	PORT_BIT(  0x80, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter busy flag

	PORT_START	// IN3 - DSW
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 1-5*" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0xc0, 0xc0, "Unknown 1-6&7" )
	PORT_DIPSETTING(    0x40, "0" )
	PORT_DIPSETTING(    0x80, "1" )
	PORT_DIPSETTING(    0xc0, "2" )
//  PORT_DIPSETTING(    0x00, "2" )

	PORT_START	// IN4 - DSW
	PORT_DIPNAME( 0x03, 0x03, "Unknown 2-0&1" )
	PORT_DIPSETTING(    0x02, "0" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x01, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x0c, 0x0c, "Unknown 2-2&3" )
	PORT_DIPSETTING(    0x08, "0" )
	PORT_DIPSETTING(    0x0c, "1" )
	PORT_DIPSETTING(    0x04, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 2-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN5 - DSW
	PORT_DIPNAME( 0x01, 0x01, "Unknown 3-0" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x04, 0x00, "Detailed Tests" )	// menu "8 OPTION" in service mode
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 3-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( rongrong )
	PORT_START	// IN0 - Player 1
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER1 )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER1 )

	PORT_START	// IN1 - Player 2
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_BUTTON2        | IPF_PLAYER2 )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_BUTTON3        | IPF_PLAYER2 )

	PORT_START	// IN2 - Coins + ?
	PORT_BIT(  0x01, IP_ACTIVE_LOW,  IPT_COIN1    )
	PORT_BIT(  0x02, IP_ACTIVE_LOW,  IPT_COIN2    )
	PORT_BIT(  0x04, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BITX( 0x08, IP_ACTIVE_LOW,  IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT(  0x10, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
	PORT_BIT(  0x20, IP_ACTIVE_LOW,  IPT_SPECIAL  )	// ? quiz365
	PORT_BIT(  0x40, IP_ACTIVE_LOW,  IPT_SPECIAL  )	// ? blitter irq flag ?
	PORT_BIT(  0x80, IP_ACTIVE_LOW,  IPT_SPECIAL  )	// ? blitter busy flag ?

  	PORT_START	// IN3 - DSW
 	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
 	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
 	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Coin_B ) )
 	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
 	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
 	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Free_Play ) )
  	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
  	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
 	PORT_DIPNAME( 0x20, 0x20, "Helps" )
 	PORT_DIPSETTING(    0x00, "2" )
 	PORT_DIPSETTING(    0x20, "3" )
 	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
 	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
 	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
  	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
  	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

  	PORT_START	// IN4 - DSW
 	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Difficulty ) )
  	PORT_DIPSETTING(    0x02, DEF_STR( Easy ) )
  	PORT_DIPSETTING(    0x03, DEF_STR( Normal ) )
  	PORT_DIPSETTING(    0x01, DEF_STR( Hard ) )
  	PORT_DIPSETTING(    0x00, DEF_STR( Hardest ) )
 	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Unknown ) )
  	PORT_DIPSETTING(    0x0c, "0" )
  	PORT_DIPSETTING(    0x08, "1" )
  	PORT_DIPSETTING(    0x04, "2" )
  	PORT_DIPSETTING(    0x00, "3" )
 	PORT_DIPNAME( 0x30, 0x30, "VS Rounds" )
  	PORT_DIPSETTING(    0x30, "1" )
  	PORT_DIPSETTING(    0x20, "2" )
  	PORT_DIPSETTING(    0x10, "3" )
//  PORT_DIPSETTING(    0x00, "3" )
 	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
  	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
  	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
 	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
  	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
  	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

  	PORT_START	// IN5 - DSW
 	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
  	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
  	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BITX(0x02, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
 	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
  	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
  	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
 	PORT_DIPNAME( 0x08, 0x08, "Select Round" )
  	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
  	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
  	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( quizchq )
	PORT_START	// IN0 - Player 1
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN1 - Player 2
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2 - Coins + ?
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_COIN1    )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_COIN2    )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BITX( 0x08, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_UNKNOWN  )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_SPECIAL  )	// ? quiz365
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_SPECIAL  )	// ? blitter irq flag ?
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_SPECIAL  )	// ? blitter busy flag ?

	PORT_START	// IN3 - DSW
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 1-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Demo_Sounds ))
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 1-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Set Date" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN4 - DSW
	PORT_DIPNAME( 0x01, 0x01, "Unknown 2-0" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 2-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	// IN5 - DSW
	PORT_DIPNAME( 0x01, 0x01, "Unknown 1-0" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 1-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 1-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 1-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END

INPUT_PORTS_START( mmpanic )
	PORT_START	// IN0 6a (68 = 1:used? 2:normal 3:goes to 69)
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_LOW, IPT_COIN1, 2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )	// tested?
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BITX(0x08, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// tested?
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "8", KEYCODE_8_PAD, IP_JOY_NONE )
	PORT_BITX(0x40, IP_ACTIVE_LOW, 0, "9", KEYCODE_9_PAD, IP_JOY_NONE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SPECIAL )	// busy?

	PORT_START	// IN1 6b (68 = 0 & 1)
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )	// tested
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "1", KEYCODE_1_PAD, IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "2", KEYCODE_2_PAD, IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "3", KEYCODE_3_PAD, IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "4", KEYCODE_4_PAD, IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "5", KEYCODE_5_PAD, IP_JOY_NONE )
	PORT_BITX(0x40, IP_ACTIVE_LOW, 0, "6", KEYCODE_6_PAD, IP_JOY_NONE )
	PORT_BITX(0x80, IP_ACTIVE_LOW, 0, "7", KEYCODE_7_PAD, IP_JOY_NONE )

	PORT_START	// IN2 - DSW
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x1c, 0x10, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x1c, "0" )
	PORT_DIPSETTING(    0x18, "1" )
	PORT_DIPSETTING(    0x14, "2" )
	PORT_DIPSETTING(    0x10, "3" )
	PORT_DIPSETTING(    0x0c, "4" )
	PORT_DIPSETTING(    0x08, "5" )
//  PORT_DIPSETTING(    0x04, "5" )
//  PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x20, 0x20, "Linked Cabinets" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 1-7*" )	// 2-0 is related to the same thing (flip?)
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN3 - DSW
	PORT_DIPNAME( 0x01, 0x01, "Unknown 2-0*" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, "Unknown 2-2&3*" )
	PORT_DIPSETTING(    0x0c, "10" )
	PORT_DIPSETTING(    0x08, "20" )
	PORT_DIPSETTING(    0x04, "40" )
	PORT_DIPSETTING(    0x00, "80" )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4*" )	// used?
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5*" )	// used?
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6*" )	// 6 & 7?
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 2-7*" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN4 - DSW
	PORT_DIPNAME( 0x01, 0x01, "Set Date" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 3-2*" )	// used?
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 3-3*" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 3-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 3-6*" )	// used?
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 3-7*" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( animaljr )
	PORT_START	// IN0 6a (68 = 1:used? 2:normal 3:goes to 69)
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_LOW, IPT_COIN1, 2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )	// tested?
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BITX(0x08, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_SPECIAL )	// busy?

	PORT_START	// IN1 6b (68 = 0 & 1)
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )	// tested ?
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "1", KEYCODE_1_PAD, IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "2", KEYCODE_2_PAD, IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "3", KEYCODE_3_PAD, IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "4", KEYCODE_4_PAD, IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "5", KEYCODE_5_PAD, IP_JOY_NONE )
	PORT_BITX(0x40, IP_ACTIVE_LOW, 0, "6", KEYCODE_6_PAD, IP_JOY_NONE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2 - DSW
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 1-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 1-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 1-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0xe0, 0x80, "Difficulty?" )
	PORT_DIPSETTING(    0xe0, "0" )
	PORT_DIPSETTING(    0xc0, "1" )
	PORT_DIPSETTING(    0xa0, "2" )
	PORT_DIPSETTING(    0x80, "3" )
	PORT_DIPSETTING(    0x60, "4" )
	PORT_DIPSETTING(    0x40, "5" )
	PORT_DIPSETTING(    0x20, "6" )
	PORT_DIPSETTING(    0x00, "7" )

	PORT_START	// IN3 - DSW
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2*" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x18, "Unknown 2-3&4" )	// used ?
//  PORT_DIPSETTING(    0x10, "0" )
	PORT_DIPSETTING(    0x00, "0" )
	PORT_DIPSETTING(    0x18, "1" )
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x00, "Pirate Fight" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Taito Copyright" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START	// IN4 - DSW
	PORT_DIPNAME( 0x01, 0x01, "Unknown 3-0*" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Tickets" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 3-2*" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x08, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 3-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 3-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 3-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


INPUT_PORTS_START( hanakanz )

	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN	)

	// keyb 2
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 A",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 E",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Yes",        IP_KEY_DEFAULT,     IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 B",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 F",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 No",         IP_KEY_DEFAULT,     IP_JOY_NONE )	// N
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Bet",        IP_KEY_DEFAULT,     IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 C",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 G",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// K
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 D",           IP_KEY_DEFAULT,    IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 H",           IP_KEY_DEFAULT,    IP_JOY_NONE )	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// L
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN				)
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 Take Score",  IP_KEY_DEFAULT,    IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 Double Up",   IP_KEY_DEFAULT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Flip",        IP_KEY_DEFAULT,    IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Big",         IP_KEY_DEFAULT,    IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Small",       IP_KEY_DEFAULT,    IP_JOY_NONE )	// "s"

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "1",              KEYCODE_1_PAD,     IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "5",              KEYCODE_5_PAD,     IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Yes",         KEYCODE_ENTER_PAD, IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "2",              KEYCODE_2_PAD,     IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "6",              KEYCODE_6_PAD,     IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 No",          KEYCODE_DEL_PAD,   IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_2,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "3",              KEYCODE_3_PAD,     IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "7",              KEYCODE_7_PAD,     IP_JOY_NONE )	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// K
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "4",              KEYCODE_4_PAD,     IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "8",              KEYCODE_8_PAD,     IP_JOY_NONE )	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// L
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN				)
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	PORT_START	// IN11 - DSW1
	PORT_DIPNAME( 0x07, 0x07, "Unknown 1-0&1&2" )
	PORT_DIPSETTING(    0x07, "0" )
	PORT_DIPSETTING(    0x06, "1" )
	PORT_DIPSETTING(    0x05, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x03, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x01, "6" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 1-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 1-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x60, "Unknown 1-5&6" )
	PORT_DIPSETTING(    0x60, "0" )
	PORT_DIPSETTING(    0x40, "1" )
	PORT_DIPSETTING(    0x20, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 1-7" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPSETTING(    0x80, "10" )

	PORT_START	// IN12 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, "Unknown 2-2&3" )
	PORT_DIPSETTING(    0x0c, "10" )
	PORT_DIPSETTING(    0x08, "20" )
	PORT_DIPSETTING(    0x04, "30" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x30, 0x30, "Unknown 2-4&5" )
	PORT_DIPSETTING(    0x30, "100" )
	PORT_DIPSETTING(    0x20, "200" )
	PORT_DIPSETTING(    0x10, "250" )
	PORT_DIPSETTING(    0x00, "300" )
	PORT_DIPNAME( 0xc0, 0xc0, "Unknown 2-6&7" )
	PORT_DIPSETTING(    0xc0, "50" )
	PORT_DIPSETTING(    0x80, "60" )
	PORT_DIPSETTING(    0x40, "70" )
	PORT_DIPSETTING(    0x00, "80" )

	PORT_START	// IN13 - DSW3
	PORT_DIPNAME( 0x03, 0x03, "Game Type" )
	PORT_DIPSETTING(    0x03, "8 Cards" )
	PORT_DIPSETTING(    0x02, "6 Cards (Bets)" )
	PORT_DIPSETTING(    0x01, "6 Cards (Bets)?" )
	PORT_DIPSETTING(    0x00, "6 Cards (Bets)??" )
	PORT_DIPNAME( 0x04, 0x04, "(C) Nihon (Censored)" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x18, "Unknown 3-3&4" )
	PORT_DIPSETTING(    0x18, "10" )
	PORT_DIPSETTING(    0x10, "20" )
	PORT_DIPSETTING(    0x08, "30" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x60, 0x60, "Unknown 3-5&6" )
	PORT_DIPSETTING(    0x60, "0" )
	PORT_DIPSETTING(    0x40, "1" )
	PORT_DIPSETTING(    0x20, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x80, 0x80, "Girl" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START	// IN14 - DSW4
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 4-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 4-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 4-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 4-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 4-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Keyboard" )
	PORT_DIPSETTING(    0x40, "Hanafuda" )
	PORT_DIPSETTING(    0x00, "Mahjong" )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 4-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN15 - DSWs top bits
	PORT_DIPNAME( 0x01, 0x01, "Unknown 1-8" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 1-9" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, "Unknown 2-8&9" )
	PORT_DIPSETTING(    0x0c, "20" )
	PORT_DIPSETTING(    0x08, "30" )
	PORT_DIPSETTING(    0x04, "40" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 3-8" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-9" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 4-8" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	// IN16 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, "Allow Bets" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN17 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, "? Hopper M." )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

INPUT_PORTS_END


INPUT_PORTS_START( hkagerou )

	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN	)

	// keyb 2
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 A",           IP_KEY_DEFAULT,      IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 E",           IP_KEY_DEFAULT,      IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Yes",         IP_KEY_DEFAULT,      IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 B",           IP_KEY_DEFAULT,      IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 F",           IP_KEY_DEFAULT,      IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 No",          IP_KEY_DEFAULT,      IP_JOY_NONE )	// N
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Bet",         IP_KEY_DEFAULT,   IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 C",           IP_KEY_DEFAULT,      IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 G",           IP_KEY_DEFAULT,      IP_JOY_NONE )	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// K
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 D",           IP_KEY_DEFAULT,      IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 H",           IP_KEY_DEFAULT,      IP_JOY_NONE )	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// L
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN				)
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 Take Score",  IP_KEY_DEFAULT,    IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 Double Up",   IP_KEY_DEFAULT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Flip",        IP_KEY_DEFAULT,    IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Big",         IP_KEY_DEFAULT,    IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Small",       IP_KEY_DEFAULT,    IP_JOY_NONE )	// "s"

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "1",              KEYCODE_1_PAD,     IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "5",              KEYCODE_5_PAD,     IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Yes",         KEYCODE_ENTER_PAD, IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "2",              KEYCODE_2_PAD,     IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "6",              KEYCODE_6_PAD,     IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 No",          KEYCODE_DEL_PAD,   IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_2,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "3",              KEYCODE_3_PAD,     IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "7",              KEYCODE_7_PAD,     IP_JOY_NONE )	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// K
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "4",              KEYCODE_4_PAD,     IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "8",              KEYCODE_8_PAD,     IP_JOY_NONE )	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// L
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN )	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN				)
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	PORT_START	// IN11 - DSW1
	PORT_DIPNAME( 0x07, 0x07, "Unknown 1-0&1&2" )
	PORT_DIPSETTING(    0x07, "0" )
	PORT_DIPSETTING(    0x06, "1" )
	PORT_DIPSETTING(    0x05, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x03, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x01, "6" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 1-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 1-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x60, "Unknown 1-5&6" )
	PORT_DIPSETTING(    0x60, "0" )
	PORT_DIPSETTING(    0x40, "1" )
	PORT_DIPSETTING(    0x20, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x80, 0x80, "Credits Per Note" )
	PORT_DIPSETTING(    0x00, "25" )
	PORT_DIPSETTING(    0x80, "50" )

	PORT_START	// IN12 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x00, "1 Coin/10 Credits" )
	PORT_DIPNAME( 0x0c, 0x0c, "Unknown 2-2&3" )
	PORT_DIPSETTING(    0x0c, "10" )
	PORT_DIPSETTING(    0x08, "20" )
	PORT_DIPSETTING(    0x04, "30" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x30, 0x30, "Unknown 2-4&5" )
	PORT_DIPSETTING(    0x30, "100" )
	PORT_DIPSETTING(    0x20, "200" )
	PORT_DIPSETTING(    0x10, "250" )
	PORT_DIPSETTING(    0x00, "300" )
	PORT_DIPNAME( 0xc0, 0xc0, "Unknown 2-6&7" )
	PORT_DIPSETTING(    0xc0, "50" )
	PORT_DIPSETTING(    0x80, "60" )
	PORT_DIPSETTING(    0x40, "70" )
	PORT_DIPSETTING(    0x00, "80" )

	PORT_START	// IN13 - DSW3
	PORT_DIPNAME( 0x01, 0x01, "Game Type?" )
	PORT_DIPSETTING(    0x01, "0" )
	PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 3-2" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "(C) Nihon (Censored)" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x18, "Unknown 3-3&4" )
	PORT_DIPSETTING(    0x18, "10" )
	PORT_DIPSETTING(    0x10, "20" )
	PORT_DIPSETTING(    0x08, "30" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x60, 0x60, "Unknown 3-5&6" )
	PORT_DIPSETTING(    0x60, "0" )
	PORT_DIPSETTING(    0x40, "1" )
	PORT_DIPSETTING(    0x20, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x80, 0x80, "Girl?" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )

	PORT_START	// IN14 - DSW4
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 4-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 4-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 4-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 4-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 4-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Keyboard" )
	PORT_DIPSETTING(    0x40, "Hanafuda" )
	PORT_DIPSETTING(    0x00, "Mahjong" )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 4-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN15 - DSWs top bits
	PORT_DIPNAME( 0x01, 0x01, "Unknown 1-8" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPSETTING(    0x01, "10" )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 1-9" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, "Unknown 2-8&9" )
	PORT_DIPSETTING(    0x0c, "20" )
	PORT_DIPSETTING(    0x08, "30" )
	PORT_DIPSETTING(    0x04, "40" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 3-8" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-9" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 4-8" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	// IN16 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, "Disable Bets" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN17 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

INPUT_PORTS_END


INPUT_PORTS_START( mjreach1 )

	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN	)

	// keyb 2
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 A",           IP_KEY_DEFAULT, IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 E",           IP_KEY_DEFAULT, IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 I",           IP_KEY_DEFAULT, IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 M",           IP_KEY_DEFAULT, IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Kan",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )	// Start 2

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 B",           IP_KEY_DEFAULT, IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 F",           IP_KEY_DEFAULT, IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 J",           IP_KEY_DEFAULT, IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 N",           IP_KEY_DEFAULT, IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Reach",       IP_KEY_DEFAULT, IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Bet",         KEYCODE_8,      IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 C",           IP_KEY_DEFAULT, IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 G",           IP_KEY_DEFAULT, IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 K",           IP_KEY_DEFAULT, IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Chii",        IP_KEY_DEFAULT, IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Ron",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 D",           IP_KEY_DEFAULT, IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 H",           IP_KEY_DEFAULT, IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 L",           IP_KEY_DEFAULT, IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Pon",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 Last Chance", IP_KEY_DEFAULT, IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 Take Score",  IP_KEY_DEFAULT, IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 Double Up",   IP_KEY_DEFAULT, IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Flip Flop",   IP_KEY_DEFAULT, IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Big",         IP_KEY_DEFAULT, IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Small",       IP_KEY_DEFAULT, IP_JOY_NONE )	// "s"

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 A",           KEYCODE_A,         IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 E",           KEYCODE_E,         IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 I",           KEYCODE_I,         IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 M",           KEYCODE_M,         IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Kan",         KEYCODE_LCONTROL,  IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )	// Start 1

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 B",           KEYCODE_B,         IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 F",           KEYCODE_F,         IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 J",           KEYCODE_J,         IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 N",           KEYCODE_N,         IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Reach",       KEYCODE_LSHIFT,    IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_7,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 C",           KEYCODE_C,         IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 G",           KEYCODE_G,         IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 K",           KEYCODE_K,         IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Chii",        KEYCODE_SPACE,     IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Ron",         KEYCODE_Z,         IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 D",           KEYCODE_D,         IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 H",           KEYCODE_H,         IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 L",           KEYCODE_L,         IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Pon",         KEYCODE_LALT,      IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 Last Chance", KEYCODE_RALT,      IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	PORT_START	// IN11 - DSW1
	PORT_DIPNAME( 0x0f, 0x07, "Pay Out Rate (%)" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPSETTING(    0x01, "53" )
	PORT_DIPSETTING(    0x02, "56" )
	PORT_DIPSETTING(    0x03, "59" )
	PORT_DIPSETTING(    0x04, "62" )
	PORT_DIPSETTING(    0x05, "65" )
	PORT_DIPSETTING(    0x06, "68" )
	PORT_DIPSETTING(    0x07, "71" )
	PORT_DIPSETTING(    0x08, "75" )
	PORT_DIPSETTING(    0x09, "78" )
	PORT_DIPSETTING(    0x0a, "81" )
	PORT_DIPSETTING(    0x0b, "84" )
	PORT_DIPSETTING(    0x0c, "87" )
	PORT_DIPSETTING(    0x0d, "90" )
	PORT_DIPSETTING(    0x0e, "93" )
	PORT_DIPSETTING(    0x0f, "96" )
	PORT_DIPNAME( 0x30, 0x30, "Odds Rate" )
	PORT_DIPSETTING(    0x30, "1 2 4 8 12 16 24 32" )
	PORT_DIPSETTING(    0x00, "1 2 3 5 8 15 30 50" )
	PORT_DIPSETTING(    0x20, "2 3 6 8 12 15 30 50" )
	PORT_DIPSETTING(    0x10, "1 2 3 5 10 25 50 100" )
	PORT_DIPNAME( 0xc0, 0xc0, "Max Bet" )
	PORT_DIPSETTING(    0xc0, "1" )
	PORT_DIPSETTING(    0x80, "5" )
	PORT_DIPSETTING(    0x40, "10" )
	PORT_DIPSETTING(    0x00, "20" )

	PORT_START	// IN12 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x00, "1 Coin/10 Credits" )
	PORT_DIPNAME( 0x0c, 0x0c, "Min Credits To Start" )
	PORT_DIPSETTING(    0x0c, "1" )
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x30, 0x30, "Payout" )
	PORT_DIPSETTING(    0x30, "300" )
	PORT_DIPSETTING(    0x20, "500" )
	PORT_DIPSETTING(    0x10, "700" )
	PORT_DIPSETTING(    0x00, "1000" )
	PORT_DIPNAME( 0x40, 0x40, "W-BET" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Last Chance" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN13 - DSW3
	PORT_DIPNAME( 0x07, 0x07, "YAKUMAN Bonus" )
	PORT_DIPSETTING(    0x07, "Cut" )
	PORT_DIPSETTING(    0x06, "1 T" )
	PORT_DIPSETTING(    0x05, "300" )
	PORT_DIPSETTING(    0x04, "500" )
	PORT_DIPSETTING(    0x03, "700" )
	PORT_DIPSETTING(    0x02, "1000" )
	PORT_DIPSETTING(    0x01, "1500" )
	PORT_DIPSETTING(    0x00, "2000" )
	PORT_DIPNAME( 0x18, 0x18, "YAKUMAN Times" )
//  PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x18, "1" )
	PORT_DIPSETTING(    0x10, "2" )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPNAME( 0x20, 0x20, "3 BAI In YAKUMAN Bonus Chance" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Auto Tsumo" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Credit Timing" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN14 - DSW4
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "In Game Music" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "Girls (Demo)" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Girls Show After 3 Renso" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "Girls (Play)" )	// Shown as always OFF in dips sheet
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Boys In Game" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Boys" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Don Den Key" )
	PORT_DIPSETTING(    0x80, "Start" )
	PORT_DIPSETTING(    0x00, "Flip/Flop" )

	PORT_START	// IN15 - DSWs top bits
	PORT_DIPNAME( 0x01, 0x01, "Credits Per Note" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, "Computer Strength" )
	PORT_DIPSETTING(    0x0c, "Weak" )
	PORT_DIPSETTING(    0x08, DEF_STR( Normal ))
	PORT_DIPSETTING(    0x04, "Strong" )
	PORT_DIPSETTING(    0x00, "Very Strong" )
	PORT_DIPNAME( 0x10, 0x10, "Game Style" )
	PORT_DIPSETTING(    0x10, "Credit" )
	PORT_DIPSETTING(    0x00, "Credit Time" )
	PORT_DIPNAME( 0x20, 0x20, "Start Method (Credit Time)" )
	PORT_DIPSETTING(    0x20, "?" )
	PORT_DIPSETTING(    0x00, "Rate" )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 4-8" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 4-9" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN16 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, "Bets?" )
	PORT_DIPSETTING(    0x40, "0" )
	PORT_DIPSETTING(    0x00, "1" )

	PORT_START	// IN17 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

INPUT_PORTS_END


INPUT_PORTS_START( mjchuuka )

	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN	)

	// keyb 2
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 A",           IP_KEY_DEFAULT, IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 E",           IP_KEY_DEFAULT, IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 I",           IP_KEY_DEFAULT, IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 M",           IP_KEY_DEFAULT, IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Kan",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )	// Start 2

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 B",           IP_KEY_DEFAULT, IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 F",           IP_KEY_DEFAULT, IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 J",           IP_KEY_DEFAULT, IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 N",           IP_KEY_DEFAULT, IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Reach",       IP_KEY_DEFAULT, IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Bet",         KEYCODE_8,      IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 C",           IP_KEY_DEFAULT, IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 G",           IP_KEY_DEFAULT, IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 K",           IP_KEY_DEFAULT, IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Chii",        IP_KEY_DEFAULT, IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Ron",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 D",           IP_KEY_DEFAULT, IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 H",           IP_KEY_DEFAULT, IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 L",           IP_KEY_DEFAULT, IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Pon",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 Last Chance", IP_KEY_DEFAULT, IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 Take Score",  IP_KEY_DEFAULT, IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 Double Up",   IP_KEY_DEFAULT, IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Flip Flop",   IP_KEY_DEFAULT, IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Big",         IP_KEY_DEFAULT, IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Small",       IP_KEY_DEFAULT, IP_JOY_NONE )	// "s"

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 A",           KEYCODE_A,         IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 E",           KEYCODE_E,         IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 I",           KEYCODE_I,         IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 M",           KEYCODE_M,         IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Kan",         KEYCODE_LCONTROL,  IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )	// Start 1

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 B",           KEYCODE_B,         IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 F",           KEYCODE_F,         IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 J",           KEYCODE_J,         IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 N",           KEYCODE_N,         IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Reach",       KEYCODE_LSHIFT,    IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_7,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 C",           KEYCODE_C,         IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 G",           KEYCODE_G,         IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 K",           KEYCODE_K,         IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Chii",        KEYCODE_SPACE,     IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Ron",         KEYCODE_Z,         IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 D",           KEYCODE_D,         IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 H",           KEYCODE_H,         IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 L",           KEYCODE_L,         IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Pon",         KEYCODE_LALT,      IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 Last Chance", KEYCODE_RALT,      IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	PORT_START	// IN11 - DSW1
	PORT_DIPNAME( 0x0f, 0x07, "Pay Out Rate (%)" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPSETTING(    0x01, "53" )
	PORT_DIPSETTING(    0x02, "56" )
	PORT_DIPSETTING(    0x03, "59" )
	PORT_DIPSETTING(    0x04, "62" )
	PORT_DIPSETTING(    0x05, "65" )
	PORT_DIPSETTING(    0x06, "68" )
	PORT_DIPSETTING(    0x07, "71" )
	PORT_DIPSETTING(    0x08, "75" )
	PORT_DIPSETTING(    0x09, "78" )
	PORT_DIPSETTING(    0x0a, "81" )
	PORT_DIPSETTING(    0x0b, "84" )
	PORT_DIPSETTING(    0x0c, "87" )
	PORT_DIPSETTING(    0x0d, "90" )
	PORT_DIPSETTING(    0x0e, "93" )
	PORT_DIPSETTING(    0x0f, "96" )
	PORT_DIPNAME( 0x30, 0x30, "Odds Rate" )
	PORT_DIPSETTING(    0x20, "1 2 3 4 6 8 10 15" )
	PORT_DIPSETTING(    0x30, "1 2 4 8 12 16 24 32" )
	PORT_DIPSETTING(    0x00, "1 2 3 5 8 15 30 50" )
	PORT_DIPSETTING(    0x10, "1 2 3 5 10 25 50 100" )
	PORT_DIPNAME( 0xc0, 0xc0, "Max Bet" )
	PORT_DIPSETTING(    0xc0, "1" )
	PORT_DIPSETTING(    0x80, "5" )
	PORT_DIPSETTING(    0x40, "10" )
	PORT_DIPSETTING(    0x00, "20" )

	PORT_START	// IN12 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x00, "1 Coin/10 Credits" )
	PORT_DIPNAME( 0x0c, 0x0c, "Min Credits To Start" )
	PORT_DIPSETTING(    0x0c, "1" )
//  PORT_DIPSETTING(    0x08, "2" ) // ? these don't let you start a game
//  PORT_DIPSETTING(    0x04, "3" )
//  PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, "10" )
	PORT_DIPSETTING(    0x00, "255" )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN13 - DSW3
	PORT_DIPNAME( 0x07, 0x07, "YAKUMAN Bonus" )
	PORT_DIPSETTING(    0x07, "Cut" )
	PORT_DIPSETTING(    0x06, "1 T" )
	PORT_DIPSETTING(    0x05, "300" )
	PORT_DIPSETTING(    0x04, "500" )
	PORT_DIPSETTING(    0x03, "700" )
	PORT_DIPSETTING(    0x02, "1000" )
	PORT_DIPSETTING(    0x01, "1000?" )
	PORT_DIPSETTING(    0x00, "1000?" )
	PORT_DIPNAME( 0x18, 0x18, "YAKUMAN Times" )
//  PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x18, "1" )
	PORT_DIPSETTING(    0x10, "2" )
	PORT_DIPSETTING(    0x08, "3?" )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, "149" )
	PORT_DIPSETTING(    0x00, "212" )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN14 - DSW4
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "In Game Music" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x00, "Girls" )
	PORT_DIPSETTING(    0x0c, DEF_STR( No ) )
	PORT_DIPSETTING(    0x08, "Dressed" )
	PORT_DIPSETTING(    0x04, "Underwear" )
	PORT_DIPSETTING(    0x00, "Nude" )
	PORT_DIPNAME( 0x10, 0x00, "Girls Speech" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN15 - DSWs top bits
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Credits Per Note" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPNAME( 0x0c, 0x0c, "Computer Strength?" )
	PORT_DIPSETTING(    0x0c, "Weak" )
	PORT_DIPSETTING(    0x08, DEF_STR( Normal ))
	PORT_DIPSETTING(    0x04, "Strong" )
	PORT_DIPSETTING(    0x00, "Very Strong" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 4-8" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 4-9" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN16 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN17 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

INPUT_PORTS_END


INPUT_PORTS_START( funkyfig )
	PORT_START	// IN0 - Keys (port 83 with port 80 = 20)
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "1", KEYCODE_1_PAD, IP_JOY_NONE )
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "2", KEYCODE_2_PAD, IP_JOY_NONE )
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "3", KEYCODE_3_PAD, IP_JOY_NONE )
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "4", KEYCODE_4_PAD, IP_JOY_NONE )
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "5", KEYCODE_5_PAD, IP_JOY_NONE )
	PORT_BITX(0x40, IP_ACTIVE_LOW, 0, "6", KEYCODE_6_PAD, IP_JOY_NONE )
	PORT_BITX(0x80, IP_ACTIVE_LOW, 0, "7", KEYCODE_7_PAD, IP_JOY_NONE )

	PORT_START	// IN1 - ? (port 83 with port 80 = 21)
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START2 )	// ?
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2 - Coins (port 82 with port 80 = 22)
	PORT_BIT_IMPULSE( 0x01, IP_ACTIVE_LOW, IPT_COIN1, 2 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )	// tested?
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BITX(0x08, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )	// tested?
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "8", KEYCODE_8_PAD, IP_JOY_NONE )
	PORT_BITX(0x40, IP_ACTIVE_LOW, 0, "9", KEYCODE_9_PAD, IP_JOY_NONE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN3 - DSW1 (low bits, port 1c with rombank = 1e)
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x1c, 0x10, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x18, "0" )
	PORT_DIPSETTING(    0x14, "1" )
	PORT_DIPSETTING(    0x1c, "2" )
	PORT_DIPSETTING(    0x10, "3" )
	PORT_DIPSETTING(    0x0c, "4" )
	PORT_DIPSETTING(    0x08, "5" )
//  PORT_DIPSETTING(    0x04, "5" )
//  PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x20, 0x20, "Linked Cabinets" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Play Rock Smash" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Yes ) )

	PORT_START	// IN4 - DSW2 (low bits, port 1c with rombank = 1d)
	PORT_DIPNAME( 0x01, 0x01, "2 Player Game" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x0c, 0x0c, "Unknown 2-2&3*" )
	PORT_DIPSETTING(    0x0c, "10" )
	PORT_DIPSETTING(    0x08, "20" )
	PORT_DIPSETTING(    0x04, "40" )
	PORT_DIPSETTING(    0x00, "80" )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4*" )	// used
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5*" )	// used
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0xc0, 0xc0, "Unknown 2-6&7*" )	// used
	PORT_DIPSETTING(    0xc0, "0" )
	PORT_DIPSETTING(    0x80, "1" )
	PORT_DIPSETTING(    0x40, "2" )
	PORT_DIPSETTING(    0x00, "3" )

	PORT_START	// IN5 - DSW1 & 2 (high bits, port 1c with rombank = 1b)
	PORT_DIPNAME( 0x01, 0x01, "Continue?" )	// related to continue
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x04, 0x04, "Debug Text" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "Country" )
	PORT_DIPSETTING(    0x08, DEF_STR( Japan ) )
	PORT_DIPSETTING(    0x00, DEF_STR( USA ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unused ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unused ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unused ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unused ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END



INPUT_PORTS_START( mjmyster )
	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN	)

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 A",           KEYCODE_A,         IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 E",           KEYCODE_E,         IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 I",           KEYCODE_I,         IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 M",           KEYCODE_M,         IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Kan",         KEYCODE_LCONTROL,  IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )	// Start 1

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 B",           KEYCODE_B,         IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 F",           KEYCODE_F,         IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 J",           KEYCODE_J,         IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 N",           KEYCODE_N,         IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Reach",       KEYCODE_LSHIFT,    IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_7,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 C",           KEYCODE_C,         IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 G",           KEYCODE_G,         IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 K",           KEYCODE_K,         IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Chii",        KEYCODE_SPACE,     IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Ron",         KEYCODE_Z,         IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 D",           KEYCODE_D,         IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 H",           KEYCODE_H,         IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 L",           KEYCODE_L,         IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Pon",         KEYCODE_LALT,      IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 Last Chance", KEYCODE_RALT,      IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	PORT_START	// IN6 - DSW1
	PORT_DIPNAME( 0x0f, 0x07, "Pay Out Rate (%)" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPSETTING(    0x01, "53" )
	PORT_DIPSETTING(    0x02, "56" )
	PORT_DIPSETTING(    0x03, "59" )
	PORT_DIPSETTING(    0x04, "62" )
	PORT_DIPSETTING(    0x05, "65" )
	PORT_DIPSETTING(    0x06, "68" )
	PORT_DIPSETTING(    0x07, "71" )
	PORT_DIPSETTING(    0x08, "75" )
	PORT_DIPSETTING(    0x09, "78" )
	PORT_DIPSETTING(    0x0a, "81" )
	PORT_DIPSETTING(    0x0b, "84" )
	PORT_DIPSETTING(    0x0c, "87" )
	PORT_DIPSETTING(    0x0d, "90" )
	PORT_DIPSETTING(    0x0e, "93" )
	PORT_DIPSETTING(    0x0f, "96" )
	PORT_DIPNAME( 0x30, 0x30, "Odds Rate" )
	PORT_DIPSETTING(    0x20, "2 3 6 8 12 15 30 50" )
	PORT_DIPSETTING(    0x30, "1 2 4 8 12 16 24 32" )
	PORT_DIPSETTING(    0x00, "1 2 3 5 8 15 30 50" )
	PORT_DIPSETTING(    0x10, "1 2 3 5 10 25 50 100" )
	PORT_DIPNAME( 0xc0, 0xc0, "Max Bet" )
	PORT_DIPSETTING(    0xc0, "1" )
	PORT_DIPSETTING(    0x80, "5" )
	PORT_DIPSETTING(    0x40, "10" )
	PORT_DIPSETTING(    0x00, "20" )

	PORT_START	// IN7 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x00, "1 Coin/10 Credits" )
	PORT_DIPNAME( 0x0c, 0x0c, "Min Credits To Start" )
	PORT_DIPSETTING(    0x0c, "1" )
//  PORT_DIPSETTING(    0x08, "2" ) // ? these don't let you start a game
//  PORT_DIPSETTING(    0x04, "3" )
//  PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x70, 0x70, "YAKUMAN Bonus" )
	PORT_DIPSETTING(    0x70, "Cut" )
	PORT_DIPSETTING(    0x60, "1 T" )
	PORT_DIPSETTING(    0x50, "300" )
	PORT_DIPSETTING(    0x40, "500" )
	PORT_DIPSETTING(    0x30, "700" )
	PORT_DIPSETTING(    0x20, "1000" )
	PORT_DIPSETTING(    0x10, "1000?" )
	PORT_DIPSETTING(    0x00, "1000?" )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN8 - DSW3
	PORT_DIPNAME( 0x03, 0x03, "YAKUMAN Times" )
//  PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x01, "3?" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x18, "Payout" )
	PORT_DIPSETTING(    0x18, "300" )
	PORT_DIPSETTING(    0x10, "500" )
	PORT_DIPSETTING(    0x08, "700" )
	PORT_DIPSETTING(    0x00, "1000" )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN9 - DSW4
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "In Game Music" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Controls ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Region ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Japan ) )
	PORT_DIPSETTING(    0x00, "Hong Kong" )

	PORT_START	// IN10 - DSWs top bits
	PORT_DIPNAME( 0x03, 0x03, "Computer Strength?" )
	PORT_DIPSETTING(    0x03, "Weak" )
	PORT_DIPSETTING(    0x02, DEF_STR( Normal ))
	PORT_DIPSETTING(    0x01, "Strong" )
	PORT_DIPSETTING(    0x00, "Very Strong" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, "149" )
	PORT_DIPSETTING(    0x00, "212" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Credits Per Note" )
	PORT_DIPSETTING(    0x40, "5" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )
INPUT_PORTS_END

INPUT_PORTS_START( hginga )

	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
//  PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN  )
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
//  PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN  )

	// keyb 2
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 A",           IP_KEY_DEFAULT, IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 E",           IP_KEY_DEFAULT, IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 M",           IP_KEY_DEFAULT, IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )	// Start 2

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 B",           IP_KEY_DEFAULT, IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 F",           IP_KEY_DEFAULT, IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 N",           IP_KEY_DEFAULT, IP_JOY_NONE )	// N
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Bet",         KEYCODE_8,      IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 C",           IP_KEY_DEFAULT, IP_JOY_NONE )	// C
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// K
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 D",           IP_KEY_DEFAULT, IP_JOY_NONE )	// D
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// L
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN				)
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 Take Score",  IP_KEY_DEFAULT, IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 Double Up",   IP_KEY_DEFAULT, IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Flip Flop",   IP_KEY_DEFAULT, IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Big",         IP_KEY_DEFAULT, IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Small",       IP_KEY_DEFAULT, IP_JOY_NONE )	// "s"

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 A",           KEYCODE_A,         IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 E",           KEYCODE_E,         IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 M",           KEYCODE_M,         IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )	// Start 1

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 B",           KEYCODE_B,         IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 F",           KEYCODE_F,         IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 N",           KEYCODE_N,         IP_JOY_NONE )	// N
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_7,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 C",           KEYCODE_C,         IP_JOY_NONE )	// C
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// K
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 D",           KEYCODE_D,         IP_JOY_NONE )	// D
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// L
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN				)
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	PORT_START	// IN11 - DSW4
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 4-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Girls" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 4-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Change Cards" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 4-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 4-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, "Cards Labels" )
	PORT_DIPSETTING(    0x80, "Numbers" )
	PORT_DIPSETTING(    0x00, "Letters" )

	PORT_START	// IN12 - DSW3
	PORT_DIPNAME( 0x03, 0x03, "Unknown 3-0&1" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x01, "3" )
	PORT_DIPSETTING(    0x00, "4" )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 3-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x18, "Unknown 3-3&4" )
	PORT_DIPSETTING(    0x18, "10" )
	PORT_DIPSETTING(    0x10, "20" )
	PORT_DIPSETTING(    0x08, "30" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0xc0, 0xc0, "Unknown 3-6&7" )
	PORT_DIPSETTING(    0xc0, "0" )
	PORT_DIPSETTING(    0x80, "1" )
	PORT_DIPSETTING(    0x40, "2" )
	PORT_DIPSETTING(    0x00, "3" )

	PORT_START	// IN13 - DSW2
	PORT_DIPNAME( 0x03, 0x03, "Unknown 2-0&1" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x30, 0x30, "Odds Rate" )
	PORT_DIPSETTING(    0x30, "2 5 6 20 50 100" )
	PORT_DIPSETTING(    0x20, "2 5 6 20 50 200" )
	PORT_DIPSETTING(    0x10, "2 5 6 20 50 250" )
	PORT_DIPSETTING(    0x00, "2 5 6 20 50 300" )
	PORT_DIPNAME( 0xc0, 0xc0, "Unknown 2-6&7" )
	PORT_DIPSETTING(    0xc0, "50" )
	PORT_DIPSETTING(    0x80, "60" )
	PORT_DIPSETTING(    0x40, "70" )
	PORT_DIPSETTING(    0x00, "80" )

	PORT_START	// IN14 - DSW1
	PORT_DIPNAME( 0x07, 0x07, "Unknown 1-0&1&2" )
	PORT_DIPSETTING(    0x07, "0" )
	PORT_DIPSETTING(    0x06, "1" )
	PORT_DIPSETTING(    0x05, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x03, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x01, "6" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 1-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 1-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x60, "Unknown 1-5&6" )
	PORT_DIPSETTING(    0x60, "0" )
	PORT_DIPSETTING(    0x40, "1" )
	PORT_DIPSETTING(    0x20, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 1-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN15 - DSWs top bits
	PORT_DIPNAME( 0x01, 0x01, "Keyboard" )
	PORT_DIPSETTING(    0x00, "Hanafuda" )	// Requires different inputs
	PORT_DIPSETTING(    0x01, "Mahjong" )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-8" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-9" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 3-8" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-9" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 4-8" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN16 - Fake DSW
	PORT_DIPNAME( 0x40, 0x40, "Allow Bets" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x40, DEF_STR( On ) )
	PORT_BIT( 0xbf, IP_ACTIVE_LOW, IPT_UNKNOWN	)
INPUT_PORTS_END

INPUT_PORTS_START( hgokou )
	PORT_START	// IN0
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN	)

	// keyb 1
	PORT_START	// IN1
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "1",              KEYCODE_1_PAD,     IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "5",              KEYCODE_5_PAD,     IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Yes",         KEYCODE_ENTER_PAD, IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1			)

	PORT_START	// IN2
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "2",              KEYCODE_2_PAD,     IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "6",              KEYCODE_6_PAD,     IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 No",          KEYCODE_DEL_PAD,   IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// Reach
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// BET

	PORT_START	// IN3
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "3",              KEYCODE_3_PAD,     IP_JOY_NONE )	// C
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START	// IN4
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "4",              KEYCODE_4_PAD,     IP_JOY_NONE )	// D
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN				)	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN				)	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN				)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN				)

	PORT_START	// IN5
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "t"
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "w"
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "b"
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "s"

	// keyb 2
	PORT_START	// IN6
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 A",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 E",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 M",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Take Score",  IP_KEY_DEFAULT,    IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2			)

	PORT_START	// IN7
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 B",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 F",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 N",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Small",       IP_KEY_DEFAULT,    IP_JOY_NONE )	// Reach
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// BET

	PORT_START	// IN8
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 C",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// C
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Big",         IP_KEY_DEFAULT,    IP_JOY_NONE )	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START	// IN9
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 D",           IP_KEY_DEFAULT,    IP_JOY_NONE )	// D
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN				)	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN				)	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Double Up",   IP_KEY_DEFAULT,    IP_JOY_NONE )	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN				)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN				)

	PORT_START	// IN10
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "t"
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "w"
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "b"
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "s"

	PORT_START	// IN11 - DSW4
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 4-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Girls" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 4-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Change Cards" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 4-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 4-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, "Cards Labels" )
	PORT_DIPSETTING(    0x80, "Numbers" )
	PORT_DIPSETTING(    0x00, "Letters" )

	PORT_START	// IN12 - DSW3
	PORT_DIPNAME( 0x03, 0x03, "Unknown 3-0&1" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x01, "3" )
	PORT_DIPSETTING(    0x00, "4" )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 3-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x18, "Unknown 3-3&4" )
	PORT_DIPSETTING(    0x18, "10" )
	PORT_DIPSETTING(    0x10, "20" )
	PORT_DIPSETTING(    0x08, "30" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0xc0, 0xc0, "Unknown 3-6&7" )
	PORT_DIPSETTING(    0xc0, "0" )
	PORT_DIPSETTING(    0x80, "1" )
	PORT_DIPSETTING(    0x40, "2" )
	PORT_DIPSETTING(    0x00, "3" )

	PORT_START	// IN13 - DSW2
	PORT_DIPNAME( 0x03, 0x03, "Unknown 2-0&1" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x30, 0x30, "Odds Rate" )
	PORT_DIPSETTING(    0x30, "2 5 6 20 50 100" )
	PORT_DIPSETTING(    0x20, "2 5 6 20 50 200" )
	PORT_DIPSETTING(    0x10, "2 5 6 20 50 250" )
	PORT_DIPSETTING(    0x00, "2 5 6 20 50 300" )
	PORT_DIPNAME( 0xc0, 0xc0, "Unknown 2-6&7" )
	PORT_DIPSETTING(    0xc0, "50" )
	PORT_DIPSETTING(    0x80, "60" )
	PORT_DIPSETTING(    0x40, "70" )
	PORT_DIPSETTING(    0x00, "80" )

	PORT_START	// IN14 - DSW1
	PORT_DIPNAME( 0x07, 0x07, "Unknown 1-0&1&2" )
	PORT_DIPSETTING(    0x07, "0" )
	PORT_DIPSETTING(    0x06, "1" )
	PORT_DIPSETTING(    0x05, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x03, "4" )
	PORT_DIPSETTING(    0x02, "5" )
	PORT_DIPSETTING(    0x01, "6" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 1-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 1-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x60, "Unknown 1-5&6" )
	PORT_DIPSETTING(    0x60, "0" )
	PORT_DIPSETTING(    0x40, "1" )
	PORT_DIPSETTING(    0x20, "2" )
	PORT_DIPSETTING(    0x00, "3" )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 1-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN15 - DSWs top bits
	PORT_DIPNAME( 0x01, 0x00, "Control Panel" )
	PORT_DIPSETTING(    0x00, "Hanafuda" )
	PORT_DIPSETTING(    0x01, "Mahjong" )	// Requires different inputs
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-8" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-9" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 3-8" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 3-9" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 4-8" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END

INPUT_PORTS_START( mjmyornt )
	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN	)

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 A",           KEYCODE_A,         IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 E",           KEYCODE_E,         IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 I",           KEYCODE_I,         IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 M",           KEYCODE_M,         IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Kan",         KEYCODE_LCONTROL,  IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )	// Start 1

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 B",           KEYCODE_B,         IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 F",           KEYCODE_F,         IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 J",           KEYCODE_J,         IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 N",           KEYCODE_N,         IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Reach",       KEYCODE_LSHIFT,    IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_7,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 C",           KEYCODE_C,         IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 G",           KEYCODE_G,         IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 K",           KEYCODE_K,         IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Chii",        KEYCODE_SPACE,     IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Ron",         KEYCODE_Z,         IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 D",           KEYCODE_D,         IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 H",           KEYCODE_H,         IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 L",           KEYCODE_L,         IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Pon",         KEYCODE_LALT,      IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 Last Chance", KEYCODE_RALT,      IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	PORT_START	// IN6 - DSW1
	PORT_DIPNAME( 0x0f, 0x07, "Pay Out Rate (%)" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPSETTING(    0x01, "53" )
	PORT_DIPSETTING(    0x02, "56" )
	PORT_DIPSETTING(    0x03, "59" )
	PORT_DIPSETTING(    0x04, "62" )
	PORT_DIPSETTING(    0x05, "65" )
	PORT_DIPSETTING(    0x06, "68" )
	PORT_DIPSETTING(    0x07, "71" )
	PORT_DIPSETTING(    0x08, "75" )
	PORT_DIPSETTING(    0x09, "78" )
	PORT_DIPSETTING(    0x0a, "81" )
	PORT_DIPSETTING(    0x0b, "84" )
	PORT_DIPSETTING(    0x0c, "87" )
	PORT_DIPSETTING(    0x0d, "90" )
	PORT_DIPSETTING(    0x0e, "93" )
	PORT_DIPSETTING(    0x0f, "96" )
	PORT_DIPNAME( 0x30, 0x30, "Odds Rate" )
	PORT_DIPSETTING(    0x20, "2 3 6 8 12 15 30 50" )
	PORT_DIPSETTING(    0x30, "1 2 4 8 12 16 24 32" )
	PORT_DIPSETTING(    0x00, "1 2 3 5 8 15 30 50" )
	PORT_DIPSETTING(    0x10, "1 2 3 5 10 25 50 100" )
	PORT_DIPNAME( 0xc0, 0xc0, "Max Bet" )
	PORT_DIPSETTING(    0xc0, "1" )
	PORT_DIPSETTING(    0x80, "5" )
	PORT_DIPSETTING(    0x40, "10" )
	PORT_DIPSETTING(    0x00, "20" )

	PORT_START	// IN7 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x00, "1 Coin/10 Credits" )
	PORT_DIPNAME( 0x0c, 0x0c, "Min Credits To Start" )
	PORT_DIPSETTING(    0x0c, "1" )
//  PORT_DIPSETTING(    0x08, "2" ) // ? these don't let you start a game
//  PORT_DIPSETTING(    0x04, "3" )
//  PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x70, 0x70, "YAKUMAN Bonus" )
	PORT_DIPSETTING(    0x70, "Cut" )
	PORT_DIPSETTING(    0x60, "1 T" )
	PORT_DIPSETTING(    0x50, "300" )
	PORT_DIPSETTING(    0x40, "500" )
	PORT_DIPSETTING(    0x30, "700" )
	PORT_DIPSETTING(    0x20, "1000" )
	PORT_DIPSETTING(    0x10, "1000?" )
	PORT_DIPSETTING(    0x00, "1000?" )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN8 - DSW3
	PORT_DIPNAME( 0x03, 0x03, "YAKUMAN Times" )
//  PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x01, "3?" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x18, "Payout" )
	PORT_DIPSETTING(    0x18, "300" )
	PORT_DIPSETTING(    0x10, "500" )
	PORT_DIPSETTING(    0x08, "700" )
	PORT_DIPSETTING(    0x00, "1000" )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN9 - DSW4
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "In Game Music" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN10 - DSWs top bits
	PORT_DIPNAME( 0x03, 0x03, "Computer Strength?" )
	PORT_DIPSETTING(    0x03, "Weak" )
	PORT_DIPSETTING(    0x02, DEF_STR( Normal ))
	PORT_DIPSETTING(    0x01, "Strong" )
	PORT_DIPSETTING(    0x00, "Very Strong" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, "149" )
	PORT_DIPSETTING(    0x00, "212" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Credits Per Note" )
	PORT_DIPSETTING(    0x40, "5" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )
INPUT_PORTS_END

INPUT_PORTS_START( mjmyuniv )
	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_SERVICE3	)	// medal out
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, DEF_STR( Service_Mode ), KEYCODE_F2, IP_JOY_NONE )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_SERVICE1	)	// analyzer
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// data clear
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2	)	// note
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN1	)
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN	)

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 A",           KEYCODE_A,         IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 E",           KEYCODE_E,         IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 I",           KEYCODE_I,         IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 M",           KEYCODE_M,         IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Kan",         KEYCODE_LCONTROL,  IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )	// Start 1

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 B",           KEYCODE_B,         IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 F",           KEYCODE_F,         IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 J",           KEYCODE_J,         IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 N",           KEYCODE_N,         IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Reach",       KEYCODE_LSHIFT,    IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_7,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 C",           KEYCODE_C,         IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 G",           KEYCODE_G,         IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 K",           KEYCODE_K,         IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Chii",        KEYCODE_SPACE,     IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Ron",         KEYCODE_Z,         IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 D",           KEYCODE_D,         IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 H",           KEYCODE_H,         IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 L",           KEYCODE_L,         IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Pon",         KEYCODE_LALT,      IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 Last Chance", KEYCODE_RALT,      IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	PORT_START	// IN6 - DSW1
	PORT_DIPNAME( 0x0f, 0x07, "Pay Out Rate (%)" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPSETTING(    0x01, "53" )
	PORT_DIPSETTING(    0x02, "56" )
	PORT_DIPSETTING(    0x03, "59" )
	PORT_DIPSETTING(    0x04, "62" )
	PORT_DIPSETTING(    0x05, "65" )
	PORT_DIPSETTING(    0x06, "68" )
	PORT_DIPSETTING(    0x07, "71" )
	PORT_DIPSETTING(    0x08, "75" )
	PORT_DIPSETTING(    0x09, "78" )
	PORT_DIPSETTING(    0x0a, "81" )
	PORT_DIPSETTING(    0x0b, "84" )
	PORT_DIPSETTING(    0x0c, "87" )
	PORT_DIPSETTING(    0x0d, "90" )
	PORT_DIPSETTING(    0x0e, "93" )
	PORT_DIPSETTING(    0x0f, "96" )
	PORT_DIPNAME( 0x30, 0x30, "Odds Rate" )
	PORT_DIPSETTING(    0x20, "2 3 6 8 12 15 30 50" )
	PORT_DIPSETTING(    0x30, "1 2 4 8 12 16 24 32" )
	PORT_DIPSETTING(    0x00, "1 2 3 5 8 15 30 50" )
	PORT_DIPSETTING(    0x10, "1 2 3 5 10 25 50 100" )
	PORT_DIPNAME( 0xc0, 0xc0, "Max Bet" )
	PORT_DIPSETTING(    0xc0, "1" )
	PORT_DIPSETTING(    0x80, "5" )
	PORT_DIPSETTING(    0x40, "10" )
	PORT_DIPSETTING(    0x00, "20" )

	PORT_START	// IN7 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x00, "1 Coin/10 Credits" )
	PORT_DIPNAME( 0x0c, 0x0c, "Min Credits To Start" )
	PORT_DIPSETTING(    0x0c, "1" )
//  PORT_DIPSETTING(    0x08, "2" ) // ? these don't let you start a game
//  PORT_DIPSETTING(    0x04, "3" )
//  PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x70, 0x70, "YAKUMAN Bonus" )
	PORT_DIPSETTING(    0x70, "Cut" )
	PORT_DIPSETTING(    0x60, "1 T" )
	PORT_DIPSETTING(    0x50, "300" )
	PORT_DIPSETTING(    0x40, "500" )
	PORT_DIPSETTING(    0x30, "700" )
	PORT_DIPSETTING(    0x20, "1000" )
	PORT_DIPSETTING(    0x10, "1000?" )
	PORT_DIPSETTING(    0x00, "1000?" )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN8 - DSW3
	PORT_DIPNAME( 0x03, 0x03, "YAKUMAN Times" )
//  PORT_DIPSETTING(    0x00, "1" )
	PORT_DIPSETTING(    0x03, "1" )
	PORT_DIPSETTING(    0x02, "2" )
	PORT_DIPSETTING(    0x01, "3?" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x18, "Payout" )
	PORT_DIPSETTING(    0x18, "300" )
	PORT_DIPSETTING(    0x10, "500" )
	PORT_DIPSETTING(    0x08, "700" )
	PORT_DIPSETTING(    0x00, "1000" )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN9 - DSW4
	PORT_DIPNAME( 0x01, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "In Game Music" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN10 - DSWs top bits
	PORT_DIPNAME( 0x03, 0x03, "Computer Strength?" )
	PORT_DIPSETTING(    0x03, "Weak" )
	PORT_DIPSETTING(    0x02, DEF_STR( Normal ))
	PORT_DIPSETTING(    0x01, "Strong" )
	PORT_DIPSETTING(    0x00, "Very Strong" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, "149" )
	PORT_DIPSETTING(    0x00, "212" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Credits Per Note" )
	PORT_DIPSETTING(    0x40, "5" )
	PORT_DIPSETTING(    0x00, "10" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x80, DEF_STR( On ) )
INPUT_PORTS_END

INPUT_PORTS_START( akamaru )
	PORT_START	// IN0 - Player 1
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER1 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN1 - Player 2
	PORT_BIT(  0x01, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT(  0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 )
	PORT_BIT(  0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 )
	PORT_BIT(  0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 )
	PORT_BIT(  0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 )
	PORT_BIT(  0x20, IP_ACTIVE_LOW, IPT_BUTTON1        | IPF_PLAYER2 )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT(  0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2 - Coins + ?
	PORT_BIT(  0x01, IP_ACTIVE_LOW,  IPT_COIN1    )
	PORT_BIT(  0x02, IP_ACTIVE_LOW,  IPT_COIN2    )
	PORT_BIT(  0x04, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BITX( 0x08, IP_ACTIVE_LOW,  IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x10, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
	PORT_BIT( 0x20, IP_ACTIVE_LOW,  IPT_SPECIAL  )
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter irq flag
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter busy flag

	PORT_START	// IN3 - DSW1
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coin_A ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Coin_B ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 1-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )

	PORT_START	// IN4 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x03, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Hardest ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 2-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END

INPUT_PORTS_START( mjflove )
	PORT_START	// IN0 - Coins + Service Keys
	PORT_BIT( 0x01, IP_ACTIVE_LOW,  IPT_COIN1    )
	PORT_BIT( 0x02, IP_ACTIVE_LOW,  IPT_COIN2    )
	PORT_BIT( 0x04, IP_ACTIVE_LOW,  IPT_SERVICE1 )
	PORT_BITX(0x08, IP_ACTIVE_LOW,  IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x10, IP_ACTIVE_LOW,  IPT_UNKNOWN  )
	PORT_BIT( 0x20, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// RTC
	PORT_BIT( 0x40, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter irq flag
	PORT_BIT( 0x80, IP_ACTIVE_HIGH, IPT_SPECIAL  )	// blitter busy flag

	// keyb 1
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 A",           KEYCODE_A,         IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 E",           KEYCODE_E,         IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 I",           KEYCODE_I,         IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 M",           KEYCODE_M,         IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Kan",         KEYCODE_LCONTROL,  IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1 )	// Start 1

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 B",           KEYCODE_B,         IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 F",           KEYCODE_F,         IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 J",           KEYCODE_J,         IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 N",           KEYCODE_N,         IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Reach",       KEYCODE_LSHIFT,    IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Bet",         KEYCODE_7,         IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 C",           KEYCODE_C,         IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 G",           KEYCODE_G,         IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 K",           KEYCODE_K,         IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Chii",        KEYCODE_SPACE,     IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Ron",         KEYCODE_Z,         IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 D",           KEYCODE_D,         IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 H",           KEYCODE_H,         IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 L",           KEYCODE_L,         IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Pon",         KEYCODE_LALT,      IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P1 Last Chance", KEYCODE_RALT,      IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P1 Take Score",  KEYCODE_RCONTROL,  IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P1 Double Up",   KEYCODE_RSHIFT,    IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Flip",        KEYCODE_X,         IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P1 Big",         KEYCODE_ENTER,     IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P1 Small",       KEYCODE_BACKSPACE, IP_JOY_NONE )	// "s"

	// keyb 2
	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 A",           IP_KEY_DEFAULT, IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 E",           IP_KEY_DEFAULT, IP_JOY_NONE )	// E
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 I",           IP_KEY_DEFAULT, IP_JOY_NONE )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 M",           IP_KEY_DEFAULT, IP_JOY_NONE )	// M
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Kan",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )	// Start 2

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 B",           IP_KEY_DEFAULT, IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 F",           IP_KEY_DEFAULT, IP_JOY_NONE )	// F
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 J",           IP_KEY_DEFAULT, IP_JOY_NONE )	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 N",           IP_KEY_DEFAULT, IP_JOY_NONE )	// N
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Reach",       IP_KEY_DEFAULT, IP_JOY_NONE )	// Reach
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Bet",         KEYCODE_8,      IP_JOY_NONE )	// BET

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 C",           IP_KEY_DEFAULT, IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 G",           IP_KEY_DEFAULT, IP_JOY_NONE )	// G
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 K",           IP_KEY_DEFAULT, IP_JOY_NONE )	// K
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Chii",        IP_KEY_DEFAULT, IP_JOY_NONE )	// Chi
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Ron",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 D",           IP_KEY_DEFAULT, IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 H",           IP_KEY_DEFAULT, IP_JOY_NONE )	// H
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 L",           IP_KEY_DEFAULT, IP_JOY_NONE )	// L
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Pon",         IP_KEY_DEFAULT, IP_JOY_NONE )	// Pon
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN			)	// nothing

	PORT_START
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 Last Chance", IP_KEY_DEFAULT, IP_JOY_NONE )	// "l"
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 Take Score",  IP_KEY_DEFAULT, IP_JOY_NONE )	// "t"
	PORT_BITX(0x04, IP_ACTIVE_LOW, 0, "P2 Double Up",   IP_KEY_DEFAULT, IP_JOY_NONE )	// "w"
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 Flip Flop",   IP_KEY_DEFAULT, IP_JOY_NONE )	// Flip Flop
	PORT_BITX(0x10, IP_ACTIVE_LOW, 0, "P2 Big",         IP_KEY_DEFAULT, IP_JOY_NONE )	// "b"
	PORT_BITX(0x20, IP_ACTIVE_LOW, 0, "P2 Small",       IP_KEY_DEFAULT, IP_JOY_NONE )	// "s"

	PORT_START	// IN11 - DSW1
	PORT_DIPNAME( 0x01, 0x01, "Unknown 2-0" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "Unknown 2-1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 2-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN12 - DSW2
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x1c, 0x1c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x18, "0" )
	PORT_DIPSETTING(    0x14, "1" )
	PORT_DIPSETTING(    0x10, "2" )
	PORT_DIPSETTING(    0x1c, "3" )
	PORT_DIPSETTING(    0x0c, "4" )
	PORT_DIPSETTING(    0x08, "5" )
	PORT_DIPSETTING(    0x04, "6" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x20, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 1-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x80, IP_ACTIVE_LOW )
INPUT_PORTS_END

INPUT_PORTS_START( hparadis )
	PORT_START	// IN0 - Player 1
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_UP )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN1 - Player 2
	PORT_BIT( 0x3f, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	// IN2
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1    )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2    )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BITX(0x08, IP_ACTIVE_LOW,  IPT_SERVICE, DEF_STR( Test ), KEYCODE_F1, IP_JOY_NONE )	/* Test */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_SERVICE2	)	// analyzer
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_SERVICE3	)	// data clear
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_UNKNOWN  )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN  )

	// keyb 1
	PORT_START	// IN3
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "1",              KEYCODE_1_PAD,     IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "5",              KEYCODE_5_PAD,     IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 Yes",         KEYCODE_ENTER_PAD, IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START1		)

	PORT_START	// IN4
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "2",              KEYCODE_2_PAD,     IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "6",              KEYCODE_6_PAD,     IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P1 No",          KEYCODE_DEL_PAD,   IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Reach
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// BET

	PORT_START	// IN5
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "3",              KEYCODE_3_PAD,     IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "7",              KEYCODE_7_PAD,     IP_JOY_NONE )	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// K
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START	// IN6
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "4",              KEYCODE_4_PAD,     IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "8",              KEYCODE_8_PAD,     IP_JOY_NONE )	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// L
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START	// IN7
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "t"
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "w"
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// Flip Flop
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "b"
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "s"

	// keyb 2
	PORT_START	// IN8
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 A",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// A
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 E",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// E
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )	// I
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 M",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// M
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Kan
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2		)

	PORT_START	// IN9
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 B",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// B
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 F",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// F
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// J
	PORT_BITX(0x08, IP_ACTIVE_LOW, 0, "P2 N",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// N
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Reach
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// BET

	PORT_START	// IN10
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 C",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// C
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 G",          IP_KEY_DEFAULT,     IP_JOY_NONE )	// G
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// K
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Chi
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// Ron
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START	// IN11
	PORT_BITX(0x01, IP_ACTIVE_LOW, 0, "P2 D",           IP_KEY_DEFAULT,    IP_JOY_NONE )	// D
	PORT_BITX(0x02, IP_ACTIVE_LOW, 0, "P2 H",           IP_KEY_DEFAULT,    IP_JOY_NONE )	// H
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// L
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN		)	// PON
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN		)
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN		)

	PORT_START	// IN12
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN	)
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "t"
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "w"
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// Flip Flop
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "b"
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN	)	// "s"

	PORT_START	// IN13 - DSW1
	PORT_DIPNAME( 0x03, 0x03, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 1C_2C ) )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Hardest ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x60, "Control Panel" )
	PORT_DIPSETTING(    0x60, "Hanafuda" )
	PORT_DIPSETTING(    0x40, "Mahjong" )
	PORT_DIPSETTING(    0x20, DEF_STR( Joystick ) )
//  PORT_DIPSETTING(    0x00, DEF_STR( Joystick ) )
	PORT_DIPNAME( 0x80, 0x80, "First Chance" )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Yes ) )

	PORT_START	// IN14 - DSW2
	PORT_DIPNAME( 0x03, 0x03, "Unknown 2-0&1" )
	PORT_DIPSETTING(    0x03, "0" )
	PORT_DIPSETTING(    0x02, "1" )
	PORT_DIPSETTING(    0x01, "2" )
	PORT_DIPSETTING(    0x00, "0?" )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "Unknown 2-4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "Unknown 2-5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "Unknown 2-6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, "Unknown 2-7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	// IN15 - DSWs top bits
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x02, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x04, 0x04, "Unknown 2-8" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Unknown 2-9" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
INPUT_PORTS_END


/***************************************************************************
                            Don Den Lover Vol.1
***************************************************************************/

static struct YM2413interface ym2413_interface =
{
	1,
	3579545,	/* ???? */
	{ YM2413_VOL(80,MIXER_PAN_CENTER,80,MIXER_PAN_CENTER) }
};

static struct AY8910interface ay8910_interface_1 =
{
	1,			/* 1 chip */
	1789772,	/* ??? */
	{ 30 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 }
};

static struct AY8910interface ay8910_interface_2 =
{
	1,			/* 1 chip */
	3579545,	/* ??? */
	{ 30 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ 0 }
};

static struct AY8910interface quiz365_ay8910_interface =
{
	1,			/* 1 chip */
	1789772,	/* ??? */
	{ 30 },
	{ quiz365_input_r },
	{ 0 },
	{ 0 },
	{ dynax_select_w }
};

static struct AY8910interface mjmyster_ay8910_interface =
{
	1,			/* 1 chip */
	3579545,	/* ??? */
	{ 30 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ dynax_select_w }
};

static struct AY8910interface mjmyster_ay8910_interface_2 =
{
	1,			/* 1 chip */
	1789772,	/* ??? */
	{ 30 },
	{ 0 },
	{ 0 },
	{ 0 },
	{ dynax_select_w }
};

static struct AY8910interface hginga_ay8910_interface =
{
	1,			/* 1 chip */
	3579545,	/* ??? */
	{ 30 },
	{ hginga_dsw_r },
	{ 0 },
	{ 0 },
	{ dynax_select_w }
};

static struct OKIM6295interface okim6295_interface =
{
	1,
	{ 8000 },	/* ? */
	{ REGION_SOUND1 },
	{ 80 }
};

static MACHINE_DRIVER_START( ddenlovr )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main",M68000,24000000 / 2)
	MDRV_CPU_MEMORY(ddenlovr_readmem,ddenlovr_writemem)
	MDRV_CPU_VBLANK_INT(irq1_line_hold,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(336, 256)
	MDRV_VISIBLE_AREA(0, 336-1, 5, 256-16+5-1)
	MDRV_PALETTE_LENGTH(0x100)

	MDRV_VIDEO_START(ddenlovr)
	MDRV_VIDEO_EOF(ddenlovr)
	MDRV_VIDEO_UPDATE(ddenlovr)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2413, ym2413_interface)
	MDRV_SOUND_ADD_TAG("ay8910", AY8910, ay8910_interface_1)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( ddenlvrj )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ddenlovr)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(ddenlvrj_readmem,ddenlvrj_writemem)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( akamaru )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ddenlovr)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(akamaru_readmem,akamaru_writemem)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( quiz365 )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ddenlovr)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(quiz365_readmem,quiz365_writemem)

	MDRV_SOUND_REPLACE("ay8910", AY8910, quiz365_ay8910_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( nettoqc )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(ddenlovr)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(nettoqc_readmem,nettoqc_writemem)
MACHINE_DRIVER_END

/***************************************************************************
                                Rong Rong
***************************************************************************/

/* the CPU is in Interrupt Mode 2
   vector can be 0xee, 0xf8 0xfa 0xfc
   rongrong: 0xf8 and 0xfa do nothing
   quizchq: 0xf8 and 0xfa are very similar, they should be triggered by the blitter
   0xee is vblank
   0xfc is from the 6242RTC
 */
static INTERRUPT_GEN( quizchq_irq )
{
	static int count;

	/* I haven't found a irq ack register, so I need this kludge to
       make sure I don't lose any interrupt generated by the blitter,
       otherwise quizchq would lock up. */
	if (cpunum_get_reg(0,Z80_IRQ_STATE))
		return;

	if ((++count % 60) == 0)
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xfc);
	else
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xee);
}

/*
static INTERRUPT_GEN( rtc_irq )
{
    cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xfc);
}
*/

static MACHINE_DRIVER_START( quizchq )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", Z80, 8000000)	/* ? */
	MDRV_CPU_MEMORY(quizchq_readmem,quizchq_writemem)
	MDRV_CPU_PORTS(quizchq_readport,quizchq_writeport)
	MDRV_CPU_VBLANK_INT(quizchq_irq,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(336, 256)
	MDRV_VISIBLE_AREA(0, 336-1, 5, 256-16+5-1)
	MDRV_PALETTE_LENGTH(0x100)
	MDRV_COLORTABLE_LENGTH(0x100)

	MDRV_VIDEO_START(ddenlovr)
	MDRV_VIDEO_EOF(ddenlovr)
	MDRV_VIDEO_UPDATE(ddenlovr)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2413, ym2413_interface)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( rongrong )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(quizchq)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(rongrong_readmem,rongrong_writemem)
	MDRV_CPU_PORTS(rongrong_readport,rongrong_writeport)
MACHINE_DRIVER_END

/***************************************************************************

    Monkey Mole Panic

***************************************************************************/

/*  the CPU is in Interrupt Mode 0:

    RST 08 is vblank
    RST 18 is from the blitter
    RST 20 is from the 6242RTC
 */
static INTERRUPT_GEN( mmpanic_irq )
{
	static int count;

	/* I haven't found a irq ack register, so I need this kludge to
       make sure I don't lose any interrupt generated by the blitter,
       otherwise the game would lock up. */
	if (cpunum_get_reg(0,Z80_IRQ_STATE))
		return;

	if ((++count % 60) == 0)
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xe7);	// RST 20, clock
	else
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xcf);	// RST 08, vblank
}

static MACHINE_DRIVER_START( mmpanic )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", Z80, 8000000)
	MDRV_CPU_MEMORY(mmpanic_readmem,mmpanic_writemem)
	MDRV_CPU_PORTS(mmpanic_readport,mmpanic_writeport)
	MDRV_CPU_VBLANK_INT(mmpanic_irq,1)

	MDRV_CPU_ADD_TAG("sound", Z80, 3579545)
	MDRV_CPU_MEMORY(mmpanic_sound_readmem,mmpanic_sound_writemem)
	MDRV_CPU_PORTS(mmpanic_sound_readport,mmpanic_sound_writeport)
	MDRV_CPU_VBLANK_INT(irq0_line_hold,1)	// NMI by main cpu

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(336, 256)
	MDRV_VISIBLE_AREA(0, 336-1, 5, 256-16+5-1)
	MDRV_PALETTE_LENGTH(0x100)
	MDRV_COLORTABLE_LENGTH(0x100)

	MDRV_VIDEO_START(mmpanic)	// extra layers
	MDRV_VIDEO_EOF(ddenlovr)
	MDRV_VIDEO_UPDATE(ddenlovr)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2413, ym2413_interface)
	MDRV_SOUND_ADD(AY8910, ay8910_interface_2)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END


/***************************************************************************

    Hana Kanzashi

***************************************************************************/

/*  the CPU is in Interrupt Mode 2
    vector can be 0xe0, 0xe2
    0xe0 is vblank
    0xe2 is from the 6242RTC
 */
static INTERRUPT_GEN( hanakanz_irq )
{
	static int count;

	/* I haven't found a irq ack register, so I need this kludge to
       make sure I don't lose any interrupt generated by the blitter,
       otherwise quizchq would lock up. */
	if (cpunum_get_reg(0,Z80_IRQ_STATE))
		return;

	if ((++count % 60) == 0)
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xe2);
	else
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xe0);
}

static MACHINE_DRIVER_START( hanakanz )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main",Z80,8000000)	// TMPZ84C015BF-8
	MDRV_CPU_MEMORY(hanakanz_readmem,hanakanz_writemem)
	MDRV_CPU_PORTS(hanakanz_readport,hanakanz_writeport)
	MDRV_CPU_VBLANK_INT(hanakanz_irq,1)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_60HZ_VBLANK_DURATION)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(336, 256)
	MDRV_VISIBLE_AREA(0, 336-1, 5, 256-11-1)
	MDRV_PALETTE_LENGTH(0x200)

	MDRV_VIDEO_START(hanakanz)	// blitter commands in the roms are shuffled around
	MDRV_VIDEO_EOF(ddenlovr)
	MDRV_VIDEO_UPDATE(ddenlovr)

	/* sound hardware */
	MDRV_SOUND_ADD(YM2413, ym2413_interface)
	MDRV_SOUND_ADD(OKIM6295, okim6295_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( hkagerou )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( hanakanz )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PORTS(hkagerou_readport,hkagerou_writeport)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( mjreach1 )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( hanakanz )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PORTS(mjreach1_readport,mjreach1_writeport)
MACHINE_DRIVER_END


/***************************************************************************
     Mahjong Chuukanejyo
***************************************************************************/

/*  the CPU is in Interrupt Mode 2
    vector can be 0xf8, 0xfa
    0xf8 is vblank
    0xfa is from the 6242RTC
 */
static INTERRUPT_GEN( mjchuuka_irq )
{
	static int count;

	/* I haven't found a irq ack register, so I need this kludge to
       make sure I don't lose any interrupt generated by the blitter,
       otherwise quizchq would lock up. */
	if (cpunum_get_reg(0,Z80_IRQ_STATE))
		return;

	if ((++count % 60) == 0)
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xfa);
	else
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xf8);
}


static MACHINE_DRIVER_START( mjchuuka )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( hanakanz )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PORTS(mjchuuka_readport,mjchuuka_writeport)
	MDRV_CPU_VBLANK_INT(mjchuuka_irq,1)

	MDRV_SOUND_ADD(AY8910, ay8910_interface_1)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( funkyfig )

	MDRV_IMPORT_FROM(mmpanic)
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(funkyfig_readmem,funkyfig_writemem)
	MDRV_CPU_PORTS(funkyfig_readport,funkyfig_writeport)
	MDRV_CPU_VBLANK_INT(mjchuuka_irq,1)

	MDRV_CPU_MODIFY("sound")
	MDRV_CPU_PORTS(funkyfig_sound_readport,mmpanic_sound_writeport)

	MDRV_VIDEO_START(ddenlovr)	// no extra layers?
MACHINE_DRIVER_END


/***************************************************************************
                        Mahjong The Mysterious World
***************************************************************************/

/*  the CPU is in Interrupt Mode 2
    vector can be 0xf8, 0xfa, 0xfc
    0xf8 is vblank
    0xfa and/or 0xfc are from the blitter (almost identical)
    NMI triggered by the RTC

    To do:

    The game randomly locks up (like quizchq?) because of some lost
    blitter interrupt I guess (nested blitter irqs?). Hence the hack
    to trigger the blitter irq every frame.
 */
static INTERRUPT_GEN( mjmyster_irq )
{
	/* I haven't found a irq ack register, so I need this kludge to
       make sure I don't lose any interrupt generated by the blitter,
       otherwise quizchq would lock up. */
	if (cpunum_get_reg(0,Z80_IRQ_STATE))
		return;

	switch( cpu_getiloops() )
	{
		case 0:	cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xf8);	break;
		case 1:	cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xfa);	break;
	}
}

static INTERRUPT_GEN( rtc_nmi_irq )
{
	cpu_set_irq_line(0, IRQ_LINE_NMI, PULSE_LINE);
}

static MACHINE_DRIVER_START( mjmyster )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( quizchq )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(mjmyster_readmem,mjmyster_writemem)
	MDRV_CPU_PORTS(mjmyster_readport,mjmyster_writeport)
	MDRV_CPU_VBLANK_INT(mjmyster_irq, 2)
	MDRV_CPU_PERIODIC_INT(rtc_nmi_irq, 1)

	MDRV_SOUND_ADD(AY8910, mjmyster_ay8910_interface)
MACHINE_DRIVER_END

/***************************************************************************
                            Hanafuda Hana Ginga
***************************************************************************/

/*  the CPU is in Interrupt Mode 2
    vector can be 0xf8, 0xfa, 0xfc, 0xee
    0xf8 is vblank
    0xfa and/or 0xfc are from the blitter (almost identical)
    0xee triggered by the RTC
 */
static INTERRUPT_GEN( hginga_irq )
{
	static int count;

	/* I haven't found a irq ack register, so I need this kludge to
       make sure I don't lose any interrupt generated by the blitter,
       otherwise hginga would lock up. */
	if (cpunum_get_reg(0,Z80_IRQ_STATE))
		return;

	if ((++count % 60) == 0)
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xee);
	else
		cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xf8);
}

static MACHINE_DRIVER_START( hginga )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( quizchq )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(hginga_readmem,hginga_writemem)
	MDRV_CPU_PORTS(hginga_readport,hginga_writeport)
	MDRV_CPU_VBLANK_INT(hginga_irq, 1)

	MDRV_SOUND_ADD(AY8910, hginga_ay8910_interface)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( hgokou )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( quizchq )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(hgokou_readmem,hgokou_writemem)
	MDRV_CPU_PORTS(hgokou_readport,hgokou_writeport)
	MDRV_CPU_VBLANK_INT(hginga_irq, 1)

	MDRV_SOUND_ADD(AY8910, ay8910_interface_1)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( mjmywrld )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( mjmyster )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(hginga_readmem,hginga_writemem)
	MDRV_CPU_PORTS(mjmywrld_readport,mjmywrld_writeport)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( mjmyuniv )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( quizchq )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(mjmyster_readmem,mjmyster_writemem)
	MDRV_CPU_PORTS(mjmyster_readport,mjmyster_writeport)
	MDRV_CPU_VBLANK_INT(mjmyster_irq, 2)
	MDRV_CPU_PERIODIC_INT(rtc_nmi_irq, 1)

	MDRV_SOUND_ADD(AY8910, mjmyster_ay8910_interface_2)
MACHINE_DRIVER_END

static MACHINE_DRIVER_START( mjmyornt )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( quizchq )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_PORTS(mjmyster_readport,mjmyster_writeport)
	MDRV_CPU_VBLANK_INT(mjmyster_irq, 2)
	MDRV_CPU_PERIODIC_INT(rtc_nmi_irq, 1)

	MDRV_SOUND_ADD(AY8910, mjmyster_ay8910_interface_2)
MACHINE_DRIVER_END

static INTERRUPT_GEN( mjflove_irq )
{
	mjflove_irq_cause = (1 << 5) | (1 << 6);

	switch (cpu_getiloops())
	{
		case 0:
			mjflove_irq_cause &= 1 << 5;
			cpu_set_irq_line(0, 0, HOLD_LINE);
			break;
		case 1:
			mjflove_irq_cause &= 1 << 6;
			cpu_set_irq_line(0, 0, HOLD_LINE);
			break;
	}
}

static MACHINE_DRIVER_START( mjflove )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( quizchq )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(rongrong_readmem,rongrong_writemem)
	MDRV_CPU_PORTS(mjflove_readport,mjflove_writeport)
	MDRV_CPU_VBLANK_INT(mjflove_irq, 2)

	MDRV_VIDEO_START(mjflove)	// blitter commands in the roms are shuffled around

	MDRV_SOUND_ADD(AY8910, ay8910_interface_1)
MACHINE_DRIVER_END

/*  It runs in IM 2, thus needs a vector on the data bus:
    0xee is vblank  */
INTERRUPT_GEN( hparadis_irq )
{
	cpu_set_irq_line_and_vector(0, 0, HOLD_LINE, 0xee);
}

static MACHINE_DRIVER_START( hparadis )
	/* basic machine hardware */
	MDRV_IMPORT_FROM( quizchq )
	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(hparadis_readmem,hparadis_writemem)
	MDRV_CPU_PORTS(hparadis_readport,hparadis_writeport)
	MDRV_CPU_VBLANK_INT(hparadis_irq, 1)
MACHINE_DRIVER_END


/***************************************************************************

Monkey Mole Panic
Nakanihon/Taito 1992
                      7001A  5563    6242
                      6295   7002
                             Z80
     8910                   5563   16MHz
     DynaX NL-001           7003              14.318MHz
                            Z80               24 MHz
          2018
                  DynaX   524256  524256       DynaX
                  1108    524256  524256       1427
                  DynaX   524256  524256       DynaX
                  1108    524256  524256       1427

     8251                      7006    7005   7004


The game asks players to slap buttons on a control panel and see mole-like creatures
get crunched on the eye-level video screen.

An on-screen test mode means the ticket dispenser can be adjusted from 1-99 tickets
and 15 score possibilities.

It also checks PCB EPROMs, switches and lamps, and the built-in income analyzer.

There are six levels of difficulty for one or two players.

The games are linkable (up to four) for competitive play.

***************************************************************************/

ROM_START( mmpanic )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "nwc7002a",     0x00000, 0x40000, 0x725b337f )
	ROM_RELOAD(               0x10000, 0x40000 )

	ROM_REGION( 0x20000, REGION_CPU2, 0 )	/* Z80 Code */
	ROM_LOAD( "nwc7003",      0x00000, 0x20000, 0x4f02ce44 )	// 1xxxxxxxxxxxxxxxx = 0xFF

	ROM_REGION( 0x280000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "nwc7004",      0x000000, 0x100000, 0x5b4ad8c5 )
	ROM_LOAD( "nwc7005",      0x100000, 0x100000, 0x9ec41956 )
	ROM_LOAD( "nwc7006a",     0x200000, 0x080000, 0x9099c571 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "nwc7001a",     0x00000, 0x40000, 0x1ae3660d )
ROM_END


/***************************************************************************

Z80 x2
OSC 24MHz, 14.31818MHz, 16MHz
Oki M6295 + YM2413 + AY-3-8910
Oki 6242 + 3.6v battery + 32.768kHz (rtc)
Toshiba TMP82C51 (USART)
Custom ICs NL-004 (x2), NL-003 (x2), NL-001
RAM 8kx8 near 7502S
RAM 8kx8 near 7503S
RAM 2kx8 near NL-001
RAM 32kx8 (x8) near NL-003 & NL-003
DIPs 10-position (x2)
PAL near 7504
PAL near 7501S

probably 7501S is damaged, I can not get a consistent read. 10 reads supplied for comparison.

***************************************************************************/

ROM_START( animaljr )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "7502s.2e",     0x00000, 0x40000, 0x4b14a4be )
	ROM_RELOAD(               0x10000, 0x40000 )

	ROM_REGION( 0x20000, REGION_CPU2, 0 )	/* Z80 Code */
	ROM_LOAD( "7503s.8e",     0x00000, 0x20000, 0xd1fac899 )	// 1xxxxxxxxxxxxxxxx = 0xFF

	ROM_REGION( 0x200000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "7504.17b",     0x000000, 0x100000, 0xb62de6a3 )
	ROM_LOAD( "7505.17d",     0x100000, 0x080000, 0x729b073f )
	ROM_LOAD( "7506s.17f",    0x180000, 0x080000, 0x1be1ae17 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "7501s_0.1h",   0x00000, 0x40000, 0x59debb66 )
ROM_END


/***************************************************************************

Quiz Channel Question (JPN ver.)
(c)1993 Nakanihon

N7311208L1-2
N73SUB

CPU:    TMPZ84C015BF-8

Sound:  YM2413
        M6295

OSC:    16MHz
    28.6363MHz
    32.768KHz ?

Custom: NL-002 - Nakanihon
    (1108F0405) - Dynax
    (1427F0071) - Dynax

Others: M6242B (RTC?)

***************************************************************************/

ROM_START( quizchq )
	ROM_REGION( 0x118000, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "nwc7302.3e",   0x00000, 0x80000, 0x14217f2d )
	ROM_RELOAD(               0x10000, 0x80000 )
	ROM_LOAD( "nwc7303.4e",   0x90000, 0x80000, 0xffc77601 )

	ROM_REGION( 0x320000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "nwc7307.s4b",  0x000000, 0x80000, 0xa09d1dbe )
	ROM_LOAD( "nwc7306.s3b",  0x080000, 0x80000, 0x52d27aac )
	ROM_LOAD( "nwc7305.s2b",  0x100000, 0x80000, 0x5f50914e )
	ROM_LOAD( "nwc7304.s1b",  0x180000, 0x80000, 0x72866919 )
	ROM_LOAD( "nwc7310.s4a",  0x200000, 0x80000, 0x5939aeab )
	ROM_LOAD( "nwc7309.s3a",  0x280000, 0x80000, 0x88c863b2 )
	ROM_LOAD( "nwc7308.s2a",  0x300000, 0x20000, 0x6eb5c81d )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "nwc7301.1f",   0x00000, 0x80000, 0x52c672e8 )	// 2 banks
ROM_END

ROM_START( quizchql )
	ROM_REGION( 0x118000, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "2.rom",        0x00000, 0x80000, 0x1bf8fb25 )
	ROM_RELOAD(               0x10000, 0x80000 )
	ROM_LOAD( "3.rom",        0x90000, 0x80000, 0x6028198f )

	ROM_REGION( 0x420000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "4.rom",        0x000000, 0x80000, 0xe6bdea31 )
	ROM_LOAD( "5.rom",        0x080000, 0x80000, 0xc243f10a )
	ROM_LOAD( "11.rom",       0x100000, 0x80000, 0xc9ae5880 )
	ROM_LOAD( "7.rom",        0x180000, 0x80000, 0xa490aa4e )
	ROM_LOAD( "6.rom",        0x200000, 0x80000, 0xfbf713b6 )
	ROM_LOAD( "8.rom",        0x280000, 0x80000, 0x68d4b79f )
	ROM_LOAD( "10.rom",       0x300000, 0x80000, 0xd56eaf0e )
	ROM_LOAD( "9.rom",        0x380000, 0x80000, 0xa11d535a )
	ROM_LOAD( "12.rom",       0x400000, 0x20000, 0x43f8e5c7 )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "1snd.rom",     0x00000, 0x80000, 0xcebb9220 )	// 2 banks
ROM_END



ROM_START( quiz365 )
	ROM_REGION( 0x180000, REGION_CPU1, 0 ) /* 68000 Code */
	ROM_LOAD16_BYTE( "7805.4b",  0x000000, 0x080000, 0x70f93543 )
	ROM_LOAD16_BYTE( "7804.4d",  0x000001, 0x080000, 0x2ae003f4 )
	ROM_LOAD16_BYTE( "7803.3b",  0x100000, 0x040000, 0x10d315b1 )
	ROM_LOAD16_BYTE( "7802.3d",  0x100001, 0x040000, 0x6616caa3 )

	ROM_REGION( 0x380000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "7810.14b", 0x000000, 0x100000, 0x4b1a4984 )
	ROM_LOAD( "7809.13b", 0x100000, 0x100000, 0x139d52ab )
	ROM_LOAD( "7808.12b", 0x200000, 0x080000, 0xa09fd4a4 )
	ROM_LOAD( "7807.11b", 0x280000, 0x080000, 0x988b3e84 )
	ROM_LOAD( "7806.10b", 0x300000, 0x080000, 0x7f9aa228 )

	ROM_REGION( 0x100000, REGION_SOUND1, 0 )	/* Samples */
	// piggy-backed sample roms dumped as 2 separate files
	ROM_LOAD( "7801.1fu",     0x000000, 0x080000, 0x53519d67 )
	ROM_LOAD( "7801.1fd",     0x080000, 0x080000, 0x448c58dd )
ROM_END

ROM_START( quiz365t )
	ROM_REGION( 0x180000, REGION_CPU1, 0 ) /* 68000 Code */
	ROM_LOAD16_BYTE( "7805.rom", 0x000000, 0x080000, 0x6db33222 )
	ROM_LOAD16_BYTE( "7804.rom", 0x000001, 0x080000, 0x46d04ace )
	ROM_LOAD16_BYTE( "7803.rom", 0x100000, 0x040000, 0x5b7a78d3 )
	ROM_LOAD16_BYTE( "7802.rom", 0x100001, 0x040000, 0xc3238a9d )

	ROM_REGION( 0x400000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "7810.rom", 0x000000, 0x100000, 0x4b1a4984 )
	ROM_LOAD( "7809.rom", 0x100000, 0x100000, 0x139d52ab )
	ROM_LOAD( "7808.rom", 0x200000, 0x080000, 0xa09fd4a4 )
	ROM_LOAD( "7806.rom", 0x280000, 0x100000, 0x75767c6f )
	ROM_LOAD( "7807.rom", 0x380000, 0x080000, 0x60fb1dfe )

	ROM_REGION( 0x100000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "7801.rom", 0x080000, 0x080000, 0x285cc62a )	// bank 2,3
	ROM_CONTINUE(         0x000000, 0x080000 )				// bank 0,1
ROM_END



/***************************************************************************

                                Rong Rong

Here are the proms for Nakanihon's Rong Rong
It's a quite nice Puzzle game.
The CPU don't have any numbers on it except for this:
Nakanihon
NL-002
3J3  JAPAN
For the sound it uses A YM2413

***************************************************************************/

ROM_START( rongrong )
	ROM_REGION( 0x118000, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "8002e.3e",     0x00000, 0x80000, 0x062fa1b6 )
	ROM_RELOAD(               0x10000, 0x80000 )
	/* 90000-10ffff empty */

	ROM_REGION( 0x280000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "8003.8c",      0x000000, 0x80000, 0xf57192e5 )
	ROM_LOAD( "8004.9c",      0x080000, 0x80000, 0xc8c0b5cb )
	ROM_LOAD( "8005e.10c",    0x100000, 0x80000, 0x11c7a23c )
	ROM_LOAD( "8006e.11c",    0x180000, 0x80000, 0x137e9b83 )
	ROM_LOAD( "8007e.12c",    0x200000, 0x80000, 0x374a1d50 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "8001w.2f",     0x00000, 0x40000, 0x8edc87a2 )

	ROM_REGION( 0x0200, REGION_USER1, 0 )
	ROM_LOAD( "gal16v8b.1e",   0x0000, 0x0117, 0xcf1b6e1d )
ROM_END

/***************************************************************************

 Rong Rong (Europe)
 Nakanihon

 Revision: 6.2.14

 CPU:
 1x Nakanihon NL-002-3D1 (main)
 1x oscillator 28.6363MHz
 1x Toshiba TMPZ84C015BF-8-9328ECZ
 1x OKI M6295-3372202 (sound)
 1x YM2413-9344HAAG (sound)
 1x Tpshiba TD62003AP-9348K
 1x DYNAX 1108F405-9401EAI
 1x DYNAX 4L02F2637-9337EAI
 1x DYNAX 1427F0071-9232EAI

 ROMs:
 1x M27C2001-12FI (8001W)(sound)
 1x M27C4001-12FI (8002E)(sound)
 2x TC534000AP (8003-8004)
 2x M27C4001-12FI (8005E-8007E)
 1x TMS 27C040-15 (8006E)
 1x GAL16V8B-25LP

***************************************************************************/

ROM_START( rongrngg )
	ROM_REGION( 0x118000, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "rr_8002g.rom", 0x00000, 0x80000, 0x9a5d2885 )
	ROM_RELOAD(               0x10000, 0x80000 )
	/* 90000-10ffff empty */

	ROM_REGION( 0x280000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "rr_8003.rom",  0x000000, 0x80000, 0xf57192e5 )
	ROM_LOAD( "rr_8004.rom",  0x080000, 0x80000, 0xc8c0b5cb )
	ROM_LOAD( "rr_8005g.rom", 0x100000, 0x80000, 0x11c7a23c )
	ROM_LOAD( "rr_8006g.rom", 0x180000, 0x80000, 0xf3de77e6 )
	ROM_LOAD( "rr_8007g.rom", 0x200000, 0x80000, 0x38a8caa3 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "rr_8001w.rom", 0x00000, 0x40000, 0x8edc87a2 )
ROM_END

/***************************************************************************

Rong Rong (Japan)
(c)1994 Nakanihon (Dynax)

N8010178L1

CPU   : TMPZ84C015BF-8
Sound : YM2413, M6295
OSC   : 28.6363MHz, ?(near CPU)
DIPs  : 10 position (x2)
Custom: NL-002
        1427F0071 (10D)
        4L02F2637 (10F)
        1108F0405 (10H)

ROMs  : 8001.2F      [9fc8a367] - Samples

        8002.3E      [27142684] - Main program

        8003.8C      [f57192e5] \
        8004.9C      [c8c0b5cb] |
        8005.10C     [d1e5f74c] |- Blitter data
        8006.11C     [bcbd1b0b] |
        8007.12C     [c76cbb69] /

***************************************************************************/

ROM_START( rongrngj )
	ROM_REGION( 0x118000, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "8002.3e", 0x00000, 0x80000, 0x27142684 )
	ROM_RELOAD(          0x10000, 0x80000 )
	/* 90000-10ffff empty */

	ROM_REGION( 0x240000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "8003.8c",  0x000000, 0x80000, 0xf57192e5 )
	ROM_LOAD( "8004.9c",  0x080000, 0x80000, 0xc8c0b5cb )
	ROM_LOAD( "8005.10c", 0x100000, 0x80000, 0xd1e5f74c )
	ROM_LOAD( "8006.11c", 0x180000, 0x80000, 0xbcbd1b0b )
	ROM_LOAD( "8007.12c", 0x200000, 0x40000, 0xc76cbb69 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "8001.2f",  0x00000, 0x40000, 0x9fc8a367 )
ROM_END

/***************************************************************************

Netto Quiz Champion (c) Nakanihon

CPU: 68HC000
Sound: OKI6295
Other: HN46505, unknown 68 pin, unknown 100 pin (x2), unknown 64 pin (part numbers scratched off).
PLDs: GAL16L8B (x2, protected)
RAM: TC524258BZ-10 (x5), TC55257BSPL-10 (x2), TC5588P-35
XTAL1: 16 MHz
XTAL2: 28.63636 MHz

***************************************************************************/

ROM_START( nettoqc )
	ROM_REGION( 0x180000, REGION_CPU1, 0 )		/* 68000 Code */
	ROM_LOAD16_BYTE( "10305.rom", 0x000000, 0x080000, 0xebb14a1f )
	ROM_LOAD16_BYTE( "10303.rom", 0x000001, 0x080000, 0x30c114c3 )
	ROM_LOAD16_BYTE( "10306.rom", 0x100000, 0x040000, 0xf19fe827 )
	ROM_LOAD16_BYTE( "10304.rom", 0x100001, 0x040000, 0xda1f56e5 )
	ROM_CONTINUE(                 0x100001, 0x040000 )	// 1ST AND 2ND HALF IDENTICAL

	ROM_REGION( 0x400000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "10307.rom", 0x000000, 0x100000, 0xc7a3b05f )
	ROM_LOAD( "10308.rom", 0x100000, 0x100000, 0x416807a1 )
	ROM_LOAD( "10309.rom", 0x200000, 0x100000, 0x81841272 )
	ROM_LOAD( "10310.rom", 0x300000, 0x080000, 0x0f790cda )
	ROM_LOAD( "10311.rom", 0x380000, 0x080000, 0x41109231 )

	ROM_REGION( 0x100000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "10301.rom", 0x000000, 0x080000, 0x52afd952 )
	ROM_LOAD( "10302.rom", 0x080000, 0x080000, 0x6e2d4660 )
ROM_END


/***************************************************************************

Don Den Lover Vol.1 -Shiro Kuro Tsukeyo!-
(c)1995 Dynax
D1120901L8

CPU: 68000(surface scratched)
Sound: YM2413(OPLL)
       YMZ284-D(AY-3-8910 without I/O ports)
       M6295
OSC: 2x ?MHz (surface-scratched)
Video: HD46505SP-2(HD68B45SP)
Others: Battery, RTC 62421B
All custom chips, PALs, GALs are surface-scratched.

ROMs:
1121.2N      [e2b8359e] \
1122.2M      [e8619d66] -- Samples

1123.2H      [d41cbed0] \
1124.2D      [6a9ec557] -- Programs

1125.2B      [0181f53c] \
1126.2A      [17ff2df4] |
1127.3D      [9c136914] |- Blitter data
1128.3C      [757c9941] |
1129.3B      [957bc57e] /

***************************************************************************/

ROM_START( ddenlvrj )
	ROM_REGION( 0x080000, REGION_CPU1, 0 )		/* 68000 Code */
	ROM_LOAD16_BYTE( "1124.2d", 0x000000, 0x040000, 0x6a9ec557 )
	ROM_LOAD16_BYTE( "1123.2h", 0x000001, 0x040000, 0xd41cbed0 )

	ROM_REGION( 0xe80000, REGION_GFX1, 0 )	/* blitter data */
	/* 000000-bfffff empty */
	ROM_LOAD( "1125.2b", 0xc00000, 0x080000, 0x0181f53c )
	ROM_LOAD( "1126.2a", 0xc80000, 0x080000, 0x17ff2df4 )
	ROM_LOAD( "1127.3d", 0xd00000, 0x080000, 0x9c136914 )
	ROM_LOAD( "1128.3c", 0xd80000, 0x080000, 0x757c9941 )
	ROM_LOAD( "1129.3b", 0xe00000, 0x080000, 0x957bc57e )

	ROM_REGION( 0x200000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "1122.2m", 0x080000, 0x080000, 0xe8619d66 )	// bank 4, 5
	ROM_LOAD( "1121.2n", 0x100000, 0x080000, 0xe2b8359e )	// bank 2, 3
ROM_END


/***************************************************************************

Don Den Lover Vol 1
(C) Dynax Inc 1995

CPU: TMP68HC000N-12
SND: OKI M6295, YM2413 (18 pin DIL), YMZ284-D (16 pin DIL. This chip is in place where a 40 pin chip is marked on PCB,
                                     possibly a replacement for some other 40 pin YM chip?)
OSC: 28.636MHz (near large GFX chip), 24.000MHz (near CPU)
DIPS: 1 x 8 Position switch. DIP info is in Japanese !
RAM: 1 x Toshiba TC5588-35, 2 x Toshiba TC55257-10, 5 x OKI M514262-70

OTHER:
Battery
RTC 72421B   4382 (18 pin DIL)
3 X PAL's (2 on daughter-board at locations 2E & 2D, 1 on main board near CPU at location 4C)
GFX Chip - NAKANIHON NL-005 (208 pin, square, surface-mounted)

***************************************************************************/

ROM_START( ddenlovr )
	ROM_REGION( 0x080000, REGION_CPU1, 0 )		/* 68000 Code */
	ROM_LOAD16_BYTE( "1134h.1a", 0x000000, 0x040000, 0x43accdff )
	ROM_LOAD16_BYTE( "1133h.1c", 0x000001, 0x040000, 0x361bf7b6 )

	ROM_REGION( 0x480000, REGION_GFX1, 0 )	/* blitter data */
	/* 000000-1fffff empty */
	ROM_LOAD( "1135h.3h", 0x200000, 0x080000, 0xee143d8e )
	ROM_LOAD( "1136h.3f", 0x280000, 0x080000, 0x58a662be )
	ROM_LOAD( "1137h.3e", 0x300000, 0x080000, 0xf96e0708 )
	ROM_LOAD( "1138h.3d", 0x380000, 0x080000, 0x633cff33 )
	ROM_LOAD( "1139h.3c", 0x400000, 0x080000, 0xbe1189ca )

	ROM_REGION( 0x200000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "1131h.1f", 0x080000, 0x080000, 0x32f68241 )	// bank 2, 3
	ROM_LOAD( "1132h.1e", 0x100000, 0x080000, 0x2de6363d )	// bank 4, 5
ROM_END

/* Korean bootleg board */
ROM_START( ddenlovb )
	ROM_REGION( 0x080000, REGION_CPU1, 0 )		/* 68000 Code */
	ROM_LOAD16_BYTE( "rom2", 0x000000, 0x040000, 0xcabdf78f )
	ROM_LOAD16_BYTE( "rom3", 0x000001, 0x040000, 0x36f8d05e )

	ROM_REGION( 0x480000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "rom4", 0x000000, 0x080000, 0xee143d8e )
	ROM_LOAD( "rom5", 0x080000, 0x080000, 0x58a662be )
	ROM_LOAD( "rom6", 0x100000, 0x080000, 0xf96e0708 )
	ROM_LOAD( "rom7", 0x180000, 0x080000, 0xb47e27ec )
	ROM_LOAD( "rom8", 0x200000, 0x080000, 0x7c7beef6 )

	ROM_REGION( 0x200000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "rom1", 0x100000, 0x080000, 0xba4723e8 )
ROM_END

static DRIVER_INIT( rongrong )
{
	/* Rong Rong seems to have a protection that works this way:
        - write 01 to port c2
        - write three times to f705 (a fixed command?)
        - write a parameter to f706
        - read the answer back from f601
        - write 00 to port c2
       The parameter is read from RAM location 60d4, and the answer
       is written back there. No matter what the protection device
       does, it seems that making 60d4 always read 0 is enough to
       bypass the protection. Actually, I'm wondering if this
       version of the game might be a bootleg with the protection
       patched. (both sets need this)
     */
	install_mem_read_handler(0, 0x60d4, 0x60d4, MRA_NOP);
}

/***************************************************************************

HANAKANZASHI
(c)1996 DYNAX.INC
CPU : Z-80 (TMPZ84C015BF-8)
SOUND : MSM6295 YM2413
REAL TIME CLOCK : MSM6242

***************************************************************************/

ROM_START( hanakanz )
	ROM_REGION( 0x90000+16*0x1000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "50720.5b",     0x00000, 0x80000, 0xdc40fcfc )
	ROM_RELOAD(               0x10000, 0x80000 )

	ROM_REGION( 0x300000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD16_BYTE( "50740.8b",     0x000000, 0x80000, 0x999e70ce )
	ROM_LOAD16_BYTE( "50730.8c",     0x000001, 0x80000, 0x54e1731d )
	ROM_LOAD16_BYTE( "50760.10b",    0x100000, 0x80000, 0x8fcb5da3 )
	ROM_LOAD16_BYTE( "50750.10c",    0x100001, 0x80000, 0x0e58bf9e )
	ROM_LOAD16_BYTE( "50780.12b",    0x200000, 0x80000, 0x6dfd8a86 )
	ROM_LOAD16_BYTE( "50770.12c",    0x200001, 0x80000, 0x118e6baf )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "50710.1c",     0x00000, 0x80000, 0x72ae072f )	// 2 banks
ROM_END


/***************************************************************************

Hana Kagerou
(c)1996 Nakanihon (Dynax)

CPU:    KL5C80A12

Sound:  YM2413
        M6295?

OSC:    20.000MHz
        28.63636MHz

Custom: (70C160F011)


NM5101.1C   samples

NM5102.5B   prg.

NM5103.8C   chr.
NM5104.8B
NM5105.10C
NM5106.10B
NM5107.12C
NM5108.12B

***************************************************************************/

ROM_START( hkagerou )
	ROM_REGION( 0x90000+16*0x1000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "nm5102.5b",    0x00000, 0x80000, 0xc56c0856 )
	ROM_RELOAD(               0x10000, 0x80000 )

	ROM_REGION( 0xe80000, REGION_GFX1, 0 )	/* blitter data */

	ROM_LOAD16_BYTE( "nm5104.8b",    0xc00000, 0x080000, 0xe91dd92b )
	ROM_LOAD16_BYTE( "nm5103.8c",    0xc00001, 0x080000, 0x4d4e248b )
	ROM_LOAD16_BYTE( "nm5106.10b",   0xd00000, 0x080000, 0x0853c32d )
	ROM_LOAD16_BYTE( "nm5105.10c",   0xd00001, 0x080000, 0xf109ec10 )
	ROM_LOAD16_BYTE( "nm5108.12b",   0xe00000, 0x040000, 0xd0a99b19 )
	ROM_LOAD16_BYTE( "nm5107.12c",   0xe00001, 0x040000, 0x65a0ebbd )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "nm5101.1c",    0x00000, 0x80000, 0xbf7a397e )	// 2 banks
ROM_END


/***************************************************************************

Mahjong Reach Ippatsu
(c)1998 Nihon System/Dynax

CPU:   KL5C80A12

Sound: YM2413
       M6295

OSC:   20.000MHz
       28.63636MHz
       32.768KHz

Custom: (70C160F011)
Others: M6242B (RTC)


52601.1C    samples

52602-N.5B  prg.

52603.8C    chr.
52604.8B
52605.10C
52606.10B
52607.12C
52608.12B

***************************************************************************/

ROM_START( mjreach1 )
	ROM_REGION( 0x90000+16*0x1000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "52602-n.5b",   0x00000, 0x80000, 0x6bef7978 )
	ROM_RELOAD(               0x10000, 0x80000 )

	ROM_REGION( 0x500000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD16_BYTE( "52604.8b",     0x000000, 0x100000, 0x6ce01bb4 )
	ROM_LOAD16_BYTE( "52603.8c",     0x000001, 0x100000, 0x16d2c169 )
	ROM_LOAD16_BYTE( "52606.10b",    0x200000, 0x100000, 0x07fe5dae )
	ROM_LOAD16_BYTE( "52605.10c",    0x200001, 0x100000, 0xb5d57163 )
	ROM_LOAD16_BYTE( "52608.12b",    0x400000, 0x080000, 0x2f93dde4 )
	ROM_LOAD16_BYTE( "52607.12c",    0x400001, 0x080000, 0x5e685c4d )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "52601.1c",     0x00000, 0x80000, 0x52666107 )	// 2 banks
ROM_END

/***************************************************************************

Mahjong Chuukanejyo
Dynax, 1995

PCB Layout
----------
D11107218L1
|-----------------------------------------------|
|10WAY           18WAY          D12101 5.5V_BATT|
|          358     358        6606              |
|      VOL    6868A                             |
|                         16MHz                 |
|           95101                   62256       |
|                        TMPZ84C015F-6          |
|                                D12102         |
|2                                        3631  |
|8                                              |
|W                                  PAL         |
|A            28.322MHz                         |
|Y                                              |
|                          PAL                  |
|             70C160F009                        |
|                           D12103      D12104  |
|              TC524256Z-12                     |
|              TC524256Z-12 D12105      D12106  |
|DIP1     DIP2 TC524256Z-12                     |
|DIP3     DIP4 TC524256Z-12 D12107      D12108  |
|-----------------------------------------------|
Notes:
      Main CPU is Toshiba TMPZ84C015F-6 (QFP100)
      95101 - Unknown 40 pin DIP, maybe equivalent to AY-3-8910?
      6868A - Unknown 18 pin DIP, maybe some other sound related chip or a PIC?
      3631  - Unknown 18 pin DIP, maybe RTC or a PIC?
      6606  - Unknown QFP44, probably OKI M6295?
      70C160F009 - QFP208 Dynax Custom


***************************************************************************/

ROM_START( mjchuuka )
	ROM_REGION( 0x90000+16*0x1000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "d12102.5b", 0x00000, 0x80000, 0x585a0a8e )
	ROM_RELOAD(            0x10000, 0x80000 )

	ROM_REGION( 0x300000, REGION_GFX1, ROMREGION_ERASEFF )	/* blitter data */
	ROM_LOAD16_BYTE( "d12103.11c", 0x000000, 0x080000, 0x83bfc841 )
	ROM_LOAD16_BYTE( "d12104.11a", 0x000001, 0x080000, 0x1bf6220a )
	ROM_LOAD16_BYTE( "d12105.12c", 0x100000, 0x080000, 0x3424c8ac )
	ROM_LOAD16_BYTE( "d12106.12a", 0x100001, 0x080000, 0x9052bd09 )
	ROM_LOAD16_BYTE( "d12107.13c", 0x280000, 0x020000, 0x184afa94 )	// 1xxxxxxxxxxxxxxxx = 0xFF
	ROM_LOAD16_BYTE( "d12108.13a", 0x280001, 0x020000, 0xf8e8558a )	// 1xxxxxxxxxxxxxxxx = 0xFF

	ROM_REGION( 0x80000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "d12101.1b", 0x00000, 0x80000, 0x9759c65e )
ROM_END


/***************************************************************************

Mahjong The Dai Chuuka Ken (China Version)
Dynax, 1995

PCB Layout
----------

D11107218L1 DYNAX INC. NAGOYA JAPAN
|-----------------------------------------------------|
|10-WAY              18-WAY                  1    5.5V|
|                               6606         x        |
|   MB3712  VOL  358                                  |
|                358                                  |
|                               16MHz        43256    |
|                 6868A                               |
|              95101            Z84C015      2        |
|                                                     |
|2                                              3631  |
|8                                                    |
|W                                                    |
|A                                         PAL        |
|Y                     28.322MHz      PAL             |
|                                                     |
|                         |---------|                 |
|                         |NAKANIHON|                 |
|                         |70C160F009   3       4     |
|                 44C251  |         |                 |
|                 44C251  |         |   5       6     |
| DSW1     DSW2   44C251  |---------|                 |
| DSW3     DSW4   44C251                7       8     |
|-----------------------------------------------------|
Notes:
      PCB uses common 10-way/18-way and 28-way Mahjong pinouts
      5.5V    - Battery
      6606    - Compatible to M6295 (QFP44)
      6868A   - compatible to YM2413 (DIP18)
      3631    - Unknown DIP18 chip (maybe RTC?)
      Z84C015 - Toshiba TMPZ84C015BF-6 Z80 compatible CPU (clock input 16.0MHz)
      44C251  - Texas Instruments TMS44C251-12SD 256k x4 Dual Port VRAM (ZIP28)
      95101   - Winbond 95101, compatible to AY-3-8910 (DIP40)
      43256   - NEC D43256 32k x8 SRAM (DIP28)
      70C160F009 - Custom Dynax graphics generator (QFP160)
      All DIPSW's have 10 switches per DIPSW
      All ROMs are 27C040
                          1   - Sound samples
                          2   - Main program
                          3,4 - Graphics
                          5-8 - unused DIP32 sockets

      The same PCB is used with 'Mahjong Zhong Hua Er Nu', with ROM locations
      as follows....
                    1 - D1111-A.1B
                    2 - D12102.5B
                    3 - D12103.11C
                    4 - D12104.11A
                    5 - D12105.12C
                    6 - D12106.12A
                    7 - D12107.13C
                    8 - D12108.13A

***************************************************************************/

ROM_START( mjdchuka )
	ROM_REGION( 0x90000+16*0x1000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "2.5b", 0x00000, 0x80000, 0x7957b4e7 )
	ROM_RELOAD(       0x10000, 0x80000 )

	ROM_REGION( 0x100000, REGION_GFX1, ROMREGION_ERASEFF )	/* blitter data */
	ROM_LOAD16_BYTE( "3.11c", 0x000000, 0x080000, 0xc66553c3 )
	ROM_LOAD16_BYTE( "4.11a", 0x000001, 0x080000, 0x972852fb )

	ROM_REGION( 0x80000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "1.1b", 0x00000, 0x80000, 0x9759c65e )
ROM_END


/***************************************************************************

The First Funky Fighter
Nakanihon, 1994

PCB Layout
----------

N7403208L-2
|------------------------------------------------------------------|
|    VR1                7401          7402  32.768kHz M6242 3V_BATT|
|    VR2          358         PAL        M6295  TC55257  PAL       |
|       YM2413          TC5563                             16MHz   |
|                 358                          7403                |
|       YM2149          Z80                         TMPZ84C015BF-8 |
|                                                                  |
|J                                                                 |
|A                                                                 |
|M          NL-002      PAL                                        |
|M                                                                 |
|A                                                     DSW(10)     |
|                                                                  |
|                                                      DSW(10)     |
|                                                                  |
|       TC5588                           28.6363MHz                |
|                                          |-ROM-sub-board-N73RSUB-|
|                                          |                       |
| DSW(4)                                   |NL-005         PAL     |
|       SN75179                            |                       |         Sub-board contains 12 sockets.
|                                          |        7404   7411 |----------- Only these 3 are populated.
|                                          |        7405   7410 /  |
|DB9   OMRON              NL-006     TC524258BZ-10  7406   7409/   |
|      G6A-474P        TC524258BZ-10 TC524258BZ-10  7407           |
|                      TC524258BZ-10 TC524258BZ-10  7408           |
|DB9                                 TC524258BZ-10         PAL     |
|                                          |             (on sub)  |
|------------------------------------------|-----------------------|

the second halves of 7408.13b, 7409.4b, 7410.3b and 7411.2b are identical

***************************************************************************/

ROM_START( funkyfig )
	ROM_REGION( 0x90000 + 0x1000*8, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "7403.3c",      0x00000, 0x80000, 0xad0f5e14 )
	ROM_RELOAD(               0x10000, 0x80000 )

	ROM_REGION( 0x20000, REGION_CPU2, 0 )	/* Z80 Code */
	ROM_LOAD( "7401.1h",      0x00000, 0x20000, 0x0f47d785 )	// 1xxxxxxxxxxxxxxxx = 0xFF

	ROM_REGION( 0x500000, REGION_GFX1, ROMREGION_ERASE00 )	/* blitter data */
	ROM_LOAD( "7404.8b",      0x000000, 0x080000, 0xaa4ddf32 )			// \ 7e6f +
	ROM_LOAD( "7405.9b",      0x080000, 0x080000, 0xfc125bd8 )			// / 35bb = b42a OK
	ROM_LOAD( "7406.10b",     0x100000, 0x080000, 0x04a214b1 )	// \ af08 +
	ROM_LOAD( "7407.11b",     0x180000, 0x080000, 0x635d4052 )
	ROM_LOAD( "7409.4b",      0x200000, 0x100000, 0x064082c3 )			// \ 15bd +
	ROM_LOAD( "7410.3b",      0x280000, 0x100000, 0x0ba67874 )			// / 2e4c = 4409 OK
	ROM_LOAD( "7408.13b",     0x300000, 0x100000, 0x9efe4c60 )			// 0f46 + 1825 = 276b OK
	ROM_LOAD( "7411.2b",      0x400000, 0x100000, 0x1e9c73dc )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "7402.1e",      0x000000, 0x040000, 0x5038cc34 )
ROM_END

/***************************************************************************

The Mysterious World
(c) 1994 DynaX

Board has a sticker labeled D7707308L1
The actual PCB is printed as D7107058L1-1

Most all chips are surface scratched

OSC: 24.000MHz, 14.318MHz
4 x 10 Switch Dipswitch
1 4 Switch Dipswitch
VR1, VR2 & Reset Switch
3.6V Ni/CD Battery
OKI M6242B - Real Time Clock

56 pin Non-JAMMA Connector
20 pin unknown Connector
36 pin unknown Connector

Sound Chips:
K-665 (OKI M6295)
YM2149F
YM2413

***************************************************************************/

ROM_START( mjmyster )
	ROM_REGION( 0x90000 + 0x1000*8, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "77t2.c3", 0x00000, 0x40000, 0xb1427cce )
	ROM_RELOAD(          0x10000, 0x40000 )

	ROM_REGION( 0x1a0000, REGION_GFX1, ROMREGION_ERASE00 )	/* blitter data */
	ROM_LOAD( "77t6.b12", 0x000000, 0x080000, 0xa287589a )
	ROM_LOAD( "77t5.b11", 0x080000, 0x080000, 0xa3475059 )
	ROM_LOAD( "77t4.b10", 0x100000, 0x080000, 0xf45c24d6 )
	ROM_LOAD( "77t3.b9",  0x180000, 0x020000, 0x8671165b )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "77t1.d1", 0x000000, 0x020000, 0x09b7a9b2 )
ROM_END


/***************************************************************************

Mahjong The Mysterious World
Dynax, 1994

PCB Layout
----------

(no number)
|-------------------------------------------------------|
|MB3712  VOL                1.1E   M6242B               |
|                           K-665  32.768kHz PAL        |
|                    YM2413        62256                |
|                                  2.3D                 |
|           DSW5(8)    YM2149               TMPZ84C015  |
|                                                       |
|M                                                      |
|A                                                 16MHz|
|H                   PAL                                |
|J                                                      |
|O     NL-002                                           |
|N                   PAL                                |
|G                                                      |
|                                      28.636MHz        |
|DSW1(10)        CY7C185                                |
|                                                       |
|DSW2(10)  1108F0405       ***       1427F0071   3.10B  |
|                                                       |
|DSW3(10)                                        4.11B  |
|           TC524256Z-10  TC524256Z-10                  |
|DSW4(10)   TC524256Z-10  TC524256Z-10  PAL PAL  5.12B  |
|-------------------------------------------------------|
Notes:
      TMPZ84C015   - Toshiba TMPZ84C015F-6 (QFP100). Clocks 16.000MHz (pins 65 & 66), 8.000MHz (pins 68 & 69)
      62256        - 32k x8 SRAM (DIP28)
      TC524256Z-10 - Toshiba TC524256Z-10 256k x4 Dual Port VRAM (ZIP28)
      1427F0071    \ Dynax Custom ICs
      1108F0405    /
      NL-002       /
      ***          - Unknown QFP100 (surface scratched)
      K-665        - == Oki M6295 (QFP44). Clock 1.02272MHz [28.636/28]. pin 7 = high
      YM2149       - Clock 3.579545MHz [28.636/8]
      YM2413       - Clock 3.579545MHz [28.636/8]
      VSync        - 61Hz
      HSync        - 15.27kHz

***************************************************************************/

ROM_START( mjmywrld )
	ROM_REGION( 0x90000 + 0x1000*8, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "2.3d", 0x00000, 0x40000, 0xa1ff31fa )
	ROM_RELOAD(       0x10000, 0x40000 )

	ROM_REGION( 0x1a0000, REGION_GFX1, ROMREGION_ERASE00 )	/* blitter data */
	ROM_LOAD( "5.12b", 0x000000, 0x100000, 0xa1f26722 )
	ROM_LOAD( "4.11b", 0x100000, 0x080000, 0xf45c24d6 )
	ROM_LOAD( "3.10b", 0x180000, 0x020000, 0x8671165b )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "1.1e", 0x00000, 0x20000, 0x09b7a9b2 )
ROM_END


/***************************************************************************

Hanafuda Hana Ginga
Dynax, 1994

PCB Layout
----------

D8102048L1 (almost same PCB as The Mysterious World)
|-------------------------------------------------------|
|MB3712  VOL      DSW1(10)      M6242B  62256      SW   |
|                             K-665  32.768kHz          |
|                 DSW2(10)                      BATTERY |
|                                  8101.2B              |
| DSW5(4)         DSW3(10)    PAL           TMPZ84C015  |
|                                                       |
|M        YM2413  DSW4(10)                              |
|A                                                 16MHz|
|H        YM2149                                        |
|J                                                      |
|O     NL-002                                 PAL       |
|N                                                      |
|G                                            8102.9A   |
|                                 28.636MHz             |
|                CY7C185                      8103.10A  |
|                                                       |
|          1108F0405       ***     1427F0071  8104.11A  |
|                                                       |
|                                                       |
|         TC524256Z-10  TC524256Z-10                    |
|         TC524256Z-10  TC524256Z-10  PAL PAL           |
|-------------------------------------------------------|
Notes:
      TMPZ84C015   - Toshiba TMPZ84C015F-6 (QFP100). Clocks 16.000MHz (pins 65 & 66), 8.000MHz (pins 68 & 69)
      62256        - 32k x8 SRAM (DIP28)
      TC524256Z-10 - Toshiba TC524256Z-10 256k x4 Dual Port VRAM (ZIP28)
      1427F0071    \ Dynax Custom ICs
      1108F0405    /
      NL-002       /
      ***          - Unknown QFP100 (surface scratched)
      K-665        - == Oki M6295 (QFP44). Clock 1.02272MHz [28.636/28]. pin 7 = high
      YM2149       - Clock 3.579545MHz [28.636/8]
      YM2413       - Clock 3.579545MHz [28.636/8]
      VSync        - 61Hz
      HSync        - 15.27kHz
      SW           - This resets the PCB and also clears the RAM (clears credits etc)
                     It is used on almost all Mahjong games by Dynax and IGS

***************************************************************************/

ROM_START( hginga )
	ROM_REGION( 0x90000 + 0x1000*8, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "8101.2b", 0x00000, 0x40000, 0x77a64b71 )
	ROM_RELOAD(          0x10000, 0x40000 )

	ROM_REGION( 0x180000, REGION_GFX1, ROMREGION_ERASEFF )	/* blitter data */
	ROM_LOAD( "8102.9a",  0x000000, 0x80000, 0x0074af23 )
	ROM_LOAD( "8103.10a", 0x080000, 0x80000, 0xa3a4ecb5 )
	ROM_LOAD( "8104.11a", 0x100000, 0x20000, 0x24513af9 )
	ROM_RELOAD(           0x120000, 0x20000 )
	ROM_RELOAD(           0x140000, 0x20000 )
	ROM_RELOAD(           0x160000, 0x20000 )
ROM_END


/***************************************************************************

Mahjong The Mysterious Orient
Dynax, 1994

PCB Layout
----------

D7107058L1-1
|-------------------------------------------------------|
|MB3712  VOL             7101.1E   M6242B               |
|                           K-665  32.768kHz PAL        |
|                    YM2413        62256                |
|                                  7102.3D              |
|           DSW5(4)    YM2149               TMPZ84C015  |
|                                                       |
|M                                                      |
|A                                           14.31818MHz|
|H                   PAL                                |
|J                                                      |
|O     NL-002                                           |
|N                   PAL                                |
|G                                                      |
|                                                       |
|DSW1(10)                  6116                         |
|                                                       |
|DSW2(10)  1108F0405               1427F0071   7103.10B |
|                                                       |
|DSW3(10)                  24MHz               7104.11B |
|          TC524256Z-10  TC524256Z-10                   |
|DSW4(10)  TC524256Z-10  TC524256Z-10 PAL PAL  7105.12B |
|-------------------------------------------------------|
Notes:
      TMPZ84C015   - Toshiba TMPZ84C015F-6 (QFP100). Clocks 14.31818 (pins 65 & 66), 7.15909MHz (pins 68 & 69)
      62256        - 32k x8 SRAM (DIP28)
      TC524256Z-10 - Toshiba TC524256Z-10 256k x4 Dual Port VRAM (ZIP28)
      1427F0071    \ Dynax Custom ICs
      1108F0405    /
      NL-002       /
      K-665        - == Oki M6295 (QFP44). Clock 1.02272MHz [14.31818/14]. pin 7 = high
      YM2149       - Clock 1.7897725MHz [14.31818/8]
      YM2413       - Clock 3.579545MHz [14.31818/4]
      VSync        - 61Hz
      HSync        - 15.27kHz

***************************************************************************/

ROM_START( mjmyornt )
	ROM_REGION( 0x90000 + 0x1000*8, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "7102.3d", 0x00000, 0x40000, 0x058f779b )
	ROM_RELOAD(          0x10000, 0x40000 )

	ROM_REGION( 0x1a0000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "7105.12b", 0x000000, 0x100000, 0x35ca0079 )
	ROM_LOAD( "7104.11b", 0x100000, 0x080000, 0x6d0fd29a )
	ROM_LOAD( "7103.10b", 0x180000, 0x020000, 0x88511487 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "7101.1e", 0x00000, 0x20000, 0x09b7a9b2 )
ROM_END


/***************************************************************************

Mahjong The Mysterious Universe
Dynax, 1994

PCB Layout
----------

D7107058L1-1
|-------------------------------------------------------|
|MB3712  VOL                1.1E   M6242B               |
|                           K-665  32.768kHz PAL        |
|                    YM2413        62256                |
|                                  2.3D                 |
|           DSW5(8)    YM2149               TMPZ84C015  |
|                                                       |
|M                                                      |
|A                                           14.31818MHz|
|H                   PAL                                |
|J                                                      |
|O     NL-002                                           |
|N                   PAL                                |
|G                                                      |
|                                                       |
|DSW1(10)                  6116                  3.9B   |
|                                                       |
|DSW2(10)  1108F0405                 1427F0071   4.10B  |
|                                                       |
|DSW3(10)                  24MHz                 5.11B  |
|           TC524256Z-10  TC524256Z-10                  |
|DSW4(10)   TC524256Z-10  TC524256Z-10  PAL PAL  6.12B  |
|-------------------------------------------------------|
Notes:
      TMPZ84C015   - Toshiba TMPZ84C015F-6 (QFP100). Clocks 14.31818 (pins 65 & 66), 7.15909MHz (pins 68 & 69)
      62256        - 32k x8 SRAM (DIP28)
      TC524256Z-10 - Toshiba TC524256Z-10 256k x4 Dual Port VRAM (ZIP28)
      1427F0071    \ Dynax Custom ICs
      1108F0405    /
      NL-002       /
      K-665        - == Oki M6295 (QFP44). Clock 1.02272MHz [14.31818/14]. pin 7 = high
      YM2149       - Clock 1.7897725MHz [14.31818/8]
      YM2413       - Clock 3.579545MHz [14.31818/4]
      VSync        - 61Hz
      HSync        - 15.27kHz

***************************************************************************/

ROM_START( mjmyuniv )
	ROM_REGION( 0x90000 + 0x1000*8, REGION_CPU1, 0 )	/* Z80 Code + space for banked RAM */
	ROM_LOAD( "2.3d", 0x00000, 0x40000, 0x3284d714 )
	ROM_RELOAD(       0x10000, 0x40000 )

	ROM_REGION( 0x600000, REGION_GFX1, 0 )	/* blitter data */
	// gap
	ROM_LOAD( "5.11b", 0x400000, 0x80000, 0xa287589a )
	ROM_LOAD( "6.12b", 0x480000, 0x80000, 0xa3475059 )
	ROM_LOAD( "3.9b",  0x500000, 0x80000, 0xf45c24d6 )
	ROM_LOAD( "4.10b", 0x580000, 0x80000, 0x57fb94cc )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "1.1e", 0x00000, 0x20000, 0x09b7a9b2 )
ROM_END

/***************************************************************************

Panel & Variety Akamaru Q Jousyou Dont-R
(c)1996 Dynax (distributed by B&F)

CPU: TMP68HC000N-12
Sound: YM2413, YMZ284-D, M6295 (VRx2, 1 for BGM, 1 for Voice)
OSC: 24.00000MHz (near CPU), 28.63636MHz (near sound section)
RTC: 62421B
Custom: NAKANIHON NL-005
PLD: Actel A1010B (printed NM500)

ROMs (on subboard):
50101.1H (TMS 27C040)
50102.1G (TMS 27C040)
50103.1F (TMS 27C040)
50104.1E (TMS 27C040)
50105.1D (TMS 27C040)
50106.1C (TMS 27C040)
50107.1B (TMS 27C040)
50108.1A (TMS 27C040)
50109.2H (TC538000)
50110.3H (TC538000)
50111.4H (TMS 27C040)
50112.2D (TC538000)
50113.3D (TMS 27C040)
50114.4D (TMS 27C040)

***************************************************************************/

ROM_START( akamaru )
	ROM_REGION( 0x200000, REGION_CPU1, 0 )		/* 68000 Code */
	ROM_LOAD16_BYTE( "50107.1b", 0x000000, 0x080000, 0x8364d627 )
	ROM_LOAD16_BYTE( "50105.1d", 0x000001, 0x080000, 0x42ff4bec )
	ROM_LOAD16_BYTE( "50108.1a", 0x100000, 0x080000, 0x1520ecad )
	ROM_LOAD16_BYTE( "50106.1c", 0x100001, 0x080000, 0x8e081747 )

	ROM_REGION( 0x480000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "50109.2h", 0x000000, 0x100000, 0xcdff154d )
	ROM_LOAD( "50112.2d", 0x100000, 0x100000, 0x7fdd9853 )
	ROM_LOAD( "50110.3h", 0x200000, 0x100000, 0x06ca4d87 )
	ROM_LOAD( "50113.3d", 0x300000, 0x080000, 0xc9a571cd )
	ROM_LOAD( "50111.4h", 0x380000, 0x080000, 0xb2de7a3c )
	ROM_LOAD( "50114.4d", 0x400000, 0x080000, 0x2af39dfd )

	ROM_REGION( 0x300000, REGION_SOUND1, 0 )	/* Samples */
	// bank 0, 1
	ROM_LOAD( "50101.1h", 0x080000, 0x080000, 0x6bef6cab )	// bank 2, 3
	ROM_LOAD( "50102.1g", 0x100000, 0x080000, 0x056ac348 )	// bank 4, 5
	// bank 6, 7
	ROM_LOAD( "50104.1e", 0x200000, 0x080000, 0x790f18c1 )	// bank 8, 9
	ROM_LOAD( "50103.1f", 0x280000, 0x080000, 0xaff3a753 )	// bank 10, 11
ROM_END

/***************************************************************************

Mahjong Fantasic Love
Nakanihon, 1996

PCB Layout
----------
no number (scratched off) Looks like Don Den Lover h/w
|--------------------------------------------------|
|UPC1242H  BATTERY                P1               |
|                 M6295      TC55257               |
|                                          *       |
|                72421B_RTC  TC55257               |
|YM2413     YMZ284-D               PAL             |
|                                                  |
|                                                  |
|J                                                 |
|A                                           24MHz |
|M           28.63636MHz                           |
|M                                     M514262     |
|A                                                 |
|                                      M514262     |
|                         |--------|               |
|                         |        |   M514262     |
|                         | NL-005 |               |
|             TC5588      |        |   M514262     |
|                         |        |               |
|                         |--------|   M514262     |
|DSW1(8)                          P2               |
|--------------------------------------------------|
Notes:
      *       - SDIP64 socket, not populated
      P1/P2   - Connector joining to ROM daughterboard
      TC55257 - 32k x8 SRAM
      TC5588  - 8k x8 SRAM
      M514262 - OKI M514262-70Z 262144-word x 4-bit Multiport DRAM
      YMZ284  - Yamaha YMZ284-D Software Controlled Melody and Effect Sound Generator (DIP16)
                Software compatible with YM2149. Clock 3.579545MHz [28.63636/8]
      YM2413  - OPLL FM Sound Generator (DIP18). Clock 3.579545MHz [28.63636/8]
      M6295   - Clock 1.02272MHz [28.63636/28]. Pin7 HIGH
      VSync   - 60Hz
      HSync   - 15.28kHz


Top Board
---------
NS5000101
|---------------------------------|
|                    16MHz        |
|       &                *        |
|                                 |
|                                 |
|                                 |
|                          DSW1(8)|
|                                 |
|50001   50003   50005   50007    |
|    50002   50004   50006        |
|                                 |
|                                 |
|50008           50010   50012    |
|    50009           50011   50013|
|                                 |
|                                 |
|---------------------------------|
Notes:
      &     - Unknown PLCC68 chip (surface scratched). Maybe CPLD/FPGA or custom? Doesn't
              look like 68000 as there is no measurable clock input on any pins.
      *     - Unknown QFP100 (surface scratched). Clock input of 16MHz. Possibly TMPZ84C015?

***************************************************************************/

ROM_START( mjflove )
	ROM_REGION( 0x90000+16*0x1000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "50004.2e", 0x00000, 0x80000, 0x20afcdec )
	ROM_RELOAD(           0x10000, 0x80000 )

	ROM_REGION( 0x3c0000, REGION_GFX1, 0 )	/* blitter data */
	ROM_LOAD( "50005.2d", 0x000000, 0x80000, 0x6494d5ad )
	ROM_LOAD( "50006.2c", 0x080000, 0x80000, 0xa1d61eb7 )
	ROM_LOAD( "50007.2b", 0x100000, 0x80000, 0xd79ea2f5 )
	ROM_LOAD( "50009.3h", 0x180000, 0x80000, 0xcba17351 )
	ROM_LOAD( "50013.3a", 0x200000, 0x80000, 0xebb7c8bd )
	ROM_LOAD( "50010.3d", 0x280000, 0x80000, 0x62c05df9 )
	ROM_LOAD( "50011.3c", 0x300000, 0x80000, 0x7e05f586 )
	ROM_LOAD( "50012.3b", 0x380000, 0x40000, 0x9853e5e4 )

	ROM_REGION( 0x200000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "50002.2h", 0x000000, 0x80000, 0x955034db )	// 0,1
	ROM_LOAD( "50003.2f", 0x080000, 0x80000, 0x8845734a )	// 2,3
	ROM_LOAD( "50001.2j", 0x100000, 0x80000, 0x6c0a93fd )	// 4,5
	ROM_LOAD( "50008.3j", 0x180000, 0x40000, 0x4a95b5eb )	// 6
	ROM_RELOAD(           0x1c0000, 0x40000 )
ROM_END

/***************************************************************************

Super Hana Paradise
1995

PCB almost like Mysterious World

PCB Layout
----------
N8010178L1
|-------------------------------------------------------|
|MA1384  VOL             1011.2F                        |
|                          62256                        |
|                   M6295   1012.3E                     |
|        VOL                                     16MHz  |
|        YM2413                       TMPZ84C015        |
|                                                       |
|M                                                      |
|A     NL-002                                   DSW1(10)|
|H                                                      |
|J                                              DSW2(10)|
|O                                                      |
|N                                                      |
|G                                               DIP32  |
|                         28.636MHz                     |
|             TC5588                             DIP32  |
|                                                       |
|          1108F0406   4L02F2637     1427F0071   DIP32  |
|                                                       |
|                                                DIP32  |
|       TC524258Z-10  TC524258Z-10                      |
|       TC524258Z-10  TC524258Z-10               DIP32  |
|-------------------------------------------------------|
Notes:
      TMPZ84C015   - Toshiba TMPZ84C015F-6 (QFP100). Clocks 16.000MHz (pins 65 & 66), 8.000MHz (pins 68 & 69)
      62256        - 32k x8 SRAM (DIP28)
      TC5588       - 8k x8 SRAM (DIP28)
      TC524258Z-10 - Toshiba TC524258Z-10 256k x4 Dual Port VRAM (ZIP28)
      1427F0071    \
      1108F0406    | Dynax Custom ICs
      NL-002       |
      4L02F2637    /
      DIP32        - Empty sockets
      M6295        - Oki M6295 (QFP44). Clock 1.02272MHz [28.636/28]. pin 7 = high
      YM2413       - Clock 3.579545MHz [28.636/8]
      VSync        - 61Hz
      HSync        - 15.27kHz

ROM Daughterboard N73RSUB
Contains the remaining ROMS, 1 PAL and nothing else
1013, 1014, 1015, 1016

--

Super Hana Paradise
(c)1994 Dynax

D10110258L1

CPU: TMPZ84C015BF-8
Sound: YM2413 M6295
OSC: 28.6363MHz 16AKSS
Custom: NL-002
        1427F0071
        4L02F2637
        1108F0405

ROMs:
1011.2F
1012.3E

Subboard
1013.1A
1014.2A
1015.3A
1016.4A

***************************************************************************/

ROM_START( hparadis )
	ROM_REGION( 0x50000+8*0x1000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "1012.3e", 0x00000, 0x40000, 0xbd3a3a8f )
	ROM_RELOAD(          0x10000, 0x40000 )

	ROM_REGION( 0x600000, REGION_GFX1, 0 )	/* blitter data */
	// unused
	ROM_LOAD( "1015.3a", 0x400000, 0x80000, 0xea7b282e )
	ROM_LOAD( "1014.2a", 0x480000, 0x80000, 0x5f057c13 )
	ROM_LOAD( "1016.4a", 0x500000, 0x80000, 0xc8e7ffb6 )
	ROM_LOAD( "1013.1a", 0x580000, 0x80000, 0xc26b0563 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "1011.2f", 0x00000, 0x40000, 0x8c852b1a )
ROM_END


/***************************************************************************

Hanafuda Hana Gokou
Dynax (Alba License), 1995

PCB almost like Mysterious World

PCB Layout
----------
N83061581L1
|-------------------------------------------------------|
|MB3714  VOL     DSW1(10)    M6242B   TC55257   BATTERY |
|                   32.768kHz 1081.2D  1082B.2B         |
|   VOL          DSW2(10)    M6295                      |
|   DSW5(4)                                      16MHz  |
|        YM2413  DSW3(10)        PAL         TMPZ84C015 |
|                                                       |
|M               DSW4(10)                               |
|A         YM2149                                       |
|H                          *                           |
|J     NL-002                           28.63636MHz     |
|O                                                 PAL  |
|N                                                      |
|G                                                      |
|2                                                      |
|8                                              1083.9A |
|                                                       |
|            &           %            #         1084.10A|
|                                                       |
|                      TC5588                    DIP32  |
|       TC524258Z-10  TC524258Z-10               DIP32  |
|       TC524258Z-10  TC524258Z-10               DIP32  |
|-------------------------------------------------------|
Notes:
      TMPZ84C015   - Toshiba TMPZ84C015F-6 (QFP100). Clocks 16.000MHz (pins 65 & 66), 8.000MHz (pins 68 & 69)
      TC55257      - 32k x8 SRAM (DIP28)
      TC5588       - 8k x8 SRAM (DIP28)
      TC524258Z-10 - Toshiba TC524258Z-10 256k x4 Dual Port VRAM (ZIP28)
      &            - Unknown QFP64. Possibly Dynax Custom 1108F0406
      %            - Unknown QFP100. Possibly Dynax Custom 4L02F2637
      #            - Unknown QFP100. Possibly Dynax Custom 1427F0071
      *            - Unknown PLCC44. Possibly MACH311 or similar CPLD
      DIP32        - Empty sockets
      M6295        - Oki M6295 (QFP44). Clock 1.02272MHz [28.63636/28]. pin 7 = HIGH
      YM2413       - Clock 3.579545MHz [28.63636/8]
      YM2149       - Clock 3.579545MHz [28.63636/8]
      VSync        - 60Hz
      HSync        - 15.36kHz

***************************************************************************/

ROM_START( hgokou )
	ROM_REGION( 0x90000+8*0x1000, REGION_CPU1, 0 )	/* Z80 Code */
	ROM_LOAD( "1082b.2b", 0x00000, 0x40000, 0xe33bc5a5 )
	ROM_RELOAD(           0x10000, 0x40000 )

	ROM_REGION( 0x200000, REGION_GFX1, 0 )	/* blitter data */
	// unused
	ROM_LOAD( "1083.9a",  0x100000, 0x80000, 0x054200c3 )
	ROM_LOAD( "1084.10a", 0x180000, 0x80000, 0x49f657e8 )

	ROM_REGION( 0x40000, REGION_SOUND1, 0 )	/* Samples */
	ROM_LOAD( "1081.2d",  0x00000, 0x40000, 0x74dede40 )
ROM_END


GAMEX( 1992, mmpanic,  0,        mmpanic,  mmpanic,  0,        ROT0, "Nakanihon + East Technology (Taito license)", "Monkey Mole Panic (USA)",                            GAME_NO_COCKTAIL )
GAMEX( 1993, funkyfig, 0,        funkyfig, funkyfig, 0,        ROT0, "Nakanihon + East Technology (Taito license)", "The First Funky Fighter",                            GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS )	// scrolling, priority?
GAMEX( 1993, quizchq,  0,        quizchq,  quizchq,  0,        ROT0, "Nakanihon",                                   "Quiz Channel Question (Ver 1.00) (Japan)",           GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS | GAME_IMPERFECT_SOUND | GAME_NOT_WORKING )
GAMEX( 1993, quizchql, quizchq,  quizchq,  quizchq,  0,        ROT0, "Nakanihon (Laxan license)",                   "Quiz Channel Question (Ver 1.23) (Taiwan?)",         GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS | GAME_IMPERFECT_SOUND | GAME_NOT_WORKING )
GAMEX( 1993, animaljr, 0,        mmpanic,  animaljr, 0,        ROT0, "Nakanihon + East Technology (Taito license)", "Animalandia Jr.",                                    GAME_NO_COCKTAIL | GAME_IMPERFECT_SOUND )
GAMEX( 1994, hginga,   0,        hginga,   hginga,   0,        ROT0, "Dynax",                                       "Hanafuda Hana Ginga",                                GAME_NO_COCKTAIL )
GAMEX( 1994, mjmyster, 0,        mjmyster, mjmyster, 0,        ROT0, "Dynax",                                       "Mahjong The Mysterious World",                       GAME_NO_COCKTAIL )
GAMEX( 1994, mjmywrld, mjmyster, mjmywrld, mjmyster, 0,        ROT0, "Dynax",                                       "Mahjong The Mysterious World (set 2)",               GAME_NO_COCKTAIL )
GAMEX( 1994, mjmyornt, 0,        mjmyornt, mjmyornt, 0,        ROT0, "Dynax",                                       "Mahjong The Mysterious Orient",                      GAME_NO_COCKTAIL )
GAMEX( 1994, mjmyuniv, 0,        mjmyuniv, mjmyster, 0,        ROT0, "Dynax",                                       "Mahjong The Mysterious Universe",                    GAME_NO_COCKTAIL )
GAMEX( 1994, quiz365,  0,        quiz365,  quiz365,  0,        ROT0, "Nakanihon",                                   "Quiz 365 (Japan)",                                   GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS | GAME_UNEMULATED_PROTECTION )
GAMEX( 1994, quiz365t, quiz365,  quiz365,  quiz365,  0,        ROT0, "Nakanihon + Taito",                           "Quiz 365 (Hong Kong & Taiwan)",                      GAME_NO_COCKTAIL | GAME_IMPERFECT_GRAPHICS | GAME_UNEMULATED_PROTECTION )
GAMEX( 1994, rongrong, 0,        rongrong, rongrong, rongrong, ROT0, "Nakanihon (Activision licence)",              "Puzzle Game Rong Rong (Europe)",                     GAME_NO_COCKTAIL | GAME_IMPERFECT_COLORS )
GAMEX( 1994, rongrngj, rongrong, rongrong, rongrong, rongrong, ROT0, "Nakanihon (Activision licence)",              "Puzzle Game Rong Rong (Japan)",                      GAME_NO_COCKTAIL | GAME_IMPERFECT_COLORS )
GAMEX( 1994, rongrngg, rongrong, rongrong, rongrong, rongrong, ROT0, "Nakanihon (Activision licence)",              "Puzzle Game Rong Rong (Germany)",                    GAME_NO_COCKTAIL | GAME_IMPERFECT_COLORS )
GAMEX( 1994, hparadis, 0,        hparadis, hparadis, 0,        ROT0, "Dynax",                                       "Super Hana Paradise (Japan)",                        GAME_NO_COCKTAIL )
GAMEX( 1995, hgokou,   0,        hgokou,   hgokou,   0,        ROT0, "Dynax (Alba licence)",					       "Hanafuda Hana Gokou (Japan)",                        GAME_NO_COCKTAIL )
GAMEX( 1995, mjdchuka, 0,        mjchuuka, mjchuuka, 0,        ROT0, "Dynax",                                       "Mahjong The Dai Chuuka Ken (China, v. D111)",        GAME_NO_COCKTAIL )
GAMEX( 1995, nettoqc,  0,        nettoqc,  nettoqc,  0,        ROT0, "Nakanihon",                                   "Nettoh Quiz Champion (Japan)",                       GAME_NO_COCKTAIL | GAME_IMPERFECT_COLORS )
GAMEX( 1995, ddenlvrj, 0,        ddenlvrj, ddenlvrj, 0,        ROT0, "Dynax",                                       "Don Den Lover Vol. 1 - Shiro Kuro Tsukeyo! (Japan)", GAME_NO_COCKTAIL )
GAMEX( 1996, ddenlovr, ddenlvrj, ddenlovr, ddenlovr, 0,        ROT0, "Dynax",                                       "Don Den Lover Vol. 1 (Hong Kong)",                   GAME_NO_COCKTAIL )
GAMEX( 1996, ddenlovb, ddenlvrj, ddenlovr, ddenlovr, 0,        ROT0, "[Dynax] (bootleg)",                           "Don Den Lover Vol. 1 (Korea, bootleg)",              GAME_NO_COCKTAIL )
GAMEX( 1996, hanakanz, 0,        hanakanz, hanakanz, 0,        ROT0, "Dynax",                                       "Hana Kanzashi (Japan)",                              GAME_NO_COCKTAIL )
GAMEX( 1996, akamaru,  0,        akamaru,  akamaru,  0,        ROT0, "Dynax (Nakanihon licence)",                   "Panel & Variety Akamaru Q Jousyou Dont-R",           GAME_NO_COCKTAIL )
GAMEX( 1996, mjflove,  0,        mjflove,  mjflove,  0,        ROT0, "Nakanihon",                                   "Mahjong Fantasic Love (Japan)",                      GAME_NO_COCKTAIL )
GAMEX( 1997, hkagerou, 0,        hkagerou, hkagerou, 0,        ROT0, "Nakanihon + Dynax",                           "Hana Kagerou [BET] (Japan)",                         GAME_NO_COCKTAIL )
GAMEX( 1998, mjchuuka, 0,        mjchuuka, mjchuuka, 0,        ROT0, "Dynax",                                       "Mahjong Chuukanejyo (China)",                        GAME_NO_COCKTAIL )
GAMEX( 1998, mjreach1, 0,        mjreach1, mjreach1, 0,        ROT0, "Nihon System",                                "Mahjong Reach Ippatsu (Japan)",                      GAME_NO_COCKTAIL )
