/***************************************************************************

	Taito Qix hardware

	driver by John Butler, Ed Mueller, Aaron Giles

	Games supported:
		* Qix (3 sets)
		* Qix II Tournament
		* Zookeeper (3 sets)
		* Space Dungeon
		* Kram (2 sets)
		* The Electric Yo-Yo (2 sets)
		* Slither (2 sets)
		* Complex X

	Known bugs:
		* none at this time

	Notes:
		* the encrypted version of Kram doesn't have the protection MCU

****************************************************************************

	Memory map

****************************************************************************

	========================================================================
	CPU #1 (all but Zookeeper)
	========================================================================
	8000-83FF   R/W   xxxxxxxx    Shared RAM with CPU #2
	8400-87FF   R/W   xxxxxxxx    Local RAM
	8800        R/W   xxxxxxxx    ACIA base address
	8C00        R/W   --------    Video FIRQ assert
	8C01        R/W   --------    Data FIRQ acknowledge
	9000-93FF   R/W   xxxxxxxx    Sound control/data IRQ PIA (pia_3)
	9400-97FF   R/W   xxxxxxxx    Coin/P1 inputs PIA (pia_0)
	9800-9BFF   R/W   xxxxxxxx    Spare/P2 inputs PIA (pia_1)
	9C00-9FFF   R/W   xxxxxxxx    P2 inputs/coin control PIA (pia_2)
	A000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by pia_3
		FIRQ generated by CPU #2
	========================================================================

	========================================================================
	CPU #1 (Zookeeper)
	========================================================================
	0000-03FF   R/W   xxxxxxxx    Shared RAM with CPU #2
	0400-07FF   R/W   xxxxxxxx    Local RAM
	0800        R/W   xxxxxxxx    ACIA base address
	0C00        R/W   --------    Video FIRQ assert
	0C01        R/W   --------    Data FIRQ acknowledge
	1000-13FF   R/W   xxxxxxxx    Sound control/data IRQ PIA (pia_3)
	1400-17FF   R/W   xxxxxxxx    Coin/P1 inputs PIA (pia_0)
	1800-1BFF   R/W   xxxxxxxx    Spare/P2 inputs PIA (pia_1)
	1C00-1FFF   R/W   xxxxxxxx    P2 inputs/coin control PIA (pia_2)
	8000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by pia_3
		FIRQ generated by CPU #2
	========================================================================

	========================================================================
	CPU #2 (all)
	========================================================================
	0000-7FFF   R/W   xxxxxxxx    Direct video RAM access
	8000-83FF   R/W   xxxxxxxx    Shared RAM with CPU #1
	8400-87FF   R/W   xxxxxxxx    CMOS NVRAM
	8800          W   xxxxxxxx    LED/Palette banking
	                  xxxxxx--       (LED control)
	                  ------xx       (palette bank select)
	8801          W   ------xx    Banked ROM select (Zookeeper only)
	8C00        R/W   --------    Data FIRQ assert
	8C01        R/W   --------    Video FIRQ acknowledge
	9000-93FF   R/W   xxxxxxxx    Palette RAM
	                  xx------       (red component)
	                  --xx----       (green component)
	                  ----xx--       (blue component)
	                  ------xx       (intensity)
	9400        R/W   xxxxxxxx    Video RAM latched access
	9401        R/W   xxxxxxxx    Video RAM access mask (Slither only)
	9402-9403     W   xxxxxxxx    Video RAM address latch
	9800        R     xxxxxxxx    Current scanline
	9C00-9FFF   R/W   xxxxxxxx    68A45 video controller
	A000-BFFF   R     xxxxxxxx    Banked ROM (Zookeeper only)
	C000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ not connected
		FIRQ generated by CPU #1
	========================================================================

	========================================================================
	CPU #3 (all but Slither)
	========================================================================
	0000-007F   R/W   xxxxxxxx    Program RAM
	2000-2003   R/W   xxxxxxxx    TMS5200 control PIA (pia_5; not used)
	4000-4003   R/W   xxxxxxxx    Sound control/DAC PIA (pia_4)
	D000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by pia_4
	========================================================================

****************************************************************************

	Qix uses two 6809 CPUs: one for data and sound and the other for
	video. Communication between the two CPUs is done using a 4K RAM
	space at $8000 (for ZooKeeper the data cpu maps it at $0000 and the
	video cpu at $8000) which both CPUs have direct access. FIRQs (fast
	interrupts) are generated by each CPU to interrupt the other at
	specific times.

	A third CPU, a 6802, is used for sample playback. It drives an 8-bit
	DAC and according to the schematics a TMS5220 speech chip, which is
	never accessed. ROM u27 is the only code needed. A sound command from
	the data CPU causes an IRQ to fire on the 6802 and the sound playback
	is started.

	The coin door switches and player controls are connected to the CPUs
	by Mototola 6821 PIAs. These devices are memory mapped as shown below.

	The screen is 256x256 with eight bit pixels (64K). The screen is
	divided into two halves each half mapped by the video CPU at
	$0000-$7FFF. The high order bit of the address latch at $9402
	specifies which half of the screen is being accessed.

	Timing is critical in the hardware. Communications between the data,
	video, and sound CPUs are all very sensitive.

	The address latch works as follows. When the video CPU accesses $9400,
	the screen address is computed by using the values at $9402 (high byte)
	and $9403 (low byte) to get a value between $0000-$FFFF. The value at
	that location is either returned or written.

	The scan line at $9800 on the video CPU records where the scan line is
	on the display (0-255). Several places in the ROM code wait until the
	scan line reaches zero before continuing.

****************************************************************************

	QIX NONVOLATILE CMOS MEMORY MAP (CPU #2 -- Video) $8400-$87ff
		$86A9 - $86AA:	When CMOS is valid, these bytes are $55AA
		$86AC - $86C3:	AUDIT TOTALS -- 4 4-bit BCD digits per setting
						(All totals default to: 0000)
						$86AC: TOTAL PAID CREDITS
						$86AE: LEFT COINS
						$86B0: CENTER COINS
						$86B2: RIGHT COINS
						$86B4: PAID CREDITS
						$86B6: AWARDED CREDITS
						$86B8: % FREE PLAYS
						$86BA: MINUTES PLAYED
						$86BC: MINUTES AWARDED
						$86BE: % FREE TIME
						$86C0: AVG. GAME [SEC]
						$86C2: HIGH SCORES
		$86C4 - $86FF:	High scores -- 10 scores/names, consecutive in memory
						Six 4-bit BCD digits followed by 3 ascii bytes
						(Default: 030000 QIX)
		$8700		 :	LANGUAGE SELECT (Default: $32)
						ENGLISH = $32  FRANCAIS = $33  ESPANOL = $34  DEUTSCH = $35
		$87D9 - $87DF:	COIN SLOT PROGRAMMING -- 2 4-bit BCD digits per setting
						$87D9: STANDARD COINAGE SETTING  (Default: 01)
						$87DA: COIN MULTIPLIERS LEFT (Default: 01)
						$87DB: COIN MULTIPLIERS CENTER (Default: 04)
						$87DC: COIN MULTIPLIERS RIGHT (Default: 01)
						$87DD: COIN UNITS FOR CREDIT (Default: 01)
						$87DE: COIN UNITS FOR BONUS (Default: 00)
						$87DF: MINIMUM COINS (Default: 00)
		$87E0 - $87EA:	LOCATION PROGRAMMING -- 2 4-bit BCD digits per setting
						$87E0: BACKUP HSTD [0000] (Default: 03)
						$87E1: MAXIMUM CREDITS (Default: 10)
						$87E2: NUMBER OF TURNS (Default: 03)
						$87E3: THRESHOLD (Default: 75)
						$87E4: TIME LINE (Default: 37)
						$87E5: DIFFICULTY 1 (Default: 01)
						$87E6: DIFFICULTY 2 (Default: 01)
						$87E7: DIFFICULTY 3 (Default: 01)
						$87E8: DIFFICULTY 4 (Default: 01)
						$87E9: ATTRACT SOUND (Default: 01)
						$87EA: TABLE MODE (Default: 00)

***************************************************************************/

#include "driver.h"
#include "qix.h"
#include "machine/6821pia.h"
#include "vidhrdw/generic.h"


/* Constants */
#define MAIN_CLOCK_OSC		20000000	/* 20 MHz */
#define SLITHER_CLOCK_OSC	21300000	/* 21.3 MHz */
#define SOUND_CLOCK_OSC		7372800		/* 7.3728 MHz */
#define COIN_CLOCK_OSC		4000000		/* 4 MHz */



/*************************************
 *
 *	Data CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( readmem_data )
	{ 0x8000, 0x83ff, qix_sharedram_r },
	{ 0x8400, 0x87ff, MRA_RAM },
	{ 0x8800, 0x8800, MRA_NOP },   /* ACIA */
	{ 0x8c00, 0x8c00, qix_video_firq_r },
	{ 0x8c01, 0x8c01, qix_data_firq_ack_r },
	{ 0x9000, 0x93ff, pia_3_r },
	{ 0x9400, 0x97ff, pia_0_r },
	{ 0x9800, 0x9bff, pia_1_r },
	{ 0x9c00, 0x9fff, pia_2_r },
	{ 0xa000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( writemem_data )
	{ 0x8000, 0x83ff, qix_sharedram_w, &qix_sharedram },
	{ 0x8400, 0x87ff, MWA_RAM },
	{ 0x8c00, 0x8c00, qix_video_firq_w },
	{ 0x8c01, 0x8c01, qix_data_firq_ack_w },
	{ 0x9000, 0x93ff, pia_3_w },
	{ 0x9400, 0x97ff, qix_pia_0_w },
	{ 0x9800, 0x9bff, pia_1_w },
	{ 0x9c00, 0x9fff, pia_2_w },
	{ 0xa000, 0xffff, MWA_ROM },
MEMORY_END


static MEMORY_READ_START( zoo_readmem_data )
	{ 0x0000, 0x03ff, qix_sharedram_r },
	{ 0x0400, 0x07ff, MRA_RAM },
	{ 0x0800, 0x0800, MRA_NOP },   /* ACIA */
	{ 0x0c00, 0x0c00, qix_video_firq_r },
	{ 0x0c01, 0x0c01, qix_data_firq_ack_r },
	{ 0x1000, 0x13ff, pia_3_r },
	{ 0x1400, 0x17ff, pia_0_r },
	{ 0x1900, 0x1bff, pia_1_r },
	{ 0x1c00, 0x1fff, pia_2_r },
	{ 0x8000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( zoo_writemem_data )
	{ 0x0000, 0x03ff, qix_sharedram_w, &qix_sharedram },
	{ 0x0400, 0x07ff, MWA_RAM },
	{ 0x0c00, 0x0c00, qix_video_firq_w },
	{ 0x0c01, 0x0c01, qix_data_firq_ack_w },
	{ 0x1000, 0x13ff, pia_3_w },
	{ 0x1400, 0x17ff, qix_pia_0_w },
	{ 0x1900, 0x1bff, pia_1_w },
	{ 0x1c00, 0x1fff, pia_2_w },
	{ 0x8000, 0xffff, MWA_ROM },
MEMORY_END



/*************************************
 *
 *	Video CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( readmem_video )
	{ 0x0000, 0x7fff, qix_videoram_r },
	{ 0x8000, 0x83ff, qix_sharedram_r },
	{ 0x8400, 0x87ff, MRA_RAM },
	{ 0x8c00, 0x8c00, qix_data_firq_r },
	{ 0x8c01, 0x8c01, qix_video_firq_ack_r },
	{ 0x9000, 0x93ff, MRA_RAM },
	{ 0x9400, 0x9400, qix_addresslatch_r },
	{ 0x9800, 0x9800, qix_scanline_r },
	{ 0xa000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( writemem_video )
	{ 0x0000, 0x7fff, qix_videoram_w },
	{ 0x8000, 0x83ff, qix_sharedram_w },
	{ 0x8400, 0x87ff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x8800, 0x8800, qix_palettebank_w },
	{ 0x8c00, 0x8c00, qix_data_firq_w },
	{ 0x8c01, 0x8c01, qix_video_firq_ack_w },
	{ 0x9000, 0x93ff, qix_paletteram_w, &paletteram },
	{ 0x9400, 0x9400, qix_addresslatch_w },
	{ 0x9402, 0x9403, MWA_RAM, &qix_videoaddress },
	{ 0x9c00, 0x9fff, MWA_RAM }, /* Video controller */
	{ 0xa000, 0xffff, MWA_ROM },
MEMORY_END


static MEMORY_READ_START( zoo_readmem_video )
	{ 0x0000, 0x7fff, qix_videoram_r },
	{ 0x8000, 0x83ff, qix_sharedram_r },
	{ 0x8400, 0x87ff, MRA_RAM },
	{ 0x8c00, 0x8c00, qix_data_firq_r },
	{ 0x8c01, 0x8c01, qix_video_firq_ack_r },
	{ 0x9000, 0x93ff, MRA_RAM },
	{ 0x9400, 0x9400, qix_addresslatch_r },
	{ 0x9800, 0x9800, qix_scanline_r },
	{ 0xa000, 0xbfff, MRA_BANK1 },
	{ 0xc000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( zoo_writemem_video )
	{ 0x0000, 0x7fff, qix_videoram_w },
	{ 0x8000, 0x83ff, qix_sharedram_w },
	{ 0x8400, 0x87ff, MWA_RAM, &generic_nvram, &generic_nvram_size },
	{ 0x8800, 0x8800, qix_palettebank_w },
	{ 0x8801, 0x8801, zoo_bankswitch_w },
	{ 0x8c00, 0x8c00, qix_data_firq_w },
	{ 0x8c01, 0x8c01, qix_video_firq_ack_w },
	{ 0x9000, 0x93ff, qix_paletteram_w, &paletteram },
	{ 0x9400, 0x9400, qix_addresslatch_w },
	{ 0x9402, 0x9403, MWA_RAM, &qix_videoaddress },
	{ 0x9c00, 0x9fff, MWA_RAM }, /* Video controller */
	{ 0xa000, 0xffff, MWA_ROM },
MEMORY_END



/*************************************
 *
 *	Sound CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( readmem_sound )
	{ 0x0000, 0x007f, MRA_RAM },
	{ 0x2000, 0x2003, pia_5_r },
	{ 0x4000, 0x4003, pia_4_r },
	{ 0xd000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( writemem_sound )
	{ 0x0000, 0x007f, MWA_RAM },
	{ 0x2000, 0x2003, pia_5_w },
	{ 0x4000, 0x4003, pia_4_w },
	{ 0xd000, 0xffff, MWA_ROM },
MEMORY_END



/*************************************
 *
 *	Coin CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( mcu_readmem )
	{ 0x0000, 0x0000, qix_68705_portA_r },
	{ 0x0001, 0x0001, qix_68705_portB_r },
	{ 0x0002, 0x0002, qix_68705_portC_r },
	{ 0x0010, 0x007f, MRA_RAM },
	{ 0x0080, 0x07ff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( mcu_writemem )
	{ 0x0000, 0x0000, qix_68705_portA_w, &qix_68705_port_out },
	{ 0x0001, 0x0001, qix_68705_portB_w },
	{ 0x0002, 0x0002, qix_68705_portC_w },
	{ 0x0004, 0x0007, MWA_RAM, &qix_68705_ddr },
	{ 0x0010, 0x007f, MWA_RAM },
	{ 0x0080, 0x07ff, MWA_ROM },
MEMORY_END



/*************************************
 *
 *	Input port definitions
 *
 *************************************/

#define COIN_PORT \
	PORT_BITX(0x01, IP_ACTIVE_LOW, IPT_SERVICE, "Test Advance",   KEYCODE_F1, IP_JOY_DEFAULT ) \
	PORT_BITX(0x02, IP_ACTIVE_LOW, IPT_SERVICE, "Test Next line", KEYCODE_F2, IP_JOY_DEFAULT ) \
	PORT_BITX(0x04, IP_ACTIVE_LOW, IPT_SERVICE, "Test Slew Up",   KEYCODE_F5, IP_JOY_DEFAULT ) \
	PORT_BITX(0x08, IP_ACTIVE_LOW, IPT_SERVICE, "Test Slew Down", KEYCODE_F6, IP_JOY_DEFAULT ) \
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN1 ) \
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2 ) \
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 ) \
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_TILT )


INPUT_PORTS_START( qix )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_COCKTAIL )
    PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNKNOWN )
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
INPUT_PORTS_END


INPUT_PORTS_START( sdungeon )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP     | IPF_8WAY )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT  | IPF_8WAY )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN   | IPF_8WAY )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT   | IPF_8WAY )
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP    | IPF_8WAY )
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT | IPF_8WAY )
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN  | IPF_8WAY )
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT  | IPF_8WAY )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2 )
    PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP     | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT  | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN   | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT   | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP    | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN  | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT  | IPF_8WAY | IPF_COCKTAIL )
INPUT_PORTS_END


INPUT_PORTS_START( elecyoyo )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
    PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( kram )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_8WAY | IPF_COCKTAIL )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_8WAY | IPF_COCKTAIL )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_COCKTAIL )
    PORT_BIT( 0x60, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
INPUT_PORTS_END


INPUT_PORTS_START( zookeep )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (PLAYER 2) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_4WAY | IPF_COCKTAIL )
	PORT_BIT( 0x30, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )
INPUT_PORTS_END


INPUT_PORTS_START( slither )
	PORT_START	/* PIA 0 Port A (PLAYER 1) */
    PORT_BIT( 0x0f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START	/* PIA 0 Port B (COIN) */
	PORT_SERVICE(0x01, IP_ACTIVE_LOW )
    PORT_BIT( 0x0e, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_TILT )

	PORT_START	/* PIA 3 Port A (PLAYER 2) */
    PORT_BIT( 0x0f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_COCKTAIL )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2  | IPF_COCKTAIL )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1  | IPF_COCKTAIL )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_COCKTAIL )

	PORT_START	/* PIA 1 Port A (TRACKBALL L/R) */
    PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_X, 50, 10, 0x00, 0xff )

	PORT_START	/* PIA 2 Port A (TRACKBALL U/D) */
    PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_Y | IPF_REVERSE, 50, 10, 0x00, 0xff )

	PORT_START	/* PIA 1 Port A (TRACKBALL L/R) */
    PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_X | IPF_COCKTAIL, 50, 10, 0x00, 0xff )

	PORT_START	/* PIA 2 Port A (TRACKBALL U/D) */
    PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_Y | IPF_REVERSE | IPF_COCKTAIL, 50, 10, 0x00, 0xff )
INPUT_PORTS_END


INPUT_PORTS_START( complexx )
	PORT_START	/* PIA 0 Port A (Left Stick) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP     | IPF_8WAY )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_RIGHT  | IPF_8WAY )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN   | IPF_8WAY )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_LEFT   | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

/* 0x80 isn't scanned unless the difficulty in the game is below 4. I
decided not to map this button, so if you set the difficulty that low,
you can't fire. If I do map that button, and you set difficulty below 4
and hit this fire button without pressing the second stick in any
direction it fires up. If you're pressing left it fires right, press
right it fires left and press down it fires down. That's just too wacky.
I gotta think it's not supposed to be hooked up at all and if the
difficulty is that low, you just can't shoot, so I think it should stay
as-is. Tim Lindquist 1-17-03 */

	PORT_START	/* PIA 0 Port B (COIN) */
	COIN_PORT

	PORT_START	/* PIA 1 Port A (SPARE) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

    PORT_START	/* PIA 1 Port B (PLAYER 1/2) */
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START	/* PIA 2 Port A (Right Stick) */
    PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_UNUSED )
    PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP    | IPF_8WAY )
    PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_RIGHT | IPF_8WAY )
    PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN  | IPF_8WAY )
    PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_LEFT  | IPF_8WAY )
INPUT_PORTS_END



/*************************************
 *
 *	Sound interfaces
 *
 *************************************/

static struct DACinterface dac_interface =
{
	1,
	{ 100 }
};


static struct SN76496interface sn76496_interface =
{
	2,
	{ SLITHER_CLOCK_OSC/4/4, SLITHER_CLOCK_OSC/4/4 },
	{ 50, 50 }
};



/*************************************
 *
 *	Machine drivers
 *
 *************************************/

static MACHINE_DRIVER_START( qix )

	/* basic machine hardware */
	MDRV_CPU_ADD_TAG("main", M6809, MAIN_CLOCK_OSC/4/4)	/* 1.25 MHz */
	MDRV_CPU_MEMORY(readmem_data,writemem_data)
	MDRV_CPU_VBLANK_INT(qix_vblank_start,1)

	MDRV_CPU_ADD_TAG("video", M6809, MAIN_CLOCK_OSC/4/4)	/* 1.25 MHz */
	MDRV_CPU_MEMORY(readmem_video,writemem_video)

	MDRV_CPU_ADD_TAG("sound", M6802, SOUND_CLOCK_OSC/2/4)	/* 0.92 MHz */
	MDRV_CPU_FLAGS(CPU_AUDIO_CPU)
	MDRV_CPU_MEMORY(readmem_sound,writemem_sound)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)

	MDRV_MACHINE_INIT(qix)
	MDRV_NVRAM_HANDLER(generic_0fill)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER)
	MDRV_SCREEN_SIZE(256, 256)
	MDRV_VISIBLE_AREA(0, 255, 8, 247)
	MDRV_PALETTE_LENGTH(1024)

	MDRV_VIDEO_START(qix)
	MDRV_VIDEO_UPDATE(qix)

	/* sound hardware */
	MDRV_SOUND_ADD_TAG("dac", DAC, dac_interface)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( mcu )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(qix)

	MDRV_CPU_ADD(M68705, COIN_CLOCK_OSC/4)	/* 1.00 MHz */
	MDRV_CPU_MEMORY(mcu_readmem,mcu_writemem)

	MDRV_MACHINE_INIT(qixmcu)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( zookeep )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(mcu)

	MDRV_CPU_MODIFY("main")
	MDRV_CPU_MEMORY(zoo_readmem_data,zoo_writemem_data)

	MDRV_CPU_MODIFY("video")
	MDRV_CPU_MEMORY(zoo_readmem_video,zoo_writemem_video)
MACHINE_DRIVER_END


static MACHINE_DRIVER_START( slither )

	/* basic machine hardware */
	MDRV_IMPORT_FROM(qix)

	MDRV_CPU_REPLACE("main", M6809, SLITHER_CLOCK_OSC/4/4)	/* 1.34 MHz */
	MDRV_CPU_REPLACE("video", M6809, SLITHER_CLOCK_OSC/4/4)	/* 1.34 MHz */
	MDRV_CPU_REMOVE("sound")

	MDRV_MACHINE_INIT(slither)

	/* video hardware */
	MDRV_VISIBLE_AREA(0, 255, 0, 255)

	/* sound hardware */
	MDRV_SOUND_REPLACE("dac", SN76496, sn76496_interface)
MACHINE_DRIVER_END



/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( qix )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u12", 0xc000, 0x0800, 0xaad35508 )
	ROM_LOAD( "u13", 0xc800, 0x0800, 0x46c13504 )
	ROM_LOAD( "u14", 0xd000, 0x0800, 0x5115e896 )
	ROM_LOAD( "u15", 0xd800, 0x0800, 0xccd52a1b )
	ROM_LOAD( "u16", 0xe000, 0x0800, 0xcd1c36ee )
	ROM_LOAD( "u17", 0xe800, 0x0800, 0x1acb682d )
	ROM_LOAD( "u18", 0xf000, 0x0800, 0xde77728b )
	ROM_LOAD( "u19", 0xf800, 0x0800, 0xc0994776 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u4",  0xc800, 0x0800, 0x5b906a09 )
	ROM_LOAD( "u5",  0xd000, 0x0800, 0x254a3587 )
	ROM_LOAD( "u6",  0xd800, 0x0800, 0xace30389 )
	ROM_LOAD( "u7",  0xe000, 0x0800, 0x8ebcfa7c )
	ROM_LOAD( "u8",  0xe800, 0x0800, 0xb8a3c8f9 )
	ROM_LOAD( "u9",  0xf000, 0x0800, 0x26cbcd55 )
	ROM_LOAD( "u10", 0xf800, 0x0800, 0x568be942 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27", 0xf800, 0x0800, 0xf3782bd0 )
ROM_END


ROM_START( qixa )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "qu12", 0xc000, 0x0800, 0x1c55b44d )
	ROM_LOAD( "qu13", 0xc800, 0x0800, 0x20279e8c )
	ROM_LOAD( "qu14", 0xd000, 0x0800, 0xbafe3ce3 )
	/* d800-dfff empty */
	ROM_LOAD( "qu16", 0xe000, 0x0800, 0xdb560753 )
	ROM_LOAD( "qu17", 0xe800, 0x0800, 0x8c7aeed8 )
	ROM_LOAD( "qu18", 0xf000, 0x0800, 0x353be980 )
	ROM_LOAD( "qu19", 0xf800, 0x0800, 0xf46a69ca )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "qu3",  0xc000, 0x0800, 0x8b4c0ef0 )
	ROM_LOAD( "qu4",  0xc800, 0x0800, 0x66a5c260 )
	ROM_LOAD( "qu5",  0xd000, 0x0800, 0x70160ea3 )
	/* d800-dfff empty */
	ROM_LOAD( "qu7",  0xe000, 0x0800, 0xd6733019 )
	ROM_LOAD( "qu8",  0xe800, 0x0800, 0x66870dcc )
	ROM_LOAD( "qu9",  0xf000, 0x0800, 0xc99bf94d )
	ROM_LOAD( "qu10", 0xf800, 0x0800, 0x88b45037 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",  0xf800, 0x0800, 0xf3782bd0 )
ROM_END


ROM_START( qixb )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "lk14.bin", 0xc000, 0x1000, 0x6d164986 )
	ROM_LOAD( "lk15.bin", 0xd000, 0x1000, 0x16c6ce0f )
	ROM_LOAD( "lk16.bin", 0xe000, 0x1000, 0x698b1f9c )
	ROM_LOAD( "lk17.bin", 0xf000, 0x1000, 0x7e3adde6 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "lk10.bin", 0xc000, 0x1000, 0x7eac67d0 )
	ROM_LOAD( "lk11.bin", 0xd000, 0x1000, 0x90ccbb6a )
	ROM_LOAD( "lk12.bin", 0xe000, 0x1000, 0xbe9b9f7d )
	ROM_LOAD( "lk13.bin", 0xf000, 0x1000, 0x51c9853b )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",      0xf800, 0x0800, 0xf3782bd0 )
ROM_END


ROM_START( qix2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u12.rmb", 0xc000, 0x0800, 0x484280fd )
	ROM_LOAD( "u13.rmb", 0xc800, 0x0800, 0x3d089fcb )
	ROM_LOAD( "u14.rmb", 0xd000, 0x0800, 0x362123a9 )
	ROM_LOAD( "u15.rmb", 0xd800, 0x0800, 0x60f3913d )
	ROM_LOAD( "u16.rmb", 0xe000, 0x0800, 0xcc139e34 )
	ROM_LOAD( "u17.rmb", 0xe800, 0x0800, 0xcf31dc49 )
	ROM_LOAD( "u18.rmb", 0xf000, 0x0800, 0x1f91ed7a )
	ROM_LOAD( "u19.rmb", 0xf800, 0x0800, 0x68e8d5a6 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u3.rmb",  0xc000, 0x0800, 0x19cebaca )
	ROM_LOAD( "u4.rmb",  0xc800, 0x0800, 0x6cfb4185 )
	ROM_LOAD( "u5.rmb",  0xd000, 0x0800, 0x948f53f3 )
	ROM_LOAD( "u6.rmb",  0xd800, 0x0800, 0x8630120e )
	ROM_LOAD( "u7.rmb",  0xe000, 0x0800, 0xbad037c9 )
	ROM_LOAD( "u8.rmb",  0xe800, 0x0800, 0x3159bc00 )
	ROM_LOAD( "u9.rmb",  0xf000, 0x0800, 0xe80e9b1d )
	ROM_LOAD( "u10.rmb", 0xf800, 0x0800, 0x9a55d360 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "u27",     0xf800, 0x0800, 0xf3782bd0 )
ROM_END


ROM_START( sdungeon )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "sd14.u14", 0xa000, 0x1000, 0x7024b55a )
	ROM_LOAD( "sd15.u15", 0xb000, 0x1000, 0xa3ac9040 )
	ROM_LOAD( "sd16.u16", 0xc000, 0x1000, 0xcc20b580 )
	ROM_LOAD( "sd17.u17", 0xd000, 0x1000, 0x4663e4b8 )
	ROM_LOAD( "sd18.u18", 0xe000, 0x1000, 0x7ef1ffc0 )
	ROM_LOAD( "sd19.u19", 0xf000, 0x1000, 0x7b20b7ac )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "sd05.u5",  0xa000, 0x1000, 0x0b2bf48e )
	ROM_LOAD( "sd06.u6",  0xb000, 0x1000, 0xf86db512 )
	ROM_LOAD( "sd07.u7",  0xc000, 0x1000, 0x7b796831 )
	ROM_LOAD( "sd08.u8",  0xd000, 0x1000, 0x5fbe7068 )
	ROM_LOAD( "sd09.u9",  0xe000, 0x1000, 0x89bc51ea )
	ROM_LOAD( "sd10.u10", 0xf000, 0x1000, 0x754de734 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "sd26.u26", 0xf000, 0x0800, 0x3df8630d )
	ROM_LOAD( "sd27.u27", 0xf800, 0x0800, 0x0386f351 )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "sd101",    0x0000, 0x0800, 0xe255af9a )
ROM_END


ROM_START( elecyoyo )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "yy14",   0xa000, 0x1000, 0x0d2edcb9 )
    ROM_LOAD( "yy15",   0xb000, 0x1000, 0xa91f01e3 )
    ROM_LOAD( "yy16-1", 0xc000, 0x1000, 0x2710f360 )
    ROM_LOAD( "yy17",   0xd000, 0x1000, 0x25fd489d )
    ROM_LOAD( "yy18",   0xe000, 0x1000, 0x0b6661c0 )
    ROM_LOAD( "yy19-1", 0xf000, 0x1000, 0x95b8b244 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "yy5",    0xa000, 0x1000, 0x3793fec5 )
    ROM_LOAD( "yy6",    0xb000, 0x1000, 0x2e8b1265 )
    ROM_LOAD( "yy7",    0xc000, 0x1000, 0x20f93411 )
    ROM_LOAD( "yy8",    0xd000, 0x1000, 0x926f90c8 )
    ROM_LOAD( "yy9",    0xe000, 0x1000, 0x2f999480 )
    ROM_LOAD( "yy10",   0xf000, 0x1000, 0xb31d20e2 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "yy27",   0xf800, 0x0800, 0x5a2aa0f3 )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "yy101",  0x0000, 0x0800, 0x3cf13038 )
ROM_END


ROM_START( elecyoy2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "yy14",  0xa000, 0x1000, 0x0d2edcb9 )
    ROM_LOAD( "yy15",  0xb000, 0x1000, 0xa91f01e3 )
    ROM_LOAD( "yy16",  0xc000, 0x1000, 0xcab19f3a )
    ROM_LOAD( "yy17",  0xd000, 0x1000, 0x25fd489d )
    ROM_LOAD( "yy18",  0xe000, 0x1000, 0x0b6661c0 )
    ROM_LOAD( "yy19",  0xf000, 0x1000, 0xd0215d2e )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "yy5",   0xa000, 0x1000, 0x3793fec5 )
    ROM_LOAD( "yy6",   0xb000, 0x1000, 0x2e8b1265 )
    ROM_LOAD( "yy7",   0xc000, 0x1000, 0x20f93411 )
    ROM_LOAD( "yy8",   0xd000, 0x1000, 0x926f90c8 )
    ROM_LOAD( "yy9",   0xe000, 0x1000, 0x2f999480 )
    ROM_LOAD( "yy10",  0xf000, 0x1000, 0xb31d20e2 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "yy27",  0xf800, 0x0800, 0x5a2aa0f3 )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "yy101", 0x0000, 0x0800, 0x3cf13038 )
ROM_END


ROM_START( kram )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "ks14-1", 0xa000, 0x1000, 0xfe69ac79 )
    ROM_LOAD( "ks15",   0xb000, 0x1000, 0x4b2c175e )
    ROM_LOAD( "ks16",   0xc000, 0x1000, 0x9500a05d )
    ROM_LOAD( "ks17",   0xd000, 0x1000, 0xc752a3a1 )
    ROM_LOAD( "ks18",   0xe000, 0x1000, 0x79158b03 )
    ROM_LOAD( "ks19-1", 0xf000, 0x1000, 0x759ea6ce )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "ks5",    0xa000, 0x1000, 0x1c472080 )
    ROM_LOAD( "ks6",    0xb000, 0x1000, 0xb8926622 )
    ROM_LOAD( "ks7",    0xc000, 0x1000, 0xc98a7485 )
    ROM_LOAD( "ks8",    0xd000, 0x1000, 0x1127c4e4 )
    ROM_LOAD( "ks9",    0xe000, 0x1000, 0xd3bc8b5e )
    ROM_LOAD( "ks10",   0xf000, 0x1000, 0xe0426444 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27",   0xf800, 0x0800, 0xc46530c8 )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "ks101.dat", 0x0000, 0x0800, 0xe53d97b7 )
ROM_END


ROM_START( kram2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "ks14", 0xa000, 0x1000, 0xa2eac1ff )
    ROM_LOAD( "ks15", 0xb000, 0x1000, 0x4b2c175e )
    ROM_LOAD( "ks16", 0xc000, 0x1000, 0x9500a05d )
    ROM_LOAD( "ks17", 0xd000, 0x1000, 0xc752a3a1 )
    ROM_LOAD( "ks18", 0xe000, 0x1000, 0x79158b03 )
    ROM_LOAD( "ks19", 0xf000, 0x1000, 0x053c5e09 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "ks5",  0xa000, 0x1000, 0x1c472080 )
    ROM_LOAD( "ks6",  0xb000, 0x1000, 0xb8926622 )
    ROM_LOAD( "ks7",  0xc000, 0x1000, 0xc98a7485 )
    ROM_LOAD( "ks8",  0xd000, 0x1000, 0x1127c4e4 )
    ROM_LOAD( "ks9",  0xe000, 0x1000, 0xd3bc8b5e )
    ROM_LOAD( "ks10", 0xf000, 0x1000, 0xe0426444 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27", 0xf800, 0x0800, 0xc46530c8 )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "ks101.dat", 0x0000, 0x0800, 0xe53d97b7 )
ROM_END


ROM_START( kram3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
    ROM_LOAD( "kr-u14", 0xa000, 0x1000, 0x02c1bd1e )
    ROM_LOAD( "kr-u15", 0xb000, 0x1000, 0x46b3ff33 )
    ROM_LOAD( "kr-u16", 0xc000, 0x1000, 0xf202b9cf )
    ROM_LOAD( "kr-u17", 0xd000, 0x1000, 0x257cea23 )
    ROM_LOAD( "kr-u18", 0xe000, 0x1000, 0xda3aed8c )
    ROM_LOAD( "kr-u19", 0xf000, 0x1000, 0x496ab571 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
    ROM_LOAD( "kr-u5",  0xa000, 0x1000, 0x9e63c2bc )
    ROM_LOAD( "kr-u6",  0xb000, 0x1000, 0xa0ff1244 )
    ROM_LOAD( "kr-u7",  0xc000, 0x1000, 0x20a15024 )
    ROM_LOAD( "ks8",    0xd000, 0x1000, 0x1127c4e4 )
    ROM_LOAD( "ks9",    0xe000, 0x1000, 0xd3bc8b5e )
    ROM_LOAD( "kr-u10", 0xf000, 0x1000, 0x0a8adbd8 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
    ROM_LOAD( "ks27",   0xf800, 0x0800, 0xc46530c8 )
ROM_END


ROM_START( zookeep )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12", 0x8000, 0x1000, 0x4e40d8dc )
	ROM_LOAD( "za13", 0x9000, 0x1000, 0xeebd5248 )
	ROM_LOAD( "za14", 0xa000, 0x1000, 0xfab43297 )
	ROM_LOAD( "za15", 0xb000, 0x1000, 0xef8cd67c )
	ROM_LOAD( "za16", 0xc000, 0x1000, 0xccfc15bc )
	ROM_LOAD( "za17", 0xd000, 0x1000, 0x358013f4 )
	ROM_LOAD( "za18", 0xe000, 0x1000, 0x37886afe )
	ROM_LOAD( "za19", 0xf000, 0x1000, 0xbbfb30d9 )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",  0x0a000, 0x1000, 0xdc0c3cbd )
	ROM_LOAD( "za3",  0x10000, 0x1000, 0xcc4d0aee )
	ROM_LOAD( "za6",  0x0b000, 0x1000, 0x27c787dd )
	ROM_LOAD( "za4",  0x11000, 0x1000, 0xec3b10b1 )

	ROM_LOAD( "za7",  0x0c000, 0x1000, 0x1479f480 )
	ROM_LOAD( "za8",  0x0d000, 0x1000, 0x4c96cdb2 )
	ROM_LOAD( "za9",  0x0e000, 0x1000, 0xa4f7d9e0 )
	ROM_LOAD( "za10", 0x0f000, 0x1000, 0x05df1a5a )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25", 0xd000, 0x1000, 0x779b8558 )
	ROM_LOAD( "za26", 0xe000, 0x1000, 0x60a810ce )
	ROM_LOAD( "za27", 0xf000, 0x1000, 0x99ed424e )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, 0x364d3557 )
ROM_END


ROM_START( zookeep2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12",     0x8000, 0x1000, 0x4e40d8dc )
	ROM_LOAD( "za13",     0x9000, 0x1000, 0xeebd5248 )
	ROM_LOAD( "za14",     0xa000, 0x1000, 0xfab43297 )
	ROM_LOAD( "za15",     0xb000, 0x1000, 0xef8cd67c )
	ROM_LOAD( "za16",     0xc000, 0x1000, 0xccfc15bc )
	ROM_LOAD( "za17",     0xd000, 0x1000, 0x358013f4 )
	ROM_LOAD( "za18",     0xe000, 0x1000, 0x37886afe )
	ROM_LOAD( "za19.red", 0xf000, 0x1000, 0xec01760e )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",      0x0a000, 0x1000, 0xdc0c3cbd )
	ROM_LOAD( "za3",      0x10000, 0x1000, 0xcc4d0aee )
	ROM_LOAD( "za6",      0x0b000, 0x1000, 0x27c787dd )
	ROM_LOAD( "za4",      0x11000, 0x1000, 0xec3b10b1 )

	ROM_LOAD( "za7",      0x0c000, 0x1000, 0x1479f480 )
	ROM_LOAD( "za8",      0x0d000, 0x1000, 0x4c96cdb2 )
	ROM_LOAD( "za9",      0x0e000, 0x1000, 0xa4f7d9e0 )
	ROM_LOAD( "za10",     0x0f000, 0x1000, 0x05df1a5a )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25",     0xd000, 0x1000, 0x779b8558 )
	ROM_LOAD( "za26",     0xe000, 0x1000, 0x60a810ce )
	ROM_LOAD( "za27",     0xf000, 0x1000, 0x99ed424e )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, 0x364d3557 )
ROM_END


ROM_START( zookeep3 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "za12",    0x8000, 0x1000, 0x4e40d8dc )
	ROM_LOAD( "za13",    0x9000, 0x1000, 0xeebd5248 )
	ROM_LOAD( "za14",    0xa000, 0x1000, 0xfab43297 )
	ROM_LOAD( "za15",    0xb000, 0x1000, 0xef8cd67c )
	ROM_LOAD( "za16",    0xc000, 0x1000, 0xccfc15bc )
	ROM_LOAD( "za17",    0xd000, 0x1000, 0x358013f4 )
	ROM_LOAD( "za18",    0xe000, 0x1000, 0x37886afe )
	ROM_LOAD( "za19",    0xf000, 0x1000, 0xbbfb30d9 )

	ROM_REGION( 0x12000, REGION_CPU2, 0 )
	ROM_LOAD( "za5",     0x0a000, 0x1000, 0xdc0c3cbd )
	ROM_LOAD( "za3",     0x10000, 0x1000, 0xcc4d0aee )
	ROM_LOAD( "za6",     0x0b000, 0x1000, 0x27c787dd )
	ROM_LOAD( "za4",     0x11000, 0x1000, 0xec3b10b1 )

	ROM_LOAD( "za7",     0x0c000, 0x1000, 0x1479f480 )
	ROM_LOAD( "za8",     0x0d000, 0x1000, 0x4c96cdb2 )
	ROM_LOAD( "zv35.9",  0x0e000, 0x1000, 0xd14123b7 )
	ROM_LOAD( "zv36.10", 0x0f000, 0x1000, 0x23705777 )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "za25",    0xd000, 0x1000, 0x779b8558 )
	ROM_LOAD( "za26",    0xe000, 0x1000, 0x60a810ce )
	ROM_LOAD( "za27",    0xf000, 0x1000, 0x99ed424e )

	ROM_REGION( 0x0800, REGION_CPU4, 0 )
	ROM_LOAD( "za_coin.bin", 0x0000, 0x0800, 0x364d3557 )
ROM_END


ROM_START( slither )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u31.cpu", 0xd800, 0x0800, 0x3dfff970 )
	ROM_LOAD( "u30.cpu", 0xe000, 0x0800, 0x8cbc5af8 )
	ROM_LOAD( "u29.cpu", 0xe800, 0x0800, 0x98c14510 )
	ROM_LOAD( "u28.cpu", 0xf000, 0x0800, 0x2762f612 )
	ROM_LOAD( "u27.cpu", 0xf800, 0x0800, 0x9306d5b1 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u41.cpu", 0xd000, 0x0800, 0xe4c60a57 )
	ROM_LOAD( "u40.cpu", 0xd800, 0x0800, 0x5dcec622 )
	ROM_LOAD( "u39.cpu", 0xe000, 0x0800, 0x69829c2a )
	ROM_LOAD( "u38.cpu", 0xe800, 0x0800, 0x6adc64c6 )
	ROM_LOAD( "u37.cpu", 0xf000, 0x0800, 0x55d31c96 )
	ROM_LOAD( "u36.cpu", 0xf800, 0x0800, 0xd5ffc013 )
ROM_END


ROM_START( slithera )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "u31.cpu", 0xd800, 0x0800, 0x3dfff970 )
	ROM_LOAD( "u30.cpu", 0xe000, 0x0800, 0x8cbc5af8 )
	ROM_LOAD( "u29.cpu", 0xe800, 0x0800, 0x98c14510 )
	ROM_LOAD( "u28.cpu", 0xf000, 0x0800, 0x2762f612 )
	ROM_LOAD( "u27.cpu", 0xf800, 0x0800, 0x9306d5b1 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "u41.cpu", 0xd000, 0x0800, 0xe4c60a57 )
	ROM_LOAD( "u40.cpu", 0xd800, 0x0800, 0x5dcec622 )
	ROM_LOAD( "u39.cpu", 0xe000, 0x0800, 0x69829c2a )
	ROM_LOAD( "u38a.cpu",0xe800, 0x0800, 0x423adfef )
	ROM_LOAD( "u37.cpu", 0xf000, 0x0800, 0x55d31c96 )
	ROM_LOAD( "u36a.cpu",0xf800, 0x0800, 0x5ac4e244 )
ROM_END

ROM_START( complexx )
	ROM_REGION( 0x10000, REGION_CPU1, 0 )
	ROM_LOAD( "cx14.bin", 0xa000, 0x1000, 0xf123a0de )
	ROM_LOAD( "cx15.bin", 0xb000, 0x1000, 0x0fcb966f )
	ROM_LOAD( "cx16.bin", 0xc000, 0x1000, 0xaa11e0e3 )
	ROM_LOAD( "cx17.bin", 0xd000, 0x1000, 0xf610856e )
	ROM_LOAD( "cx18.bin", 0xe000, 0x1000, 0x8f8c3984 )
	ROM_LOAD( "cx19.bin", 0xf000, 0x1000, 0x13af3ba8 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "cx5.bin",  0xa000, 0x1000, 0x62a2b87b )
	ROM_LOAD( "cx6.bin",  0xb000, 0x1000, 0xdfa7c088 )
	ROM_LOAD( "cx7.bin",  0xc000, 0x1000, 0xc8bd6759 )
	ROM_LOAD( "cx8.bin",  0xd000, 0x1000, 0x14a57221 )
	ROM_LOAD( "cx9.bin",  0xe000, 0x1000, 0xfc2d4a9f )
	ROM_LOAD( "cx10.bin", 0xf000, 0x1000, 0x96e0c1ad )

	ROM_REGION( 0x10000, REGION_CPU3, 0 )
	ROM_LOAD( "cx26.bin",    0xe000, 0x1000, 0xf4f19c9b )
	ROM_LOAD( "cx27.bin",    0xf000, 0x1000, 0x7e177328 )
ROM_END




/*************************************
 *
 *	Game-specific initialization
 *
 *************************************/

static DRIVER_INIT( kram )
{
	/* we need to override one PIA handler to prevent controls from getting disabled */
	install_mem_write_handler(0, 0x9400, 0x97ff, zookeep_pia_0_w);
}


static DRIVER_INIT( zookeep )
{
	/* we need to override two PIA handlers to prevent controls from getting disabled */
	install_mem_write_handler(0, 0x1400, 0x17ff, zookeep_pia_0_w);
	install_mem_write_handler(0, 0x1c00, 0x1fff, zookeep_pia_2_w);
}


static DRIVER_INIT( slither )
{
	install_mem_write_handler(1, 0x9401, 0x9401, slither_vram_mask_w);
}



/*************************************
 *
 *	Game drivers
 *
 *************************************/

GAME( 1981, qix,      0,        qix,     qix,      0,        ROT270, "Taito America Corporation", "Qix (set 1)" )
GAME( 1981, qixa,     qix,      qix,     qix,      0,        ROT270, "Taito America Corporation", "Qix (set 2)" )
GAME( 1981, qixb,     qix,      qix,     qix,      0,        ROT270, "Taito America Corporation", "Qix (set 3)" )
GAME( 1981, qix2,     qix,      qix,     qix,      0,        ROT270, "Taito America Corporation", "Qix II (Tournament)" )
GAME( 1981, sdungeon, 0,        mcu,     sdungeon, 0,        ROT270, "Taito America Corporation", "Space Dungeon" )
GAME( 1982, elecyoyo, 0,        mcu,     elecyoyo, 0,        ROT270, "Taito America Corporation", "The Electric Yo-Yo (set 1)" )
GAME( 1982, elecyoy2, elecyoyo, mcu,     elecyoyo, 0,        ROT270, "Taito America Corporation", "The Electric Yo-Yo (set 2)" )
GAME( 1982, kram,     0,        mcu,     kram,     kram,     ROT0,   "Taito America Corporation", "Kram (set 1)" )
GAME( 1982, kram2,    kram,     mcu,     kram,     kram,     ROT0,   "Taito America Corporation", "Kram (set 2)" )
GAMEX(1982, kram3,    kram,     qix,     kram,     kram,     ROT0,   "Taito America Corporation", "Kram (encrypted)", GAME_UNEMULATED_PROTECTION )
GAME( 1982, zookeep,  0,        zookeep, zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 1)" )
GAME( 1982, zookeep2, zookeep,  zookeep, zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 2)" )
GAME( 1982, zookeep3, zookeep,  zookeep, zookeep,  zookeep,  ROT0,   "Taito America Corporation", "Zoo Keeper (set 3)" )
GAME( 1982, slither,  0,        slither, slither,  slither,  ROT270, "Century II",                "Slither (set 1)" )
GAME( 1982, slithera, slither,  slither, slither,  slither,  ROT270, "Century II",                "Slither (set 2)" )
GAME( 1984, complexx, 0,        qix,     complexx, 0,        ROT270, "Taito America Corporation", "Complex X" )
