/***************************************************************************

	Atari Gauntlet hardware

	driver by Aaron Giles

	Games supported:
		* Gauntlet (1985) [14 sets]
		* Gauntlet 2-player Version (1985) [6 sets]
		* Gauntlet II (1986) [2 sets]
		* Gauntlet II 2-player Version (1986) [3 sets]
		* Vindicators Part II (1988) [3 sets]

	Known bugs:
		* none at this time

****************************************************************************

	Memory map

****************************************************************************

	========================================================================
	MAIN CPU
	========================================================================
	000000-037FFF   R     xxxxxxxx xxxxxxxx   Program ROM
	038000-03FFFF   R     xxxxxxxx xxxxxxxx   Slapstic-protected ROM
	040000-07FFFF   R     xxxxxxxx xxxxxxxx   Program ROM
	800000-801FFF   R/W   xxxxxxxx xxxxxxxx   Program RAM
	802000-802FFF   R/W   -------- xxxxxxxx   EEPROM
	803000          R     -------- xxxxxxxx   Input port 1
	803002          R     -------- xxxxxxxx   Input port 2
	803004          R     -------- xxxxxxxx   Input port 3
	803006          R     -------- xxxxxxxx   Input port 4
	803008          R     -------- -xxxx---   Status port
	                R     -------- -x------      (VBLANK)
	                R     -------- --x-----      (Sound command buffer full)
	                R     -------- ---x----      (Sound response buffer full)
	                R     -------- ----x---      (Self test)
	80300E          R     -------- xxxxxxxx   Sound response read
	803100            W   -------- --------   Watchdog reset
	80312E            W   -------- -------x   Sound CPU reset
	803140            W   -------- --------   VBLANK IRQ acknowledge
	803150            W   -------- --------   EEPROM enable
	803170            W   -------- xxxxxxxx   Sound command write
	900000-901FFF   R/W   xxxxxxxx xxxxxxxx   Playfield RAM (64x64 tiles)
	                R/W   x------- --------      (Horizontal flip)
	                R/W   -xxx---- --------      (Palette select)
	                R/W   ----xxxx xxxxxxxx      (Tile index)
	902000-903FFF   R/W   xxxxxxxx xxxxxxxx   Motion object RAM (1024 entries x 4 words)
	                R/W   -xxxxxxx xxxxxxxx      (0: Tile index)
	                R/W   xxxxxxxx x-------      (1024: X position)
	                R/W   -------- ----xxxx      (1024: Palette select)
	                R/W   xxxxxxxx x-------      (2048: Y position)
	                R/W   -------- -x------      (2048: Horizontal flip)
	                R/W   -------- --xxx---      (2048: Number of X tiles - 1)
	                R/W   -------- -----xxx      (2048: Number of Y tiles - 1)
	                R/W   ------xx xxxxxxxx      (3072: Link to next object)
	904000-904FFF   R/W   xxxxxxxx xxxxxxxx   Spare video RAM
	905000-905FFF   R/W   xxxxxxxx xxxxxxxx   Alphanumerics RAM (64x32 tiles)
	                R/W   x------- --------      (Opaque/transparent)
	                R/W   -xxxxx-- --------      (Palette select)
	                R/W   ------xx xxxxxxxx      (Tile index)
	905F6E          R/W   xxxxxxxx x-----xx   Playfield Y scroll/tile bank select
	                R/W   xxxxxxxx x-------      (Playfield Y scroll)
	                R/W   -------- ------xx      (Playfield tile bank select)
	910000-9101FF   R/W   xxxxxxxx xxxxxxxx   Alphanumercs palette RAM (256 entries)
	                R/W   xxxx---- --------      (Intensity)
	                R/W   ----xxxx --------      (Red)
	                R/W   -------- xxxx----      (Green)
	                R/W   -------- ----xxxx      (Blue)
	910200-9103FF   R/W   xxxxxxxx xxxxxxxx   Motion object palette RAM (256 entries)
	910400-9105FF   R/W   xxxxxxxx xxxxxxxx   Playfield palette RAM (256 entries)
	910600-9107FF   R/W   xxxxxxxx xxxxxxxx   Extra palette RAM (256 entries)
	930000            W   xxxxxxxx x-------   Playfield X scroll
	========================================================================
	Interrupts:
		IRQ4 = VBLANK
		IRQ6 = sound CPU communications
	========================================================================


	========================================================================
	SOUND CPU
	========================================================================
	0000-0FFF   R/W   xxxxxxxx   Program RAM
	1000          W   xxxxxxxx   Sound response write
	1010        R     xxxxxxxx   Sound command read
	1020        R     ----xxxx   Coin inputs
	            R     ----x---      (Coin 1)
	            R     -----x--      (Coin 2)
	            R     ------x-      (Coin 3)
	            R     -------x      (Coin 4)
	1020          W   xxxxxxxx   Mixer control
	              W   xxx-----      (TMS5220 volume)
	              W   ---xx---      (POKEY volume)
	              W   -----xxx      (YM2151 volume)
	1030        R     xxxx----   Sound status read
	            R     x-------      (Sound command buffer full)
	            R     -x------      (Sound response buffer full)
	            R     --x-----      (TMS5220 ready)
	            R     ---x----      (Self test)
	1030          W   x-------   YM2151 reset
	1031          W   x-------   TMS5220 data strobe
	1032          W   x-------   TMS5220 reset
	1033          W   x-------   TMS5220 frequency
	1800-180F   R/W   xxxxxxxx   POKEY communications
	1810-1811   R/W   xxxxxxxx   YM2151 communications
	1820          W   xxxxxxxx   TMS5220 data latch
	1830        R/W   --------   IRQ acknowledge
	4000-FFFF   R     xxxxxxxx   Program ROM
	========================================================================
	Interrupts:
		IRQ = timed interrupt
		NMI = latch on sound command
	========================================================================

****************************************************************************/


#include "driver.h"
#include "machine/atarigen.h"
#include "gauntlet.h"



/*************************************
 *
 *	Statics
 *
 *************************************/

static UINT8 speech_val;
static UINT8 last_speech_write;
static data16_t sound_reset_val;



/*************************************
 *
 *	Initialization & interrupts
 *
 *************************************/

static void update_interrupts(void)
{
	int newstate = 0;

	if (atarigen_video_int_state)
		newstate |= 4;
	if (atarigen_sound_int_state)
		newstate |= 6;

	if (newstate)
		cpu_set_irq_line(0, newstate, ASSERT_LINE);
	else
		cpu_set_irq_line(0, 7, CLEAR_LINE);
}


static void scanline_update(int scanline)
{
	/* sound IRQ is on 32V */
	if (scanline & 32)
		atarigen_6502_irq_gen();
	else
		atarigen_6502_irq_ack_r(0);
}


static MACHINE_INIT( gauntlet )
{
	last_speech_write = 0x80;
	sound_reset_val = 1;

	atarigen_eeprom_reset();
	atarigen_slapstic_reset();
	atarigen_interrupt_reset(update_interrupts);
	atarigen_scanline_timer_reset(scanline_update, 32);
	atarigen_sound_io_reset(1);
}



/*************************************
 *
 *	Controller reads
 *
 *************************************/

static int fake_inputs(int real_port, int fake_port)
{
	int result = readinputport(real_port);
	int fake = readinputport(fake_port);

	if (fake & 0x01)			/* up */
	{
		if (fake & 0x04)		/* up and left */
			result &= ~0x20;
		else if (fake & 0x08)	/* up and right */
			result &= ~0x10;
		else					/* up only */
			result &= ~0x30;
	}
	else if (fake & 0x02)		/* down */
	{
		if (fake & 0x04)		/* down and left */
			result &= ~0x80;
		else if (fake & 0x08)	/* down and right */
			result &= ~0x40;
		else					/* down only */
			result &= ~0xc0;
	}
	else if (fake & 0x04)		/* left only */
		result &= ~0x60;
	else if (fake & 0x08)		/* right only */
		result &= ~0x90;

	return result;
}


static READ16_HANDLER( vindctr2_port01_r )
{
	return fake_inputs(0 + offset, 6 + offset);
}


static READ16_HANDLER( port4_r )
{
	int temp = readinputport(4);
	if (atarigen_cpu_to_sound_ready) temp ^= 0x0020;
	if (atarigen_sound_to_cpu_ready) temp ^= 0x0010;
	return temp;
}



/*************************************
 *
 *	Sound reset
 *
 *************************************/

static WRITE16_HANDLER( sound_reset_w )
{
	if (ACCESSING_LSB)
	{
		int oldword = sound_reset_val;
		COMBINE_DATA(&sound_reset_val);

		if ((oldword ^ sound_reset_val) & 1)
		{
			cpu_set_reset_line(1, (sound_reset_val & 1) ? CLEAR_LINE : ASSERT_LINE);
			atarigen_sound_reset();
		}
	}
}



/*************************************
 *
 *	Sound I/O inputs
 *
 *************************************/

static READ_HANDLER( switch_6502_r )
{
	int temp = 0x30;

	if (atarigen_cpu_to_sound_ready) temp ^= 0x80;
	if (atarigen_sound_to_cpu_ready) temp ^= 0x40;
	if (tms5220_ready_r()) temp ^= 0x20;
	if (!(readinputport(4) & 0x0008)) temp ^= 0x10;

	return temp;
}



/*************************************
 *
 *	Sound TMS5220 write
 *
 *************************************/

static WRITE_HANDLER( tms5220_w )
{
	speech_val = data;
}



/*************************************
 *
 *	Sound control write
 *
 *************************************/

static WRITE_HANDLER( sound_ctl_w )
{
	switch (offset & 7)
	{
		case 0:	/* music reset, bit D7, low reset */
			break;

		case 1:	/* speech write, bit D7, active low */
			if (((data ^ last_speech_write) & 0x80) && (data & 0x80))
				tms5220_data_w(0, speech_val);
			last_speech_write = data;
			break;

		case 2:	/* speech reset, bit D7, active low */
			if (((data ^ last_speech_write) & 0x80) && (data & 0x80))
				tms5220_reset();
			break;

		case 3:	/* speech squeak, bit D7 */
			data = 5 | ((data >> 6) & 2);
			tms5220_set_frequency(ATARI_CLOCK_14MHz/2 / (16 - data));
			break;
	}
}



/*************************************
 *
 *	Sound mixer write
 *
 *************************************/

static WRITE_HANDLER( mixer_w )
{
	atarigen_set_ym2151_vol((data & 7) * 100 / 7);
	atarigen_set_pokey_vol(((data >> 3) & 3) * 100 / 3);
	atarigen_set_tms5220_vol(((data >> 5) & 7) * 100 / 7);
}



/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

static MEMORY_READ16_START( main_readmem )
	{ 0x000000, 0x07ffff, MRA16_ROM },
	{ 0x800000, 0x801fff, MRA16_RAM },
	{ 0x802000, 0x802fff, atarigen_eeprom_r },
	{ 0x803000, 0x803001, input_port_0_word_r },
	{ 0x803002, 0x803003, input_port_1_word_r },
	{ 0x803004, 0x803005, input_port_2_word_r },
	{ 0x803006, 0x803007, input_port_3_word_r },
	{ 0x803008, 0x803009, port4_r },
	{ 0x80300e, 0x80300f, atarigen_sound_r },
	{ 0x900000, 0x905fff, MRA16_RAM },
	{ 0x910000, 0x9107ff, MRA16_RAM },
MEMORY_END


static MEMORY_WRITE16_START( main_writemem )
	{ 0x000000, 0x07ffff, MWA16_ROM },
	{ 0x800000, 0x801fff, MWA16_RAM },
	{ 0x802000, 0x802fff, atarigen_eeprom_w, &atarigen_eeprom, &atarigen_eeprom_size },
	{ 0x803100, 0x803101, watchdog_reset16_w },
	{ 0x80312e, 0x80312f, sound_reset_w },
	{ 0x803140, 0x803141, atarigen_video_int_ack_w },
	{ 0x803150, 0x803151, atarigen_eeprom_enable_w },
	{ 0x803170, 0x803171, atarigen_sound_w },
	{ 0x900000, 0x901fff, atarigen_playfield_w, &atarigen_playfield },
	{ 0x902000, 0x903fff, atarimo_0_spriteram_w, &atarimo_0_spriteram },
	{ 0x904000, 0x904fff, MWA16_RAM },
	{ 0x905f6e, 0x905f6f, gauntlet_yscroll_w, &atarigen_yscroll },
	{ 0x905000, 0x905f7f, atarigen_alpha_w, &atarigen_alpha },
	{ 0x905f80, 0x905fff, atarimo_0_slipram_w, &atarimo_0_slipram },
	{ 0x910000, 0x9107ff, paletteram16_IIIIRRRRGGGGBBBB_word_w, &paletteram16 },
	{ 0x930000, 0x930001, gauntlet_xscroll_w, &atarigen_xscroll },
MEMORY_END



/*************************************
 *
 *	Sound CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( sound_readmem )
	{ 0x0000, 0x0fff, MRA_RAM },
	{ 0x1010, 0x101f, atarigen_6502_sound_r },
	{ 0x1020, 0x102f, input_port_5_r },
	{ 0x1030, 0x103f, switch_6502_r },
	{ 0x1800, 0x180f, pokey1_r },
	{ 0x1811, 0x1811, YM2151_status_port_0_r },
	{ 0x1830, 0x183f, atarigen_6502_irq_ack_r },
	{ 0x4000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( sound_writemem )
	{ 0x0000, 0x0fff, MWA_RAM },
	{ 0x1000, 0x100f, atarigen_6502_sound_w },
	{ 0x1020, 0x102f, mixer_w },
	{ 0x1030, 0x103f, sound_ctl_w },
	{ 0x1800, 0x180f, pokey1_w },
	{ 0x1810, 0x1810, YM2151_register_port_0_w },
	{ 0x1811, 0x1811, YM2151_data_port_0_w },
	{ 0x1820, 0x182f, tms5220_w },
	{ 0x1830, 0x183f, atarigen_6502_irq_ack_w },
	{ 0x4000, 0xffff, MWA_ROM },
MEMORY_END



/*************************************
 *
 *	Port definitions
 *
 *************************************/

INPUT_PORTS_START( gauntlet )
	PORT_START	/* 803000 */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 803002 */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 803004 */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 803006 */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START4 )
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER4 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER4 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER4 | IPF_8WAY )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER4 | IPF_8WAY )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER4 | IPF_8WAY )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER4 | IPF_8WAY )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 803008 */
	PORT_BIT( 0x0007, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_SERVICE( 0x0008, IP_ACTIVE_LOW )
	PORT_BIT( 0x0030, IP_ACTIVE_HIGH, IPT_SPECIAL )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW,  IPT_VBLANK )
	PORT_BIT( 0xff80, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START	/* 1020 (sound) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( vindctr2 )
	PORT_START	/* 803000 */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP    | IPF_PLAYER1 | IPF_2WAY )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP   | IPF_PLAYER1 | IPF_2WAY )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN  | IPF_PLAYER1 | IPF_2WAY )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN | IPF_PLAYER1 | IPF_2WAY )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 803002 */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_UP    | IPF_PLAYER2 | IPF_2WAY )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_UP   | IPF_PLAYER2 | IPF_2WAY )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICKLEFT_DOWN  | IPF_PLAYER2 | IPF_2WAY )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICKRIGHT_DOWN | IPF_PLAYER2 | IPF_2WAY )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 803004 */
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0xfffc, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 803006 */
	PORT_BIT( 0xffff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 803008 */
	PORT_BIT( 0x0007, IP_ACTIVE_HIGH, IPT_UNUSED )
	PORT_SERVICE( 0x0008, IP_ACTIVE_LOW )
	PORT_BIT( 0x0030, IP_ACTIVE_HIGH, IPT_SPECIAL )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW,  IPT_VBLANK )
	PORT_BIT( 0xff80, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START	/* 1020 (sound) */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* single joystick */
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_CHEAT | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_CHEAT | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_CHEAT | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_CHEAT | IPF_PLAYER1 )

	PORT_START	/* single joystick */
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_CHEAT | IPF_PLAYER2 )
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_CHEAT | IPF_PLAYER2 )
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_CHEAT | IPF_PLAYER2 )
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_CHEAT | IPF_PLAYER2 )
INPUT_PORTS_END



/*************************************
 *
 *	Graphics definitions
 *
 *************************************/

static struct GfxLayout anlayout =
{
	8,8,
	RGN_FRAC(1,1),
	2,
	{ 0, 4 },
	{ 0, 1, 2, 3, 8, 9, 10, 11 },
	{ 0*16, 1*16, 2*16, 3*16, 4*16, 5*16, 6*16, 7*16 },
	8*16
};


static struct GfxLayout pfmolayout =
{
	8,8,
	RGN_FRAC(1,4),
	4,
	{ RGN_FRAC(3,4), RGN_FRAC(2,4), RGN_FRAC(1,4), RGN_FRAC(0,4) },
	{ 0, 1, 2, 3, 4, 5, 6, 7 },
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8 },
	8*8
};


static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX2, 0, &pfmolayout,  256, 32 },
	{ REGION_GFX1, 0, &anlayout,      0, 64 },
	{ -1 }
};



/*************************************
 *
 *	Sound definitions
 *
 *************************************/

static struct YM2151interface ym2151_interface =
{
	1,
	ATARI_CLOCK_14MHz/4,
	{ YM3012_VOL(48,MIXER_PAN_LEFT,48,MIXER_PAN_RIGHT) },
	{ 0 }
};


static struct POKEYinterface pokey_interface =
{
	1,
	ATARI_CLOCK_14MHz/8,
	{ 32 },
};


static struct TMS5220interface tms5220_interface =
{
	ATARI_CLOCK_14MHz/2/11,	/* potentially ATARI_CLOCK_14MHz/2/9 as well */
	80,
	0
};



/*************************************
 *
 *	Machine driver
 *
 *************************************/

static MACHINE_DRIVER_START( gauntlet )

	/* basic machine hardware */
	MDRV_CPU_ADD(M68010, ATARI_CLOCK_14MHz/2)
	MDRV_CPU_MEMORY(main_readmem,main_writemem)
	MDRV_CPU_VBLANK_INT(atarigen_video_int_gen,1)

	MDRV_CPU_ADD(M6502, ATARI_CLOCK_14MHz/8)
	MDRV_CPU_MEMORY(sound_readmem,sound_writemem)

	MDRV_FRAMES_PER_SECOND(60)
	MDRV_VBLANK_DURATION(DEFAULT_REAL_60HZ_VBLANK_DURATION)

	MDRV_MACHINE_INIT(gauntlet)
	MDRV_NVRAM_HANDLER(atarigen)

	/* video hardware */
	MDRV_VIDEO_ATTRIBUTES(VIDEO_TYPE_RASTER | VIDEO_NEEDS_6BITS_PER_GUN | VIDEO_UPDATE_BEFORE_VBLANK)
	MDRV_SCREEN_SIZE(42*8, 30*8)
	MDRV_VISIBLE_AREA(0*8, 42*8-1, 0*8, 30*8-1)
	MDRV_GFXDECODE(gfxdecodeinfo)
	MDRV_PALETTE_LENGTH(1024)

	MDRV_VIDEO_START(gauntlet)
	MDRV_VIDEO_UPDATE(gauntlet)

	/* sound hardware */
	MDRV_SOUND_ATTRIBUTES(SOUND_SUPPORTS_STEREO)
	MDRV_SOUND_ADD(YM2151, ym2151_interface)
	MDRV_SOUND_ADD(POKEY, pokey_interface)
	MDRV_SOUND_ADD(TMS5220, tms5220_interface)
MACHINE_DRIVER_END



/*************************************
 *
 *	ROM definition(s)
 *
 *************************************/

ROM_START( gaunts )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1507-9a.037",  0x000000, 0x008000, 0xb5183228 )
	ROM_LOAD16_BYTE( "1508-9b.037",  0x000001, 0x008000, 0xafd3c501 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "1509-7a.037",  0x040000, 0x008000, 0x69e50ae9 )
	ROM_LOAD16_BYTE( "1510-7b.037",  0x040001, 0x008000, 0x54e2692c )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntlet )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1307-9a.037",  0x000000, 0x008000, 0x46fe8743 )
	ROM_LOAD16_BYTE( "1308-9b.037",  0x000001, 0x008000, 0x276e15c4 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "1409-7a.037",  0x040000, 0x008000, 0x6fb8419c )
	ROM_LOAD16_BYTE( "1410-7b.037",  0x040001, 0x008000, 0x931bd2a0 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntj )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1307-9a.037",  0x000000, 0x008000, 0x46fe8743 )
	ROM_LOAD16_BYTE( "1308-9b.037",  0x000001, 0x008000, 0x276e15c4 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "1309-7a.037",  0x040000, 0x008000, 0xe8ba39d8 )
	ROM_LOAD16_BYTE( "1310-7b.037",  0x040001, 0x008000, 0xa204d997 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntj12 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1207-9a.037",  0x000000, 0x008000, 0x6dc0610d )
	ROM_LOAD16_BYTE( "1208-9b.037",  0x000001, 0x008000, 0xfaa306eb )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "1109-7a.037",  0x040000, 0x008000, 0x500194fb )
	ROM_LOAD16_BYTE( "1110-7b.037",  0x040001, 0x008000, 0xb2969076 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntg )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1007-9a.037",  0x000000, 0x008000, 0x6a224cea )
	ROM_LOAD16_BYTE( "1008-9b.037",  0x000001, 0x008000, 0xfa391dab )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "1009-7a.037",  0x040000, 0x008000, 0x75d1f966 )
	ROM_LOAD16_BYTE( "1010-7b.037",  0x040001, 0x008000, 0x28a4197b )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntr9 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "907-9a.037",   0x000000, 0x008000, 0xc13a6399 )
	ROM_LOAD16_BYTE( "908-9b.037",   0x000001, 0x008000, 0x417607d9 )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "909-7a.037",   0x040000, 0x008000, 0xfb1cdc1c )
	ROM_LOAD16_BYTE( "910-7b.037",   0x040001, 0x008000, 0xf188e7b3 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntgr8 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "807-9a.037",   0x000000, 0x008000, 0x671c0bc2 )
	ROM_LOAD16_BYTE( "808-9b.037",   0x000001, 0x008000, 0xf2842af4 )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "809-7a.037",   0x040000, 0x008000, 0x05642d60 )
	ROM_LOAD16_BYTE( "810-7b.037",   0x040001, 0x008000, 0x36d295e3 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntr7 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "207-9a.037",   0x000000, 0x008000, 0xfd871f81 )
	ROM_LOAD16_BYTE( "208-9b.037",   0x000001, 0x008000, 0xbcb2fb1d )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "709-7a.037",   0x040000, 0x008000, 0x73e1ad79 )
	ROM_LOAD16_BYTE( "710-7b.037",   0x040001, 0x008000, 0xfd248cea )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntgr6 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "307-9a.037",   0x000000, 0x008000, 0x759827c9 )
	ROM_LOAD16_BYTE( "308-9b.037",   0x000001, 0x008000, 0xd71262d1 )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "609-7a.037",   0x040000, 0x008000, 0xcd3381de )
	ROM_LOAD16_BYTE( "610-7b.037",   0x040001, 0x008000, 0x2cff932a )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntr5 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "207-9a.037",   0x000000, 0x008000, 0xfd871f81 )
	ROM_LOAD16_BYTE( "208-9b.037",   0x000001, 0x008000, 0xbcb2fb1d )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "409-7a.037",   0x040000, 0x008000, 0xc57377b3 )
	ROM_LOAD16_BYTE( "510-7b.037",   0x040001, 0x008000, 0x1cac2071 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntr4 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "207-9a.037",   0x000000, 0x008000, 0xfd871f81 )
	ROM_LOAD16_BYTE( "208-9b.037",   0x000001, 0x008000, 0xbcb2fb1d )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "409-7a.037",   0x040000, 0x008000, 0xc57377b3 )
	ROM_LOAD16_BYTE( "410-7b.037",   0x040001, 0x008000, 0x6b971a27 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntgr3 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "307-9a.037",   0x000000, 0x008000, 0x759827c9 )
	ROM_LOAD16_BYTE( "308-9b.037",   0x000001, 0x008000, 0xd71262d1 )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "309-7a.037",   0x040000, 0x008000, 0x7f03696b )
	ROM_LOAD16_BYTE( "310-7b.037",   0x040001, 0x008000, 0x8d7197fc )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntr2 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "207-9a.037",   0x000000, 0x008000, 0xfd871f81 )
	ROM_LOAD16_BYTE( "208-9b.037",   0x000001, 0x008000, 0xbcb2fb1d )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "209-7a.037",   0x040000, 0x008000, 0xd810a7dc )
	ROM_LOAD16_BYTE( "210-7b.037",   0x040001, 0x008000, 0xfbba7290 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gauntr1 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "107-9a.037",   0x000000, 0x008000, 0xa5885e14 )
	ROM_LOAD16_BYTE( "108-9b.037",   0x000001, 0x008000, 0x0087f1ab )
	ROM_LOAD16_BYTE( "105-10a.037",  0x038000, 0x004000, 0x4642cd95 )
	ROM_LOAD16_BYTE( "106-10b.037",  0x038001, 0x004000, 0xc8df945e )
	ROM_LOAD16_BYTE( "109-7a.037",   0x040000, 0x008000, 0x55d87198 )
	ROM_LOAD16_BYTE( "110-7b.037",   0x040001, 0x008000, 0xf84ad06d )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gaunt2p )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "507-9a.041",   0x000000, 0x008000, 0x8784133f )
	ROM_LOAD16_BYTE( "508-9b.041",   0x000001, 0x008000, 0x2843bde3 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "609-7a.041",   0x040000, 0x008000, 0x5b4ee415 )
	ROM_LOAD16_BYTE( "610-7b.041",   0x040001, 0x008000, 0x41f5c9e2 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gaunt2pj )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "507-9a.041",   0x000000, 0x008000, 0x8784133f )
	ROM_LOAD16_BYTE( "508-9b.041",   0x000001, 0x008000, 0x2843bde3 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "509-7a.041",   0x040000, 0x008000, 0xfb2ef226 )
	ROM_LOAD16_BYTE( "510-7b.041",   0x040001, 0x008000, 0xa69be8da )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gaunt2pg )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "407-9a.041",   0x000000, 0x008000, 0xcde72140 )
	ROM_LOAD16_BYTE( "408-9b.041",   0x000001, 0x008000, 0x4ab1af62 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "409-7a.041",   0x040000, 0x008000, 0x44e01459 )
	ROM_LOAD16_BYTE( "410-7b.041",   0x040001, 0x008000, 0xb58d96d3 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gaun2pr3 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "207-9a.041",   0x000000, 0x008000, 0x0e1af1b4 )
	ROM_LOAD16_BYTE( "208-9b.041",   0x000001, 0x008000, 0xbf51a238 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "309-7a.041",   0x040000, 0x008000, 0x5acbcd2b )
	ROM_LOAD16_BYTE( "310-7b.041",   0x040001, 0x008000, 0x1889ab77 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gaun2pj2 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "207-9a.041",   0x000000, 0x008000, 0x0e1af1b4 )
	ROM_LOAD16_BYTE( "208-9b.041",   0x000001, 0x008000, 0xbf51a238 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "209-7a.041",   0x040000, 0x008000, 0xddc9b56f )
	ROM_LOAD16_BYTE( "210-7b.041",   0x040001, 0x008000, 0xffe78a4f )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gaun2pg1 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "107-9a.041",   0x000000, 0x008000, 0x3faf74d8 )
	ROM_LOAD16_BYTE( "108-9b.041",   0x000001, 0x008000, 0xf1e6d815 )
	ROM_LOAD16_BYTE( "205-10a.037",  0x038000, 0x004000, 0x6d99ed51 )
	ROM_LOAD16_BYTE( "206-10b.037",  0x038001, 0x004000, 0x545ead91 )
	ROM_LOAD16_BYTE( "109-7a.041",   0x040000, 0x008000, 0x56d0c5b8 )
	ROM_LOAD16_BYTE( "110-7b.041",   0x040001, 0x008000, 0x3b9ae397 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "120-16r.037",  0x004000, 0x004000, 0x6ee7f3cc )
	ROM_LOAD( "119-16s.037",  0x008000, 0x008000, 0xfa19861f )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "104-6p.037",   0x000000, 0x002000, 0x9e2a5b59 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "111-1a.037",   0x000000, 0x008000, 0x91700f33 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "113-1l.037",   0x010000, 0x008000, 0xd497d0a8 )
	ROM_LOAD( "114-1mn.037",  0x018000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "115-2a.037",   0x020000, 0x008000, 0x9510b898 )
	ROM_LOAD( "116-2b.037",   0x028000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "117-2l.037",   0x030000, 0x008000, 0x29a5db41 )
	ROM_LOAD( "118-2mn.037",  0x038000, 0x008000, 0x8bf3b263 )
ROM_END


ROM_START( gaunt2 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1307-9a.037",  0x000000, 0x008000, 0x46fe8743 )
	ROM_LOAD16_BYTE( "1308-9b.037",  0x000001, 0x008000, 0x276e15c4 )
	ROM_LOAD16_BYTE( "1105-10a.043", 0x038000, 0x004000, 0x45dfda47 )
	ROM_LOAD16_BYTE( "1106-10b.043", 0x038001, 0x004000, 0x343c029c )
	ROM_LOAD16_BYTE( "1109-7a.043",  0x040000, 0x008000, 0x58a0a9a3 )
	ROM_LOAD16_BYTE( "1110-7b.043",  0x040001, 0x008000, 0x658f0da8 )
	ROM_LOAD16_BYTE( "1121-6a.043",  0x050000, 0x008000, 0xae301bba )
	ROM_LOAD16_BYTE( "1122-6b.043",  0x050001, 0x008000, 0xe94aaa8a )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1120-16r.043", 0x004000, 0x004000, 0x5c731006 )
	ROM_LOAD( "1119-16s.043", 0x008000, 0x008000, 0xdc3591e7 )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1104-6p.043",  0x000000, 0x002000, 0x1343cf6f )

	ROM_REGION( 0x60000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "1111-1a.043",  0x000000, 0x008000, 0x09df6e23 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "1123-1c.043",  0x010000, 0x004000, 0xe4c98f01 )
	ROM_RELOAD(               0x014000, 0x004000 )
	ROM_LOAD( "1113-1l.043",  0x018000, 0x008000, 0x33cb476e )
	ROM_LOAD( "114-1mn.037",  0x020000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "1124-1p.043",  0x028000, 0x004000, 0xc4857879 )
	ROM_RELOAD(               0x02c000, 0x004000 )
	ROM_LOAD( "1115-2a.043",  0x030000, 0x008000, 0xf71e2503 )
	ROM_LOAD( "116-2b.037",   0x038000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "1125-2c.043",  0x040000, 0x004000, 0xd9c2c2d1 )
	ROM_RELOAD(               0x044000, 0x004000 )
	ROM_LOAD( "1117-2l.043",  0x048000, 0x008000, 0x9e30b2e9 )
	ROM_LOAD( "118-2mn.037",  0x050000, 0x008000, 0x8bf3b263 )
	ROM_LOAD( "1126-2p.043",  0x058000, 0x004000, 0xa32c732a )
	ROM_RELOAD(               0x05c000, 0x004000 )
ROM_END


ROM_START( gaunt2g )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1007-9a.037",  0x000000, 0x008000, 0x6a224cea )
	ROM_LOAD16_BYTE( "1008-9b.037",  0x000001, 0x008000, 0xfa391dab )
	ROM_LOAD16_BYTE( "1105-10a.043", 0x038000, 0x004000, 0x45dfda47 )
	ROM_LOAD16_BYTE( "1106-10b.043", 0x038001, 0x004000, 0x343c029c )
	ROM_LOAD16_BYTE( "2209-7a.043",  0x040000, 0x008000, 0x577f4101 )
	ROM_LOAD16_BYTE( "2210-7b.043",  0x040001, 0x008000, 0x03254cf4 )
	ROM_LOAD16_BYTE( "2221-6a.043",  0x050000, 0x008000, 0xc8adcf1a )
	ROM_LOAD16_BYTE( "2222-6b.043",  0x050001, 0x008000, 0x7788ff84 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1120-16r.043", 0x004000, 0x004000, 0x5c731006 )
	ROM_LOAD( "1119-16s.043", 0x008000, 0x008000, 0xdc3591e7 )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1104-6p.043",  0x000000, 0x002000, 0x1343cf6f )

	ROM_REGION( 0x60000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "1111-1a.043",  0x000000, 0x008000, 0x09df6e23 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "1123-1c.043",  0x010000, 0x004000, 0xe4c98f01 )
	ROM_RELOAD(               0x014000, 0x004000 )
	ROM_LOAD( "1113-1l.043",  0x018000, 0x008000, 0x33cb476e )
	ROM_LOAD( "114-1mn.037",  0x020000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "1124-1p.043",  0x028000, 0x004000, 0xc4857879 )
	ROM_RELOAD(               0x02c000, 0x004000 )
	ROM_LOAD( "1115-2a.043",  0x030000, 0x008000, 0xf71e2503 )
	ROM_LOAD( "116-2b.037",   0x038000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "1125-2c.043",  0x040000, 0x004000, 0xd9c2c2d1 )
	ROM_RELOAD(               0x044000, 0x004000 )
	ROM_LOAD( "1117-2l.043",  0x048000, 0x008000, 0x9e30b2e9 )
	ROM_LOAD( "118-2mn.037",  0x050000, 0x008000, 0x8bf3b263 )
	ROM_LOAD( "1126-2p.043",  0x058000, 0x004000, 0xa32c732a )
	ROM_RELOAD(               0x05c000, 0x004000 )
ROM_END


ROM_START( gaunt22p )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1307-9a.037",  0x000000, 0x008000, 0x46fe8743 )
	ROM_LOAD16_BYTE( "1308-9b.037",  0x000001, 0x008000, 0x276e15c4 )
	ROM_LOAD16_BYTE( "1105-10a.043", 0x038000, 0x004000, 0x45dfda47 )
	ROM_LOAD16_BYTE( "1106-10b.043", 0x038001, 0x004000, 0x343c029c )
	ROM_LOAD16_BYTE( "2109-7a.044",  0x040000, 0x008000, 0x1102ab96 )
	ROM_LOAD16_BYTE( "2110-7b.044",  0x040001, 0x008000, 0xd2203a2b )
	ROM_LOAD16_BYTE( "2121-6a.044",  0x050000, 0x008000, 0x753982d7 )
	ROM_LOAD16_BYTE( "2122-6b.044",  0x050001, 0x008000, 0x879149ea )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1120-16r.043", 0x004000, 0x004000, 0x5c731006 )
	ROM_LOAD( "1119-16s.043", 0x008000, 0x008000, 0xdc3591e7 )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1104-6p.043",  0x000000, 0x002000, 0x1343cf6f )

	ROM_REGION( 0x60000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "1111-1a.043",  0x000000, 0x008000, 0x09df6e23 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "1123-1c.043",  0x010000, 0x004000, 0xe4c98f01 )
	ROM_RELOAD(               0x014000, 0x004000 )
	ROM_LOAD( "1113-1l.043",  0x018000, 0x008000, 0x33cb476e )
	ROM_LOAD( "114-1mn.037",  0x020000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "1124-1p.043",  0x028000, 0x004000, 0xc4857879 )
	ROM_RELOAD(               0x02c000, 0x004000 )
	ROM_LOAD( "1115-2a.043",  0x030000, 0x008000, 0xf71e2503 )
	ROM_LOAD( "116-2b.037",   0x038000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "1125-2c.043",  0x040000, 0x004000, 0xd9c2c2d1 )
	ROM_RELOAD(               0x044000, 0x004000 )
	ROM_LOAD( "1117-2l.043",  0x048000, 0x008000, 0x9e30b2e9 )
	ROM_LOAD( "118-2mn.037",  0x050000, 0x008000, 0x8bf3b263 )
	ROM_LOAD( "1126-2p.043",  0x058000, 0x004000, 0xa32c732a )
	ROM_RELOAD(               0x05c000, 0x004000 )
ROM_END


ROM_START( gaun22p1 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1307-9a.037",  0x000000, 0x008000, 0x46fe8743 )
	ROM_LOAD16_BYTE( "1308-9b.037",  0x000001, 0x008000, 0x276e15c4 )
	ROM_LOAD16_BYTE( "1105-10a.043", 0x038000, 0x004000, 0x45dfda47 )
	ROM_LOAD16_BYTE( "1106-10b.043", 0x038001, 0x004000, 0x343c029c )
	ROM_LOAD16_BYTE( "1109-7a.044",  0x040000, 0x008000, 0x31f805eb )
	ROM_LOAD16_BYTE( "1110-7b.044",  0x040001, 0x008000, 0x5285c0e2 )
	ROM_LOAD16_BYTE( "1121-6a.044",  0x050000, 0x008000, 0xd1f3b32a )
	ROM_LOAD16_BYTE( "1122-6b.044",  0x050001, 0x008000, 0x3485785f )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1120-16r.043", 0x004000, 0x004000, 0x5c731006 )
	ROM_LOAD( "1119-16s.043", 0x008000, 0x008000, 0xdc3591e7 )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1104-6p.043",  0x000000, 0x002000, 0x1343cf6f )

	ROM_REGION( 0x60000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "1111-1a.043",  0x000000, 0x008000, 0x09df6e23 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "1123-1c.043",  0x010000, 0x004000, 0xe4c98f01 )
	ROM_RELOAD(               0x014000, 0x004000 )
	ROM_LOAD( "1113-1l.043",  0x018000, 0x008000, 0x33cb476e )
	ROM_LOAD( "114-1mn.037",  0x020000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "1124-1p.043",  0x028000, 0x004000, 0xc4857879 )
	ROM_RELOAD(               0x02c000, 0x004000 )
	ROM_LOAD( "1115-2a.043",  0x030000, 0x008000, 0xf71e2503 )
	ROM_LOAD( "116-2b.037",   0x038000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "1125-2c.043",  0x040000, 0x004000, 0xd9c2c2d1 )
	ROM_RELOAD(               0x044000, 0x004000 )
	ROM_LOAD( "1117-2l.043",  0x048000, 0x008000, 0x9e30b2e9 )
	ROM_LOAD( "118-2mn.037",  0x050000, 0x008000, 0x8bf3b263 )
	ROM_LOAD( "1126-2p.043",  0x058000, 0x004000, 0xa32c732a )
	ROM_RELOAD(               0x05c000, 0x004000 )
ROM_END


ROM_START( gaun22pg )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1007-9a.037",  0x000000, 0x008000, 0x6a224cea )
	ROM_LOAD16_BYTE( "1008-9b.037",  0x000001, 0x008000, 0xfa391dab )
	ROM_LOAD16_BYTE( "1105-10a.043", 0x038000, 0x004000, 0x45dfda47 )
	ROM_LOAD16_BYTE( "1106-10b.043", 0x038001, 0x004000, 0x343c029c )
	ROM_LOAD16_BYTE( "2209-7a.044",  0x040000, 0x008000, 0x9da52ecd )
	ROM_LOAD16_BYTE( "2210-7b.044",  0x040001, 0x008000, 0x63d0f6a7 )
	ROM_LOAD16_BYTE( "2221-6a.044",  0x050000, 0x008000, 0x8895b31b )
	ROM_LOAD16_BYTE( "2222-6b.044",  0x050001, 0x008000, 0xa4456cc7 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1120-16r.043", 0x004000, 0x004000, 0x5c731006 )
	ROM_LOAD( "1119-16s.043", 0x008000, 0x008000, 0xdc3591e7 )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1104-6p.043",  0x000000, 0x002000, 0x1343cf6f )

	ROM_REGION( 0x60000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "1111-1a.043",  0x000000, 0x008000, 0x09df6e23 )
	ROM_LOAD( "112-1b.037",   0x008000, 0x008000, 0x869330be )
	ROM_LOAD( "1123-1c.043",  0x010000, 0x004000, 0xe4c98f01 )
	ROM_RELOAD(               0x014000, 0x004000 )
	ROM_LOAD( "1113-1l.043",  0x018000, 0x008000, 0x33cb476e )
	ROM_LOAD( "114-1mn.037",  0x020000, 0x008000, 0x29ef9882 )
	ROM_LOAD( "1124-1p.043",  0x028000, 0x004000, 0xc4857879 )
	ROM_RELOAD(               0x02c000, 0x004000 )
	ROM_LOAD( "1115-2a.043",  0x030000, 0x008000, 0xf71e2503 )
	ROM_LOAD( "116-2b.037",   0x038000, 0x008000, 0x11e0ac5b )
	ROM_LOAD( "1125-2c.043",  0x040000, 0x004000, 0xd9c2c2d1 )
	ROM_RELOAD(               0x044000, 0x004000 )
	ROM_LOAD( "1117-2l.043",  0x048000, 0x008000, 0x9e30b2e9 )
	ROM_LOAD( "118-2mn.037",  0x050000, 0x008000, 0x8bf3b263 )
	ROM_LOAD( "1126-2p.043",  0x058000, 0x004000, 0xa32c732a )
	ROM_RELOAD(               0x05c000, 0x004000 )
ROM_END


ROM_START( vindctr2 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1186-9a.059",  0x000000, 0x008000, 0xaf138263 )
	ROM_LOAD16_BYTE( "1187-9b.059",  0x000001, 0x008000, 0x44baff64 )
	ROM_LOAD16_BYTE( "1196-10a.059", 0x038000, 0x004000, 0xc92bf6dd )
	ROM_LOAD16_BYTE( "1197-10b.059", 0x038001, 0x004000, 0xd7ace347 )
	ROM_LOAD16_BYTE( "3188-7a.059",  0x040000, 0x008000, 0x10f558d2 )
	ROM_LOAD16_BYTE( "3189-7b.059",  0x040001, 0x008000, 0x302e24b6 )
	ROM_LOAD16_BYTE( "2190-6a.059",  0x050000, 0x008000, 0xe7dc2b74 )
	ROM_LOAD16_BYTE( "2191-6b.059",  0x050001, 0x008000, 0xed8ed86e )
	ROM_LOAD16_BYTE( "2192-5a.059",  0x060000, 0x008000, 0xeec2c93d )
	ROM_LOAD16_BYTE( "2193-5b.059",  0x060001, 0x008000, 0x3fbee9aa )
	ROM_LOAD16_BYTE( "1194-3a.059",  0x070000, 0x008000, 0xe6bcf458 )
	ROM_LOAD16_BYTE( "1195-3b.059",  0x070001, 0x008000, 0xb9bf245d )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1160-16s.059", 0x004000, 0x004000, 0xeef0a003 )
	ROM_LOAD( "1161-16r.059", 0x008000, 0x008000, 0x68c74337 )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1198-6p.059",  0x000000, 0x004000, 0xf99b631a )

	ROM_REGION( 0xc0000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "1162-1a.059",  0x000000, 0x008000, 0xdd3833ad )
	ROM_LOAD( "1166-1b.059",  0x008000, 0x008000, 0xe2db50a0 )
	ROM_LOAD( "1170-1c.059",  0x010000, 0x008000, 0xf050ab43 )
	ROM_LOAD( "1174-1d.059",  0x018000, 0x008000, 0xb6704bd1 )
	ROM_LOAD( "1178-1ef.059", 0x020000, 0x008000, 0xd3006f05 )
	ROM_LOAD( "1182-1j.059",  0x028000, 0x008000, 0x9046e985 )

	ROM_LOAD( "1163-1l.059",  0x030000, 0x008000, 0xd505b04a )
	ROM_LOAD( "1167-1mn.059", 0x038000, 0x008000, 0x1869c76d )
	ROM_LOAD( "1171-1p.059",  0x040000, 0x008000, 0x1b229c2b )
	ROM_LOAD( "1175-1r.059",  0x048000, 0x008000, 0x73c41aca )
	ROM_LOAD( "1179-1st.059", 0x050000, 0x008000, 0x9b7cb0ef )
	ROM_LOAD( "1183-1u.059",  0x058000, 0x008000, 0x393bba42 )

	ROM_LOAD( "1164-2a.059",  0x060000, 0x008000, 0x50e76162 )
	ROM_LOAD( "1168-2b.059",  0x068000, 0x008000, 0x35c78469 )
	ROM_LOAD( "1172-2c.059",  0x070000, 0x008000, 0x314ac268 )
	ROM_LOAD( "1176-2d.059",  0x078000, 0x008000, 0x061d79db )
	ROM_LOAD( "1180-2ef.059", 0x080000, 0x008000, 0x89c1fe16 )
	ROM_LOAD( "1184-2j.059",  0x088000, 0x008000, 0x541209d3 )

	ROM_LOAD( "1165-2l.059",  0x090000, 0x008000, 0x9484ba65 )
	ROM_LOAD( "1169-2mn.059", 0x098000, 0x008000, 0x132d3337 )
	ROM_LOAD( "1173-2p.059",  0x0a0000, 0x008000, 0x98de2426 )
	ROM_LOAD( "1177-2r.059",  0x0a8000, 0x008000, 0x9d0824f8 )
	ROM_LOAD( "1181-2st.059", 0x0b0000, 0x008000, 0x9e62b27c )
	ROM_LOAD( "1185-2u.059",  0x0b8000, 0x008000, 0x9d62f6b7 )
ROM_END


ROM_START( vindc2r2 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1186-9a.059",  0x000000, 0x008000, 0xaf138263 )
	ROM_LOAD16_BYTE( "1187-9b.059",  0x000001, 0x008000, 0x44baff64 )
	ROM_LOAD16_BYTE( "1196-10a.059", 0x038000, 0x004000, 0xc92bf6dd )
	ROM_LOAD16_BYTE( "1197-10b.059", 0x038001, 0x004000, 0xd7ace347 )
	ROM_LOAD16_BYTE( "2188-7a.059",  0x040000, 0x008000, 0xd4e0ef1f )
	ROM_LOAD16_BYTE( "2189-7b.059",  0x040001, 0x008000, 0xdcbbe2aa )
	ROM_LOAD16_BYTE( "2190-6a.059",  0x050000, 0x008000, 0xe7dc2b74 )
	ROM_LOAD16_BYTE( "2191-6b.059",  0x050001, 0x008000, 0xed8ed86e )
	ROM_LOAD16_BYTE( "2192-5a.059",  0x060000, 0x008000, 0xeec2c93d )
	ROM_LOAD16_BYTE( "2193-5b.059",  0x060001, 0x008000, 0x3fbee9aa )
	ROM_LOAD16_BYTE( "1194-3a.059",  0x070000, 0x008000, 0xe6bcf458 )
	ROM_LOAD16_BYTE( "1195-3b.059",  0x070001, 0x008000, 0xb9bf245d )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1160-16s.059", 0x004000, 0x004000, 0xeef0a003 )
	ROM_LOAD( "1161-16r.059", 0x008000, 0x008000, 0x68c74337 )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1198-6p.059",  0x000000, 0x004000, 0xf99b631a )

	ROM_REGION( 0xc0000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "1162-1a.059",  0x000000, 0x008000, 0xdd3833ad )
	ROM_LOAD( "1166-1b.059",  0x008000, 0x008000, 0xe2db50a0 )
	ROM_LOAD( "1170-1c.059",  0x010000, 0x008000, 0xf050ab43 )
	ROM_LOAD( "1174-1d.059",  0x018000, 0x008000, 0xb6704bd1 )
	ROM_LOAD( "1178-1ef.059", 0x020000, 0x008000, 0xd3006f05 )
	ROM_LOAD( "1182-1j.059",  0x028000, 0x008000, 0x9046e985 )

	ROM_LOAD( "1163-1l.059",  0x030000, 0x008000, 0xd505b04a )
	ROM_LOAD( "1167-1mn.059", 0x038000, 0x008000, 0x1869c76d )
	ROM_LOAD( "1171-1p.059",  0x040000, 0x008000, 0x1b229c2b )
	ROM_LOAD( "1175-1r.059",  0x048000, 0x008000, 0x73c41aca )
	ROM_LOAD( "1179-1st.059", 0x050000, 0x008000, 0x9b7cb0ef )
	ROM_LOAD( "1183-1u.059",  0x058000, 0x008000, 0x393bba42 )

	ROM_LOAD( "1164-2a.059",  0x060000, 0x008000, 0x50e76162 )
	ROM_LOAD( "1168-2b.059",  0x068000, 0x008000, 0x35c78469 )
	ROM_LOAD( "1172-2c.059",  0x070000, 0x008000, 0x314ac268 )
	ROM_LOAD( "1176-2d.059",  0x078000, 0x008000, 0x061d79db )
	ROM_LOAD( "1180-2ef.059", 0x080000, 0x008000, 0x89c1fe16 )
	ROM_LOAD( "1184-2j.059",  0x088000, 0x008000, 0x541209d3 )

	ROM_LOAD( "1165-2l.059",  0x090000, 0x008000, 0x9484ba65 )
	ROM_LOAD( "1169-2mn.059", 0x098000, 0x008000, 0x132d3337 )
	ROM_LOAD( "1173-2p.059",  0x0a0000, 0x008000, 0x98de2426 )
	ROM_LOAD( "1177-2r.059",  0x0a8000, 0x008000, 0x9d0824f8 )
	ROM_LOAD( "1181-2st.059", 0x0b0000, 0x008000, 0x9e62b27c )
	ROM_LOAD( "1185-2u.059",  0x0b8000, 0x008000, 0x9d62f6b7 )
ROM_END


ROM_START( vindc2r1 )
	ROM_REGION( 0x80000, REGION_CPU1, 0 )	/* 8*64k for 68000 code */
	ROM_LOAD16_BYTE( "1186-9a.059",  0x000000, 0x008000, 0xaf138263 )
	ROM_LOAD16_BYTE( "1187-9b.059",  0x000001, 0x008000, 0x44baff64 )
	ROM_LOAD16_BYTE( "1196-10a.059", 0x038000, 0x004000, 0xc92bf6dd )
	ROM_LOAD16_BYTE( "1197-10b.059", 0x038001, 0x004000, 0xd7ace347 )
	ROM_LOAD16_BYTE( "1188-7a.059",  0x040000, 0x008000, 0x52294cad )
	ROM_LOAD16_BYTE( "1189-7b.059",  0x040001, 0x008000, 0x577a705f )
	ROM_LOAD16_BYTE( "1190-6a.059",  0x050000, 0x008000, 0x7be01bb1 )
	ROM_LOAD16_BYTE( "1191-6b.059",  0x050001, 0x008000, 0x91922a02 )
	ROM_LOAD16_BYTE( "1192-5a.059",  0x060000, 0x008000, 0xe4f59d72 )
	ROM_LOAD16_BYTE( "1193-5b.059",  0x060001, 0x008000, 0xe901c618 )
	ROM_LOAD16_BYTE( "1194-3a.059",  0x070000, 0x008000, 0xe6bcf458 )
	ROM_LOAD16_BYTE( "1195-3b.059",  0x070001, 0x008000, 0xb9bf245d )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )	/* 64k for 6502 code */
	ROM_LOAD( "1160-16s.059", 0x004000, 0x004000, 0xeef0a003 )
	ROM_LOAD( "1161-16r.059", 0x008000, 0x008000, 0x68c74337 )

	ROM_REGION( 0x04000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "1198-6p.059",  0x000000, 0x004000, 0xf99b631a )

	ROM_REGION( 0xc0000, REGION_GFX2, ROMREGION_DISPOSE | ROMREGION_INVERT )
	ROM_LOAD( "1162-1a.059",  0x000000, 0x008000, 0xdd3833ad )
	ROM_LOAD( "1166-1b.059",  0x008000, 0x008000, 0xe2db50a0 )
	ROM_LOAD( "1170-1c.059",  0x010000, 0x008000, 0xf050ab43 )
	ROM_LOAD( "1174-1d.059",  0x018000, 0x008000, 0xb6704bd1 )
	ROM_LOAD( "1178-1ef.059", 0x020000, 0x008000, 0xd3006f05 )
	ROM_LOAD( "1182-1j.059",  0x028000, 0x008000, 0x9046e985 )

	ROM_LOAD( "1163-1l.059",  0x030000, 0x008000, 0xd505b04a )
	ROM_LOAD( "1167-1mn.059", 0x038000, 0x008000, 0x1869c76d )
	ROM_LOAD( "1171-1p.059",  0x040000, 0x008000, 0x1b229c2b )
	ROM_LOAD( "1175-1r.059",  0x048000, 0x008000, 0x73c41aca )
	ROM_LOAD( "1179-1st.059", 0x050000, 0x008000, 0x9b7cb0ef )
	ROM_LOAD( "1183-1u.059",  0x058000, 0x008000, 0x393bba42 )

	ROM_LOAD( "1164-2a.059",  0x060000, 0x008000, 0x50e76162 )
	ROM_LOAD( "1168-2b.059",  0x068000, 0x008000, 0x35c78469 )
	ROM_LOAD( "1172-2c.059",  0x070000, 0x008000, 0x314ac268 )
	ROM_LOAD( "1176-2d.059",  0x078000, 0x008000, 0x061d79db )
	ROM_LOAD( "1180-2ef.059", 0x080000, 0x008000, 0x89c1fe16 )
	ROM_LOAD( "1184-2j.059",  0x088000, 0x008000, 0x541209d3 )

	ROM_LOAD( "1165-2l.059",  0x090000, 0x008000, 0x9484ba65 )
	ROM_LOAD( "1169-2mn.059", 0x098000, 0x008000, 0x132d3337 )
	ROM_LOAD( "1173-2p.059",  0x0a0000, 0x008000, 0x98de2426 )
	ROM_LOAD( "1177-2r.059",  0x0a8000, 0x008000, 0x9d0824f8 )
	ROM_LOAD( "1181-2st.059", 0x0b0000, 0x008000, 0x9e62b27c )
	ROM_LOAD( "1185-2u.059",  0x0b8000, 0x008000, 0x9d62f6b7 )
ROM_END



/*************************************
 *
 *	Driver initialization
 *
 *************************************/

static void common_init(int slapstic, int vindctr2)
{
	atarigen_eeprom_default = NULL;
	atarigen_slapstic_init(0, 0x038000, slapstic);

	/* swap the top and bottom halves of the main CPU ROM images */
	atarigen_swap_mem(memory_region(REGION_CPU1) + 0x000000, memory_region(REGION_CPU1) + 0x008000, 0x8000);
	atarigen_swap_mem(memory_region(REGION_CPU1) + 0x040000, memory_region(REGION_CPU1) + 0x048000, 0x8000);
	atarigen_swap_mem(memory_region(REGION_CPU1) + 0x050000, memory_region(REGION_CPU1) + 0x058000, 0x8000);
	atarigen_swap_mem(memory_region(REGION_CPU1) + 0x060000, memory_region(REGION_CPU1) + 0x068000, 0x8000);
	atarigen_swap_mem(memory_region(REGION_CPU1) + 0x070000, memory_region(REGION_CPU1) + 0x078000, 0x8000);

	/* indicate whether or not we are vindicators 2 */
	vindctr2_screen_refresh = vindctr2;
}


static DRIVER_INIT( gauntlet )
{
	common_init(104, 0);
}


static DRIVER_INIT( gaunt2p )
{
	common_init(107, 0);
}


static DRIVER_INIT( gauntlet2 )
{
	common_init(106, 0);
}


static DRIVER_INIT( vindctr2 )
{
	UINT8 *gfx2_base = memory_region(REGION_GFX2);
	UINT8 *data = malloc(0x8000);
	int i;

	common_init(118, 1);

	/* install our special ports */
	install_mem_read16_handler(0, 0x803000, 0x803003, vindctr2_port01_r);

	/* highly strange -- the address bits on the chip at 2J (and only that
	   chip) are scrambled -- this is verified on the schematics! */
	if (data)
	{
		memcpy(data, &gfx2_base[0x88000], 0x8000);
		for (i = 0; i < 0x8000; i++)
		{
			int srcoffs = (i & 0x4000) | ((i << 11) & 0x3800) | ((i >> 3) & 0x07ff);
			gfx2_base[0x88000 + i] = data[srcoffs];
		}
		free(data);
	}
}



/*************************************
 *
 *	Game driver(s)
 *
 *************************************/

GAME( 1985, gauntlet, 0,        gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (rev 14)" )
GAME( 1985, gaunts,   gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (Spanish, rev 15)" )
GAME( 1985, gauntj,   gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (Japanese, rev 13)" )
GAME( 1985, gauntg,   gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (German, rev 10)" )
GAME( 1985, gauntj12, gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (Japanese, rev 12)" )
GAME( 1985, gauntr9,  gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (rev 9)" )
GAME( 1985, gauntgr8, gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (German, rev 8)" )
GAME( 1985, gauntr7,  gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (rev 7)" )
GAME( 1985, gauntgr6, gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (German, rev 6)" )
GAME( 1985, gauntr5,  gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (rev 5)" )
GAME( 1985, gauntr4,  gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (rev 4)" )
GAME( 1985, gauntgr3, gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (German, rev 3)" )
GAME( 1985, gauntr2,  gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (rev 2)" )
GAME( 1985, gauntr1,  gauntlet, gauntlet, gauntlet, gauntlet,  ROT0, "Atari Games", "Gauntlet (rev 1)" )

GAME( 1985, gaunt2p,  gauntlet, gauntlet, gauntlet, gaunt2p,   ROT0, "Atari Games", "Gauntlet (2 Players, rev 6)" )
GAME( 1985, gaunt2pj, gauntlet, gauntlet, gauntlet, gaunt2p,   ROT0, "Atari Games", "Gauntlet (2 Players, Japanese, rev 5)" )
GAME( 1985, gaunt2pg, gauntlet, gauntlet, gauntlet, gaunt2p,   ROT0, "Atari Games", "Gauntlet (2 Players, German, rev 4)" )
GAME( 1985, gaun2pr3, gauntlet, gauntlet, gauntlet, gaunt2p,   ROT0, "Atari Games", "Gauntlet (2 Players, rev 3)" )
GAME( 1985, gaun2pj2, gauntlet, gauntlet, gauntlet, gaunt2p,   ROT0, "Atari Games", "Gauntlet (2 Players, Japanese, rev 2)" )
GAME( 1985, gaun2pg1, gauntlet, gauntlet, gauntlet, gaunt2p,   ROT0, "Atari Games", "Gauntlet (2 Players, German, rev 1)" )

GAME( 1986, gaunt2,   0,        gauntlet, gauntlet, gauntlet2, ROT0, "Atari Games", "Gauntlet II" )
GAME( 1986, gaunt2g,  gaunt2,   gauntlet, gauntlet, gauntlet2, ROT0, "Atari Games", "Gauntlet II (German)" )

GAME( 1986, gaunt22p, gaunt2,   gauntlet, gauntlet, gauntlet2, ROT0, "Atari Games", "Gauntlet II (2 Players, rev 2)" )
GAME( 1986, gaun22p1, gaunt2,   gauntlet, gauntlet, gauntlet2, ROT0, "Atari Games", "Gauntlet II (2 Players, rev 1)" )
GAME( 1986, gaun22pg, gaunt2,   gauntlet, gauntlet, gauntlet2, ROT0, "Atari Games", "Gauntlet II (2 Players, German)" )

GAME( 1988, vindctr2, 0,        gauntlet, vindctr2, vindctr2,  ROT0, "Atari Games", "Vindicators Part II (rev 3)" )
GAME( 1988, vindc2r2, vindctr2, gauntlet, vindctr2, vindctr2,  ROT0, "Atari Games", "Vindicators Part II (rev 2)" )
GAME( 1988, vindc2r1, vindctr2, gauntlet, vindctr2, vindctr2,  ROT0, "Atari Games", "Vindicators Part II (rev 1)" )
